/***************************************************************************
*	Brutal Chess Pre-Alpha Build
*	- chessplayer.cpp
*
*	Authors: Maxwell Lazaroff, Michael Cook, and Joe Flint
*	Date Created : May 3th, 2005
*	Last Modified: November 5th, 2005
*
*	- description - Implements the way that players work.  Performs move
*	validity checks.
***************************************************************************/

#include <cmath>
#include <string>
#include <vector>
#include <iostream>
#include <cstdlib>

#include "chessplayer.h"

using namespace std;

bool
ChessPlayer::check_move(const BoardMove & move, const Board &board, bool white )
/**************************************************************************
*	check_move
*
*	- Checks to see whether the move stays within the bounds of the board,
*	whether the move is valid, and whether the player is in check after
*	making the move.
**************************************************************************/
{
	if(!is_in(move, board))
		return false;
	
	// Makes a copy of the board.
	Board boardcopy = board;
	boardcopy.updateBoard(move);

	// Finding the king.
	int kingCol = -1, kingRow = -1;
	bool foundKing = false;
	for( int i = 0; foundKing == false && i < 8; i++ )
		for( int j = 0; foundKing == false && j < 8; j++ ) 
			if( boardcopy._board[i][j].getUnit() == KING
				&& ( ( boardcopy._board[i][j].getColor() == WHITE && white )
				|| ( boardcopy._board[i][j].getColor() == BLACK && !white )))
			{
				kingRow = i;
				kingCol = j;
				foundKing = true;
			}

	// Checking whether the move can be taken or not.
	if (is_valid(move, board, white) && 
		!boardcopy.inDanger(white, kingRow, kingCol))
		return true;

	return false;
}

vector<ChessPlayer*> ChessPlayer::_players;

vector<string>
ChessPlayer::names()
{
    vector<string> ret;
    for (uint i=0; i<_players.size(); i++)
        ret.push_back(_players[i]->_name);
    return ret;
}

void
ChessPlayer::add_player(ChessPlayer* player)
{
    if (player == 0)
        cerr << "NimPlayer::add_player: error: player is null" << endl;
    else if (lookup(player->_name))
    {
        cerr << "NimPlayer::add_player: error: can't add "
        << player->_name << " again." << endl;
    }
    else
    {
        _players.push_back(player);
    }
}

ChessPlayer*
ChessPlayer::lookup(const string& player_name)
{
    for (uint i=0; i<_players.size(); i++)
    {
        if (_players[i]->_name == player_name)
            return _players[i];
    }
    return 0;
}

bool
ChessPlayer::is_in(const BoardMove & move, const Board & board)
{
    if(    move.getStartx() < 8 && move.getStartx() >=0
		&& move.getStarty() < 8 && move.getStarty() >=0
	    && move.getEndx() < 8 && move.getEndy() >= 0
		&& move.getEndy() < 8 && move.getEndy() >= 0   )
		return true;
	else
		return false;
			
}

/***************************************************************************
*	ChessPlayer::is_valid and its helpers
***************************************************************************/

bool
pawnMoveIsValid(const BoardMove &move, const Board &board, bool &is_white)
/***************************************************************************
*	pawnMoveIsValid                       Helper for->ChessPlayer.is_valid()
*	
*	- Checks to see if the move passed in is valid for a pawn to make.  It
*	first checks to see if the pawn hasn't moved and is moving within its
*	initial column. Then checks to see if its trying to move out of its
*	column (diagonally).  Also checks for En Passant.
*
***************************************************************************/
{

	uint startrow = move.getStartx();
	uint startcol = move.getStarty();
	uint endrow   = move.getEndx  ();
	uint endcol   = move.getEndy  ();
	uint strtMvCt = board._board[startrow][startcol].getMovecount();

	// Checking to make sure that the player actually MOVED.
	if (startrow == endrow && startcol == endcol)
		return false;
	
	if(strtMvCt == 0 && endcol == startcol)
	{
		// Condition where pawn hasn't moved yet.
		if(is_white && endrow == 4 && strtMvCt == 0)
		{
			if(   board._board[5][startcol].getUnit() == EMPTY 
			   && board._board[4][startcol].getUnit() == EMPTY)
				return true;
			else
				return false;
		}
		else if (!is_white && endrow == 3 && strtMvCt == 0)
		{
			if(   board._board[2][startcol].getUnit() == EMPTY
			   && board._board[3][startcol].getUnit() == EMPTY)
				return true;
			else
				return false;
		}
		else
		{
			// Normal Move
			if (board._board[endrow][endcol].getUnit() == EMPTY && is_white)
			{
				if (startrow-endrow == 1)
					return true;
				else
					return false;
			}
			else if (board._board[endrow][endcol].getUnit() == EMPTY && !is_white)
			{
				if (endrow-startrow == 1)
					return true;
				else
					return false;
			}
			else
				return false;
		}
	}
	else if(endcol == (startcol-1) || endcol == (startcol+1))
	{
		// Condition where white pawn moves.
		if (is_white)
		{
			// En-Passant.			
			if(   board._board[endrow][endcol].getUnit() == EMPTY
			   && board._board[startrow][endcol].getColor() == BLACK
			   && board._board[startrow][endcol].getUnit() == PAWN
			   && board._board[startrow][endcol].getMovecount() == 1
			   && board._board[startrow][endcol].getJustMoved() == true
			   && startrow == 3 && endrow == 2 )
				return true;
			
			// Otherwise, we see if we are taking a piece.
			if(   board._board[endrow][endcol].getUnit() != EMPTY
			   && board._board[endrow][endcol].getColor() == BLACK
			   && endrow == startrow - 1 )
				return true;
			else
				return false;
		}
		
		// Condition where black pawn moves.
		else if (!is_white){
			// En-Passant.
			if(   board._board[endrow][endcol].getUnit() == EMPTY
			   && board._board[startrow][endcol].getColor() == WHITE
			   && board._board[startrow][endcol].getUnit() == PAWN
			   && board._board[startrow][endcol].getMovecount() == 1
			   && board._board[startrow][endcol].getJustMoved() == true
			   && startrow == 4 && endrow == 5 )
				return true;
			
			// Otherwise, we see if we are taking a piece.
			if(   board._board[endrow][endcol].getUnit() != EMPTY
			   && board._board[endrow][endcol].getColor() == WHITE
			   && endrow == startrow + 1 )
				return true;
			else
				return false;
		}
	}	
	else if( endcol == startcol )
	{
		// Normal Move
		if (board._board[endrow][endcol].getUnit() == EMPTY && is_white)
		{
			if (startrow-endrow == 1)
				return true;
			else
				return false;
		}
		else if (board._board[endrow][endcol].getUnit() == EMPTY && !is_white)
		{
			if (endrow-startrow == 1)
				return true;
			else
				return false;
		}
		else
			return false;
	} else {
		return false;
	}
	
	cerr << "Error in chessplayer.cpp::pawnMoveIsValid()\n";
	return false;
}

bool
rookMoveIsValid(const BoardMove &move, const Board &board, bool &is_white)
/***************************************************************************
*	rookMoveIsValid                       Helper for->ChessPlayer.is_valid()
*	
*	- Checks to make sure that the move input provided by the player is 
*	valid for the rook unit.
***************************************************************************/
{
	// Getting move coordinates.
	uint startrow    = move.getStartx();
	uint startcol    = move.getStarty();
	uint endrow      = move.getEndx  ();
	uint endcol      = move.getEndy  ();
	Color startcolor = board._board[startrow][startcol].getColor();
	Color endcolor   = board._board[endrow][endcol].getColor();
	
	// Finding length of the moves.
	uint rowDiff = abs((int)startrow-(int)endrow);
	uint colDiff = abs((int)startcol-(int)endcol);
	
	// Checking to make sure that the player actually MOVED.
	if (startrow == endrow && startcol == endcol)
		return false;
		
	if (rowDiff == 0)
	{
		// Player only moved one spot, no need to loop.
		if (colDiff == 1)
		{
			if(board._board[endrow][endcol].getUnit() == EMPTY)
				return true;
			if (startcolor == endcolor)
				return false;
			else
				return true;
		}
		
		// Player moved more than one spot, so loop over those in between
		if (startcol > endcol)
		{
			for (uint i = startcol-1; i > endcol; i--)
				if (board._board[startrow][i].getUnit() != EMPTY){return false;}
			if(board._board[endrow][endcol].getUnit() == EMPTY)
				return true;
			if (startcolor != endcolor) { return true; }
			else { return false; }
		}
		else // startcol < endcol
		{
			for (uint i = startcol+1; i < endcol; i++)
				if (board._board[startrow][i].getUnit() != EMPTY){return false;}
			if(board._board[endrow][endcol].getUnit() == EMPTY)
				return true;
			if (startcolor != endcolor) { return true; }
			else { return false; }
		}
	}
	else if (colDiff == 0)
	{
		// Player only moved one spot, no need to loop.
		if (rowDiff == 1)
		{
			if(board._board[endrow][endcol].getUnit() == EMPTY)
				return true;
			if (startcolor == endcolor)
				return false;
			else
				return true;
		}
		
		// Player moved more than one spot, so loop over those in between.
		if (startrow > endrow)
		{
			for (uint i = startrow-1; i > endrow; i--)
				if (board._board[i][startcol].getUnit() != EMPTY){return false;}
			if(board._board[endrow][endcol].getUnit() == EMPTY)
				return true;
			if (startcolor != endcolor) { return true; }
			else { return false; }
		}
		else // startrow < endrow
		{
			for (uint i = startrow+1; i < endrow; i++)
				if (board._board[i][startcol].getUnit() != EMPTY){return false;}
			if(board._board[endrow][endcol].getUnit() == EMPTY)
				return true;
			if (startcolor != endcolor) { return true; }
			else { return false; }
		}
	}
	else // Player tried to move diagonally.
		return false;
	
	cerr << "Error in chessplayer.cpp::rookMoveIsValid()\n";
	return false;
}

bool
knightMoveIsValid(const BoardMove &move, const Board &board, bool &is_white)
/***************************************************************************
*	knightMoveIsValid                     Helper for->ChessPlayer.is_valid()
*	
*	- Checks to make sure that the move input provided by the player is 
*	valid for the knight unit.
***************************************************************************/
{
	// Getting move coordinates.
	uint startrow    = move.getStartx();
	uint startcol    = move.getStarty();
	uint endrow      = move.getEndx  ();
	uint endcol      = move.getEndy  ();
	Color startcolor = board._board[startrow][startcol].getColor();
	Color endcolor   = board._board[endrow][endcol].getColor();
	
	// Checking to make sure that the player actually MOVED.
	if (startrow == endrow && startcol == endcol)
		return false;
	
	uint rowDiff = abs((int)startrow-(int)endrow);
	uint colDiff = abs((int)startcol-(int)endcol);
	
	if (rowDiff == 2)
	{
		if (colDiff == 1 && startcolor != endcolor)
			return true;
		else
			return false;
	} 
	else if (rowDiff == 1)
	{
		if (colDiff == 2 && startcolor != endcolor)
			return true;
		else
			return false;
	}			

	return false;
}

bool
bishopMoveIsValid(const BoardMove &move, const Board &board, bool &is_white)
/***************************************************************************
*	bishopMoveIsValid                     Helper for->ChessPlayer.is_valid()
*	
*	- Checks to make sure that the move input provided by the player is 
*	valid for the bishop unit.
***************************************************************************/
{
	// Getting move coordinates.
	uint startrow    = move.getStartx();
	uint startcol    = move.getStarty();
	uint endrow      = move.getEndx  ();
	uint endcol      = move.getEndy  ();
	Color endcolor   = board._board[endrow][endcol].getColor();
	
	// Checking to make sure that the player actually MOVED.
	if (startrow == endrow || startcol == endcol)
		return false;
	
	// Make sure the bishop isn't trying to take a piece of its own color.
    if (endcolor == WHITE && is_white)
        return false;

	// Still makin sure the bishop isn't trying to take its own color.
    if (endcolor == BLACK && !is_white)
        return false;
	
	// Computing how many rows the bishop moved.
	uint rowDiff = abs((int)startrow-(int)endrow);
	uint colDiff = abs((int)startcol-(int)endcol);
	
	if(rowDiff != colDiff)
		return false;

	// If the bishop moved 'down' the board, check all the spaces 
	// diagonal from it.
	if(endcol > startcol)
	{
		if(endrow > startrow)
		{
			for(uint i = 1; i < rowDiff; i++)
			{
				if(board._board[startrow+i][startcol+i].getUnit() != EMPTY)
					return false;
			}
		}
		else //endrow < startrow
		{
			for(uint i = 1; i < rowDiff; i++)
			{
				if(board._board[startrow-i][startcol+i].getUnit() != EMPTY)
					return false;
			}
		}
	}
	else if (endcol < startcol)
	{
		if(endrow > startrow)
		{
			for(uint i = 1; i < rowDiff; i++)
			{
				if(board._board[startrow+i][startcol-i].getUnit() != EMPTY)
					return false;
			}
		}
		else //endrow < startrow
		{
			for(uint i = 1; i < rowDiff; i++)
			{
				if(board._board[startrow-i][startcol-i].getUnit() != EMPTY)
					return false;
			}
		}
	}

	return true;
}


bool
queenMoveIsValid(const BoardMove &move, const Board &board, bool &is_white)
/***************************************************************************
*	queenMoveIsValid                     Helper for->ChessPlayer.is_valid()
*	
*	- Checks to make sure that the move input provided by the player is 
*	valid for the queen unit.
***************************************************************************/
{
    if(rookMoveIsValid(move, board, is_white)
       || bishopMoveIsValid(move, board, is_white))
        return true;
    else
        return false;
}

bool
kingMoveIsValid(const BoardMove &move, const Board &board, bool &is_white)
/***************************************************************************
*	kingMoveIsValid                     Helper for->ChessPlayer.is_valid()
*	
*	- Checks to make sure that the move input provided by the player is 
*	valid for the king unit.
***************************************************************************/
{
	// Getting move coordinates.
	uint startrow    = move.getStartx();
	uint startcol    = move.getStarty();
	uint endrow      = move.getEndx  ();
	uint endcol      = move.getEndy  ();
	int row = static_cast<int>(startrow);
	int col = static_cast<int>(startcol);
	Color endcolor   = board._board[endrow][endcol].getColor();
	
	// Checking to make sure that the player actually MOVED.
	if (startrow == endrow && startcol == endcol)
		return false;
		
    if (endcolor == WHITE && is_white)
        return false;

    if (endcolor == BLACK && !is_white)
        return false;
        
    if( is_white )
    {
        if(!board.inDanger(is_white, row, col))
        {
            if(board._board[startrow][startcol].getMovecount() == 0
			   && endrow == 7 && endcol == 2
               && board._board[7][0].getMovecount() == 0
               && board._board[7][1].getUnit() == EMPTY
               && board._board[7][2].getUnit() == EMPTY
               && board._board[7][3].getUnit() == EMPTY)
                return true;

            if(board._board[startrow][startcol].getMovecount() == 0
			   && endrow == 7 && endcol == 6
               && board._board[7][7].getMovecount() == 0
               && board._board[7][6].getUnit() == EMPTY
               && board._board[7][5].getUnit() == EMPTY)
                return true;
        }
    }
    else
    {
        if(!board.inDanger(is_white, row, col))
        {
            if(board._board[startrow][startcol].getMovecount() == 0
			   && endrow == 0 && endcol == 2
               && board._board[0][0].getMovecount() == 0
               && board._board[0][1].getUnit() == EMPTY
               && board._board[0][2].getUnit() == EMPTY
			   && board._board[0][3].getUnit() == EMPTY)
                return true;

			if(board._board[startrow][startcol].getMovecount() == 0
			   && endrow == 0 && endcol == 6
               && board._board[0][7].getMovecount() == 0
               && board._board[0][6].getUnit() == EMPTY
               && board._board[0][5].getUnit() == EMPTY)
                return true;
        }
    }

	if(    (startrow == endrow   && startcol == endcol+1)
		|| (startrow == endrow   && startcol == endcol-1)
		|| (startrow == endrow+1 && startcol == endcol+1)
		|| (startrow == endrow+1 && startcol == endcol-1)
		|| (startrow == endrow+1 && startcol == endcol  )
		|| (startrow == endrow-1 && startcol == endcol+1)
		|| (startrow == endrow-1 && startcol == endcol-1)
		|| (startrow == endrow-1 && startcol == endcol  ))
		return true;
	else
		return false;
}

bool
ChessPlayer::is_valid(const BoardMove &move, const Board &board, bool is_white)
/***************************************************************************
*	ChessPlayer::is_valid
*
*	- Checks the type of piece that the player is trying to move, and calls
*	the appropriate helper function.
***************************************************************************/
{
	uint startrow = move.getStartx();
	uint startcol = move.getStarty();
	
    if(board._board[startrow][startcol].getColor() == WHITE && !is_white)
		return false;
	if(board._board[startrow][startcol].getColor() == BLACK && is_white)
		return false;
	
	Board boardcopy = board;
	boardcopy.updateBoard(move);

	// Look for the king, so inDanger can be called
	int kingi = -1, kingj = -1;

	for( int i = 0; i < 8; i++ )
		for( int j = 0; j < 8; j++ ) 
			if( boardcopy._board[i][j].getUnit() == KING
				&& ( ( boardcopy._board[i][j].getColor() == WHITE && is_white )
				|| ( boardcopy._board[i][j].getColor() == BLACK && !is_white )))
			{
				kingi = i;
				kingj = j;
			}

	// Can't have mate without check.
	if( boardcopy.inDanger( is_white, kingi, kingj ) )
		return false;
	
	
	switch(board._board[startrow][startcol].getUnit())
	{

		case EMPTY:
			cout << "There is no piece at row:" << startrow << 
		                            " column: " << startcol << endl;
			break;
		case PAWN:
			return pawnMoveIsValid(move, board, is_white);
			break;
		case KNIGHT:
			return knightMoveIsValid(move, board, is_white);
			break;
		case BISHOP:
            return bishopMoveIsValid(move, board, is_white);
			break;
		case ROOK:
			return rookMoveIsValid(move, board, is_white);
			break;
		case QUEEN:
            return queenMoveIsValid(move, board, is_white);
			break;
		case KING:
            return kingMoveIsValid(move, board, is_white);
			break;
		default:
			return false;
			break;
	}		
	return false;
}

bool
ChessPlayer::inCheckmate(const Board &board, bool is_white)
/***************************************************************************
*	ChessPlayer::inCheckmate()
*
*	- Checks the board to see if 'this' ChessPlayer is in checkmate.
***************************************************************************/
{
	// Look for the king, so inDanger can be called
	int kingi = -1, kingj = -1;

	for( int i = 0; i < 8; i++ )
		for( int j = 0; j < 8; j++ ) 
			if( board._board[i][j].getUnit() == KING
				&& ( ( board._board[i][j].getColor() == WHITE && is_white )
				|| ( board._board[i][j].getColor() == BLACK && !is_white )))
			{
				kingi = i;
				kingj = j;
			}

	// Can't have mate without check.
	if( !board.inDanger( is_white, kingi, kingj ) )
		return false;
	
	// Look for pieces to move
	for( int i = 0; i < 8; i++ )
		for( int j = 0; j < 8; j++ ) {
			if( ( board._board[i][j].getColor() == WHITE && is_white )
				|| ( board._board[i][j].getColor() == BLACK && !is_white ) )
			{
				// Found one, now try to move it
				vector<BoardMove> posmoves;
				bool isking = false;
				// Get the possible moves for each piece
				switch( board._board[i][j].getUnit() ) {
					case PAWN:
						posmoves.push_back( BoardMove( i, j, i+1, j ) );
						posmoves.push_back( BoardMove( i, j, i-1, j ) );
						posmoves.push_back( BoardMove( i, j, i+2, j ) );
						posmoves.push_back( BoardMove( i, j, i-2, j ) );
						posmoves.push_back( BoardMove( i, j, i+1, j+1 ) );
						posmoves.push_back( BoardMove( i, j, i-1, j+1 ) );
						posmoves.push_back( BoardMove( i, j, i+1, j-1 ) );
						posmoves.push_back( BoardMove( i, j, i-1, j-1 ) );
						break;
					case KNIGHT:
						posmoves.push_back( BoardMove( i, j, i+1, j+2 ) );
						posmoves.push_back( BoardMove( i, j, i+1, j-2 ) );
						posmoves.push_back( BoardMove( i, j, i+2, j+1 ) );
						posmoves.push_back( BoardMove( i, j, i+2, j-1 ) );
						posmoves.push_back( BoardMove( i, j, i-1, j+2 ) );
						posmoves.push_back( BoardMove( i, j, i-1, j-2 ) );
						posmoves.push_back( BoardMove( i, j, i-2, j+1 ) );
						posmoves.push_back( BoardMove( i, j, i-2, j-1 ) );
						break;
					case QUEEN:
					case BISHOP:
						for( int a = 0; a < 8; a++ ) {
							posmoves.push_back( BoardMove( i, j, i+a, j+a ) );
							posmoves.push_back( BoardMove( i, j, i-a, j-a ) );
							posmoves.push_back( BoardMove( i, j, i+a, j-a ) );
							posmoves.push_back( BoardMove( i, j, i-a, j+a ) );
						}
						if( board._board[i][j].getUnit() == BISHOP )
							break;
					case ROOK:
						for( int x = 0; x < 8; x++ )
							if( x != i )
								posmoves.push_back( BoardMove( i, j, x, j ) );
						for( int y = 0; y < 8; y++ )
							if( y != j )
								posmoves.push_back( BoardMove( i, j, i, y ) );
						break;
					case KING:
						isking = true;
						posmoves.push_back( BoardMove( i, j, i+1, j ) );
						posmoves.push_back( BoardMove( i, j, i+1, j+1 ) );
						posmoves.push_back( BoardMove( i, j, i, j+1 ) );
						posmoves.push_back( BoardMove( i, j, i-1, j+1 ) );
						posmoves.push_back( BoardMove( i, j, i-1, j ) );
						posmoves.push_back( BoardMove( i, j, i-1, j-1 ) );
						posmoves.push_back( BoardMove( i, j, i, j-1 ) );
						posmoves.push_back( BoardMove( i, j, i+1, j-1 ) );
					default:
						break;
				}
				for( uint k = 0; k < posmoves.size(); k++ ) {
					BoardMove m = posmoves[k];
					//cout << m.getStartx() << " " << m.getStarty();
					//cout << " to " << m.getEndx() << " " << m.getEndy() << endl;
					if( check_move(m, board, is_white) )
						return false;
				}
			}
		}
	return true;
}

bool
ChessPlayer::inStalemate(const Board &board, bool is_white )
/***************************************************************************
*	ChessPlayer::inStalemate()
*
*	- Checks the board to see if a stalemate has occurred.
***************************************************************************/
{	
	// If only kings are on the board, we're stalemated.
    bool onlyKings = true;
	for( int i = 0; i < 8; i++ )
	{
		for( int j = 0; j < 8; j++ )
		{
		  if(board._board[i][j].getUnit() != KING &&
			 board._board[i][j].getUnit() != EMPTY)
		      onlyKings = false;
        }
    }
    if(onlyKings)
        return true;
	
	// Look for pieces to move.
	for( int i = 0; i < 8; i++ )
	{
		for( int j = 0; j < 8; j++ )
        {
			if( board._board[i][j].getColor() != BLANK )
			{   
				// Found one, now try to move it
				vector<BoardMove> posmoves;
				// Get the possible moves for each piece
				switch( board._board[i][j].getUnit() ) {
					case PAWN:
						posmoves.push_back( BoardMove( i, j, i+1, j ) );
						posmoves.push_back( BoardMove( i, j, i-1, j ) );
						posmoves.push_back( BoardMove( i, j, i+2, j ) );
						posmoves.push_back( BoardMove( i, j, i-2, j ) );
						posmoves.push_back( BoardMove( i, j, i+1, j+1 ) );
						posmoves.push_back( BoardMove( i, j, i-1, j+1 ) );
						posmoves.push_back( BoardMove( i, j, i+1, j-1 ) );
						posmoves.push_back( BoardMove( i, j, i-1, j-1 ) );
						break;
					case KNIGHT:
						posmoves.push_back( BoardMove( i, j, i+1, j+2 ) );
						posmoves.push_back( BoardMove( i, j, i+1, j-2 ) );
						posmoves.push_back( BoardMove( i, j, i+2, j+1 ) );
						posmoves.push_back( BoardMove( i, j, i+2, j-1 ) );
						posmoves.push_back( BoardMove( i, j, i-1, j+2 ) );
						posmoves.push_back( BoardMove( i, j, i-1, j-2 ) );
						posmoves.push_back( BoardMove( i, j, i-2, j+1 ) );
						posmoves.push_back( BoardMove( i, j, i-2, j-1 ) );
						break;
					case QUEEN:
					case BISHOP:
						for( int a = 0; a < 8; a++ ) {
							posmoves.push_back( BoardMove( i, j, i+a, j+a ) );
							posmoves.push_back( BoardMove( i, j, i-a, j-a ) );
							posmoves.push_back( BoardMove( i, j, i+a, j-a ) );
							posmoves.push_back( BoardMove( i, j, i-a, j+a ) );
						}
						if( board._board[i][j].getUnit() == BISHOP )
							break;
					case ROOK:
						for( int x = 0; x < 8; x++ )
							if( x != i )
								posmoves.push_back( BoardMove( i, j, x, j ) );
						for( int y = 0; y < 8; y++ )
							if( y != j )
								posmoves.push_back( BoardMove( i, j, i, y ) );
						break;
					case KING:
						posmoves.push_back( BoardMove( i, j, i+1, j ) );
						posmoves.push_back( BoardMove( i, j, i+1, j+1 ) );
						posmoves.push_back( BoardMove( i, j, i, j+1 ) );
						posmoves.push_back( BoardMove( i, j, i-1, j+1 ) );
						posmoves.push_back( BoardMove( i, j, i-1, j ) );
						posmoves.push_back( BoardMove( i, j, i-1, j-1 ) );
						posmoves.push_back( BoardMove( i, j, i, j-1 ) );
						posmoves.push_back( BoardMove( i, j, i+1, j-1 ) );
					default:
						break;
				}

				for( uint k = 0; k < posmoves.size(); k++ )
                {
					BoardMove m = posmoves[k];
					if(check_move(m, board, is_white))
					{
                        return false;
					}
				}
			}
		}
    }
    return true;
}

// End of file chessplayer.cpp
