# Copyright (C) 2005 by Aaron Bentley

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
import errno
import os
import shutil
import sys

from bzrlib.bzrdir import BzrDir
from bzrlib.osutils import has_symlinks, isdir
from bzrlib.workingtree import WorkingTree

def is_detritus(subp):
    return subp.endswith('.THIS') or subp.endswith('.BASE') or\
        subp.endswith('.OTHER') or subp.endswith('~') or subp.endswith('.tmp')

def iter_deletables(tree, unknown=True, ignored=False, detritus=False):
    """Iterate through files that may be deleted"""
    for subp in tree.extras():
        if detritus and is_detritus(subp):
            yield tree.abspath(subp), subp
            continue
        if tree.is_ignored(subp):
            if ignored:
                yield tree.abspath(subp), subp
        else:
            if unknown:
                yield tree.abspath(subp), subp

def clean_tree(directory, ignored=False, detritus=False, dry_run=False,
               to_file=sys.stdout):
    tree = WorkingTree.open_containing(directory)[0]
    deletables = iter_deletables(tree, ignored=ignored, detritus=detritus)
    delete_items(deletables, dry_run=dry_run, to_file=to_file)

def delete_items(deletables, dry_run=False, to_file=sys.stdout):
    """Delete files in the deletables iterable"""
    has_deleted = False
    for path, subp in deletables:
        if not has_deleted:
            print >> to_file, "deleting paths:"
            has_deleted = True
        print >> to_file, ' ', subp
        if not dry_run:
            if isdir(path):
                shutil.rmtree(path)
            else:
                os.unlink(path)
    if not has_deleted:
        print >> to_file, "No files deleted."

def test_suite():
    from bzrlib.tests import TestCaseInTempDir
    from unittest import makeSuite
    import os.path
    from StringIO import StringIO
    class TestCleanTree(TestCaseInTempDir):
        def test_symlinks(self):
            if has_symlinks() is False:
                return
            os.mkdir('branch')
            BzrDir.create_standalone_workingtree('branch')
            os.symlink(os.path.realpath('no-die-please'), 'branch/die-please')
            os.mkdir('no-die-please')
            assert os.path.exists('branch/die-please')
            os.mkdir('no-die-please/child')

            clean_tree('branch', to_file=StringIO())
            assert os.path.exists('no-die-please')
            assert os.path.exists('no-die-please/child')

        def test_iter_deletable(self):
            """Files are selected for deletion appropriately"""
            os.mkdir('branch')
            tree = BzrDir.create_standalone_workingtree('branch')
            file('branch/file.BASE', 'wb').write('contents')
            self.assertEqual(len(list(iter_deletables(tree))), 1)
            file('branch/file~', 'wb').write('contents')
            file('branch/file.pyc', 'wb').write('contents')
            dels = [r for a,r in iter_deletables(tree)]
            assert 'file~' not in dels
            assert 'file.pyc' not in dels
            dels = [r for a,r in iter_deletables(tree, detritus=True)]
            assert 'file~' in dels
            assert 'file.pyc' not in dels
            dels = [r for a,r in iter_deletables(tree, ignored=True)]
            assert 'file~' in dels
            assert 'file.BASE' in dels
            assert 'file.pyc' in dels
            dels = [r for a,r in iter_deletables(tree, unknown=False)]
            assert 'file.BASE' not in dels

    return makeSuite(TestCleanTree)
