/*
        Source: plugins.h 
        Author: Adam G. Bowen (agbowen@bealenet.com) 
        $Revision: 1.9 $
        $Date: 1999/05/17 14:26:35 $


 * Header file for plugins for Netsaint program.
 *
 * $Log: plugins.h,v $
 * Revision 1.9  1999/05/17 14:26:35  netsaint
 * Added #define CHECK_TELNET.
 *
 * Revision 1.8  1999/05/14 02:24:04  netsaint
 * Changed the get_var subroutine to:
 * 	int get_var(char *LINE_NAME, char *file_name, char *var_name)
 * LINE_NAME = values to look for in the file.
 * file_name = the file to search.
 * var_name = pointer to variable in calling program to copy data.
 * Changed the check_network subroutine into 4 new subroutines:
 * 	open_socket
 * 	send_socket
 * 	recv_socket
 * 	close_socket
 * This will allow more felxibility of the socket functions.
 *
 * Revision 1.7  1999/05/07 15:21:52  netsaint
 * Changed the exit_error subroutine from type void to type int.  This
 * subroutine will now return an int value.
 * Added the check_output_file subroutine.  This subroutine takes a pointer to
 * a file name as an argument and checks if the file exists, if the file is
 * readable, if the file is not zero size, and if the file can be opened.  If
 * any of these test fail, the subrotuine returns with a STATE_UNKNOWN error.
 * Added subroutine get_command_name.  This subroutine takes the command_name
 * (argv[0]) as an argument and strips off all leading directories and returns
 * a pointer to the command_name.
 * Added subroutine check_consistency.  This subroutine takes 2 integers as
 * arguments.  The first is warn_value, the second is critical_value.  The
 * subroutine checks that warn_value < critical_value.  If this test fails,
 * the subroutine returns a STATE_UNKNOWN error.
 * Added the following includes:
 *      <ctype.h>
 *      <time.h>
 *
 * Revision 1.6  1999/05/06 15:13:34  netsaint
 * Corrected typos in error messages.
 *
 * Revision 1.5  1999/05/03 14:56:49  netsaint
 * Added the #define TIME_OUT statement.  TIME_OUT is used when setting the
 * alarm prior to attempting a network connection.  Also added the exit_error
 * subroutine.  This is used to print error messages recived while attempting
 * network connections.
 *
 * Revision 1.4  1999/04/29 18:19:25  netsaint
 * Added subroutine alarm_signal, this is ued to trap the SIGALRM signal.
 * Added check_net subroutine, this is used by plugins to make a socket
 * connection to a remote host before running the rsh command.  This should
 * prevent problems with plugins hanging on down remote hosts.  Added the
 * print_error subroutine.  This will print the contents of the error file
 * if the file exists and is larger than zero.
 *
 * Revision 1.3  1999/04/27 17:24:00  netsaint
 * Added #define SUNDAY 0 and subroutine strupr to convert string to upper case.
 *
 * Revision 1.2  1999/04/23 16:05:01  netsaint
 * Added #define statement for CONFIG_DIR.
 *
 * Revision 1.1  1999/04/23 13:17:54  netsaint
 * Initial revision
 *
*/

#include <ctype.h>
#include <errno.h>
#include <netdb.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/timeb.h>
#include <sys/types.h>
#include <sys/wait.h>

/* service state return codes */

#define STATE_CRITICAL 2
#define STATE_WARNING  1
#define STATE_OK       0
#define STATE_UNKNOWN  -1

#define TRUE           1
#define FALSE          0
#define EXISTS         4
#define READABLE       6

#define SATURDAY       6
#define SUNDAY         0

#define TIME_OUT       5

#define MAX_CHARS      256

#define CONFIG_DIR     "/usr/local/netsaint/config/"
#define CHECK_TELNET   "check_telnet"

int get_var(char *LINE_NAME, char *file_name, char *var_name)
{
  char temp_input_buffer[MAX_CHARS];
  char tok_sep[] = "|";
  char temp_var_name[MAX_CHARS];
  char *temp_var_value;

  FILE *temp_fp;

  int return_value = STATE_UNKNOWN;

  strcpy(var_name,"NOT_FOUND");

  temp_fp=fopen(file_name,"r");
  while(fgets(temp_input_buffer,MAX_CHARS-1,temp_fp) != NULL)
  {
         strcpy(temp_var_name,strtok(temp_input_buffer,tok_sep));
         temp_var_value=strtok(NULL,tok_sep);
         if(strcmp(LINE_NAME,temp_var_name) == 0)
         {
                strcpy(var_name,temp_var_value);
                return_value = STATE_OK;
         }
  }
  return return_value;
}
void strupr (char *string_name)
{
  do
  {
         *string_name=toupper(*string_name);
  }
  while(*string_name++);
  return;
}
void alarm_signal(int sig_num)
{
  printf("No response received from host in %d seconds, possible host down or network error.\n",TIME_OUT);
  exit(STATE_CRITICAL);
}
int open_socket(int * socket_name, char *host, int port_num, char *protocol)
{
  int return_value = STATE_OK;

  struct protoent *proto_ptr;
  struct sockaddr_in remote_addr;
  struct hostent *host_addr;

  if(isdigit(host[0]))
  {
    inet_aton(host, &remote_addr.sin_addr);
  }
  else
  {
    if((host_addr = gethostbyname(host)) == NULL)
    {
      switch(errno)
      {
        case HOST_NOT_FOUND:
          return_value = 1;
          break;
        case TRY_AGAIN:
          return_value = 2;
          break;
        case NO_RECOVERY:
          return_value = 3;
          break;
        case NO_ADDRESS:
          return_value = 4;
          break;
        default:
          return_value = 5;
      }
    }
    else
    {
      memcpy(&remote_addr.sin_addr, host_addr->h_addr, host_addr->h_length);
    }
  }

  remote_addr.sin_family=AF_INET;
  remote_addr.sin_port=htons(port_num);

  if((proto_ptr = getprotobyname(protocol)) == NULL)
  {
    return_value = 6;
  }
  if((*socket_name=socket(AF_INET,SOCK_STREAM,proto_ptr->p_proto)) < 0)
  {
    return_value = 7;
  }
  if((connect(*socket_name,(struct sockaddr *)&remote_addr,sizeof(remote_addr))) < 0)
  {
    switch(errno)
    {
      case ECONNREFUSED:
        return_value = 8;
        break;
      case ETIMEDOUT:
        return_value = 9;
        break;
      case ENETUNREACH:
        return_value = 10;
        break;
      default:
        return_value = 11;
        break;
    }
    close(*socket_name);
  }
  return return_value;
}
int recv_socket(int *socket_name, char *expected)
{
  char input_buffer[MAX_CHARS];

  int return_value = STATE_OK;
  if((recv(*socket_name,&input_buffer,MAX_CHARS-1,0)) < 0)
  {
    return_value = 12;
  }
  if(strstr(input_buffer,expected) == NULL)
  {
    return_value = 13;
  }
  return return_value;
}
int send_socket(int *socket_name, char *send_buffer)
{
  int return_value = STATE_OK;
  if (strcmp(send_buffer,"\n") == 0)
  {
    if((send(*socket_name,"\n",strlen("\n"),0)) <0)
    {
      return_value = 14;
    }
  }
  else
  {
    if((send(*socket_name,&send_buffer,strlen(send_buffer),0)) <0)
    {
      return_value = 14;
    }
  }
  return return_value;
}
int close_socket(int *socket_name)
{
  close(*socket_name);
  return STATE_OK;
}
void print_error(char *file_name)
{
  char input_buffer[MAX_CHARS];

  FILE *temp_fp;
  
  struct stat file_stat;

  if (access(file_name, READABLE) == 0 )
  {
    if(stat(file_name, &file_stat) == 0)
    {
      if(file_stat.st_size != 0)
      {
        if((temp_fp=fopen(file_name,"r")) != NULL)
        {
          printf("The following errors were produced:  ");
          while(fgets(input_buffer,MAX_CHARS-1,temp_fp) != NULL)
          {
            printf("%s",input_buffer);
          }
          fclose(temp_fp);
        }
      }
    }
  }
  printf("\n");
  return;
}
int exit_error(int error_code, char *ip_address, char *protocol, int port)
{

  int return_value;

  switch(error_code)
  {
    case 1:
      printf("Hostname %s not found\n",ip_address);
      return_value=STATE_WARNING;
      break;
    case 2:
      printf("Hostname %s not found, but try again\n",ip_address);
      return_value=STATE_WARNING;
      break;
    case 3:
      printf("DNS error looking up hostname %s\n",ip_address);
      return_value=STATE_WARNING;
      break;
    case 4:
      printf("No ip address for host %s\n",ip_address);
      return_value=STATE_WARNING;
      break;
    case 5:
      printf("Unknow host name lookup error\n");
      return_value=STATE_WARNING;
      break;
    case 6:
      printf("Unable to map protocol %s\n",protocol);
      return_value=STATE_WARNING;
      break;
    case 7:
      printf("Unable to create socket to host %s on port %d\n",ip_address,port);
      return_value=STATE_CRITICAL;
      break;
    case 8:
      printf("Connection refused by %s on port %d\n",ip_address,port);
      return_value=STATE_CRITICAL;
      break;
    case 9:
      printf("Timed out while creating socket to host %s on port %d\n",ip_address,port);
      return_value=STATE_CRITICAL;
      break;
    case 10:
      printf("Network unreachable for host %s\n",ip_address);
      return_value=STATE_CRITICAL;
      break;
    case 11:
      printf("Unknown connect error to %s on port %d\n",ip_address,port);
      return_value=STATE_CRITICAL;
      break;
    case 12:
      printf("Could not read form port %s on host %d\n",ip_address,port);
      return_value=STATE_CRITICAL;
      break;
    case 13:
      printf("Invalid response string received from host %s on port %d\n",ip_address,port);
      return_value=STATE_WARNING;
      break;
    case 14:
      printf("Error occured while sending data to host %s on port %d\n",ip_address,port);
      return_value=STATE_WARNING;
      break;
    default:
      printf("Unknown error received from host %s on port %d\n",ip_address,port);
      return_value=STATE_UNKNOWN;
  }
  return return_value;
}
int check_output_file(char *file_name)
{

  FILE *temp_fp;

  int return_value;

  struct stat file_stat;

  if (access(file_name, READABLE) != 0 )
  {
    printf("Unable to read output file for system.  ");
    return_value = STATE_UNKNOWN;
  }
  else if(stat(file_name, &file_stat) != 0)
  {
    printf("Unable to retrive status of output file %s.  ",file_name);
    return_value = STATE_UNKNOWN;
  }
  else if(file_stat.st_size == 0)
  {
    printf("Output file %s has zero size.  ",file_name);
    return_value = STATE_UNKNOWN;
  }
  else if((temp_fp=fopen(file_name,"r")) == NULL)
  {
    printf("Unable to open output file for system.  ");
    return_value = STATE_UNKNOWN;
  }
  else
  {
    fclose(temp_fp);
    return_value = STATE_OK;
  }
  return return_value;
}
char *get_command_name(char *command_line)
{
  char *return_string;
  char *sub_string;
  if((sub_string=strrchr(command_line,'/')) != NULL)
  {
    return_string=++sub_string;
  }
  else
  {
    return_string=command_line;
  }
  return return_string;
}
int check_consistency(float warn_value, float critical_value)
{

  int return_value;

  if(warn_value >= critical_value)
  {
    printf("Inconsistency in parameters: warning value %.2f >= critical value %.2f.\n", warn_value, critical_value);
    return_value = STATE_UNKNOWN;
  }
  else
  {
    return_value = STATE_OK;
  }
  return return_value;
}
