/** @file
 * Common GUI Library - Darwin Cursor routines.
 *
 * @todo Move this up somewhere so that the two SDL GUIs can use this code too.
 */

/*
 * Copyright (C) 2006-2007 Sun Microsystems, Inc.
 *
 * This file is part of VirtualBox Open Source Edition (OSE), as
 * available from http://www.virtualbox.org. This file is free software;
 * you can redistribute it and/or modify it under the terms of the GNU
 * General Public License (GPL) as published by the Free Software
 * Foundation, in version 2 as it comes in the "COPYING" file of the
 * VirtualBox OSE distribution. VirtualBox OSE is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY of any kind.
 *
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa
 * Clara, CA 95054 USA or visit http://www.sun.com if you need
 * additional information or have any questions.
 */


/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#include "DarwinCursor.h"
#include <Carbon/Carbon.h>
#include <QuickTime/ImageCompression.h>

#include <VBox/err.h>
#include <iprt/assert.h>
#include <iprt/process.h>
#include <iprt/string.h>
#include <iprt/stream.h>
#if 1
# include <iprt/asm.h>
# include <iprt/alloca.h>
#endif


/*******************************************************************************
*   Internal Functions                                                         *
*******************************************************************************/
//static void darwinCursorDumpFormatInfo(uint32_t Format);
#if 0
static void darwinCursorDump(unsigned cx, unsigned cy, unsigned xHot, unsigned yHot, bool fHasAlpha,
                             const void *pvAndMask, const void *pvShape);
#endif


/**
 * Creates a cursor.
 *
 * @returns VBox status code.
 * @param   cx          The cursor width.
 * @param   cy          The cursor height.
 * @param   xHot        The x-coordinate of the hot spot.
 * @param   yHot        The y-coordinate of the hot spot.
 * @param   fHasAlpha   Set if the image has an alpha channel.
 * @param   pvAndMask   The 1-bit and mask.
 * @param   pvShape     The 32-bit shape image.
 * @param   pCursor     Where to store the cursor handle.
 */
int DarwinCursorCreate(unsigned cx, unsigned cy, unsigned xHot, unsigned yHot, bool fHasAlpha,
                       const void *pvAndMask, const void *pvShape, PDARWINCURSOR pCursor)
{
#if 0
    darwinCursorDump(cx, cy, xHot, yHot, fHasAlpha, pvAndMask, pvShape);
#endif

    /*
     * Minimal input validation and output init.
     */
    AssertPtr(pCursor);
    Assert(pCursor->szName[0] == '\0' || strncmp(pCursor->szName, "VBoxCursor", sizeof("VBoxCursor") - 1));
    pCursor->szName[0] = '\0';


    /*
     * The cursor needs to be one of these size 16x16, 32x32, 48x48 or 64x64,
     * QuickTime could do this for us but it's better if we just take care of it.
     */
    const unsigned cxDst = RT_ALIGN(cx, 16);
    const unsigned cyDst = RT_ALIGN(cy, 16);
    Rect rct;
    SetRect(&rct, 0, 0, cxDst, cyDst);

    unsigned y, x;
    int rc = 0;

    /*
     * The shape data.
     * We must convert it to big-endian because QuickTime seems to ignore whatever I set
     */
    const unsigned cbDstDataLine = cxDst * 4;
    const unsigned cbDstData = cbDstDataLine * cyDst;

    uint32_t *pu32DstData = (uint32_t *)alloca(cbDstData);

    /* convert the data from little-endian ARGB to big-endian ARGB and do the size adjustment. */
    const uint32_t *pu32Src = (const uint32_t *)pvShape;
    uint32_t *pu32Dst = pu32DstData;
    for (y = 0;
         y < cy;
         y++, pu32Dst += cxDst, pu32Src += cx)
    {
        for (x = 0; x < cx; x++)
            pu32Dst[x] = RT_BE2H_U32(pu32Src[x]);
        memset(pu32Dst + cx, 0, (cxDst - cx) * 4);
    }
    memset(pu32Dst, 0, (cyDst - cy) * cbDstDataLine);


    /*
     * The AND mask.
     */
    PixMapHandle hMask = NULL;
    GWorldPtr    pGWMask = NULL;
    if (!fHasAlpha)
    {
#if 1
        /* Convert the AND mask to the alpha channel. */
        pu32Dst = pu32DstData;
        unsigned cbSrcMaskLine = RT_ALIGN(cx, 8) / 8;
        const uint8_t *pbSrcMask = (const uint8_t *)pvAndMask;
        for (y = 0;
             y < cy;
             y++, pbSrcMask += cbSrcMaskLine, pu32Dst += cxDst)
        {
            for (x = 0; x < cx; x++)
            {
                if (!(pbSrcMask[x / 8] & (1 << (7 - (x % 8)))))
                    pu32Dst[x] |= 0x000000ff;
                else
                {
                    /* this isn't quite right, but it's the best we can do I think... */
                    if (pu32Dst[x] & 0xffffff00)
                        pu32Dst[x] = 0x000000ff;
                    else
                        pu32Dst[x] = 0x00000000;
                }
            }
        }

#else
        /* Convert the and + xor mask to an 8-bit Mac mask - doesn't get inversion working :-/ */
        unsigned cbSrcMaskLine = RT_ALIGN(cx, 8) / 8;
        unsigned cbDstMaskLine = RT_ALIGN(cxDst, 4);
        uint8_t *pbMacMask = (uint8_t *)alloca(cbDstMaskLine * cyDst);

        pu32Dst = pu32DstData;
        const uint8_t *pbSrcMask = (const uint8_t *)pvAndMask;
        uint8_t *pbDstMask = pbMacMask;
        for (y = 0;
             y < cy;
             y++, pbDstMask += cbDstMaskLine, pbSrcMask += cbSrcMaskLine, pu32Dst += cxDst)
        {
            for (x = 0; x < cx; x++)
            {
                if (!(pbSrcMask[x / 8] & (1 << (7 - (x % 8)))))
                    pbDstMask[x] = 0xff;
                else
                    pbDstMask[x] = 0x00;
            }
            memset(pbDstMask + cx, 0, cxDst - cx);
        }
        memset(pbDstMask, 0, (cyDst - cy) * cbDstMaskLine);

        /* create the pixmap */
        rc = NewGWorldFromPtr(&pGWMask, k8IndexedGrayPixelFormat, &rct, 0, 0, 0 /*alignPix | stretchPix | newDepth | kNativeEndianPixMap*/,
                              (char *)pbMacMask, cbDstMaskLine);
        AssertMsg(!rc, ("%d (%#x)\n", rc, rc));
        if (!rc)
            hMask = GetGWorldPixMap(pGWMask);
        else
            pGWMask = NULL;
#endif
    }

    /*
     * Create the data pixmap.
     */
    PixMapHandle hData = NULL;
    GWorldPtr pGWData = NULL;
    if (!rc)
    {
        Assert(QTGetPixelSize(k32ARGBPixelFormat) == 32);
        rc = NewGWorldFromPtr(&pGWData, k32ARGBPixelFormat, &rct, 0, 0, 0 /*alignPix | stretchPix | newDepth | kNativeEndianPixMap*/,
                              (char *)pu32DstData, cbDstDataLine);

        AssertMsg(!rc, ("%d (%#x)\n", rc, rc));
        if (!rc)
            hData = GetGWorldPixMap(pGWData);
        else
            pGWData = NULL;
        Assert(hData);
    }

    /*
     * Create the cursor
     */
    if (!rc)
    {
        static uint64_t s_seqNumber = 0;
        RTStrPrintf(pCursor->szName, sizeof(pCursor->szName),
                    "VBoxCursor-%RTproc-%RX64", RTProcSelf(), ++s_seqNumber);

        Point HotPt;
        HotPt.h = xHot;
        HotPt.v = yHot;
        rc = QDRegisterNamedPixMapCursor(hData, hMask, HotPt, pCursor->szName);
        AssertMsg(!rc, ("%d (%#x)\n", rc, rc));
        if (rc)
            pCursor->szName[0] = '\0';
    }

    /* cleanup */
    if (pGWData)
        DisposeGWorld(pGWData);
    if (pGWMask)
        DisposeGWorld(pGWMask);

    return !rc ? VINF_SUCCESS : VERR_GENERAL_FAILURE;
}


/**
 * Destroys a cursor created by DarwinCursorCreate.
 *
 * @returns VBox status code.
 * @param   pCursor     The cursor to destroy.
 */
int DarwinCursorDestroy(PDARWINCURSOR pCursor)
{
    int rc = VINF_SUCCESS;
    if (    pCursor
        &&  pCursor->szName[0])
    {
        rc = QDUnregisterNamedPixMapCursor(pCursor->szName);
        AssertMsg(!rc, ("%d (%#x)\n", rc, rc));
        pCursor->szName[0] = '\0';
    }
    return rc;
}


/**
 * Sets the mouse cursor shape.
 *
 * @returns VBox status code.
 * @param   pCursor     The new cursor shape.
 */
int DarwinCursorSet(PDARWINCURSOR pCursor)
{
    int rc = QDSetNamedPixMapCursor(pCursor->szName);
    AssertMsg(!rc, ("%d (%#x)\n", rc, rc));
    if (!rc)
        return VINF_SUCCESS;
    return VERR_GENERAL_FAILURE;
}


int DarwinCursorHide(void);
int DarwinCursorShow(void);

/**
 * Set the cursor handle to Null (for init).
 *
 * @param   pCursor     The cursor handle to set.
 */
void DarwinCursorClearHandle(PDARWINCURSOR pCursor)
{
    pCursor->szName[0] = '\0';
}


/**
 * Checks if the cursor handle is Null or not.
 *
 * @returns true if Null, otherwise false.
 * @param   pCursor     The cursor handle to test.
 */
bool DarwinCursorIsNull(PDARWINCURSOR pCursor)
{
    return pCursor->szName[0] == '\0';
}


#if 0
/**
 * Dumps the QuickTime format info for a given fourcc image format.
 *
 * @param   Format      The format;
 */
static void darwinCursorDumpFormatInfo(uint32_t Format)
{
    char szFmt[5] = { RT_BYTE4(Format), RT_BYTE3(Format), RT_BYTE2(Format), RT_BYTE1(Format), 0 };
    RTPrintf("Format='%.4s' (%#x)\n", szFmt, Format);

    ICMPixelFormatInfo info;
    memset(&info, 0, sizeof(info));
    info.size = sizeof(info);
    OSErr rc = ICMGetPixelFormatInfo(Format, &info);
    if (!rc)
    {
        RTPrintf("size=%d\n", info.size);
        RTPrintf("formatFlags=%lx", info.formatFlags);
        if (info.formatFlags & kICMPixelFormatIsPlanarMask)
            RTPrintf(" plans=%ld", info.formatFlags & kICMPixelFormatIsPlanarMask);
        if (info.formatFlags & kICMPixelFormatIsIndexed)        RTPrintf(" kICMPixelFormatIsIndexed");
        if (info.formatFlags & kICMPixelFormatIsSupportedByQD)  RTPrintf(" kICMPixelFormatIsSupportedByQD");
        if (info.formatFlags & kICMPixelFormatIsMonochrome)     RTPrintf(" kICMPixelFormatIsMonochrome");
        if (info.formatFlags & kICMPixelFormatHasAlphaChannel)  RTPrintf(" kICMPixelFormatHasAlphaChannel");
        RTPrintf("\n");
        if (info.formatFlags & kICMPixelFormatIsPlanarMask)
            for (unsigned i = 0; i < (info.formatFlags & kICMPixelFormatIsPlanarMask); i++)
                RTPrintf("plane #%d: bitsPerPixel=%d horizontalSubsampling=%d verticalSubsampling=%d\n",
                         i, info.bitsPerPixel[i], info.horizontalSubsampling[i], info.verticalSubsampling[i]);
        else
            RTPrintf("bitsPerPixel=%d\n"
                     "horizontalSubsampling=%d\n"
                     "verticalSubsampling=%d\n",
                     info.bitsPerPixel[0],
                     info.horizontalSubsampling[0],
                     info.verticalSubsampling[0]);
        RTPrintf("defaultGammaLevel=%d\n", info.defaultGammaLevel);
        RTPrintf("cmpCount=%d\n", info.cmpCount);
        RTPrintf("cmpSize=%d\n", info.cmpSize);
    }
    else
        RTPrintf("ICMGetPixelFormatInfo->%d (%#x)\n", rc, rc);
}
#endif


#if 0
static void darwinCursorDump(unsigned cx, unsigned cy, unsigned xHot, unsigned yHot, bool fHasAlpha,
                             const void *pvAndMask, const void *pvShape)
{
    RTPrintf("int MyCursor(PDARWINCURSOR pCursor)\n"
             "{\n");


    unsigned cbMaskLine = RT_ALIGN(cx, 8) / 8;
    RTPrintf("    static const uint8_t s_abMask[%d] =\n"
             "    {\n",
             cbMaskLine * cy);
    const uint8_t *pbMask = (const uint8_t *)pvAndMask;
    for (unsigned y = 0; y < cy; y++, pbMask += cbMaskLine)
    {
        RTPrintf("       ");
        for (unsigned x = 0; x < cx; x += 8)
            RTPrintf(" 0x%02x,", pbMask[x / 8]);
        RTPrintf("   /*");

        for (unsigned x = 0; x < cx; x++)
            RTPrintf(" %c", pbMask[x / 8] & (1 << (7 - (x % 8))) ? 'x' : ' ');
        RTPrintf(" */\n");
    }
    RTPrintf("    };\n");


    unsigned cbShapeLine = cx * sizeof(uint32_t);
    RTPrintf("    const uint32_t s_au32Shape[%d] =\n"
             "    {\n",
             cbShapeLine / sizeof(uint32_t) * cy);
    const uint32_t *pu32Src = (const uint32_t *)pvShape;
    for (unsigned y = 0; y < cy; y++, pu32Src += cbShapeLine / sizeof(*pu32Src), pbMask += cbMaskLine)
    {
        RTPrintf("       ");
        for (unsigned x = 0; x < cx; x++)
            RTPrintf(" 0x%08x,", pu32Src[x]);
        RTPrintf("   /*");

        for (unsigned x = 0; x < cx; x++)
        {
            if (fHasAlpha)
                RTPrintf(" %02x", pu32Src[x] >> 24);
            else
                RTPrintf(" %c", pbMask[x / 8] & (1 << (7 - (x % 8)))
                                ? ' '
                                : (pu32Src[x] & 0xffffff) == 0
                                ? 'Y'
                                : 'X');

        }
        RTPrintf(" */\n");
    }
    RTPrintf("    };\n");


    RTPrintf("    return DarwinCursorCreate(%d, %d, %d, %d, %RTbool, s_abMask, s_au32Shape, pCursor);\n"
             "}\n\n",
             cx, cy, xHot, yHot, fHasAlpha);
}
#endif


/*
                bool locked = LockPixels (crsrMask);
                uint8_t *dstLine = (uint8_t *)GetPixBaseAddr (crsrMask);
                unsigned short rowBytes = GetPixRowBytes (crsrMask);
                for (unsigned y = 0; y < me->height(); y++, dstLine += rowBytes)
                {

                }
                UnlockPixels (crsrMask);

*/



#ifdef TESTING

#include <iprt/thread.h>
#include <iprt/time.h>

int MyCursor0(PDARWINCURSOR pCursor)
{
    static const uint8_t s_abMask[128] =
    {
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
    };
    const uint32_t s_au32Shape[1024] =
    {
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y X X X Y X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y X X X Y X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
    };
    return DarwinCursorCreate(32, 32, 10, 10, false, s_abMask, s_au32Shape, pCursor);
}

int MyCursor1(PDARWINCURSOR pCursor)
{
    static const uint8_t s_abMask[234] =
    {
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xfb, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x   x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf9, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x     x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x       x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x7f, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x         x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x3f, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x           x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x1f, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x             x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x0f, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x               x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x07, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x                 x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x03, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x                   x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x01, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x                     x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x00, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x                       x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x00, 0x7f, 0xff, 0xff,   /* x x x x x x x x x x x x x                         x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x07, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x                 x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x07, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x                 x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0xc3, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x       x x         x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf9, 0xc3, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x     x x x         x x x x x x x x x x x x x x x x x x x */
        0xff, 0xfb, 0xe1, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x   x x x x x         x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xe1, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x         x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xf0, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x         x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xf0, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x         x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xf9, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x     x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
    };
    const uint32_t s_au32Shape[1599] =
    {
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0x02000000, 0x03000000, 0x02000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 02 03 02 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0x03000000, 0x07000000, 0x0b000000, 0x09000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff 03 07 0b 09 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xff000000, 0x0f000000, 0x19000000, 0x17000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff 0f 19 17 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xff000000, 0x27000000, 0x29000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff 27 29 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x39000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff 39 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0x42000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff 42 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0x42000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff 42 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0x42000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff 42 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0x42000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff 42 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0x42000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff ff 42 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0x42000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff ff ff 42 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0x42000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff ff ff ff 42 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0x3a000000, 0x29000000, 0x17000000, 0x09000000, 0x02000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff ff ff ff ff 3a 29 17 09 02 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x51000000, 0x4e000000, 0x48000000, 0x3e000000, 0x34000000, 0x27000000, 0x19000000, 0x0b000000, 0x03000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff 51 4e 48 3e 34 27 19 0b 03 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x4e000000, 0x4b000000, 0x41000000, 0x30000000, 0x22000000, 0x17000000, 0x0f000000, 0x07000000, 0x02000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff 4e 4b 41 30 22 17 0f 07 02 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xff000000, 0x35000000, 0x43000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x49000000, 0x3f000000, 0x2a000000, 0x17000000, 0x0b000000, 0x06000000, 0x03000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff 35 43 ff ff ff ff 49 3f 2a 17 0b 06 03 01 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xff000000, 0x1a000000, 0x31000000, 0x39000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x46000000, 0x42000000, 0x2f000000, 0x18000000, 0x08000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff 1a 31 39 ff ff ff ff 46 42 2f 18 08 01 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0x08000000, 0x16000000, 0x27000000, 0x29000000, 0x20000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x46000000, 0x39000000, 0x22000000, 0x0e000000, 0x03000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff 08 16 27 29 20 ff ff ff ff 46 39 22 0e 03 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x06000000, 0x0f000000, 0x19000000, 0x17000000, 0x0f000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x46000000, 0x42000000, 0x2f000000, 0x18000000, 0x08000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 06 0f 19 17 0f ff ff ff ff 46 42 2f 18 08 01 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x03000000, 0x07000000, 0x0b000000, 0x09000000, 0x05000000, 0x02000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x46000000, 0x39000000, 0x22000000, 0x0e000000, 0x03000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 03 07 0b 09 05 02 ff ff ff ff 46 39 22 0e 03 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0x02000000, 0x03000000, 0x02000000, 0x01000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x45000000, 0x3f000000, 0x2a000000, 0x13000000, 0x05000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 02 03 02 01 00 ff ff ff ff 45 3f 2a 13 05 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0xff000000, 0xff000000, 0x2c000000, 0x3a000000, 0x38000000, 0x27000000, 0x12000000, 0x05000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 ff ff 2c 3a 38 27 12 05 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x03000000, 0x0c000000, 0x1b000000, 0x27000000, 0x27000000, 0x1b000000, 0x0c000000, 0x03000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 03 0c 1b 27 27 1b 0c 03 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0x05000000, 0x0c000000, 0x12000000, 0x12000000, 0x0c000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 05 0c 12 12 0c 05 01 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0x03000000, 0x05000000, 0x05000000, 0x03000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 03 05 05 03 01 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
    };
    return DarwinCursorCreate(41, 39, 13, 13, true , s_abMask, s_au32Shape, pCursor);
}

int MyCursor2(PDARWINCURSOR pCursor)
{
    static const uint8_t s_abMask[234] =
    {
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xfb, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x   x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf1, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x       x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xe0, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x           x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xc0, 0x7f, 0xff, 0xff, 0xff,   /* x x x x x x x x x x               x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0x80, 0x3f, 0xff, 0xff, 0xff,   /* x x x x x x x x x                   x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0x80, 0x3f, 0xff, 0xff, 0xff,   /* x x x x x x x x x                   x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf1, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x       x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf1, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x       x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf1, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x       x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf1, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x       x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf1, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x       x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf1, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x       x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf1, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x       x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf1, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x       x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf1, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x       x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0x80, 0x3f, 0xff, 0xff, 0xff,   /* x x x x x x x x x                   x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0x80, 0x3f, 0xff, 0xff, 0xff,   /* x x x x x x x x x                   x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xc0, 0x7f, 0xff, 0xff, 0xff,   /* x x x x x x x x x x               x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xe0, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x           x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf1, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x       x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xfb, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x   x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
    };
    const uint32_t s_au32Shape[1599] =
    {
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0x02000000, 0x03000000, 0x02000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 02 03 02 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0x05000000, 0x0a000000, 0x0d000000, 0x0a000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff 05 0a 0d 0a 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xff000000, 0xffffffff, 0x1a000000, 0x20000000, 0x1a000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff 1a 20 1a 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xff000000, 0xff000000, 0xff000000, 0xffffffff, 0x36000000, 0x2f000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff 36 2f 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xffffffff, 0x41000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff 41 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xffffffff, 0x42000000, 0x31000000, 0x1d000000, 0x0c000000, 0x03000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff ff 42 31 1d 0c 03 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0x46000000, 0x38000000, 0x25000000, 0x11000000, 0x05000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff ff 46 38 25 11 05 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x05000000, 0x10000000, 0xffffffff, 0xff000000, 0xffffffff, 0x48000000, 0x4b000000, 0x48000000, 0x3e000000, 0x31000000, 0x21000000, 0x10000000, 0x05000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 05 10 ff ff ff 48 4b 48 3e 31 21 10 05 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x03000000, 0x09000000, 0xffffffff, 0xff000000, 0xffffffff, 0x3f000000, 0x45000000, 0x3f000000, 0x2d000000, 0x1e000000, 0x12000000, 0x09000000, 0x03000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 03 09 ff ff ff 3f 45 3f 2d 1e 12 09 03 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0x03000000, 0xffffffff, 0xff000000, 0xffffffff, 0x39000000, 0x41000000, 0x39000000, 0x21000000, 0x10000000, 0x06000000, 0x03000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 01 03 ff ff ff 39 41 39 21 10 06 03 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xff000000, 0xffffffff, 0x36000000, 0x3f000000, 0x36000000, 0x1b000000, 0x09000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff 36 3f 36 1b 09 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xff000000, 0xffffffff, 0x36000000, 0x3f000000, 0x36000000, 0x1b000000, 0x09000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff 36 3f 36 1b 09 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xff000000, 0xffffffff, 0x36000000, 0x3f000000, 0x36000000, 0x1b000000, 0x09000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff 36 3f 36 1b 09 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xff000000, 0xffffffff, 0x36000000, 0x3f000000, 0x36000000, 0x1b000000, 0x09000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff 36 3f 36 1b 09 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xff000000, 0xffffffff, 0x36000000, 0x3f000000, 0x36000000, 0x1b000000, 0x09000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff 36 3f 36 1b 09 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0x03000000, 0xffffffff, 0xff000000, 0xffffffff, 0x39000000, 0x41000000, 0x39000000, 0x21000000, 0x10000000, 0x06000000, 0x03000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 01 03 ff ff ff 39 41 39 21 10 06 03 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0x2d000000, 0x1e000000, 0x12000000, 0x09000000, 0x03000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff ff 2d 1e 12 09 03 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xffffffff, 0x3e000000, 0x31000000, 0x21000000, 0x10000000, 0x05000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff ff 3e 31 21 10 05 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xffffffff, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xffffffff, 0x4d000000, 0x46000000, 0x38000000, 0x25000000, 0x11000000, 0x05000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff 4d 46 38 25 11 05 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x03000000, 0xffffffff, 0xff000000, 0xff000000, 0xff000000, 0xffffffff, 0x4f000000, 0x4c000000, 0x42000000, 0x31000000, 0x1d000000, 0x0c000000, 0x03000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 03 ff ff ff ff ff 4f 4c 42 31 1d 0c 03 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0x05000000, 0xffffffff, 0xff000000, 0xffffffff, 0x41000000, 0x47000000, 0x41000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 01 05 ff ff ff 41 47 41 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0x05000000, 0xffffffff, 0x1f000000, 0x2f000000, 0x36000000, 0x2f000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 01 05 ff 1f 2f 36 2f 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0x05000000, 0x0f000000, 0x1a000000, 0x20000000, 0x1a000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 01 05 0f 1a 20 1a 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0x05000000, 0x0a000000, 0x0d000000, 0x0a000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 01 05 0a 0d 0a 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0x02000000, 0x03000000, 0x02000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 02 03 02 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
    };
    return DarwinCursorCreate(41, 39, 13, 13, true , s_abMask, s_au32Shape, pCursor);
}

int MyCursor3(PDARWINCURSOR pCursor)
{
    static const uint8_t s_abMask[234] =
    {
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xfb, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x   x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf9, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x     x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x       x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x7f, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x         x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x3f, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x           x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x1f, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x             x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x0f, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x               x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x07, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x                 x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x03, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x                   x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x01, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x                     x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x00, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x                       x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x00, 0x7f, 0xff, 0xff,   /* x x x x x x x x x x x x x                         x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x07, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x                 x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0x07, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x                 x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf8, 0xc3, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x       x x         x x x x x x x x x x x x x x x x x x x */
        0xff, 0xf9, 0xc3, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x     x x x         x x x x x x x x x x x x x x x x x x x */
        0xff, 0xfb, 0xe1, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x   x x x x x         x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xe1, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x         x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xf0, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x         x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xf0, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x         x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xf9, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x     x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
    };
    const uint32_t s_au32Shape[1599] =
    {
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0x02000000, 0x03000000, 0x02000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 02 03 02 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0x03000000, 0x07000000, 0x0b000000, 0x09000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff 03 07 0b 09 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xff000000, 0x0f000000, 0x19000000, 0x17000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff 0f 19 17 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xff000000, 0x27000000, 0x29000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff 27 29 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x39000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff 39 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0x42000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff 42 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0x42000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff 42 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0x42000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff 42 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0x42000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff 42 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0x42000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff ff 42 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0x42000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff ff ff 42 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0x42000000, 0x32000000, 0x1f000000, 0x0f000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff ff ff ff 42 32 1f 0f 05 01 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0xff000000, 0x3a000000, 0x29000000, 0x17000000, 0x09000000, 0x02000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff ff ff ff ff 3a 29 17 09 02 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xffffffff, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x51000000, 0x4e000000, 0x48000000, 0x3e000000, 0x34000000, 0x27000000, 0x19000000, 0x0b000000, 0x03000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff 51 4e 48 3e 34 27 19 0b 03 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x4e000000, 0x4b000000, 0x41000000, 0x30000000, 0x22000000, 0x17000000, 0x0f000000, 0x07000000, 0x02000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff ff ff ff ff ff 4e 4b 41 30 22 17 0f 07 02 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xffffffff, 0xff000000, 0x35000000, 0x43000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x49000000, 0x3f000000, 0x2a000000, 0x17000000, 0x0b000000, 0x06000000, 0x03000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff ff 35 43 ff ff ff ff 49 3f 2a 17 0b 06 03 01 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0xff000000, 0x1a000000, 0x31000000, 0x39000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x46000000, 0x42000000, 0x2f000000, 0x18000000, 0x08000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff ff 1a 31 39 ff ff ff ff 46 42 2f 18 08 01 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0xff000000, 0x08000000, 0x16000000, 0x27000000, 0x29000000, 0x20000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x46000000, 0x39000000, 0x22000000, 0x0e000000, 0x03000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 ff 08 16 27 29 20 ff ff ff ff 46 39 22 0e 03 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x06000000, 0x0f000000, 0x19000000, 0x17000000, 0x0f000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x46000000, 0x42000000, 0x2f000000, 0x18000000, 0x08000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 06 0f 19 17 0f ff ff ff ff 46 42 2f 18 08 01 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x03000000, 0x07000000, 0x0b000000, 0x09000000, 0x05000000, 0x02000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x46000000, 0x39000000, 0x22000000, 0x0e000000, 0x03000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 03 07 0b 09 05 02 ff ff ff ff 46 39 22 0e 03 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0x02000000, 0x03000000, 0x02000000, 0x01000000, 0x00000000, 0xff000000, 0xffffffff, 0xffffffff, 0xff000000, 0x45000000, 0x3f000000, 0x2a000000, 0x13000000, 0x05000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 02 03 02 01 00 ff ff ff ff 45 3f 2a 13 05 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0xff000000, 0xff000000, 0x2c000000, 0x3a000000, 0x38000000, 0x27000000, 0x12000000, 0x05000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 ff ff 2c 3a 38 27 12 05 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x03000000, 0x0c000000, 0x1b000000, 0x27000000, 0x27000000, 0x1b000000, 0x0c000000, 0x03000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 03 0c 1b 27 27 1b 0c 03 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0x05000000, 0x0c000000, 0x12000000, 0x12000000, 0x0c000000, 0x05000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 05 0c 12 12 0c 05 01 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x01000000, 0x03000000, 0x05000000, 0x05000000, 0x03000000, 0x01000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 03 05 05 03 01 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 */
    };
    return DarwinCursorCreate(41, 39, 13, 13, true , s_abMask, s_au32Shape, pCursor);
}

int MyCursor4(PDARWINCURSOR pCursor)
{
    static const uint8_t s_abMask[128] =
    {
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0x7f, 0xff, 0xff, 0xff,   /*   x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0x3f, 0xff, 0xff, 0xff,   /*     x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0x5f, 0xff, 0xff, 0xff,   /*   x   x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0x6f, 0xff, 0xff, 0xff,   /*   x x   x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0x77, 0xff, 0xff, 0xff,   /*   x x x   x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0x7b, 0xff, 0xff, 0xff,   /*   x x x x   x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0x7d, 0xff, 0xff, 0xff,   /*   x x x x x   x x x x x x x x x x x x x x x x x x x x x x x x x */
        0x7e, 0xff, 0xff, 0xff,   /*   x x x x x x   x x x x x x x x x x x x x x x x x x x x x x x x */
        0x7f, 0x7f, 0xff, 0xff,   /*   x x x x x x x   x x x x x x x x x x x x x x x x x x x x x x x */
        0x7f, 0xbf, 0xff, 0xff,   /*   x x x x x x x x   x x x x x x x x x x x x x x x x x x x x x x */
        0x7f, 0xdf, 0xff, 0xff,   /*   x x x x x x x x x   x x x x x x x x x x x x x x x x x x x x x */
        0x7f, 0xef, 0xff, 0xff,   /*   x x x x x x x x x x   x x x x x x x x x x x x x x x x x x x x */
        0x7f, 0xf7, 0xff, 0xff,   /*   x x x x x x x x x x x   x x x x x x x x x x x x x x x x x x x */
        0x7f, 0xfb, 0xff, 0xff,   /*   x x x x x x x x x x x x   x x x x x x x x x x x x x x x x x x */
        0x7f, 0x01, 0xff, 0xff,   /*   x x x x x x x               x x x x x x x x x x x x x x x x x */
        0x77, 0x7f, 0xff, 0xff,   /*   x x x   x x x   x x x x x x x x x x x x x x x x x x x x x x x */
        0x6b, 0xbf, 0xff, 0xff,   /*   x x   x   x x x   x x x x x x x x x x x x x x x x x x x x x x */
        0x5b, 0xbf, 0xff, 0xff,   /*   x   x x   x x x   x x x x x x x x x x x x x x x x x x x x x x */
        0x3d, 0xdf, 0xff, 0xff,   /*     x x x x   x x x   x x x x x x x x x x x x x x x x x x x x x */
        0x7d, 0xdf, 0xff, 0xff,   /*   x x x x x   x x x   x x x x x x x x x x x x x x x x x x x x x */
        0xfe, 0xef, 0xff, 0xff,   /* x x x x x x x   x x x   x x x x x x x x x x x x x x x x x x x x */
        0xfe, 0xef, 0xff, 0xff,   /* x x x x x x x   x x x   x x x x x x x x x x x x x x x x x x x x */
        0xff, 0x77, 0xff, 0xff,   /* x x x x x x x x   x x x   x x x x x x x x x x x x x x x x x x x */
        0xff, 0x77, 0xff, 0xff,   /* x x x x x x x x   x x x   x x x x x x x x x x x x x x x x x x x */
        0xff, 0x8f, 0xff, 0xff,   /* x x x x x x x x x       x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
        0xff, 0xff, 0xff, 0xff,   /* x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x x */
    };
    const uint32_t s_au32Shape[1024] =
    {
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y X X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y X X X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y X X X X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y X X X X X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y X X X X X X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y X X X X X X X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y X X X X X X X X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y X X X X X X X X X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y X X X X X X X X X X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y X X X X X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y X X X Y X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y X X Y Y Y X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y X Y Y Y Y X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00ffffff, 0x00ffffff, 0x00ffffff, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y X X X Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
        0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000,   /* Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y Y */
    };
    return DarwinCursorCreate(32, 32, 0, 4, false, s_abMask, s_au32Shape, pCursor);
}

#include <qapplication.h>
#include <qmainwindow.h>

class CursorTestWindow : public QMainWindow
{
    DARWINCURSOR Cursor;

public:
    CursorTestWindow()
        : QMainWindow(0)
    {
        int rc = MyCursor4(&Cursor); AssertRC(rc);
        resize(500, 500);
        rc = DarwinCursorSet(&Cursor); AssertRC(rc);

        PixMapHandle hData = 0;
        Point hot;
        QDGetCursorData(true, &hData, &hot);

    }
};

int main(int argc, char **argv)
{
    QApplication app(argc, argv);
    CursorTestWindow wnd;

    app.setMainWidget(&wnd);
    //wnd.showMaximized();
    wnd.show();
    return app.exec();
}

/*

#!/bin/bash
set -e
set -x

g++ -g \
    -framework IOKit -framework Carbon -framework QuickTime \
    -DRT_OS_DARWIN -DIN_RING3 -DQT_NO_DEBUG -DQT_THREAD_SUPPORT -DTESTING \
    -I include/ \
    -I ../../../../include \
    -I /Developer/SDKs/MacOSX10.4u.sdk/Developer/Headers/ \
    -I ../../../../tools/darwin.x86/qt/v3.3.8/include \
    ../../../../out/darwin.x86/debug/lib/RuntimeR3.a \
    ../../../../out/darwin.x86/debug/lib/SUPR3.a \
    -liconv \
    ../../../../out/darwin.x86/debug/dist/VirtualBox.app/Contents/MacOS/libqt-mt.3.dylib \
    -o ../../../../out/darwin.x86/debug/dist/VirtualBox.app/Contents/MacOS/VirtualBox \
    src/darwin/DarwinCursor.cpp

../../../../out/darwin.x86/debug/dist/VirtualBox.app/Contents/MacOS/VirtualBox

*/

#endif
