/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.explain;

import java.util.*;

/**
 * This class represents a node in an {@link
 * com.redhat.rhdb.explain.ExplainTree ExplainTree}.
 *
 * Every node has a name, a type, various details, and possibly
 * children. The name of the node can be anything, although it is
 * usually a human-friendly name based on type and subtype such as
 * "Hash Join" or "Append". Type is an integer that describes which
 * type of node the instance is (see NODE_* constants). Details include
 * such things as rows returned, width of each row in bytes, and so
 * on. Several <code>String</code> constants defined; these are keys
 * for accessing details.
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 */
public class ExplainTreeNode implements java.io.Serializable {

	// ATTENTION:
	// Make sure your changes are backward compatible.
	// Otherwise change the fingerprint, but them VISE will
	// refuse to read plans saved with previous versions
	static final long serialVersionUID = -7903950019386599034L;

	// detail keys
	
	/** Key for number of rows returned */
	public static final String ROWS = "rows";

	/** Key for width of result rows */
	public static final String WIDTH = "width";

	/** Key for startup cost */
	public static final String STARTUP_COST = "startup cost";

	/** Key for total cost */
	public static final String TOTAL_COST = "total cost";

	/** Key for node cost */
	public static final String NODE_COST = "node cost";

	/** Key for type of node */
	public static final String TYPE = "type";

	/** Key for subtype */
	public static final String SUBTYPE = "subtype";

	/** Key for table used */
	public static final String TABLE = "table";

	/** Key for table alias */
	public static final String TABLE_ALIAS = "table alias";

	/** Key for index(es) used */
	public static final String INDEXES = "indexes";

	/** Key for direction of index scan */
	public static final String DIRECTION = "direction";

	/** Key for setop all */
	public static final String SETOP_ALL = "all";

	/** Key for subquery on */
	public static final String SUBQUERY_ON = "on";

	/** Key for startup time (EXPLAIN ANALYZE) */
	public static final String STARTUP_TIME = "startup time";

	/** Key for total time (EXPLAIN ANALYZE) */
	public static final String TOTAL_TIME = "total time";

	/** Key for node time (EXPLAIN ANALYZE) */
	public static final String NODE_TIME = "node time";

	/** Key for actual rows returned (EXPLAIN ANALYZE) */
	public static final String ACTUAL_ROWS = "actual rows";

	/** Key for actual loops done (EXPLAIN ANALYZE) */
	public static final String ACTUAL_LOOPS = "loops";

	/** Key for "never executed" node (EXPLAIN ANALYZE) */
	public static final String ACTUAL_NOTE = "note";

	/** Key for total runtime of query (EXPLAIN ANALYZE) */
	public static final String TOTAL_RUNTIME = "total runtime";
        
        // Qualifier expressions

	/** Key for Filter */
	public static final String FILTER = "filter";

	/** Key for Join Filter */
	public static final String JOIN_FILTER = "join filter";

	/** Key for One-Time Filter */
	public static final String ONE_TIME_FILTER = "one-time filter";

	/** Key for Sort Key */
	public static final String SORT_KEY = "sort key";

	/** Key for Index Cond */
	public static final String INDEX_COND = "index cond";

	/** Key for Index Cond */
	public static final String MERGE_COND = "merge cond";

	/** Key for Index Cond */
	public static final String HASH_COND = "hash cond";

	// types
	
	/** Unknown node. */
	public static final int NODE_UNKNOWN = -1;
	
	/** Result node. */
	public static final int NODE_RESULT = 0;

	/** Append node. */
	public static final int NODE_APPEND = 1;

	/** Join node. */
	public static final int NODE_JOIN = 2;
	
	/** Scan node. */
	public static final int NODE_SCAN = 3;
	
	/** Materialize node. */
	public static final int NODE_MATERIALIZE = 4;

	/** Sort node. */
	public static final int NODE_SORT = 5;
	
	/** Group node. */
	public static final int NODE_GROUP = 6;

	/** Aggregate node. */
	public static final int NODE_AGGREGATE = 7;
	
	/** Unique node. */
	public static final int NODE_UNIQUE = 8;

	/** SetOp node. */
	public static final int NODE_SETOP = 9;

	/** Limit node. */
	public static final int NODE_LIMIT = 10;

	/** Hash node. */
	public static final int NODE_HASH = 11;

	/** InitPlan node. */
	public static final int NODE_INITPLAN = 12;
	
	/** SubPlan node. */
	public static final int NODE_SUBPLAN = 13;

	private final int max = 13, min = -1;
	private String name;
	private HashMap details;
	private ArrayList children;
	private ExplainTreeNode parent;
	private int type;

	/**
	 * Creates a new <code>ExplainTreeNode</code> instance. The type
	 * of the node is UNKNOWN.
	 */
	public ExplainTreeNode()
	{
		parent = null;
		name = null;
		details = new HashMap(7);
		children = new ArrayList(2);
		type = NODE_UNKNOWN;
	}

	/**
	 * Creates a new <code>ExplainTreeNode</code> instance of the
	 * given type.
	 *
	 * @param type an <code>int</code> value
	 */
	public ExplainTreeNode(int type)
	{
		this(type, null);
	}
	
	/**
	 * Creates a new <code>ExplainTreeNode</code> instance with a
	 * given name. The type of the node is UNKNOWN.
	 *
	 * @param name a <code>String</code> value
	 */
	public ExplainTreeNode(String name)
	{
		this(NODE_UNKNOWN, name);
	}

	/**
	 * Creates a new <code>ExplainTreeNode</code> instance with a
	 * given name and type.
	 *
	 * @param type an <code>int</code> value
	 * @param name a <code>String</code> value
	 */
	public ExplainTreeNode(int type, String name)
	{
		this();
		
		if (type < min || type > max)
			type = NODE_UNKNOWN;
		
		this.type = type;
		this.name = name;
	}

	/**
	 * Gets the name of the node.
	 *
	 * @return a <code>String</code> value
	 */
	public String getName()
	{
		return name;
	}

	/**
	 * Sets the name of the node.
	 *
	 * @param n a <code>String</code> value
	 */
	public void setName(String n)
	{
		name = n;
	}

	/**
	 * Returns a <code>Set</code> of detail keys.
	 * getDetail can be used to retrieve values corresponding
	 * to individual keys in the set.
	 *
	 * @return a <code>Set</code> value
	 */
	public Set getDetailTypes()
	{
		return details.keySet();
	}

	/**
	 * Set detail type <i>d</i> as having value <i>v</i>. A new detail
	 * will be added without complaint; a detail that is already a detail
	 * of the node will be replaced.
	 *
	 * @param d a <code>String</code> value
	 * @param v an <code>Object</code> value
	 */
	public void setDetail(String d, Object v)
	{
		details.put(d, v);
	}

	/**
	 * Return the value of detail <i>d</i>.
	 *
	 * @param d a <code>String</code> value
	 * @return a <code>String</code> value
	 */
	public String getDetail(String d)
	{
		return (String) details.get(d);
	}

	/**
	 * Set <code>ExplainTreeNode</code> <i>c</i> to be a child
	 * of this node.
	 *
	 * @param c an <code>ExplainTreeNode</code> value
	 *
	 * @exception IllegalArgumentException if the child to the added is the node itself.
	 */
	public void addChild(ExplainTreeNode c) throws IllegalArgumentException
	{
		if (this == c)
			throw new IllegalArgumentException("A node can't be a child of itself.");

		children.add(c);
	}

	/**
	 * Removes the child node specified by the given index.
	 * Indexes start at 0.
	 *
	 * @param n an <code>int</code> value
	 */
	public void removeChild(int n)
	{
		if (n >= 0 && n < children.size())
			children.remove(n);
	}

	/**
	 * Removes the given child node.
	 *
	 * @param child an <code>ExplainTreeNode</code> value
	 */
	public void removeChild(ExplainTreeNode child)
	{
		if (children.contains(child))
			children.remove(child);
	}
	
	/**
	 * Returns the number of children that this node has.
	 *
	 * @return an <code>int</code> value
	 */
	public int getChildCount()
	{
		return children.size();
	}
	
	/**
	 * Returns the n'th child of this node. Indexes starts at 0.
	 *
	 * @param n an <code>int</code> value
	 * @return an <code>ExplainTreeNode</code> value or null if
	 * the given child doesn't exist.
	 */
	public ExplainTreeNode getChild(int n)
	{
		if (n >= 0 && n < children.size())
			return (ExplainTreeNode) children.get(n);
		return null;
	}

	/**
	 * Returns an enumeration for iterating over the children of the current
	 * node.
	 *
	 * @return an <code>Enumeration</code> value
	 */
	public Enumeration getChildren() {
		return Collections.enumeration(children);
	}

	/**
	 * For the specified child, return its index. Indexes begin at 0.
	 *
	 * @param child an <code>ExplainTreeNode</code> value
	 * @return an <code>int</code> value
	 */
	public int getIndexOfChild(ExplainTreeNode child)
	{
		if (children.contains(child))
			return children.indexOf(child);

		return -1;
	}
	
	/**
	 * Gets the parent of the current node.
	 *
	 * @return an <code>ExplainTreeNode</code> value or null
	 * if the node has no parent.
	 */
	public ExplainTreeNode getParent()
	{
		return parent;
	}

	/**
	 * Set the parent of the current node.
	 *
	 * @param n an <code>ExplainTreeNode</code> value
	 *
	 * @exception IllegalArgumentException if the parent is the node itself.
	 */
	public void setParent(ExplainTreeNode n) throws IllegalArgumentException
	{
		if (this == n)
			throw new IllegalArgumentException("A node can't be the parent of itself");

		parent = n;
	}

	/**
	 * Gets the type of the node (see NODE_* constants).
	 *
	 * @return an <code>int</code> value
	 */
	public int getType()
	{
		return type;
	}

	/**
	 * Sets the type of the node (see NODE_* constants). 
	 *
	 * @param t an <code>int</code> value
	 *
	 * @exception IllegalArgumentException if t is not a valid node type.
	 */
	public void setType(int t) throws IllegalArgumentException
	{
		if (t < min || t > max)
			throw new IllegalArgumentException("Bad node type specified");

		this.type = t;
	}

	/**
	 * Is the node a leaf node?
	 *
	 * @return a <code>boolean</code> value
	 */
	public boolean isLeaf()
	{
		if (getChildCount() == 0)
			return true;

		return false;
	}

	/**
	 * Get a String that would be good for displaying a tooltip for the
	 * node. For most nodes this is the total time, or cost,
	 * whichever is available.  If time nor cost is available,
	 * then it is just the name.
	 *
	 * @return a <code>String</code> value
	 */
	public String getToolTipText()
	{
		String text;

		if (getDetail(TOTAL_TIME) != null)
			text = getDetail(TOTAL_TIME);
		else if (getDetail(TOTAL_COST) != null)
			text = getDetail(TOTAL_COST);
		else
			text = "";
	
		return text.toString();
	}

	/**
	 * Get a String that would be good for displaying a header for the
	 * node. For all nodes except for Index Scan and Seq Scan, this is
	 * just the node name. For Index and Seq Scan, it is "(Index|Seq) Scan on
	 * (<alias>|<table if no alias>)".
	 *
	 * @return a <code>String</code> value
	 */
	public String getDisplayText()
	{
		StringBuffer text = new StringBuffer();
		text.append(name);

		if (name.equals("Index Scan") ||
			name.equals("Seq Scan"))
		{
			if (getDetail(TABLE) != null)
				text.append(" on " + getDetail(TABLE));
			if (getDetail(TABLE_ALIAS) != null)
				text.append(" (" + getDetail(TABLE_ALIAS) + ")");
		}

		return text.toString();
	}
	
	/**
	 * A friendly String version of the node.
	 *
	 * @return a <code>String</code> value
	 * @see #getDisplayText
	 */
	public String toString()
	{
		return getDisplayText();
	}
}//ExplainTreeNode
