/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.misc;

import java.io.*;
import java.util.*;

/**
 * This class contains utility methods that may be used by many
 * different classes in many different packages.
 *
 * @author Liam Stewart
 * @version 1.2.0
 */
public class RHDBUtils {
	
	private final static boolean isJava14 = (System.getProperty("java.specification.version").equals("1.4"));
	
	/**
	 * True if the Java specification is 1.4
	 * Used everywhere to select between preferred code that uses
	 * 1.4 methods and compatibility code that exists to allow 1.3 execution
	 *
	 * @return a <code>boolean</code> value
	 */
	 public static boolean isJava14()
	 {
	 	return isJava14;
	 }

	/**
	 * Load the given File. The contents of the file will be returned
	 * as a String.
	 *
	 * @param f a <code>File</code> value
	 * @return a <code>String</code> value
	 * @exception IOException if an error occurs
	 */
	public static String load(File f) throws IOException
	{
		String text;
		FileReader filein = new FileReader(f);
		BufferedReader reader = new BufferedReader(filein);
		StringWriter strw = new StringWriter();
		PrintWriter writer = new PrintWriter(strw);
		String s;

		while ((s = reader.readLine()) != null)
			writer.println(s);

		reader.close();
		filein.close();

		text = strw.toString();
		strw.close();

		return text;
	}

	/**
	 * Saves the contents of the given String to the given File.
	 *
	 * @param f a <code>File</code> value
	 * @param text a <code>String</code> value
	 * @exception Exception if an error occurs
	 */
	public static void save(File f, String text) throws Exception
	{
		FileWriter fileout = new FileWriter(f);
		BufferedWriter writer = new BufferedWriter(fileout);
		StringReader strr = new StringReader(text);
		int ch;

		while ((ch = strr.read()) != -1)
			writer.write(ch);

		strr.close();
		writer.close();
		fileout.close();
	}

	/**
	 * Given a File, return the extention, not including the dot. Eg:
	 * if the name of the File f was foo.sql, getExension(f) would
	 * return "sql". Returns null if the File does not have an
	 * extension.
	 *
	 * @param f a <code>File</code> value
	 * @return a <code>String</code> value
	 */
	public static String getExtension(File f)
	{
		String name = f.getName();
		int dot = name.lastIndexOf('.');

		if (dot < 0 || dot > name.length() - 1)
			return null;

		return name.substring(dot + 1);
	}

	/**
	 * Add the given extension to a File, but only if the file does
	 * not already have an extension. Returns a new File. Eg: if the
	 * name of the File f was "foo" and addExtension(f, "sql") was
	 * called, the new File would be named "foo.sql". If f called
	 * "foo.bar", no extension would be added.
	 *
	 * @param f a <code>File</code> value
	 * @param ext a <code>String</code> value
	 * @return a <code>File</code> value
	 */
	public static File addExtension(File f, String ext)
	{
		if (RHDBUtils.getExtension(f) != null)
			return f;

		return new File(f.getParent(), f.getName() + "." + ext);
	}

	/**
	 * Describe <code>stringRemove</code> method here.
	 *
	 * @param from a <code>String</code> value
	 * @param toss a <code>char</code> value
	 * @return a <code>String</code> value
	 */
	public static String stringRemove(String from, char toss)
	{
		char[] chars = from.toCharArray();
		int len = chars.length;
		int put = 0;
		for (int i = 0; i < len; i++)
			if (chars[i] != toss)
				chars[put++] = chars[i];
		return new String(chars, 0, put);
	}

	/**
	 * Attempts to fit the given string into the given space (# of
	 * characters). Newlines added as neccessary.
	 *
	 * @param s a <code>String</code> value
	 * @param width an <code>int</code> value
	 * @return a <code>String</code> value
	 */
	public static String stringFitWidth(String s, int width)
	{
		if (s == null)
			return s;

		StringReader strread = new StringReader(s);
		BufferedReader reader = new BufferedReader(strread);
		StringWriter strwrite = new StringWriter();
		PrintWriter writer = new PrintWriter(strwrite);
		String line;

		try {
			while ((line = reader.readLine()) != null)
			{
				if (line.length() <= width)
					writer.println(line);
				else
				{
					StringBuffer buf = new StringBuffer();
					StringTokenizer strtok = new StringTokenizer(line, " \t\n\f\r", true);
					String tok;
					
					while (strtok.hasMoreTokens())
					{
						tok = strtok.nextToken();
						if (buf.length() + tok.length() > width)
						{
							writer.println(buf.toString().trim());
							buf = new StringBuffer();
						}
						buf.append(tok);
					}
					
					writer.println(buf.toString().trim());
				}
			}

			writer.close();
		} catch (Exception ex) {
			return s;
		}

		return strwrite.toString();
	}

	/**
	 * Removes all text from the matching portion to the end of the
	 * String. Eg. If String s was "hello bob and marge" and incl was
	 * "bob", the String returned would be "hello ".
	 *
	 * @param str a <code>String</code> value
	 * @param incl a <code>String</code> value
	 * @return a <code>String</code> value
	 */
	public static String removeIncluding(String str, String incl)
	{
		if (str == null)
			return str;

		int i = str.indexOf(incl);
		if (i == -1)
			return str;

		return str.substring(0, i);
	}

}// RHDBUtils
