/**********************************************************************
 ** Readobj functions: Functions that provide for reading in an object
 **
 **
 **    
 ** Last reviewed: 
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef READOBJ_C
#define READOBJ_C

#include "config.h"
#include "sysdep.h"
#include "global.h"
#include "mudtypes.h"
#include "strings.h"
#include "lexer.h"
#include "errlog.h"
#include "mudobject.h"
#include "readobj.h"
#include "skill.h"
#include "spell.h"
#include "newfuncts.h"



/***********************************************************************
 ** read_skill - reads in a skill
 **
 ** Parameters: read_file - the file to read in from
 **             error_log - the error log to write any errors to
 **             build_format - if we are reading this in .bldr file format
 **
 ** Returns: a pointer to the Spell MudObject that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *read_skill(FILE *read_file, ErrLog *error_log, int build_format)
{
    Skill        *the_skill;
    token_record *the_token;
    Strings      holder;

    the_token = get_token(read_file,'\0');

    the_skill = new_Skill(the_token->the_string);
   
    if (the_skill->read_ability_attrib(read_file, error_log, build_format) <= 0)
    {
       delete_Skill(the_skill);
       return NULL;
    }

    if (the_skill->read_skill_attrib(read_file, error_log) <= 0)
    {
       delete_Skill(the_skill);
       return NULL;
    }

    return (Entity *)the_skill;
}


/***********************************************************************
 ** read_spell - reads in a spell
 **
 ** Parameters: read_file - the file to read in from
 **             error_log - the error log to write any errors to
 **             build_format - if we are reading this in .bldr file format
 **
 ** Returns: a pointer to the Spell MudObject that we read in
 **          NULL if an error is found
 **
 ***********************************************************************/
Entity *read_spell(FILE *read_file, ErrLog *error_log, int build_format)
{
    Spell        *the_spell;
    token_record *the_token;
    Strings      holder;

    the_token = get_token(read_file,'\0');

    the_spell = new Spell(the_token->the_string);
   
    if (the_spell->
            read_ability_attrib(read_file, error_log, build_format) <= 0)
    {
       delete the_spell;
       return NULL;
    }

    if (the_spell->read_spell_attrib(read_file, error_log) <= 0)
    {
       delete the_spell;
       return NULL;
    }

    return (Entity *)the_spell;
}


/***********************************************************************
 ** read_ability - Reads in an ability from the ability file 
 **
 ** Parameters: read_file - the file we are reading the mudobject from
 **             the_tok - the tokens that we get to read
 **             error_log - the log we write any errors we encounter to
 **             build_format - are we reading this in builder format?
 **
 ** Returns: a pointer to the mudobject we read in
 **          NULL if an error is found
 **
 ***********************************************************************/

Entity *read_ability(FILE *read_file, token_record *the_tok, 
                                      ErrLog *error_log, int build_format)
{
   Strings holder;

   switch(the_tok->token_type)
   {
      case T_SPELL:
         return read_spell(read_file, error_log, build_format);
         break;
      case T_SKILL:
         return read_skill(read_file, error_log, build_format);
         break;
      default:
         holder.sprintf("Invalid type %s in abilities file.", 
                                                    the_tok->the_string);
         error_log->log_err(holder.str_show(), "read_ability");
         break;
   }
   return NULL;
}



#endif





