/**********************************************************************
 ** Timing - controls timing of the mud, from events to when day and
 **          night end.  All events that happen without user intervention
 **          will go here, with the exception of socket handling.  These
 **          functions are handled by CGameTimer on a windows system
 **
 ** Last reviewed:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef TIMING_C
#define TIMING_C

#include "config.h"
#include "sysdep.h"
#include "mudtypes.h"
#include "timing.h"
#include "global.h"

/***********************************************************************
 ** Timing (constructor) - Constructs the timing object
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Timing::Timing()
{
   gettimeofday(&initial_time, NULL);

   last_heal_players = last_heal_mobiles = last_level = 
                         last_magic_players = last_endur_players = 0;
}



/***********************************************************************
 ** ~Timing (destructor) - cleans up and destroys the timing object
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Timing::~Timing()
{ 
}


/***********************************************************************
 ** init_times - initializes all timers to the current time
 **
 ** Parameters: None
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Timing::init_times()
{ 
   timeval cur_time;
   gettimeofday(&cur_time, NULL);

   last_loop.tv_usec = cur_time.tv_usec;
   last_loop.tv_sec = last_second = last_hour = start_day = cur_time.tv_sec;

   return 1;
}

/***********************************************************************
 ** cycle_elapsed - the milliseconds that have elapsed since last cycle
 **
 ** Parameters: None
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

long Timing::cycle_elapsed()
{
   long    return_val   = 0;
   timeval new_time;

   gettimeofday(&new_time, NULL);
   if (new_time.tv_sec > last_loop.tv_sec)
   {
      return_val = ((1000000 - last_loop.tv_usec) + new_time.tv_usec + 
                    (1000000 * ((new_time.tv_sec - last_loop.tv_sec) - 1)));
   }
   else
      return_val = (new_time.tv_usec - last_loop.tv_usec);

   return return_val;
}


/***********************************************************************
 ** set_cycle - sets the cycle value to the current time
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Timing::set_cycle()
{
   gettimeofday(&last_loop, NULL);
}



/***********************************************************************
 ** handle_events - handles any events that need to be executed right
 **                 now
 **
 ** Parameters: None
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Timing::handle_events()
{ 
   float time_num[5] = {1, 0.1, 0.3, 0.55, .65}; 
   char *time_change[5] = {
          "The sky begins to lighten with the coming of the sun.\n",
          "The sun rises above the horizon.\n",
          "The sun has peaked in the sky.\n",
          "The sun disappears beneath the horizon.\n",
          "The last bit of light vanishes, leaving a dark sky.\n"};

   timeval new_time;

   gettimeofday(&new_time, NULL);

   /* if a second hasnt elapsed, we do nothing here */
   if (new_time.tv_sec <= last_second)
   {
      return 0;
   }
   
   /* else, a new second is upon us, handle second events first */
   mainstruct->cycle_world();
   last_second = new_time.tv_sec;

   /* next handle possible time of day progressions */
   if ((new_time.tv_sec - start_day) > 
       (time_num[mainstruct->get_tod()] * the_config.secs_in_day))
   {
      mainstruct->send_all_outside(time_change[mainstruct->get_tod()]);
      mainstruct->increment_tod();
      if (mainstruct->get_tod() == Dawn)
         start_day = new_time.tv_sec;   
   }

   /* reset these heal times, since we probably already healed */
   if ((time(0) - last_heal_players) >= the_config.secs_player_heal)
      last_heal_players = time(0);

   if ((time(0) - last_heal_mobiles) >= the_config.secs_mobile_heal)
      last_heal_mobiles = time(0);

   if ((time(0) - last_magic_players) >= the_config.secs_player_magic)
      last_magic_players = time(0);

   if ((time(0) - last_endur_players) >= the_config.secs_player_endur)
      last_endur_players = time(0);

   if ((time(0) - last_level) >= the_config.secs_level_check)
      last_level = time(0);


   return 1;
}


/***********************************************************************
 ** plr_timesincehealed - the seconds since the last player heal
 **
 **
 ***********************************************************************/

time_t Timing::plr_TimeSinceHealed()
{ 
   if (!last_heal_players)
      return the_config.secs_player_heal;

   return (time(0) - last_heal_players);
}


/***********************************************************************
 ** plr_timesincelevel - the seconds since the last level upgrade check
 **
 **
 ***********************************************************************/

time_t Timing::plr_TimeSinceLevel()
{ 
   if (!last_level)
      return the_config.secs_level_check;

   return (time(0) - last_level);
}


/***********************************************************************
 ** mob_timesincehealed - the seconds since the last mobile heal
 **
 **
 ***********************************************************************/

time_t Timing::mob_TimeSinceHealed()
{ 
   if (!last_heal_mobiles)
      return the_config.secs_mobile_heal;

   return (time(0) - last_heal_mobiles);
}

/***********************************************************************
 ** plr_timesincemagic - the seconds since the last player magic up
 **
 **
 ***********************************************************************/

time_t Timing::plr_TimeSinceMagic()
{ 
   if (!last_magic_players)
      return the_config.secs_player_magic;

   return (time(0) - last_magic_players);
}


/***********************************************************************
 ** plr_timesinceendur - the seconds since the last player endurance up
 **
 **
 ***********************************************************************/

time_t Timing::plr_TimeSinceEndur()
{ 
   if (!last_endur_players)
      return the_config.secs_player_endur;

   return (time(0) - last_endur_players);
}


/***********************************************************************
 ** set_tod_time - sets the appropriate time for tod if we are manually
 **                changing tod, so it doesn't go back to where it was
 **
 ** Parameters: the_time - the time of day we are setting it to
 **
 ***********************************************************************/

void Timing::set_tod_time(TimeOfDay the_time)
{ 
   float time_num[5] = {
      ( float )1,
      ( float )0.1,
      ( float )0.3,
      ( float )0.55,
      ( float ).65
   };
   
   long  span;
   int   new_time;

   new_time = ((int) the_time) - 1;
   if (new_time == -1)
      new_time = 4; 
   span = (long) (time_num[new_time] * the_config.secs_in_day);

   start_day = last_second - span;
}


#endif









