// ---------------------------------------------------------------------------
// - Operator.cpp                                                            -
// - aleph engine - operator builtin functions implementation                -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Real.hpp"
#include "Builtin.hpp"
#include "Boolean.hpp"
#include "Runnable.hpp"
#include "Exception.hpp"

namespace aleph {

  // this procedure extract and evaluate the calling object for an operator
  static inline Object* getobj (Runnable* robj, Nameset* nset, Cons* args, 
				const char* opname) {
    Object* car = nilp;
    Object* obj = nilp;
    if ((args == nilp) || (args->length () != 2))
      throw Exception ("argument-error",
		       "missing or too many arguments with operator", opname);
    car = args->getcar ();
    obj = (car == nilp) ? nilp : car->eval (robj,nset);
    if (obj == nilp) 
      throw Exception ("type-error", "invalid nil object with operator",
		       opname);
    return obj;
  }

  // this procedure evaluates the cadr and return it
  static inline Object* getarg (Runnable* robj, Nameset* nset, Cons* args) {
    Object* car = (args == nilp) ? nilp : args->getcadr ();
    return (car == nilp) ? nilp : car->eval (robj, nset);
  }

  // operator + 

  Object* builtin_add (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj    = getobj (robj, nset, args, "+");
    Object* arg    = getarg (robj, nset, args);
    Object* result = obj->oper (robj, Object::ADD, arg);
    Object::cref (obj);
    Object::cref (arg);
    return result;
  }

  // operator -

  Object* builtin_sub (Runnable* robj, Nameset* nset, Cons* args) {
    long argc = 0;
    if ((args == nilp) || (((argc = args->length ()) != 1) && (argc != 2)))
      throw Exception ("argument-error",
		       "missing or too many arguments with operator -");
    Object* car = args->getcar ();
    Object* obj = (car == nilp) ? nilp : car->eval (robj,nset);
    if (obj == nilp) 
      throw Exception ("type-error", "invalid nil object with operator -");
    if (argc == 1) {
      Object* result = obj->oper (robj, Object::MINUS, (Object*) nilp);
      Object::cref (obj);
      return result;
    }
    Object* arg = getarg (robj, nset, args);
    Object* result = obj->oper (robj, Object::SUB, arg);
    Object::cref (obj);
    Object::cref (arg);
    return result;
  }

  // operator * 

  Object* builtin_mul (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj    = getobj (robj, nset, args, "==");
    Object* arg    = getarg (robj, nset, args);
    Object* result = obj->oper (robj, Object::MUL, arg);
    Object::cref (obj);
    Object::cref (arg);
    return result;
  }

  // operator /

  Object* builtin_div (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj    = getobj (robj, nset, args, "==");
    Object* arg    = getarg (robj, nset, args);
    Object* result = obj->oper (robj, Object::DIV, arg);
    Object::cref (obj);
    Object::cref (arg);
    return result;
  }

  // operator ==

  Object* builtin_eql (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj    = getobj (robj, nset, args, "==");
    Object* arg    = getarg (robj, nset, args);
    Object* result = obj->oper (robj, Object::EQL, arg);
    Object::cref (obj);
    Object::cref (arg);
    return result;
  }

  // operator !=

  Object* builtin_neq (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj    = getobj (robj, nset, args, "==");
    Object* arg    = getarg (robj, nset, args);
    Object* result = obj->oper (robj, Object::NEQ, arg);
    Object::cref (obj);
    Object::cref (arg);
    return result;
  }
  
  // operator >=

  Object* builtin_geq (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj    = getobj (robj, nset, args, "==");
    Object* arg    = getarg (robj, nset, args);
    Object* result = obj->oper (robj, Object::GEQ, arg);
    Object::cref (obj);
    Object::cref (arg);
    return result;
  }
  
  // operator >

  Object* builtin_gth (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj    = getobj (robj, nset, args, "==");
    Object* arg    = getarg (robj, nset, args);
    Object* result = obj->oper (robj, Object::GTH, arg);
    Object::cref (obj);
    Object::cref (arg);
    return result;
  }
  
  // operator <=

  Object* builtin_leq (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj    = getobj (robj, nset, args, "==");
    Object* arg    = getarg (robj, nset, args);
    Object* result = obj->oper (robj, Object::LEQ, arg);
    Object::cref (obj);
    Object::cref (arg);
    return result;
  }

  // operator <

  Object* builtin_lth (Runnable* robj, Nameset* nset, Cons* args) {
    Object* obj    = getobj (robj, nset, args, "==");
    Object* arg    = getarg (robj, nset, args);
    Object* result = obj->oper (robj, Object::LTH, arg);
    Object::cref (obj);
    Object::cref (arg);
    return result;
  }
  

  // process an assert command - this does nothing if the assert flag is false

  Object* builtin_assert (Runnable* robj, Nameset* nset, Cons* args) {
    // do nothing if assert flag is false
    if (robj->getassert () == false) return nilp;

    // check for nil first
    Object* car1 = (args == nilp) ? nilp : args->getcar ();
    Object* obj1 = (car1 == nilp) ? nilp : car1->eval (robj, nset);
    Object* car2 = (args == nilp) ? nilp : args->getcadr ();
    Object* obj2 = (car2 == nilp) ? nilp : car2->eval (robj, nset);
    if ((obj1 == nilp) && (obj2 == nilp)) return nilp;
    
    // normal compare
    bool status = false;
    if (obj1 != nilp) {
      Object*  aobj = obj1->oper (robj, Object::EQL, obj2);
      Boolean* bobj = dynamic_cast <Boolean*> (aobj);
      status = bobj->toboolean ();
      Object::cref (bobj);
    }
    Object::cref (obj1);
    Object::cref (obj2);
    if (status == true) return nilp;
    Exception e ("assert-error");
    e.setabf (true);
    throw e;  
  }
}
