# This module is a part of Alicq instant messenger package
# It provides storing and retieving messages in history database
#
# Author: Ihar Viarheichyk

variable base history

# History item flag:
# R - incoming
# s - added to queue to send
# S - sent, but not acknowledged
# a - acknowledged bt ICQ server
# A - acknowledged by remote client
array set types {R incoming s outgoing S sent a server A client }
foreach {k v} [array get types] { set types($v) $k }

namespace eval meta {
	set description "History storage"
	set author "Ihar Viarheichyk <iverg@mail.ru>"
	set icon [namespace current]::icon
	image create photo $icon -data "
		R0lGODlhEAANAMIAAAAAALYA/////2hoaLYA/7YA/7YA/7YA/yH5BAEAAAQAL
		AAAAAAQAA0AAAM2SLrcDjDKBoO9kdVroQDLxnlQKElfSQBC67bRp7CvS4LzGU
		PDXNu83uoHAwyEOZ0RqTg6n8wEADs="
	# Provide this icon as image for any 'history' widget as well	
	if {[package vsatisfies [package present Tk] 8.4]} {
		option add *history.image $icon widgetDefault
	}
}

namespace eval limit::meta {
	array set size {
		type integer save change property Global:History|limit:size
		default 64536
		description "History chunk maximal size, bytes"
	}
	array set time {
		type integer save change property Global:History|limit:time
		default 14400
		description "History chunk maximal age, seconds"
	}
}

# Initiate asynchronous history request
handler HistoryRequest Request {uid {chunk {}}} {
	variable base
	set uin [lindex [split $uid :] end]
	# Determine chunk
	if {$chunk==""} {
		set chunk [lindex [chunks $uid] 0]
		if {$chunk==""} return
	}	
	set res [catch {
		set fd [open $chunk]
		fconfigure $fd -blocking no -encoding binary
		fileevent $fd readable [nc ParseHistory $fd $uid $chunk]
	} reason]	
	if {$res!=0} {Event Error history $reason }
}

# Get list of available history chunks, ordered from newer to older
handler HistoryChunks chunks {uid} {
	variable base
	variable chunk_cache
	if {[info exists chunk_cache($uid)]} { return $chunk_cache($uid) }
	set uin [lindex [split $uid :] end]
	set chunk_cache($uid) [lsort -decreasing -dictionary\
		[glob -nocomplain $base/$uin/$uin.* $base/$uin.*]]
}

# Proceed history file asynchronously
proc ParseHistory {fd uid chunk} {
	set name [namespace current]::suck-$fd
	upvar #0 $name data
	set ret [gets $fd line]
	if {$ret==-1} {
		if {[eof $fd]} {
			close $fd
			compose $name $uid $chunk
			unset data
		}
		return
	}
	set prefix [string index $line 0]
	if {$prefix=="\["} {
		if {[info exists data(msg)]} { 
			compose $name $uid $chunk
			unset data(msg)
		}
		set l [string range [string map {" " ""} $line] 1 end-1]
		foreach key {type class msgid _ time} val [split $l |] {
			set data($key) $val
		}
	} elseif {$prefix==":"} {
		append data(msg) [string range $line 1 end] "\n"
	}
}

# Compose history responce and generate proper event
proc compose {name uid ch} {
	variable types
	upvar #0 $name data
	if {[string is integer $data(class)]} {
		set data(class) text
		set enc [encoding system]
	} else { set enc utf-8 }
	if {![info exists data(msg)]} { set data(msg) "" }
	if {$data(type)!="R"} { set u Me } else { set u $uid }
	Event $uid|HistoryItem $data(class) $u $data(time)\
		[list [encoding convertfrom $enc $data(msg)]]\
		[list $data(msgid) $types($data(type)) $data(class)] $ch
}

# Update message flag (sent, received, acknowledged)
# TODO:
#  1. Last update mark. Acknowledgements are normally subsequent, thus
#     caching last position can speed-up search for next acknowledgements.
#  2. Do error handling
#  3. Make it background
handler *|Acknowledgement UpdateFlag {type uid msgid} {
	variable types
	set flag $types($type)
	if {[string length $flag]>1} return
	foreach chunk [chunks $uid] {
		set fd [open $chunk RDWR]
		fconfigure $fd -encoding utf-8
		for {set pos 0} {[gets $fd line]!=-1} {set pos [tell $fd]} {
			if {[string index $line 0]!="\["} continue
			set l [string range [string map {" " ""} $line] 1 end-1]
			set id [lindex [split $l |] 2]
			if {$id==$msgid} {
				seek $fd [expr $pos+2]
				puts -nonewline $fd $flag
				close $fd
				return
			}
		}
		close $fd
	}
}

proc text2text {txt} {set txt}
proc authrequest2text {msg} {lindex $msg end}

# Save message to history
proc Message {type class uid time message {msgid 0}} {
	variable base
	variable chunk_cache
	if {[llength [info commands ${class}2text]]} {
		 set message [${class}2text $message]
	} else { set message [join $message "\n"] }
	# If msgid is 0 (incoming message) create random id
	if {$msgid==0} { set msgid [clock seconds]:[clock clicks] }
	set txt "\[ $type | $class | $msgid | 0000 | $time \]\n"
	foreach line [split $message "\n"] { append txt ":$line\n" }
	# Determine chunk
	set chunks [chunks $uid]
	set uin [lindex [split $uid :] end]
	if {[llength $chunks]} { 
		 set chunk [lindex $chunks 0] 
	} else { 
		set chunk [file join $base $uin ${uin}.1]
		CheckDir $chunk
		set chunk_cache($uid) [list $chunk]
	}
	if {[file exists $chunk]} {
		if {[file size $chunk]>$limit::size ||
		    $time-[file mtime $chunk]>$limit::time} {
		  	set ext [string range [file extension $chunk] 1 end]
			if {[string is integer $ext]} { incr ext } else {
				set ext ${ext}.1
			}
			if {[file dirname $chunk]==$base} {
				set chunk [file join $base $uin ${uin}.${ext}]
				CheckDir $chunk
			} else { set chunk [file root $chunk].$ext }
			set chunk_cache($uid) [concat [list $chunk]\
				$chunk_cache($uid)]
		}
	}
	if {[catch {
		set fd [open $chunk a+]
		fconfigure $fd -encoding utf-8
		puts -nonewline $fd $txt
		close $fd
	} reason]} {
		Event Error history "History write error: $reason"
	}
}

proc CheckDir {chunk} {
	set name [file dirname $chunk]
	if {![file isdirectory $name]} { file mkdir $name }
}

if {![file exists $base]} {file mkdir $base}

hook Incoming [namespace code {Message R}]
hook Outgoing [namespace code {Message s}]

