/// ========================================================================
// copyright (C) 1999-2003 by Tobias Erbsland <te@profzone.ch>
// ------------------------------------------------------------------------
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2 of the License, or (at your
// option) any later version.
// ========================================================================

#ifndef PARAMETER_H
#define PARAMETER_H
#ifdef _MSC_VER
#include "win32/config_win32.h"
#endif
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>
#include <map>
#include <list>

/** This is the class on the top of the whole class hierarchy of this
  * project. Every object is inherited by parameter. This class provides
  * a universal interface for commandoline parameters.
  *@short Abstract class for the parameter administration.
  *@author Tobias Erbsland
  */
class Parameter
{

private:
	/** internal use. */
	const char * const str_unknown;
	/** internal use. */
	const char * const str_none;

public:
	/** Every Parameter can have one this this four Types.
	  *
	  * FLAG is a local flag like "--xxx-flagname". The initial value of such local
	  * flag should be "0". If the flag is set, the value is set to "1".
	  *
	  * STRING is a local string like "--xxx-stringname=yyy". If the string is set,
	  * the vaule will by yyy in this example.
	  *
	  * G_FLAG and G_STRING are the global versions of the types above.
	  */
	enum param_e { FLAG, G_FLAG, STRING, G_STRING, INTEGER, G_INTEGER, DATE, G_DATE };

	/** This struct is the base for every parameter.
	  */
	struct param_t
	{
		/** This is a short info describes this parameter.
		  */
		std::string Info;
		/** A short Helptext to this Parameter.
		  */
		std::string Help;
		/** The typ of this parameter.
		  */
		param_e Typ;
		/** Initial Value of this Parameter.
		  */
		std::string Value;
	};

	/** This struct is used to store the global dependence of parameter
	  * objects and "parameters".
	  */
	struct param_user_t
	{
		/** Copy of the parameter information.
		  */
		param_t Param;
		/** A list of pointer to all parameter objects use the parameter above.
		  */
		std::list< Parameter * > User;
	};

protected:
	/** This is the primary parameter map every object own. The function
	  * Param() let everyony access this map for read and write.
	  */
	std::map< std::string, param_t > mapParam;

	/** This protected function is normaly used in the Constructor of every
	  * parameter object to add all needed parameters. A exception is the
	  * Plugin object. In this object a function beforeParam() exist to
	  * add new parameters on thy fly.
	  */
	void addParam( const std::string &prefix, const std::string &info, const std::string &help,
		const param_e typ, const std::string &value );

	/** With this function every parameter object can check out a submitted
	  * integer parameter.
	  */
	long getIntParam( const std::string &prefix );

	/** With this function every parameter object can check out a submitted
	  * string parameter.
	  */
	std::string getStringParam( const std::string &prefix );

	/** This function is like getIntParam, only with Flags. This function is
	  * usable too with STRING parameters.
	  */
	bool getFlagParam( const std::string &prefix );

public:

	Parameter();

	/** This function passes a reference to the parameter map inside every
	  * object.
	  */
	virtual std::map< std::string, param_t > &Param( void ) { return mapParam; };

	/** This function returns a unique object prefix. Use only lowercase
	  * letters (a-z) and keep this name short as possible.
	  */
	virtual const char * Prefix( void ) const = 0;

	/** Select a short Name for the parameter Object.
	  */
	virtual const char * Name( void ) const { return str_unknown; };

	/** Version up to 8 chars in the format V-.-
	  */
	virtual const char * Version( void ) const { return str_unknown; };

	/** Name of authors of the Object with Email in format "Firstname Lastname &lt;email&gt;"
	  * up to 40 chars.
	  */
	virtual const char * Author( void ) const { return str_unknown; };

	/** A short helptext, or description to this object.
	  */
	virtual const char * Help( void ) const { return str_none; };
	
	/** Virtual deconstructor.
	  */
	virtual ~Parameter();
};

#endif
