/* $Id: ArkDataClass.cpp,v 1.27 2002/10/11 01:10:02 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <Ark/ArkDataClass.h>
#include <Ark/ArkLexer.h>
#include <Ark/ArkSystem.h>


#include <sstream>


namespace Ark
{

static const char *EnumToString (Entry::Type tp)
{
  switch (tp)
  {
    case Entry::STRING:  return "STRING";
    case Entry::INTEGER: return "INTEGER";
    case Entry::SCALAR:  return "SCALAR";
    case Entry::VECTOR2: return "VECTOR2";
    case Entry::VECTOR3: return "VECTOR3";
    case Entry::ENTITY: return "ENTITY";

    default: return NULL;
  }
}

//---------------------------------------------------------------------------
// EntryDef : an EntryDef is a definition for an entry in a DataDef.
// It contains informations about the name of the entry, its type, its
// default value, and so on...
//---------------------------------------------------------------------------

EntryDef::EntryDef (const String &name,
                    const String &desc,
                    const String &type)
{
  m_Name = name;
  m_Desc = desc;
  m_Type = Entry::UNKNOWN;
  m_DefVal = this;

  if (type == "STRING")
    m_Type = Entry::STRING;
  else if (type == "INTEGER")
    m_Type = Entry::INTEGER;
  else if (type == "SCALAR")
    m_Type = Entry::SCALAR;
  else if (type == "VECTOR2")
    m_Type = Entry::VECTOR2;
  else if (type == "VECTOR3")
    m_Type = Entry::VECTOR3;

  m_DefVal.m_Type = m_Type;
}

bool EntryDef::ReadDefault (Lexer &lexer)
{
  return m_DefVal.Read (lexer);
}

/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
void
Entry::Destroy ()
{
  if (!d_void)
    return;

  switch (m_Type)
  {
    case STRING:  delete d_str; break;
    case VECTOR2: delete d_vector2; break;
    case VECTOR3: delete d_vector3; break;
    default: break;
  }

  d_void = NULL;
}

Entry &
Entry::operator = (const Entry &src)
{
  Destroy ();

  m_Type = src.m_Type;
  switch (m_Type)
  {
    case STRING:  assert(src.d_str); d_str = new String (*src.d_str); break;
    case VECTOR2: assert(src.d_vector2); d_vector2 = new Vector2 (*src.d_vector2); break;
    case VECTOR3: assert(src.d_vector3); d_vector3 = new Vector3 (*src.d_vector3); break;
    case SCALAR:  d_scalar = src.d_scalar; break;
    case INTEGER: d_integer = src.d_integer; break;
    default: d_void = NULL;
  }

  return *this;
}

Entry::Entry (const Entry &src)
{
  m_Type = src.m_Type;
  switch (m_Type)
  {
     case STRING:  d_str = new String (*src.d_str); break;
     case VECTOR2: d_vector2 = new Vector2 (*src.d_vector2); break;
     case VECTOR3: d_vector3 = new Vector3 (*src.d_vector3); break;
     case SCALAR:  d_scalar = src.d_scalar; break;
     case INTEGER: d_integer = src.d_integer; break;
     default: d_void = NULL;
  }
}

Entry::Entry (EntryDef *def)
{
  assert (def != NULL);
  m_Type = def->m_Type;
  d_void = NULL;
}

Entry::Entry (Type tp)
{
  m_Type = tp;
  d_void = NULL;
}

Entry::~Entry ()
{
  Destroy();
}

static Vector3 GetVec3 (Lexer *lexer)
{
  scalar sX, sY, sZ;

  lexer->CheckToken ("{");
  lexer->ReadScalar(&sX);
  lexer->CheckToken (",");
  lexer->ReadScalar(&sY);
  lexer->CheckToken (",");
  lexer->ReadScalar(&sZ);
  lexer->CheckToken ("}");

  return Vector3 (sX, sY, sZ);
}

static Vector2 GetVec2 (Lexer *lexer)
{
  scalar sX, sY;

  lexer->CheckToken ("{");
  lexer->ReadScalar(&sX);
  lexer->CheckToken (",");
  lexer->ReadScalar(&sY);
  lexer->CheckToken ("}");

  return Vector2 (sX, sY);
}

bool
Entry::Read (Lexer &lexer)
{
  Destroy();

  if (m_Type == STRING)
  {
    String str = lexer.GetToken (Lexer::STRING);

    bool error = str.empty();
       
    d_str = new String( UnquoteString(str) );

    return !error;
  }
  else if (m_Type == VECTOR3)
  {
    d_vector3 = new Vector3 (GetVec3 (&lexer));
    return true;
  }
  else if (m_Type == VECTOR2)
  {
    d_vector2 = new Vector2 (GetVec2 (&lexer));
    return true;
  }
  else if (m_Type == SCALAR)
  {
    return lexer.ReadScalar(&d_scalar);
  }
  else if (m_Type == INTEGER)
  {
    return lexer.ReadInteger(&d_integer);
  }

  return false;
}

String
Entry::ToString () const
{
    std::ostringstream os;

  if (m_Type == STRING)
    return *d_str;
  else if (m_Type == VECTOR3)
    os << "{" << d_vector3->X << ", " << d_vector3->Y << ", " << d_vector3->Z << "}";
  else if (m_Type == VECTOR2)
    os << "{" << d_vector2->X << ", " << d_vector2->Y << "}";
  else if (m_Type == SCALAR)
    os << d_scalar;
  else if (m_Type == INTEGER)
    os << d_integer;

  return os.str();
}

/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/

DataDef::DataDef ()
{}

DataDef::~DataDef ()
{}

void
DataDef::Dump ()
{
    std::vector<EntryDef>::iterator i;

  for (i = m_Entries.begin(); i != m_Entries.end(); i++)
  {
    Sys()->Log ("  %s %s desc (%s) defaults (%s);\n",
                           EnumToString(i->m_Type),
                           i->m_Name.c_str(),
                           i->m_Desc.c_str(),
                           i->m_DefVal.ToString().c_str());
  }
}

bool
DataDef::Read (Lexer &lexer)
{
  lexer.CheckToken ("{");

  while (true)
  {
    String type = lexer.GetToken();

    if (type == "}" || type.empty())
      break;

    String name = lexer.GetToken(Lexer::IDENTIFIER);
    if (name.empty())
      break;

    lexer.CheckToken ("{");

    lexer.CheckToken("desc");
    lexer.CheckToken ("=");
    String desc = UnquoteString( lexer.GetToken(Lexer::STRING) );
    String next;

    // For multiple line strings
    if (! desc.empty() )
    {
      while (true)
      {
        next = lexer.GetToken();

        if ((! next.empty() ) && ( next[0] == '"' ))
	{
          desc += UnquoteString( next );
	}
        else
          break;
	
      }
    }

    if (next != ";")
      break;

    EntryDef edef = EntryDef (name, desc, type);

    lexer.CheckToken("default");
    lexer.CheckToken ("=");
    if (!edef.ReadDefault (lexer))
      break;
    lexer.CheckToken (";");
    lexer.CheckToken ("}");

    m_Entries.insert (m_Entries.begin(), edef);
  }

  return true;
}

/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/

ClassDef::ClassDef (const String &name, const String &parent)
{
  m_ParentName = parent;
  m_Parent = NULL;
  m_Name = name;
}

ClassDef::~ClassDef ()
{}

bool
ClassDef::Read (Lexer &lexer)
{
  return DataDef::Read (lexer);
}

EntryDef *
ClassDef::FindEntryDef (const String &name)
{
   ClassDef *cd = this;
   while (cd)
   {
      EntryDef *def = cd->GetEntryDef (name);
      if (def)
	 return def;

      cd = cd->m_Parent;
   }

   return NULL;
}

/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
ClassList::ClassList ()
{}

ClassList::~ClassList ()
{}

ClassDef *
ClassList::Find (const String &name)
{
  for (std::vector<ClassDef>::iterator i = m_Classes.begin();
       i != m_Classes.end();
       ++i)
  {
    if (i->m_Name == name)
      return &*i;
  }

  return NULL;
}

void
ClassList::Dump ()
{
  for (std::vector<ClassDef>::iterator i = m_Classes.begin();
       i != m_Classes.end();
       ++i)
  {
    if (i->m_Parent)
      Sys()->Log ("%s inherits %s\n", i->m_Name.c_str(), i->m_Parent->m_Name.c_str());
    else
      Sys()->Log ("%s\n", i->m_Name.c_str());

    Sys()->Log ("{\n");
    i->Dump ();
    Sys()->Log ("}\n");
  }
}

bool
ClassList::Load (const String &file)
{
  AutoReadStream stream(file);
  std::ifstream& f = stream.Get();

  if (!f.is_open() || f.bad())
     return false;
  
  Lexer lexer (file, f);
  do
  {
     String name = lexer.GetToken(Lexer::IDENTIFIER);
     
     if ( name.empty() )
	break;
     
     String inh = lexer.GetToken();
     String sparent;
     
     if (inh == "inherits")
     {
	sparent = UnquoteString(lexer.GetToken(Lexer::IDENTIFIER));

	if (Find (sparent) == 0)
	{
	    std::ostringstream os;
	    os << "There's no class called '" << sparent << "'";
	    lexer.Error( os.str() );
	}
     }
     else
	lexer.UngetToken ();
     
    ClassDef *cdef = new ClassDef (name, sparent);
    if (!cdef->Read (lexer))
    {
       delete cdef;
       break;
    }
    lexer.CheckToken (";");
    
    m_Classes.insert (m_Classes.end(), *cdef);
  } while (1);
  
  // Resolve inheritance (?) dependencies
  for (std::vector<ClassDef>::iterator i = m_Classes.begin();
       i != m_Classes.end();
       ++i)
  {
     if (! i->m_ParentName.empty() )
	i->m_Parent = Find (i->m_ParentName);
  }
  
  return true;
}

bool
ClassList::LoadTemplates (const String &file)
{
   AutoReadStream stream (file);
   std::ifstream& f = stream.Get();

   if (!f.is_open() || f.eof())
      return false;
   
   Lexer lexer (file, f);
   
   while (true)
   {
      String s = lexer.GetToken();
      
      if (s != "template")
	 break;
      
      s = lexer.GetToken(Lexer::IDENTIFIER);
      if (s.empty())
	 break;
      
      Read (lexer, &m_Templates [s]);
   }
      
   return true;
}

bool
ClassList::Read (Lexer &lexer, std::map<String, Entry> *values)
{
   assert (values != NULL);
   
   String kw = lexer.GetToken();

   if (kw != "implements")
   {
      lexer.UngetToken();
      return false;
   }

   String s = lexer.GetToken (Lexer::IDENTIFIER);
   
   ClassDef *cls = Find (s);
   if (!cls)
   {
       std::ostringstream os;
       os << "Cannot find class called '" << s << "'";
       lexer.Error( os.str() );
       return false;
   }
   
   // This EntryList can use a template (through keyword 'uses'), if so,
   // try to find it in our list
   s = lexer.GetToken();
   if (s == ",")
   {
      lexer.CheckToken ("uses");
      s = lexer.GetToken (Lexer::IDENTIFIER);
      
      std::map<String, EntryList>::iterator i = m_Templates.find(s);
      if (i == m_Templates.end())
      {
	  std::ostringstream os;
	  os << "Cannot find template '" << s << "'";
	  lexer.Error( os.str() );
	  return false;
      }
      
      *values = std::map<String, Entry> (i->second);

      // Now, set the template..
      Entry e (Entry::STRING);
      e.d_str = new String(s);
      values->insert (std::pair<String,Entry>("template", e));
   }
   else
      lexer.UngetToken();
   
   // Fill values with defaults (if not already set by template
   for (ClassDef *cli = cls; cli != NULL; cli = cli->m_Parent)
   {
	  std::vector<EntryDef>::iterator i;
      for (i = cli->m_Entries.begin(); i != cli->m_Entries.end(); i++)
      {
		EntryList::iterator v = values->find (i->m_Name);
	 	// Not in list -> add it
		if (v == values->end())
			values->insert(std::pair<String,Entry>(i->m_Name, i->m_DefVal));
      }
   }
   
   // Now, set the class..
   Entry e (Entry::STRING);
   e.d_str = new String (cls->m_Name);
   values->insert (std::pair<String,Entry>("class", e));
   
   // Read data
   lexer.CheckToken ("{");

   while (true)
   {
    String name = lexer.GetToken();
    
    if (name == "}" || name.empty() )
      break;
    
    if (!lexer.CheckToken("="))
       return false;
    
    EntryList::iterator i = values->find (name);
    if (i == values->end())
    {
	std::ostringstream os;
	os << "Cannot find definition for entry '" << name << "'";
	lexer.Error( os.str() );
	return false;
    }
    else if (i->second.Read (lexer) == false)
    {
	std::ostringstream os;
	os << "Cannot read value for entry '" << name << "'";
	lexer.Error( os.str() );
	return false;
    }
    
    if (!lexer.CheckToken(";"))
       return false;
   }
   
   return true;
}

}
