/* $Id: ArkEntity.h,v 1.43 2003/02/14 05:21:18 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_ENTITY_H
#define ARK_ENTITY_H

#include <list>
#include <vector>

#include <Ark/ArkMath.h>
#include <Ark/ArkCache.h>
#include <Ark/ArkModelState.h>
#include <Ark/ArkEntityHelp.h>

namespace Ark
{

   class World;
   
   typedef int EntityID;
      
   /// Forward declaration for vector and iterator typedefs
   class Entity;

   /// List of Pointers to Entity objects
   typedef std::vector< Entity* > EntityList;

   /// Iterator to an EntityList
   typedef EntityList::iterator EntityLI;

   /// Const iterator to an EntityList
   typedef EntityList::const_iterator EntityLCI;

   /// Entity collision list
   typedef std::list<EntityCollision> EntityCollisionLinkedList;
   typedef EntityCollisionLinkedList::iterator EntityCollisionListIterator;

   //  ========================================================================
   /// An entity (= any object) in the world
   //  ========================================================================
   
   class ARK_DLL_API Entity
   {
	 /// World needs to be able to set the entity identifier.
	 friend class World;

	 /// World updater needs to access entity attachments.
	 friend class WorldUpdater;
 
	 friend class Path;

	 /// FIXME: try to avoid reference to implementation code here.
	 friend class HeightField;
	 
      public:
	 /** Used when sending entities. This is a bitmask of what has
	  * changed since the last update :
	  * \arg \c NAME :       the name of the entity.
	  * \arg \c POSITION :   the position of the entity
	  * \arg \c PATH :       the control points of the curve defining the
	  *                      trajectory of the entity have changed, or
	  *                      there is no more control points, so a new
	  *                      trajectory has been computed.
	  * \arg \c PSYS :       The particle systems bound to this entity have
	  *                      changed.
	  * \arg \c MODEL :      the model for this entity is not the same as
	  *                      it was. Of course if the models changes, the
	  *                      modelstate changes too.
	  * \arg \c MODELSTATE : the current animation state of the model.
	  *
	  *
	  * \arg \c GOAL :       this one is a bit special since it is used by
	  *                      the client to know when the goal has to be
	  *                      sent to the server.
	  */
	 enum
	 {
	    NAME       = (1 << 0),
	    POSITION   = (1 << 1),
	    PATH       = (1 << 2),
	    MODELSTATE = (1 << 3),
	    MODEL      = (1 << 4) | MODELSTATE,
	    PSYS       = (1 << 5),
	    GOAL       = (1 << 6),
	    EVERYTHING = NAME | POSITION | PATH | MODEL | MODELSTATE | PSYS
	 };

	 /** Flags.
	  * \arg \c  JUSTCREATED is set when the entity has been created but
	  *          not added to the world yet.
	  * \arg \c  AIDRIVEN is set when the entity is controlled by AI.
	  * \arg \c  VISIBLE should be set if the entity can be seen;
	  * \arg \c  COLLISION should be set if it can hit other objects in
	  *                    the world ;
	  * \arg \c  GRAVITY should be set if this entity reacts to gravity ;
	  * \arg \c  INWORLD is set if the entity has been added to the world ;
	  * \arg \c  DEAD is set on death.
	  * \arg \c  STATIC is set if the entity is static (ie scenery).
	  */
	 enum
	 {
	    JUSTCREATED = (1 << 0),
	    AIDRIVEN    = (1 << 1),
	    VISIBLE     = (1 << 1),
	    COLLISION   = (1 << 2),
	    GRAVITY     = (1 << 3),
	    PATHFINDING = (1 << 4),
	    INWORLD     = (1 << 5),
	    DEAD        = (1 << 6),
	    STATIC      = (1 << 7),
	 };

      protected:
	 /// Destroy the given entity
	 virtual ~Entity ();
	 
      public:
	 /**
	  * Create a new entity belonging to 'world'. It should then be added
	  * to the world, when every properties have been set (such as model,
	  * name, pos) using World::Add().
	  */
	 Entity (World *world);
	 
	 /**
	  * Read all informations avalaible from the given file
	  * It can be used to restore the data previously written in a file,
	  * or to receive data from a socket.
	  */
	 virtual void Read  (Stream &stream);
	 
	 /**
	  * Write some informations concerning the entity in the given file.
	  * (an information is sent only if it as changed since the last time
	  * ClearChanged() was called, or if all is true).
	  */
	 virtual void Write (WriteStream &stream, bool all = false);
   
	 /**
	  * Does everything needed to update this entity, after an elapsed
	  * time of "dtime". Most of time it just moves the entity according
	  * to the path/trajectory sent by the server, and updates the
	  * particle systems.
	  */
	 virtual void Update (scalar dt);
	
	 /// Clear the changed bitfield.
         void ClearChanged() {m_Changed=0;}

	 bool HasChanged() const {return m_Changed!=false;}
	
	 void SetChanged(int fl) { m_Changed |= fl;}

	 // == Setting parameters ===========================================
	 /**
	  * Changes the model (sets the 'model changed' flag, and re-init the
	  * modelstate)
	  */
	 void SetModel (const String &model);
	 
	 /// Set the entity position and set the related flag
	 void SetPosition (const Vector3 &pos);
	 
	 /**
	  * Set the goals this entity wants to reach
	  * \param goal is the position it wants to reach
	  */
	 void SetGoal (const Vector3 &goal);
	 
	 /**
	  * Set the goals this entity wants to reach
	  * \param goal is the entity it wants to reach
	  */
	 void SetGoal (Entity *goal);
	 
	 /**
	  * This function should be called when an entity is not used any more,
	  * instead of destroying it. If the entity belongs to a World, it is
	  * marked  as dead, and will be destroyed at the next world update.
	  * If not, it is  immediatly destroyed.
	  */
	 void SetDead ();

	 // == Geometry ===================================================

	 /// Return the bounding box of the entity, in world space.
	 BBox &GetBBox ()
	    {return m_BBox;}

	 // == Particle systems ===========================================

	 /**
	  * Set then nth particle system for this entity. Have a look at
	  * the documentation of ParticleSys::ParticleSys() to learn what
	  * the different parameters refer to.
	  */
	 void PsysSet (int n,
		       const Ark::String &posvel, 
		       const Ark::String &defs,
		       const Ark::String &part);

	 /**
	  * Stop the nth particle set : this one will be deleted when
	  * there will be no more particles in it.
	  */
	 void PsysStop (int n);
			      
	 /**
	  * Returns true if there is a particle system numbered 'n', 
	  * false otherwise.
	  */
	 bool PsysExists (int n);

      
	 // == Messages ===========================================
	 /** 
	  * Add a message wich will be sent the next time the entity is
	  * updated by a WorldUpdater.
	  *  \param dest is the entity this message will be sent to.
	  *  \param message is the content of the message.
	  *  \param answer_list is a list of possible answers (this can
	  *         be a way for a NPC to ask a question with a limited set
	  *         of answers);
	  */
	 virtual bool AddMessage (Entity *dest, const String &message,
		 std::vector<String> *answer_list = NULL);

	 // == Collisions ===========================================

	 /**
	  * Add a collision to the entity's collision list. This function
	  * is normally called by the world's DetectCollision(). 
	  *   \param with is a pointer to the hit entity.
	  *   \param pair is a description of the collision (bodyparts,
	  *          triangle).
	  *   \param potential tells wether this collision is only a bounding
	  *          box one. such collisions might be ignored by the physic
	  *          or simulation code, but are hints for the AI and the
	  *          pathfinding.
	  */
	 virtual bool AddCollision (Entity *with, const ColPair &pair,
				    bool potential = false);

	 /**
	  * Retrieve a collision and remove if of the collision list. This
	  * function is used to send "onHit" events to the scripting language.
	  *   \param with will be filled with a pointer to the hit entity.
	  *   \param pair will be filled with a description of the collision.
	  *   \return true if there was still a collision in the collision
	  *           list, false otherwise. In this second case, the two
	  *           parameters remains unchanged.
	  */
	 virtual bool PeekCollision (Entity **with, ColPair *pair, 
				     bool *potential);

	 /**
	  * Same as above, except it takes an EntityCollision structure.
	  */
	 virtual bool PeekCollision (EntityCollision *col)
	 {
	    assert (col != NULL);
	    return PeekCollision (&col->m_Collider, &col->m_Pair,
	                          &col->m_Potential);
	 }

	 // == Attachments ================================================

	 /**
	  * Attach this entity to another. If the given entity is nil, then
	  * the entity is simply detached.
	  *  \param ent pointer to the entity this one will be bound to.
	  *  \param attachment name of the attachment point.
	  */
	 void AttachToEntity (Entity *ent, const String &attachment);

	 /**
	  * This function returns true if the entity is attached to another.
	  * It is used by the world update function, to ensure the
	  * transformation of the entity to which this one is attached is
	  * computed before.
	  */
	 bool IsAttached ();
	 
     public:
	 /**
	  * Visual appearance of the entity. Model state (with current
	  * animation, position, etc). 
	  */
	 ModelState m_MState;

	 /// The world this entity belongs to.
	 World *m_World;

	 /**
	  * This can be used by the world to save some more informations
	  * about the entity position in the world (octree or quadtree
	  * node, for example).
	  */
	 void *m_WorldData;

	 /// Input message queue.
	 std::vector <EntityMessage> m_Messages;

      protected:
	 /**
	  * Entity this one is attached to. This can be used to bind a sword
	  * to the hand of a horseman, for example.
	  */
	 Entity *m_AttachmentEnt;

	 /**
	  * Name of the attachment point in the entity to which this one is
	  * bound.
	  */
	 String m_Attachment;

	 /**
	  * Outgoing message queue... This is filled by the AddMessage()
	  * function, and emptied by the Engine when it effectively
	  * send those message...
	  */
	 std::vector <EntityMessage> m_OutMessages;

	 /**
	  * Current collisions with other entity. This is filled by the
	  * AddCollision() function, and emptied by the PeekCollision()
	  * function.
	  */
	 EntityCollisionLinkedList m_Collisions;
	 
	 /**
	  * Contains a bitfield specifying what have changed since the
	  * previous reset.
	  */
	 int8 m_Changed;
	 
	 /// The path the entity is following
	 Path m_Path;

	 /// Entity class. Used by AI functions.
	 String m_Class;

	 /// Particle systems bound to this entity.
	 std::vector<EntityPSys*> m_Particles;

	 /// Next goal...
	 EntityGoal m_Goal;

	 /// Bounding box..
	 BBox m_BBox;

      public:
	 /**
	  * Temporary ID assigned by the server to the entity to be able
	  * to send shorter updates to the client (ie without the name)
	  */
	 EntityID m_ID;

	 /// Entity flags
	 int8 m_Flags;

	 /**
	  * Entity name (this can be ""), which will be displayed on player's
	  * screen and short name, which will be used as an identifier and must
	  * consequently be unique.
	  */
	 String m_Name, m_ShortName;
   };

}

#endif
