/* $Id: ArkLexer.h,v 1.9 2002/10/11 01:10:03 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_LEXER_H
#define ARK_LEXER_H

#include <Ark/Ark.h>
#include <Ark/ArkStream.h>

namespace Ark
{

//  ==========================================================================
/// This is a simple lexical analyzer that will tokenize a text
/// as a C parser would.
//  ==========================================================================

   class ARK_DLL_API Lexer
      {
	protected:
	 /** Skip blanks and comments in the file */
	 int SkipBlanks ();
	 
	 /** Get a character */
	 int GetChar();
	 
	 /** Flag the last char as not read */
	 void UngetChar();
	 
	public:
	 /** Used to specify which token type is wanted to GetToken() :
	  * \arg \c UNKNOW : any
	  * \arg \c STRING : only a string ("test")
	  * \arg \c SYMBOL : only a symbol (math symbol: /+-*,
	  *                  separator: :;,., etc.)
	  * \arg \c IDENTIFIER : only an identifier (like a variable or
	  *                  function name).
	  * \arg \c CONSTANT : only a numerical constant (float or integer).
	  */
	 enum Type
	 { UNKNOW = -1, STRING, SYMBOL, IDENTIFIER, CONSTANT };
	 
	public:
	 /** Create a lexer reading in 'file' */
	 Lexer (const String& name, Stream &file);
	 
	 /** Destroy the lexer */
	 ~Lexer ();
	 
	 /** Check that the next token is the one we wait for.
	  * \return false if the given string and the read token aren't the
	  * same
	  */
	 bool CheckToken(const String &waitfor);
	 
	 /** Get a token of the given type
	  * \param type the token type, if it is UNKNOW all token types can
	  * be read; else only a token of the right type will be read (and
	  * a warning will be emitted if a wrong token is read)
	  *
	  * \return a string containing the read token
	  */
	 String GetToken(Type type = UNKNOW);

	 /** Flag the token char as not read */
	 void UngetToken();
	 
	 /** Print the error description, along with a file name and the
	  * current line number.
	  */
	 void Error (const String& desc) const;
	 
	 /// Return the current file name.
	 const String& Name() const 
	    { return m_Name; }
	 
	 /// Return the current line.
	 int Line() const
	    { return m_LineNumber; }

	public:
	 /// Helper for integers
	 bool ReadInteger (int *to);
	 
	 /// Helper for scalars
	 bool ReadScalar (scalar *to);
	 
	 /// Helper for vectors
	 bool ReadScalarVector (scalar *to, size_t n);
	 
	private:
	 String m_Name;
	 Stream& m_File;
	 String m_LastToken;

	 int m_LastChar;
	 String m_CurrentLine;
	 unsigned int m_LinePosition;
	 unsigned int m_LineSize;
	 unsigned int m_LineNumber;

	 bool m_TokenAvail;
	 bool m_CharAvail;
      };
   
}

#endif
