/*
 * atanks - obliterate each other with oversize weapons
 * Copyright (C) 2003  Thomas Hudson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * */


#include "environment.h"
#include "globaldata.h"
#include "player.h"
#include "tank.h"
#include "menu.h"
#include "lineseq.h"

// When defined draws AI 'planning'
//#define AI_PLANNING_DEBUG	1

PLAYER::PLAYER (GLOBALDATA *global, ENVIRONMENT *env)
{
	_global = global;
	_env = env;
	money = 15000;
	score = 0;
	played = 0;
	won = 0;
	tank = NULL;
	_turnStage = 0;
	selected = FALSE;

	_targetMatrix = new double[global->screenWidth];

	nm[0] = 99;
	for (int count = 1; count < WEAPONS; count++)
		nm[count] = 0;

	for (int count = 0; count < ITEMS; count++)
		ni[count] = 0;

	strncpy (_name, "New", 11);
	type = HUMAN_PLAYER;

	// 10% of time set to perplay weapon preferences
	preftype = (rand () % 10)?ALWAYS_PREF:PERPLAY_PREF;
	generatePreferences ();
	vengeful = rand () % 100;
	vengeanceThreshold = (double)(rand () % 1000) / 1000.0;
	revenge = NULL;
	selfPreservation = (double)(rand () % 3000) / 1000;
	painSensitivity = (double)(rand () % 3000) / 1000;
	defensive = (double)((rand () % 10000) - 5000) / 5000.0;

	color = rand () % WHITE;
	typeText[0] = "Human";
	typeText[1] = "Useless";
	typeText[2] = "Guesser";
	typeText[3] = "Range Finder";
	typeText[4] = "Targetter";
	typeText[5] = "Deadly";
	preftypeText[0] = "Per Game";
	preftypeText[1] = "Only Once";
	initMenuDesc ();
}

PLAYER::PLAYER (GLOBALDATA *global, ENVIRONMENT *env, FILE *file)
{
	_global = global;
	_env = env;
	money = 15000;
	score = 0;
	tank = NULL;
	_turnStage = 0;
	selected = FALSE;

	_targetMatrix = new double[global->screenWidth];

	nm[0] = 99;
	for (int count = 1; count < WEAPONS; count++)
		nm[count] = 0;

	for (int count = 0; count < ITEMS; count++)
		ni[count] = 0;

	loadFromFile (file);
	type = (int)type;

	typeText[0] = "Human";
	typeText[1] = "Useless";
	typeText[2] = "Guesser";
	typeText[3] = "Range Finder";
	typeText[4] = "Targetter";
	typeText[5] = "Deadly";
	preftypeText[0] = "Per Game";
	preftypeText[1] = "Only Once";
	initMenuDesc ();
}

void PLAYER::setComputerValues ()
{
	switch ((int)type) {
		case HUMAN_PLAYER:
			break;
		case USELESS_PLAYER:
			rangeFindAttempts = 1;
			retargetAttempts = 0;
			focusRate = 2.0;
			errorMultiplier = 2.0;
			break;
		case GUESSER_PLAYER:
			rangeFindAttempts = 1;
			retargetAttempts = 0;
			focusRate = 0.6;
			errorMultiplier = 1.0;
			break;
		case RANGEFINDER_PLAYER:
			rangeFindAttempts = 10;
			retargetAttempts = 2;
			focusRate = 0.1;
			errorMultiplier = 0.3;
			break;
		case TARGETTER_PLAYER:
			rangeFindAttempts = 20;
			retargetAttempts = 5;
			focusRate = 1.0;
			errorMultiplier = 0.1;
			break;
		case DEADLY_PLAYER:
			rangeFindAttempts = 40;
			retargetAttempts = 10;
			focusRate = 1.0;
			errorMultiplier = 0.02;
			break;
		default:
			rangeFindAttempts = 0;
			retargetAttempts = 0;
			focusRate = 0;
			errorMultiplier = 1.0;
			break;
	}
}

int displayPlayerName (GLOBALDATA *global, ENVIRONMENT *env, int x, int y, void *data);
void PLAYER::initMenuDesc ()
{
	int destroyPlayer (GLOBALDATA *global, ENVIRONMENT *env, void *data);
	menudesc = new MENUDESC;
	if (!menudesc) {
		perror ("player.cc: Failed allocating memory for menudesc in PLAYER::initMenuDesc");
		exit (1);
	}
	menudesc->title = _name;

	// Name,Color
	menudesc->numEntries = 7;
	menudesc->okayButton = TRUE;
	menudesc->quitButton = FALSE;

	menuopts = new MENUENTRY[menudesc->numEntries];
	if (!menuopts) {
		perror ("player.cc: Failed allocating memory for menuopts in PLAYER::initMenuDesc");
		exit (1);
	}

	// Player name
	menuopts[0].name = "Name";
	menuopts[0].displayFunc = NULL;
	menuopts[0].color = color;
	menuopts[0].value = (double*)_name;
	menuopts[0].specialOpts = NULL;
	menuopts[0].type = OPTION_TEXTTYPE;
	menuopts[0].viewonly = FALSE;
	menuopts[0].x = _global->halfWidth - 3;
	menuopts[0].y = _global->halfHeight - 68;

	// Player colour
	menuopts[1].name = "Colour";
	menuopts[1].displayFunc = NULL;
	menuopts[1].color = WHITE;
	menuopts[1].value = (double*)&color;
	menuopts[1].specialOpts = NULL;
	menuopts[1].type = OPTION_COLORTYPE;
	menuopts[1].viewonly = FALSE;
	menuopts[1].x = _global->halfWidth - 3;
	menuopts[1].y = _global->halfHeight - 48;

	// Player type (human, computer)
	menuopts[2].name = "Type";
	menuopts[2].displayFunc = NULL;
	menuopts[2].color = WHITE;
	menuopts[2].value = (double*)&type;
	menuopts[2].min = 0;
	menuopts[2].max = LAST_PLAYER_TYPE - 1;
	menuopts[2].increment = 1;
	menuopts[2].defaultv = 0;
	menuopts[2].format = "%s";
	menuopts[2].specialOpts = typeText;
	menuopts[2].type = OPTION_SPECIALTYPE;
	menuopts[2].viewonly = FALSE;
	menuopts[2].x = _global->halfWidth - 3;
	menuopts[2].y = _global->halfHeight - 28;
	
	// Player preftype (human, computer)
	menuopts[3].name = "Generate Pref.";
	menuopts[3].displayFunc = NULL;
	menuopts[3].color = WHITE;
	menuopts[3].value = (double*)&preftype;
	menuopts[3].min = 0;
	menuopts[3].max = ALWAYS_PREF;
	menuopts[3].increment = 1;
	menuopts[3].defaultv = 0;
	menuopts[3].format = "%s";
	menuopts[3].specialOpts = preftypeText;
	menuopts[3].type = OPTION_SPECIALTYPE;
	menuopts[3].viewonly = FALSE;
	menuopts[3].x = _global->halfWidth - 3;
	menuopts[3].y = _global->halfHeight - 8;

	menuopts[4].name = "Played";
	menuopts[4].displayFunc = NULL;
	menuopts[4].color = WHITE;
	menuopts[4].value = (double*)&played;
	menuopts[4].format = "%.0f";
	menuopts[4].specialOpts = NULL;
	menuopts[4].type = OPTION_DOUBLETYPE;
	menuopts[4].viewonly = TRUE;
	menuopts[4].x = _global->halfWidth - 3;
	menuopts[4].y = _global->halfHeight + 12;

	menuopts[5].name = "Won";
	menuopts[5].displayFunc = NULL;
	menuopts[5].color = WHITE;
	menuopts[5].value = (double*)&won;
	menuopts[5].format = "%.0f";
	menuopts[5].specialOpts = NULL;
	menuopts[5].type = OPTION_DOUBLETYPE;
	menuopts[5].viewonly = TRUE;
	menuopts[5].x = _global->halfWidth - 3;
	menuopts[5].y = _global->halfHeight + 32;

	menuopts[6].name = "Delete This Player";
	menuopts[6].displayFunc = NULL;
	menuopts[6].color = WHITE;
	menuopts[6].value = (double*)destroyPlayer;
	menuopts[6].data = (void*)this;
	menuopts[6].type = OPTION_ACTIONTYPE;
	menuopts[6].viewonly = FALSE;
	menuopts[6].x = _global->halfWidth - 3;
	menuopts[6].y = _global->halfHeight + 100;

	menudesc->entries = menuopts;
}

PLAYER::~PLAYER ()
{
	if (tank)
		delete tank;

	delete menuopts;
	delete menudesc;
}

// Following are config file definitions
#define	GFILE_LATEST_VERSION	9

#define GFILE_NAME		_name, sizeof (char), 11, file
#define	GFILE_WEAPONPREFERENCE1	_weaponPreference, sizeof (int), 51, file
#define	GFILE_WEAPONPREFERENCE2	_weaponPreference, sizeof (int), 54, file
#define	GFILE_WEAPONPREFERENCE3	_weaponPreference, sizeof (int), 58, file
#define	GFILE_WEAPONPREFERENCE4	_weaponPreference, sizeof (int), 60, file
#define	GFILE_WEAPONPREFERENCE5	_weaponPreference, sizeof (int), 62, file
#define	GFILE_TYPE		&type, sizeof (double), 1, file
#define	GFILE_VENGEFUL		&vengeful, sizeof (int), 1, file
#define	GFILE_VENGEANCETHRESHOLD &vengeanceThreshold, sizeof (double), 1, file
#define	GFILE_COLOR		&color, sizeof (int), 1, file
#define	GFILE_COLOR2		&color2, sizeof (int), 1, file
#define	GFILE_PLAYED		&played, sizeof (double), 1, file
#define	GFILE_WON		&won, sizeof (double), 1, file
#define GFILE_PREFTYPE		&preftype, sizeof (double), 1, file
#define	GFILE_SELFPRESERVATION	&selfPreservation, sizeof (double), 1, file
#define	GFILE_PAINSENSITIVITY	&painSensitivity, sizeof (double), 1, file
#define	GFILE_DEFENSIVE		&defensive, sizeof (double), 1, file

#define	GFILE_VERSION_1(a)	a (GFILE_NAME);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_WEAPONPREFERENCE1);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);

#define	GFILE_VERSION_2(a)	a (GFILE_NAME);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_COLOR2);\
				a (GFILE_WEAPONPREFERENCE2);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);

#define	GFILE_VERSION_3(a)	a (GFILE_NAME);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_COLOR2);\
				a (GFILE_WEAPONPREFERENCE3);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);

#define	GFILE_VERSION_4(a)	a (GFILE_NAME);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_COLOR2);\
				a (GFILE_WEAPONPREFERENCE4);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);

#define	GFILE_VERSION_5(a)	a (GFILE_NAME);\
				a (GFILE_VENGEFUL);\
				a (GFILE_VENGEANCETHRESHOLD);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_COLOR2);\
				a (GFILE_WEAPONPREFERENCE4);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);

#define	GFILE_VERSION_6(a)	a (GFILE_NAME);\
				a (GFILE_VENGEFUL);\
				a (GFILE_VENGEANCETHRESHOLD);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_COLOR2);\
				a (GFILE_WEAPONPREFERENCE4);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);\
				a (GFILE_PREFTYPE);

#define	GFILE_VERSION_7(a)	a (GFILE_NAME);\
				a (GFILE_VENGEFUL);\
				a (GFILE_VENGEANCETHRESHOLD);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_COLOR2);\
				a (GFILE_WEAPONPREFERENCE4);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);\
				a (GFILE_PREFTYPE);\
				a (GFILE_SELFPRESERVATION);\
				a (GFILE_PAINSENSITIVITY);

#define	GFILE_VERSION_8(a)	a (GFILE_NAME);\
				a (GFILE_VENGEFUL);\
				a (GFILE_VENGEANCETHRESHOLD);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_COLOR2);\
				a (GFILE_WEAPONPREFERENCE5);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);\
				a (GFILE_PREFTYPE);\
				a (GFILE_SELFPRESERVATION);\
				a (GFILE_PAINSENSITIVITY);

#define	GFILE_VERSION_9(a)	a (GFILE_NAME);\
				a (GFILE_VENGEFUL);\
				a (GFILE_VENGEANCETHRESHOLD);\
				a (GFILE_TYPE);\
				a (GFILE_COLOR);\
				a (GFILE_COLOR2);\
				a (GFILE_WEAPONPREFERENCE5);\
				a (GFILE_PLAYED);\
				a (GFILE_WON);\
				a (GFILE_PREFTYPE);\
				a (GFILE_SELFPRESERVATION);\
				a (GFILE_PAINSENSITIVITY);\
				a (GFILE_DEFENSIVE);

#define	GFILE_WRITE_LATEST	GFILE_VERSION_9(fwrite)
#define	GFILE_READ_VERSION_1	GFILE_VERSION_1(fread)
#define	GFILE_READ_VERSION_2	GFILE_VERSION_2(fread)
#define	GFILE_READ_VERSION_3	GFILE_VERSION_3(fread)
#define	GFILE_READ_VERSION_4	GFILE_VERSION_4(fread)
#define	GFILE_READ_VERSION_5	GFILE_VERSION_5(fread)
#define GFILE_READ_VERSION_6	GFILE_VERSION_6(fread)
#define GFILE_READ_VERSION_7	GFILE_VERSION_7(fread)
#define GFILE_READ_VERSION_8	GFILE_VERSION_8(fread)
#define GFILE_READ_VERSION_9	GFILE_VERSION_9(fread)

int PLAYER::saveToFile (FILE *file)
{
	int version = GFILE_LATEST_VERSION;
	if (!file)
		return (-1);

	fwrite (&version, sizeof (int), 1, file);

	GFILE_WRITE_LATEST

	return (ferror (file));
}

int PLAYER::loadFromFile (FILE *file)
{
	int version;
	if (!file)
		return (-1);

	fread (&version, sizeof (int), 1, file);

	switch (version) {
		case 1:
			GFILE_READ_VERSION_1;
			break;
		case 2:
			GFILE_READ_VERSION_2;
			break;
		case 3:
			GFILE_READ_VERSION_3;
			break;
		case 4:
			GFILE_READ_VERSION_4;
			break;
		case 5:
			GFILE_READ_VERSION_5;
			break;
		case 6:
			GFILE_READ_VERSION_6;
			break;
		case 7:
			GFILE_READ_VERSION_7;
			break;
		case 8:
			GFILE_READ_VERSION_8;
			break;
		case 9:
			GFILE_READ_VERSION_9;
			break;
		default:
			break;
	}
	
	if (version < 6) {
		// Apply change between v 5 and 6 of the player config
		// 10% of time set to perplay weapon preferences
		preftype = (rand () % 10)?ALWAYS_PREF:PERPLAY_PREF;
		generatePreferences ();
	}
	if (version < 7) {
		// Apply change between v 6 and 7
		selfPreservation = (double)(rand () % 3000) / 1000;
		painSensitivity = (double)(rand () % 3000) / 1000;
	}
	if (version < 8) {
		// Apply changes between 7 and 8
		generatePreferences ();
	}
	if (version < 9) {
		// Apply changes between 8 and 9
		defensive = (double)((rand () % 10000) - 5000) / 5000.0;
	}

	return (ferror (file));
}

void PLAYER::exitShop ()
{
	double tmpDM = 0;

	damageMultiplier = 1.0;
	tmpDM += ni[ITEM_INTENSITY_AMP] * item[ITEM_INTENSITY_AMP].vals[0];
	tmpDM += ni[ITEM_VIOLENT_FORCE] * item[ITEM_VIOLENT_FORCE].vals[0];
	if (tmpDM > 0)
		damageMultiplier += pow (tmpDM, 0.6);
}

void PLAYER::newRound ()
{
	setComputerValues ();
	if (!tank) {
		tank = new TANK(_global, _env);
		if (!tank) {
			perror ("player.cc: Failed allocating memory for tank in PLAYER::newRound");
			exit (1);
		}
		tank->player = this;
	}
	tank->newRound ();
}

void PLAYER::initialise ()
{
	long int totalPrefs;
	int rouletteCount;

	nm[0] = 99;
	for (int count = 1; count < WEAPONS; count++)
		nm[count] = 0;

	for (int count = 0; count < ITEMS; count++)
		ni[count] = 0;

	totalPrefs = 0;
	for (int weapCount = 0; weapCount < THINGS; weapCount++)
		totalPrefs += _weaponPreference[weapCount];

	rouletteCount = 0;
	for (int weapCount = 0; weapCount < THINGS; weapCount++) {
		int weapRSpace = (int)((double)_weaponPreference[weapCount] / totalPrefs * NUM_ROULETTE_SLOTS);
		int weapRCount = 0;

		if (weapRSpace < 1)
			weapRSpace = 1;
		while (weapRCount < weapRSpace && rouletteCount + weapRCount < NUM_ROULETTE_SLOTS) {
			_rouletteWheel[rouletteCount + weapRCount] = weapCount;
			weapRCount++;
		}
		rouletteCount += weapRSpace;
	}
	while (rouletteCount < NUM_ROULETTE_SLOTS)
		_rouletteWheel[rouletteCount++] = rand () % THINGS;
}

void PLAYER::generatePreferences ()
{
	for (int weapCount = 0; weapCount < THINGS; weapCount++) {
		_weaponPreference[weapCount] = (rand () % MAX_WEAP_PROBABILITY);
	}
	// Add greater chance of buying parachutes
	_weaponPreference[WEAPONS + ITEM_PARACHUTE] += MAX_WEAP_PROBABILITY;
}

int PLAYER::selectRandomItem ()
{
	return (_rouletteWheel[rand () % NUM_ROULETTE_SLOTS]);
}

void PLAYER::setName (char *name)
{
	strncpy (_name, name, 10);
}

int PLAYER::controlTank (TANK *ctank)
{
	if (key[KEY_F1])
		save_bmp ("scrnshot.bmp", _env->db, NULL);

	if (key_shifts & KB_CTRL_FLAG && ctrlUsedUp) {
		if (key[KEY_LEFT] || key[KEY_RIGHT] ||
			key[KEY_UP] || key[KEY_DOWN])
			ctrlUsedUp = TRUE;
		else
			ctrlUsedUp = FALSE;
	} else {
		ctrlUsedUp = FALSE;
	}

	if (_global->computerPlayersOnly &&
		((int)_global->skipComputerPlay >= SKIP_AUTOPLAY)) {
		if (_env->stage == 3)
			return (-1);
	}

	k = 0;
	if (keypressed () && !fi) {
		k = readkey ();

		if ((_env->stage == 3) &&
			(k >> 8 == KEY_ENTER ||
			 k >> 8 == KEY_ESC ||
			 k >> 8 == KEY_SPACE))
			return (-1);
		if (k >> 8 == KEY_ESC) {
			void clockadd ();
			install_int_ex (clockadd, SECS_TO_TIMER(6000));
			int mm = _env->ingamemenu ();
			install_int_ex (clockadd, BPS_TO_TIMER (100));
			_env->make_update (0, 0, _global->screenWidth, _global->screenHeight);
			_env->make_bgupdate (0, 0, _global->screenWidth, _global->screenHeight);
			
			//Main Menu
			if (mm == 1) 
			{
				_global->command = GLOBAL_COMMAND_MENU;
				return (-1);
			}
			else if (mm == 2)  //Quit game
			{
				_global->command = GLOBAL_COMMAND_QUIT;
				return (-1);
			}
		}
	}
	if ((int)type == HUMAN_PLAYER || !ctank) {
		return (humanControls (ctank));
	} else if (!_env->stage) {
		return (computerControls (ctank));
	}
	return (0);
}

int PLAYER::chooseItemToBuy ()
{
	//loop through all weapon preferences
	for (int count = 0; count < THINGS; count++) {
		int currItem = selectRandomItem ();
		int itemNum = currItem - WEAPONS;
		char cumulative;

		//is item available for purchase or selected first item
		//first item is free
		if ( (!_env->isItemAvailable (currItem)) || (currItem==0))
			continue;

		//determine the likelyhood of purchasing this selection
		//less likely the more of the item is owned		
		//if have zero of item, it is a fifty/fifty chance of purchase	
		int nextMod,curramt,newamt;
		if(currItem < WEAPONS) {
		  curramt = nm[currItem];
		  newamt = weapon[currItem].amt;
		} else {
		  curramt = ni[currItem - WEAPONS];
		  newamt = item[currItem - WEAPONS].amt;
		}

		if ((itemNum >= ITEM_INTENSITY_AMP &&
			itemNum <= ITEM_VIOLENT_FORCE) ||
			(itemNum >= ITEM_ARMOUR &&
			 itemNum <= ITEM_PLASTEEL)) {
			cumulative = TRUE;
		} else {
			cumulative = FALSE;
		}
		  
		if (!cumulative) {
			nextMod = curramt / newamt;
		} else {
			nextMod = 1;
		}
		if (nextMod <= 0) {
			nextMod = 1;
		}

		//if amount available for purchuse is more then current amount
		//already bought 33% chance of purchase
		/*if(newamt > curramt)
			nextMod = 3;
		else if (curramt == 0) //don't have any 50% chance of purchase
			nextMod = 2;
		else //have at one/more purchase full 5% chance of purchase
			nextMod = 20;*/
		
		if (rand () % nextMod)
			continue;
			
		//weapon
		if (currItem < WEAPONS)
		{
			//don't buy if already maxed out
			if(nm[currItem] >=99) continue;
			//purchase the item
			if (money >= weapon[currItem].cost)
			{
			  money -= weapon[currItem].cost;
			  nm[currItem] += weapon[currItem].amt;
			  //don't allow more than 99
			  if(nm[currItem] > 99) nm[currItem] = 99;
			  return currItem;
			}
		}
		else  //item
		{
			//don't buy if already maxed out
			if(ni[currItem-WEAPONS] >=99) continue;
			//purchase the item
			if (money >= item[currItem - WEAPONS].cost) {
				money -= item[currItem - WEAPONS].cost;
				ni[currItem - WEAPONS] += item[currItem- WEAPONS].amt;
				//don't allow more than 99
				if(ni[currItem - WEAPONS] > 99) ni[currItem - WEAPONS] = 99;
				return (currItem);
			}
		}
	}
	return (-1);
}

const char *PLAYER::selectRevengePhrase (double scale)
{
	static const LINESEQ says( DATA_DIR "/revenge.txt", LINESEQ::skip_blanks );

	return says.random( );
}

const char *PLAYER::selectGloatPhrase (double scale)
{
	static const LINESEQ says( DATA_DIR "/gloat.txt", LINESEQ::skip_blanks );

	return says.random( );

}

int PLAYER::traceShellTrajectory (TANK *ctank, int *estimateX, int *estimateY, double targetX, double targetY, double x, double y, double xv, double yv)
{
	TANK *ltank;
	int hitSomething = 0;
	double distance;
	char direction = (char)copysign (1, xv);
	double drag;
	int ticks = 0;
	int maxTicks = 500;
	double startX, startY;

	drag = weapon[ctank->cw].drag;
	if (ni[ITEM_DIMPLEP]) {
		drag *= item[ITEM_DIMPLEP].vals[0];
	} else if (ni[ITEM_SLICKP]) {
		drag *= item[ITEM_SLICKP].vals[0];
	}

	while ((!hitSomething) && (ticks < maxTicks) && (y < MENU ||
		 (getpixel (_env->terrain, (int)x, (int)y) == PINK))) {
		for (int objCount = 0; (ltank = (TANK*)_env->getNextOfClass (TANK_CLASS, &objCount)) && ltank; objCount++) {
			if (ltank->player != this) {
				double xaccel = 0, yaccel = 0;
				ltank->repulse (x, y, &xaccel, &yaccel);
				xv += xaccel;
				yv += yaccel;
			}
		}

		startX = x;
		startY = y;
		if (x + xv < 1 || x + xv > (_global->screenWidth-1)) {
			xv = -xv;	//bounce on the border
			direction = -direction;
		} else {
			// motion - wind affected
			double accel = (_env->wind - xv) / weapon[ctank->cw].mass * drag * _env->viscosity;
			xv += accel;
			x += xv;
		}
		if (y + yv >= _global->screenHeight)
		{
			yv = -yv * 0.5;
			xv *= 0.95;
			if (fabs(xv) + fabs(yv) < 0.8)
				hitSomething = 1;
		}
		y += yv;
		yv += _env->gravity * (100.0 / FRAMES_PER_SECOND);

		if (	(y > _global->screenHeight + 100) ||
			(x > _global->screenWidth + 100) ||
			(x < 0 - 100)) {
			hitSomething = 1;
			break;
		}
		if (checkPixelsBetweenTwoPoints (_global, _env, &startX, &startY, x, y)) {
			x = startX;
			y = startY;
			hitSomething = 1;
			break;
		}
#ifdef	AI_PLANNING_DEBUG
		// Plot trajectories for debugging purposes
		//circlefill (screen, (int)x, (int)y, 2, color);
#endif
		ticks++;
	}
#ifdef	AI_PLANNING_DEBUG
	// Targetting circle for debugging purposes
	circlefill (screen, (int)x, (int)y, 10, color);
	circlefill (screen, (int)targetX, (int)targetY, 20, color);
#endif

	*estimateX = (int)x;
	*estimateY = (int)y;
	distance = vector_length_f ((float)(x - targetX), (float)(y - targetY), 0.0);
	if (ticks > maxTicks)
		_targetPower /= 2;
	if ((targetX - x < 0) && (direction < 0))
		distance = -distance;
	else if ((targetX - x > 0) && (direction > 0))
		distance = -distance;
	return ((int)distance);
}

int PLAYER::rangeFind (TANK *ctank, int *estimateX, int *estimateY, int maxAttempts)
{
	int findCount;
	int spread = weapon[ctank->cw].spread;
	int overshoot = 1;
	int z;

	findCount = 0;
	while (overshoot && (findCount < maxAttempts)) {
		spread = 1;
		for (z = 0; z < spread; z++) {
			double mxv,myv;
			int ca;

			ca = _targetAngle + ((SPREAD * z) - (SPREAD * (spread - 1) / 2));
			mxv = _global->slope[ca][0] * _targetPower * (100.0 / FRAMES_PER_SECOND) / 100;
			myv = _global->slope[ca][1] * _targetPower * (100.0 / FRAMES_PER_SECOND) / 100;

			overshoot = (int)(traceShellTrajectory (
				ctank,
				estimateX, estimateY,
				_targetX, _targetY,
				ctank->x + (_global->slope[ca][0] * GUNLENGTH) - mxv,
				ctank->y + (_global->slope[ca][1] * GUNLENGTH) - myv,
					mxv, myv));
			if (overshoot < 0) {
				_targetPower += (int)((rand () % (abs (overshoot) + 10)) * focusRate);
				if (_targetPower > MAX_POWER)
					_targetPower = 2000;
			} else {
				_targetPower -= (int)((rand () % (abs (overshoot) + 10)) * focusRate);
				if (_targetPower < 0)
					_targetPower = 0;
			}
			if (abs (overshoot) < abs (ctank->smallestOvershoot)) {
				ctank->smallestOvershoot = overshoot;
				ctank->bestPower = _targetPower;
				ctank->bestAngle = _targetAngle;
			}
			findCount++;
		}
	}
	return (overshoot);
}

int PLAYER::calculateDirectAngle (int dx, int dy)
{
	double angle;

	angle = atan2 ((double)dx, (double)dy) / PI * 180;
	angle += (rand () % 40 - 20) * errorMultiplier;

	if (angle < 0)
		angle = angle + 360;
	if (angle < 90)
		angle = 90;
	else if (angle > 270)
		angle = 270;

	return ((int)angle);
}

int PLAYER::calculateAngle (int dx, int dy)
{
	double angle;

	angle = atan2 ((double)dx, (double)dy - abs (dx)) / PI * 180;
	angle += (rand () % 40 - 20) * errorMultiplier;

	if (angle < 0)
		angle = angle + 360;
	if (angle < 90)
		angle = 135;
	else if (angle > 270)
		angle = 225;

	return ((int)angle);
}

int PLAYER::calculatePowerGivenAngle (int dx, int dy, int angle)
{
	double airTime, dxTime;
	double requiredPower;
	double slopeX, slopeY;
	double powerMult;
	double minYv, minV;

	slopeX = _global->slope[angle][1];
	slopeY = _global->slope[angle][0];

	/*
		The greater dx, the longer the missile will be in the air.
		Steeper angle, longer air time.
		The greater dy, the shorter air time.
		Double the gravity, double the power.
		Following wind = less power needed.
	*/


	//maxHeight = 0.5 * (yv * yv) / _env->gravity * (100.0 / FRAMES_PER_SECOND);
	// calculate minimum V to reach height dy
	if (dy > 0)
		minYv = sqrt (dy * _env->gravity * 2 * (100.0 / FRAMES_PER_SECOND));
	else
		minYv = 0;
	if ((slopeY != 0) && (minYv != 0))
		minV = fabs (minYv / slopeY);
	else
		minV = 0;
	if (slopeX != 0) {
		dxTime = (dx/fabs(slopeX));
	} else {
		// entirely down to the elements now
		dxTime = (dx / 0.000001);
	}
	//windAccel = (_env->wind / 20 * 0.2 * _env->viscosity);
	//dxTime = dxTime - (0.5 * windAccel * (dxTime*dxTime));
	if (dx < 0) {
		if (dxTime > 0)
			return (-1); // Won't ever get there
	} else {
		if (dxTime < 0)
			return (-1); // Won't ever get there
	}
	// Low target, less power
	// xdistance proportional to sqrt(dy)
	airTime = fabs(dxTime) + ((dy * slopeY) * _env->gravity * (100.0 / FRAMES_PER_SECOND)) * 2;

	// Less airTime doesn't necessarily mean less power
	// Horizontal firing means more power needed even though
	//   airTime is minimised
	powerMult = 1;
//	powerMult = (slopeX * slopeY);
//	if (powerMult == 0)
//		powerMult = 0.01;
	requiredPower = (sqrt (airTime * _env->gravity * (100.0 / FRAMES_PER_SECOND)) / powerMult) * 100;
	requiredPower += (int)((rand () % 100 - 50) * errorMultiplier);
	return ((int)requiredPower);
}

int PLAYER::adjustAngleGivenClearance (TANK *ctank, int xdist, int ydist, int clearance)
{
	while (!ctank->shootClearance (_targetAngle, clearance)) {
		if (xdist > 0) {
			if (_targetAngle < 270) {
				_targetAngle++;
			} else {
				_targetAngle += (int)((double)(rand () % 40 - 20) / (rangeFindAttempts + 1));
				break;
			}
		} else {
			if (_targetAngle > 90) {
				_targetAngle--;
			} else {
				_targetAngle += (int)((double)(rand () % 40 - 20) / (rangeFindAttempts + 1));
				break;
			}
		}
	}
	if (_targetAngle > 270)
		_targetAngle = 270 - (_targetAngle - 270);
	else if (_targetAngle < 90)
		_targetAngle = 90 + (90 - _targetAngle);

	return (_targetAngle);
}

/*
 * Add the estimated scores for a given tank with a given weapon
 *   to _targetMatrix
 * Take into account how much damage is likely to be done and whether it will
 *   be enough to destroy the tank.
 * Positional correction to account for wind effect on 2nd stage.
 */
int PLAYER::addTankDamageToMatrix (TANK *ctank, TANK *ltank, int weapNum)
{
	int radius = weapon[weapNum].radius;
	double wdamage = calcTotalEffectiveDamage (weapNum) *
					weapon[weapNum].spread;
	int tankX = (int)ltank->x;
	int tankY = (int)ltank->y;
	double windAffect;

	if ((weapNum >= RIOT_BOMB &&
		 weapNum <= HVY_RIOT_BOMB)) {
		// wdamage should probably be renamed to something
		//   more accurate
		wdamage = calcDefenseValue (ctank, ltank) *
			radius * (defensive + 1);
	}

	// Adjust tanks pos according to drag on submunition
	if (weapon[weapNum].submunition >= 0) {
		windAffect = (weapon[weapon[weapNum].submunition].drag * _env->wind / _env->gravity);
	} else {
		windAffect = 0;
	}

	if (ltank->player == this) {
		wdamage = -wdamage * painSensitivity;
		radius = (int)(radius * selfPreservation);
	} else if (ltank->player == revenge) {
		wdamage *= 1 + (double)vengeful / 50.0;
	}

	for (int cx = 0; cx < _global->screenWidth; cx++) {
		// Weapon specific calculations...
		int damage = 0;
		double distance;
		int cy = _env->surface[cx];
		int effectiveTankX = (int)(tankX + ((tankY - cy) * windAffect));
		if (weapNum >= RIOT_CHARGE && weapNum <= RIOT_BLAST) {
			// add points only within range and above ctank
			if (	(cy > ctank->y - (radius / 2)) ||
				(abs (ctank->x - cx) > radius))
				continue;
		} else if (weapNum >= SML_ROLLER && weapNum <= DTH_ROLLER) {
			// Only aim rollers above other tanks
			if (cy > tankY + TANKHEIGHT &&
				abs (ctank->x - cx) > radius)
				continue;
		} else if (weapNum >= SML_LAZER && weapNum <= LRG_LAZER) {
			// Lazer can only be aimed above horizontal
			if (cy > ctank->y)
				continue;
		} else if (weapNum >= SHAPED_CHARGE && weapNum <= CUTTER) {
			double ydist = abs ((cy - tankY) * 20);
			if (ydist > TANKHEIGHT/2)
				ydist -= TANKHEIGHT/2;
			else
				ydist = 0;
			distance = sqrt (pow ((float)cx - effectiveTankX, 2) + pow (ydist, 2));
		}

		if (weapNum < SHAPED_CHARGE || weapNum > CUTTER) {
			distance = sqrt (pow ((float)cx - effectiveTankX, 2) + pow ((float)cy - tankY, 2));
		}

		// Estimate fall damage
		if (cy - tankY > 0 && abs (cx - effectiveTankX) < radius) {
			if (ltank->player == this)
				damage -= radius - abs (cx - effectiveTankX);
			else
				damage += radius - abs (cx - effectiveTankX);
		}

		if (distance <= (radius + TANKHEIGHT/2)) {

			damage += (int) (wdamage * (1.0 - (distance / (radius + TANKHEIGHT/2)) / 2));
		}

		// Getting rid of a tank is tactically advantageous as 
		//   it is one less to shoot you.
		// This could probably use a personality affector
		//   (extent to which a player would rather finish a tank
		//   off than cause up to x amount of damage total).
		if (ltank->l + ltank->sh < damage) {
			if (ltank->player == this)
				damage -= 50;
			else
				damage += 50;
		}

		_targetMatrix[cx] += damage;
	}
	return (0);
}


/*
 * Calculate the damage matrix for the given weapon
 */
void PLAYER::calcDamageMatrix (TANK *ctank, int weapNum)
{
	int objCount = 0;
	TANK *ltank = NULL;

	for (int cx = 0; cx < _global->screenWidth; cx++)
		_targetMatrix[cx] = 0;

	while (((ltank = (TANK*)_env->getNextOfClass (TANK_CLASS, &objCount))) || ltank) {
		if (ltank->l <= 0)
			continue;
		if (weapNum < WEAPONS) {
			addTankDamageToMatrix (ctank, ltank, weapNum);
		} else {
			int itemNum = weapNum - WEAPONS;
			if (itemNum  >= ITEM_VENGEANCE && itemNum <= ITEM_FATAL_FURY) {
				// add sqrt distances for each tank * potential damage
				long int totalEffectiveDamage = calcTotalEffectiveDamage (itemNum);
				_targetMatrix[(int)ctank->x] += sqrt (abs (ctank->x - ltank->x)) * totalEffectiveDamage;

			}
		}
		objCount++;
	}
}

double PLAYER::selectTarget ()
{
	int targetXCoord, targetYCoord;

	return (selectTarget (&targetXCoord, &targetYCoord));
}

double PLAYER::selectTarget (int *targetXCoord, int *targetYCoord)
{
	double highestScore = 0;
	int startAtX = rand () % _global->screenWidth;

	*targetXCoord = 0;
	for (int xCount = 0; xCount < _global->screenWidth; xCount++) {
		int xPos = (startAtX + xCount) % _global->screenWidth;
		if (_targetMatrix[xPos] > highestScore) {
			*targetXCoord = xPos;
			highestScore = _targetMatrix[xPos];
		}
	}
	*targetYCoord = _env->surface[*targetXCoord];

#ifdef	AI_PLANNING_DEBUG
	// Plot damage matrix for debugging purposes
	for (int xCount = 0; xCount < _global->screenWidth; xCount++) {
		vline (screen, xCount, _env->surface[xCount], _env->surface[xCount] + 20, makecol ((int)(_targetMatrix[xCount] / highestScore * 200), (int)(_targetMatrix[xCount] / highestScore * 200), (int)(_targetMatrix[xCount] / highestScore * 200)));
	}
#endif

	return (highestScore);
}

/*
 * calculate defense versus offense.
 * This can be based on turns to kill as a ratio (self to enemy).
 * If the ratio is too far in favour of the enemy, be defensive.
 * The ratio will largely be based on the armour of each tank as
 *   it's the only thing we know for sure. Perhaps later we could
 *   use known characteristics for the given player (fave weapon,
 *   defensiveness, accuracy etc.).
 */
double PLAYER::calcDefenseValue (TANK *ctank, TANK *ltank)
{
	double strengthRatio;
	double armourRatio;

	if (ctank->l > 0 && ltank->l > 0) {
		if (ctank->l + ctank->sh <= ltank->l + ltank->sh) {
			armourRatio = (double)(ctank->l + ctank->sh) /
					(double)(ltank->l + ltank->sh);
		} else {
			armourRatio = -(double)(ltank->l + ltank->sh) /
					(double)(ctank->l + ctank->sh);
		}
	}

	strengthRatio = armourRatio;

	return (strengthRatio);
}

int PLAYER::computerSelectItem (TANK *ctank)
{
	int cw = 0; // Current Weapon

	if (ctank->howBuried () > 135) {
		if (nm[RIOT_BLAST] > 0)
			cw = RIOT_BLAST;
		else if (nm[RIOT_CHARGE] > 0)
			cw = RIOT_CHARGE;
		else if (nm[RIOT_BOMB] > 0)
			cw = RIOT_BOMB;
		else if (nm[HVY_RIOT_BOMB] > 0)
			cw = HVY_RIOT_BOMB;
		else if (ni[ITEM_TELEPORT] > 0)
			cw = WEAPONS + ITEM_TELEPORT;
		else if (nm[SML_LAZER] > 0)
			cw = SML_LAZER;
		else if (nm[MED_LAZER] > 0)
			cw = MED_LAZER;
		else if (nm[LRG_LAZER] > 0)
			cw = LRG_LAZER;
	}
	if (cw == 0) {
		double highestScore = -1;
		int highestScorer = 0;
		int targetXCoord, targetYCoord;
		for (cw = 0; cw < THINGS; cw++) {

			if (cw < WEAPONS) {
				if (!nm[cw]) {
					continue;
				} else {
					double score;
					calcDamageMatrix (ctank, cw);
					score = selectTarget (&targetXCoord,
							&targetYCoord);
					if (weapon[cw].cost > 0)
						score = (score *
							_weaponPreference[cw]) /
							weapon[cw].cost;
					if (score > highestScore) {
						highestScorer = cw;
						highestScore = score;
						_targetX = targetXCoord;
						_targetY = targetYCoord;
					}
				}
			} else {
				double score;
				int itemNum = cw - WEAPONS;

				if ((!ni[itemNum]) ||
					(!item[itemNum].selectable))
					continue;

				if (itemNum == ITEM_TELEPORT) {
					if (!ni[ITEM_PARACHUTE])
						continue;
				}

				if ((itemNum >= ITEM_VENGEANCE) &&
					(itemNum <= ITEM_FATAL_FURY)) {
					if (ctank->l + ctank->sh < 20) {
						calcDamageMatrix (ctank, cw);
						score = selectTarget ();
						score /= item[itemNum].cost;
					}
				}
				if (score > highestScore) {
					highestScorer = cw;
					highestScore = score;
				}
			}
			//break;
		}
		cw = highestScorer;
	}

	if ((cw < WEAPONS && !nm[cw]) || (cw >= THINGS) || (!ni[cw - WEAPONS]))
		cw = 0;

	ctank->cw = cw;
	_global->updateMenu = 1;

	return (cw);
}

int PLAYER::computerControls (TANK *ctank)
{
	// At the most basic: select target, select weapon, aim and fire
	ctank->requireUpdate ();
	if (_turnStage == SELECT_WEAPON) {
		computerSelectItem (ctank);
		_turnStage++;

	} else if (_turnStage == SELECT_TARGET) {
		selectTarget ();
		_turnStage++;

	} else if (_turnStage == CALCULATE_ATTACK) {
		int overshoot;
		int lastOvershoot;
		int estimateX, estimateY;
		int xdist = (int)(_targetX - ctank->x);
		int ydist = (int)(_targetY - ctank->y);

		ctank->smallestOvershoot = _global->screenWidth;
		if (ctank->howBuried () > 135) {
			_targetAngle = 180;
			_targetPower = MAX_POWER / 10;
		} else {
			if (ctank->cw >= SML_LAZER && ctank->cw <= LRG_LAZER) {
				_targetAngle = calculateDirectAngle (xdist, ydist);
			} else {
				_targetAngle = calculateAngle (xdist, ydist);
				_targetAngle = adjustAngleGivenClearance (ctank, xdist, ydist, 150);
				_targetPower = calculatePowerGivenAngle (xdist, ydist, _targetAngle);
				if (rangeFindAttempts > 0)
					overshoot = rangeFind (ctank, &estimateX, &estimateY, rangeFindAttempts);
				else
					overshoot = 0;
				if (abs (overshoot) > weapon[ctank->cw].radius) {
					for (int retargetCount = 0; (retargetCount < retargetAttempts) && (abs (overshoot) > weapon[ctank->cw].radius); retargetCount++) {
						if (overshoot != 0 && (float)abs (overshoot - lastOvershoot) / abs (overshoot) < 0.01) {
							// There could be a
							//   more useful test?
							_targetAngle = (rand () % 180) + 90;
							_targetPower = MAX_POWER / 2;
						} else {
							/*computerSelectItem (ctank);
							selectTarget (&_targetX, &_targetY);
							xdist = (int)(_targetX - ctank->x);
							ydist = (int)(_targetY - ctank->y);
							*/

							_targetAngle = calculateAngle (xdist, ydist);
							_targetAngle = adjustAngleGivenClearance (ctank, xdist, ydist, 150);
							_targetPower = calculatePowerGivenAngle (xdist, ydist, _targetAngle);
						}
						if (rangeFindAttempts > 0)
							overshoot = rangeFind (ctank, &estimateX, &estimateY, rangeFindAttempts);
						else
							overshoot = 0;
						lastOvershoot = overshoot;
					}
				}
				if (abs (ctank->smallestOvershoot) > weapon[ctank->cw].radius * 3) {
					_targetAngle = calculateAngle (xdist, ydist);
					_targetAngle = adjustAngleGivenClearance (ctank, xdist, ydist, (int)(weapon[ctank->cw].radius * 1.5));
					_targetPower = (MAX_POWER / 2);
				} else {
					_targetAngle = ctank->bestAngle;
					_targetPower = ctank->bestPower;
				}
			}
			/*if (_targetAngle == -1) {
				computerSelectItem (ctank);
				selectTarget (&_targetX, &_targetY);
				_turnStage--;
			}*/
		}
		_turnStage++;

	} else if (_turnStage == AIM_WEAPON) {
		if (_targetAngle > ctank->a && ctank->a < 270) {
			// Left
			ctank->a++;
			_global->updateMenu = 1;
		} else if (_targetAngle < ctank->a && ctank->a > 90) {
			// Right
			ctank->a--;
			_global->updateMenu = 1;
		} else if (_targetPower < (ctank->p - 3) && ctank->p > 0) {
			// Reduce power
			ctank->p -= 5;
			_global->updateMenu = 1;
		} else if (_targetPower > (ctank->p + 3) && ctank->p < MAX_POWER) {
			// Increase Power
			ctank->p += 5;
			_global->updateMenu = 1;
		} else {
			_turnStage++;
		}

	} else if (fi) {
		// if (fi) don't do any of the following
	} else if (_turnStage == FIRE_WEAPON) {
		ctank->activateCurrentSelection ();
		_turnStage = 0;
	}
	return (0);
}

int PLAYER::humanControls (TANK *ctank)
{
	if (ctank) {
		if (!_env->mouseclock && mouse_b & 1 && mouse_x >= 250
			&& mouse_x < 378 && mouse_y >= 11 && mouse_y < 19) {
			_global->updateMenu = 1;
			if (ctank->fs) {
				ctank->sht++;
			}
			ctank->fs = 1;
			if (ctank->sht > SHIELDS - 1) {
				ctank->sht = 0;
			}
		}
		if (!_env->mouseclock && mouse_b & 1 && mouse_x >= 250
			&& mouse_x < 378 && mouse_y >= 21
			&& mouse_y < 29 && ctank->player->ni[ctank->sht] > 0 && (ctank->fs || ctank->sh > 0)) {
			_global->updateMenu = 1;
			ctank->ds = ctank->sht;
			ctank->player->ni[ctank->sht]--;
			ctank->sh = (int)item[ctank->sht].vals[SHIELD_ENERGY];
		}
	
		ctank->requireUpdate ();
	}
	
	//Keyboard control
	if (!_env->stage) {
		if (ctank) {
			if (key[KEY_LEFT] && !ctrlUsedUp && ctank->a < 270) {
				ctank->a++;
				_global->updateMenu = 1;
				if (key_shifts & KB_CTRL_FLAG)
					ctrlUsedUp = TRUE;
			}
			if (key[KEY_RIGHT] && !ctrlUsedUp && ctank->a > 90) {
				ctank->a--;
				_global->updateMenu = 1;
				if (key_shifts & KB_CTRL_FLAG)
					ctrlUsedUp = TRUE;
			}
			if (key[KEY_DOWN] && !ctrlUsedUp && ctank->p > 0) {
				ctank->p -= 5;
				_global->updateMenu = 1;
				if (key_shifts & KB_CTRL_FLAG)
					ctrlUsedUp = TRUE;
			}
			if (key[KEY_UP] && !ctrlUsedUp && ctank->p < MAX_POWER) {
				ctank->p += 5;
				_global->updateMenu = 1;
				if (key_shifts & KB_CTRL_FLAG)
					ctrlUsedUp = TRUE;
			}
		}
	}
	if (k && !fi) {
		if (!_env->stage) {
			if (ctank) {
				if (k >> 8 == KEY_N) {
					ctank->a = 180;
					_global->updateMenu = 1;
				}
				if (k >> 8 == KEY_TAB) {
					_global->updateMenu = 1;
					while (1) {
						ctank->cw++;
						if (ctank->cw >= THINGS)
							ctank->cw = 0;
						if (ctank->cw < WEAPONS) {
							if (ctank->player->nm[ctank->cw])
								break;
						} else {
							if (item[ctank->cw - WEAPONS].selectable && ctank->player->ni[ctank->cw - WEAPONS])
								break;

						}
					}
					//calcDamageMatrix (ctank, ctank->cw);
					//selectTarget ();
				}
				if ((k >> 8 == KEY_SPACE) &&
					(((ctank->cw < WEAPONS) && (ctank->player->nm[ctank->cw] > 0)) ||
						((ctank->cw < THINGS) && (ctank->player->ni[ctank->cw - WEAPONS] > 0)))) {
					ctank->activateCurrentSelection ();
				}
			}
		}
		if ((_env->stage == 3) && (k >> 8 == KEY_ENTER || k >> 8 == KEY_ESC || k >> 8 == KEY_SPACE))
			return (-1);
	}
	return (0);
}
