#!/bin/sh
# Remove an existing new Axyl website instance.
# This script will remove an Axyl website which is on the path given
# on the command-line. Some checks will be made to make sure the
# given path is indeed an Axyl website root.
#
# Questions are then asked as to whether database, Apache configs etc.
# should also be removed, in a step-by-step approach.
#
# Paul Waite
#
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
set -e

echo ""
echo "REMOVE AXYL WEBSITE"
echo ""

# Get common funcs and config vars etc.
# Acquire the local configuration..
CONFDIR=/etc/axyl
CONF=${CONFDIR}/axyl.conf
if [ ! -f $CONF ]
then
  echo "Axyl configuration file $CONF not found!"
  exit 2
else
  . $CONF
  if [ ! -d $AXYL_HOME ] ; then
    echo "FATAL: the Axyl root directory '$AXYL_HOME' does not exist."
    exit 6
  fi
  . ${AXYL_HOME}/install/axyl-common.sh
fi

# Get path to website to remove..
WWW_PATH=
if [ "$1" != "" ]
then
  WWW_PATH=$1
else
  echo "Enter path to the Axyl website which is to be REMOVED"
  echo -n "? "
  read ANS
  if [ "$ANS" = "" ]
  then
    echo "Exiting.. no path specified."
    exit 0
  else
    WWW_PATH=$ANS
  fi
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# SANITY CHECK
if [ ! -f ${WWW_PATH}/application.xml ]
then
  echo "ERROR: application.xml not found. No Axyl website found at"
  echo "$WWW_PATH !"
  exit 1
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# INSTALLATION NAME
INST_NAME=`perl -ne 's/[\s]*<definition name="APP_PREFIX">([\w]+)<.*$/$1/ && print;' ${WWW_PATH}/application.xml`
echo ""
echo "Confirm the name of the Axyl website installation to be removed."
echo "(this is the APP_PREFIX as defined in application.xml)"
echo ""
echo -n "Axyl website installation name [${INST_NAME}] :"
read ANS
[ "$ANS" = "" ] && ANS=$INST_NAME
INST_NAME=$ANS
if [ "$INST_NAME" = "" ]
then
  echo "A name is mandatory. Aborting.."
  exit 1
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# DATABASE DETAILS
# Notes: Only copes with a single database defined for the website.
echo ""
DB_TYPE=`grep "add_database" -A 1 ${WWW_PATH}/application.xml | egrep -E -o "\"type\"(.*?)>(.*?)<" | egrep -E -o ">(.*?)<" | tr -d "<>"`
DB_NAME=`grep "add_database" -A 2 ${WWW_PATH}/application.xml | egrep -E -o "\"name\"(.*?)>(.*?)<" | egrep -E -o ">(.*?)<" | tr -d "<>"`
DB_HOST=`grep "\"host\"" ${WWW_PATH}/application.xml | egrep -E -o ">(.*?)<" | tr -d "<>"`
if [ "$DB_TYPE" = "postgres" -a "$DB_HOST" = "" ]
then
  echo -n "Drop Postgres database '${DB_NAME}' Y or N [N] :"
  read ANS
  if [ "$ANS" = "Y" -o "$ANS" = "y" ]
  then
    su postgres -c "dropdb $DB_NAME"
    echo "done."
  fi
else
  if [ "$DB_TYPE" = "postgres" -a "$DB_HOST" != "" ]
  then
    echo "Database '${DB_NAME}' is served remotely from host '${DB_HOST}'"
    echo "You will have to drop this database yourself, on that Host."
  else
    echo -n "The website is served by database '${DB_NAME}' ($DB_TYPE)"
    if [ "$DB_HOST" != "" ]
    then
      echo " on remote host '${DB_HOST}'"
    else
      echo ""
    fi
    echo "If it is to be removed, then you must do this manually."
  fi
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# LUCENE and LUCENE DOMAIN PROPERTIES FILE
if [ -f $AXYL_DATA/lucene/servers/axyl/etc/application/${INST_NAME}.properties ]
then
  echo ""
  echo -n "Remove Lucene domain properties file? Y or N [N] :"
  read ANS
  if [ "$ANS" = "Y" -o "$ANS" = "y" ]
  then
    [ -x /etc/init.d/axyl-lucene ] && /etc/init.d/axyl-lucene stop
    rm -f $AXYL_DATA/lucene/servers/axyl/etc/application/${INST_NAME}.properties
    [ -x /etc/init.d/axyl-lucene ] && /etc/init.d/axyl-lucene start
    echo "done."
  fi
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# APACHE CONFIGURATION
if [ -f ${APACHE_CONFDIR}/conf.d/${INST_NAME}.* ]
then
  echo ""
  echo -n "Deleting apache virtual host..."
  rm -f ${APACHE_CONFDIR}/conf.d/${INST_NAME}.*
  /etc/init.d/apache restart
  echo "done."
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# The Website docroot itself..
echo ""
echo -n "Deleteing the whole ${INST_NAME} hirearchy..."
rm -rf ${WWW_PATH}
echo "done."

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# Update Axyl websites database
echo ""
echo "updating Axyl website database.."
${AXYL_HOME}/install/unregister-axyl-website.sh $INST_NAME

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# FINISHED
echo ""
echo "Axyl website $INST_NAME has been removed."

# END