/* cmd-commit.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "libfsutils/file-contents.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/patch-logs.h"
#include "libarch/archive.h"
#include "libarch/editor.h"
#include "libarch/proj-tree-lint.h"
#include "libarch/commit.h"
#include "commands/cmd.h"
#include "commands/commit.h"
#include "commands/cmdutils.h"
#include "commands/version.h"


/* __STDC__ prototypes for static functions */
static void arch_make_edit_log (t_uchar *tree_root, t_uchar *archive, t_uchar *version);

static t_uchar * last_tree_level (t_uchar * tree_root, t_uchar * archive, t_uchar *version);


static t_uchar * usage = N_("[options] [[archive]/version] [-- file ...]");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_archive, "A", "archive", 1, \
      N_("Override `my-default-archive'")) \
  OP (opt_dir, "d", "dir DIR", 1, \
      N_("cd to DIR first")) \
  OP (opt_log, "l", "log FILE", 1, \
      N_("commit with log file FILE")) \
  OP (opt_summary, "s", "summary TEXT", 1, \
      N_("log with summary TEXT plus log-for-merge output") ) \
  OP (opt_log_msg, "L", "log-message TEXT", 1, \
      N_("log with TEXT plus log-for-merge output")) \
  OP (opt_strict, 0, "strict", 0, \
      N_("strict tree-lint")) \
  OP (opt_seal, 0, "seal", 0, \
      N_("create a version-0 revision")) \
  OP (opt_fix, 0, "fix", 0, \
      N_("create a versionfix revision")) \
  OP (opt_out_of_date, 0, "out-of-date-ok", 0, \
      N_("commit even if out of date")) \
  OP (opt_just_commit, 0, "just-commit", 0, \
      N_("only commit a simple changeset, do not add other files (such as ancestry," \
	 "cached revisions.")) \
  OP (opt_file_list, 0, "file-list FILE", 1, \
      N_("commit only changes to files listed in FILE")) \
  OP (opt_unescaped, 0, "unescaped", 0, \
      N_("show filenames in unescaped form"))


t_uchar arch_cmd_commit_help[] =
  ("archive a changeset-based revision\n"

   "Archive a revision of the project tree containing DIR (or the current\n"
   "directory)\n"
   "\n"
   "If there are any automated ChangeLog files, update them before\n"
   "computing the patch set.\n"
   "\n"
   "If --log-message is specified without --summary, then TEXT is used both\n"
   "as the summary and the first line of the log body.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_commit (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * default_archive = 0;
  t_uchar * dir = 0;
  t_uchar * log_file = 0;
  t_uchar * log_text = 0;
  t_uchar * summary = 0;
  t_uchar * not_writable_archive = 0;
  int strict = 0;
  int seal = 0;
  int fix = 0;
  int out_of_date_ok = 0;
  t_uchar * file_list_file = 0;
  int opt_end_with_double_dash = 0;
  int escape_classes = arch_escape_classes;
  int result = 0;
  int just_commit = 0;

  dir = str_save (0, ".");

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_commit_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;

      if (o == opt_double_dash)
        {
          opt_end_with_double_dash = 1;
          break;
        }

      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_archive:
          {
            default_archive = str_save (0, option->arg_string);
            break;
          }

        case opt_dir:
          {
            dir = str_save (0, option->arg_string);
            break;
          }

        case opt_log:
          {
            lim_free (0, log_file);
            log_file = str_save (0, option->arg_string);
            lim_free (0, log_text);
            log_text = 0;
            break;
          }

        case opt_log_msg:
          {
            lim_free (0, log_text);
            log_text = str_save (0, option->arg_string);
            lim_free (0, log_file);
            log_file = 0;
            break;
          }

        case opt_summary:
          {
            lim_free (0, summary);
            summary = str_save (0, option->arg_string);
            lim_free (0, log_file);
            log_file = 0;
            break;
          }

        case opt_strict:
          {
            strict = 1;
            break;
          }

        case opt_seal:
          {
            seal = 1;
            break;
          }

        case opt_fix:
          {
            fix = 1;
            break;
          }

	case opt_just_commit:
	  {
	    just_commit = 1;
	    break;
	  }

        case opt_out_of_date:
          {
            out_of_date_ok = 1;
            break;
          }

        case opt_file_list:
          {
            lim_free (0, file_list_file);
            file_list_file = str_save (0, option->arg_string);
            break;
          }

	case opt_unescaped:
	  {
	    escape_classes = 0;
	    break;
	  }
        }
    }

  {
    t_uchar * log = 0;
    t_uchar * vsnspec = 0;
    rel_table file_list = 0;
    t_uchar * tree_root = 0;
    t_uchar * archive = 0;
    t_uchar * version = 0;
    t_uchar * last_level = 0;
    t_uchar * revision = 0;
    struct arch_archive * arch = 0;

    if (default_archive)
      {
        if (!arch_valid_archive_name (default_archive))
          {
            safe_printfmt (2, "%s: invalid archive name (%s)\n",
                           argv[0], default_archive);
            exit (1);
          }
      }

    tree_root = arch_tree_root (0, dir, 1);

    if (!tree_root)
      {
        safe_printfmt (2, "%s: not in project tree (%s)\n", argv[0], dir);
        exit (1);
      }
    
    arch_tree_ensure_no_rejects(tree_root);


    if (argc == 1 || opt_end_with_double_dash)
      /* No args, or file arguments but no version spec.  */
      {
        vsnspec = arch_tree_version (tree_root);
        if (!vsnspec)
          {
            safe_printfmt (2, "%s: project tree has no default version\n  tree: %s\n",
                           argv[0], tree_root);
            exit (1);
          }
      }
    else
      /* Version spec specified.  */
      vsnspec = str_save (0, argv[1]);

    if (file_list_file)
      {
        int in_fd = safe_open (file_list_file, O_RDONLY, 0);
        file_list = rel_read_table (in_fd, 1, argv[0], file_list_file);
        safe_close (in_fd);
      }

    if (argc > 2 || opt_end_with_double_dash)
      /* File arguments.  */
      {
        int argx = opt_end_with_double_dash ? 1 : 2;

        if (!opt_end_with_double_dash && str_cmp (argv[argx], "--") == 0)
          argx++;

        if (argx == argc)
          /* --, but no files specified; should this be an error?  */
          goto usage_error;

        while (argx < argc)
          rel_add_records (&file_list, rel_make_record (argv[argx++], 0), 0);
      }

    if (!arch_valid_package_name (vsnspec, arch_maybe_archive, arch_req_version, 0))
      {
        safe_printfmt (2, "%s: invalid version name -- %s\n",
                       argv[0], vsnspec);
        exit (1);
      }

    archive = arch_parse_package_name (arch_ret_archive, default_archive, vsnspec);
    version = arch_parse_package_name (arch_ret_non_archive, 0, vsnspec);

    if (arch_is_system_package_name (version))
      {
        safe_printfmt (2, "%s: user's can not commit to system versions\n  version: %s\n", argv[0], version);
        exit (2);
      }

    if (log_text || summary)
      {
        if (! summary)
          summary = log_text;
        log = arch_auto_log_message (tree_root, archive, version,
                                     summary, log_text);
      }
    else
      {
        if (!log_file)
          {
            log_file = arch_make_log_file (tree_root, archive, version);
            if (safe_access (log_file, F_OK))
	      arch_make_edit_log (tree_root, archive, version);
          }

        log = file_contents (log_file);
      }

    if (!arch_valid_log_file (log))
      {
        safe_printfmt (2, "%s: invalid log file (%s)\n",
                       argv[0], (log_file ? log_file : log_text));
        exit (1);
      }

    if (!arch_has_patch_log (tree_root, archive, version))
      {
        safe_printfmt (2, "%s: tree has no patch log for version\n    tree: %s\n    version: %s/%s\n (maybe you need to run import?)",
                       argv[0], tree_root, archive, version);
        exit (1);
      }

    {
      rel_table log_entries = 0;

      log_entries = arch_logs (tree_root, archive, version, 0);

      if (!log_entries)
        {
          safe_printfmt (2, "%s: tree has no patch log entries for version\n    tree: %s\n    version: %s/%s (maybe you need to run import?)",
                         argv[0], tree_root, archive, version);
          exit (1);
        }
      rel_free_table (log_entries);
    }


    {
      struct arch_tree_lint_result * lint = 0;
      int status;

      lint = arch_tree_lint (tree_root);
      status = arch_print_tree_lint_report (2, lint, escape_classes);

      if ((status < 0) || (strict && status))
        {
          safe_printfmt (2, "%s: commit aborted\n", argv[0]);
          exit (1);
        }
    }

    arch = arch_archive_connect (archive, 0);

    /*
     * Make sure that the archive we're going to commit to is writable.
     * If not, then print out the error and bail.
     */
    not_writable_archive = arch_archive_not_writable (arch, 0);
    if (not_writable_archive)
      panic(not_writable_archive);


    
    if (out_of_date_ok)
      {
	last_level = arch_archive_last_level (arch, version);
      }
    else
      last_level = last_tree_level (tree_root, archive, version);

    if (!last_level)
      {
        safe_printfmt (2, "%s: version has no revisions -- use revimport instead\n",
                       argv[0]);
        exit (1);
      }

    revision = arch_next_revision (version, last_level, seal, fix, argv[0]);

    if (!out_of_date_ok)
      {
	t_uchar * alt_revision = 0;
	if (!fix)
	{
	  alt_revision = arch_next_revision(version, last_level, !seal, fix, argv[0]);
	}

	if (arch_revision_exists (arch, revision) || (alt_revision && arch_revision_exists(arch, alt_revision)))
          {
            safe_printfmt (2, "%s: tree is not up-to-date (missing latest revision is %s/%s--%s)\n",
                           argv[0], archive, version, arch_archive_last_level (arch, version));
            exit (1);
          }

        lim_free (0, alt_revision);
      }

    if (!arch_commit (1, arch, revision, tree_root, log, file_list, just_commit, escape_classes))
      {
	
	safe_printfmt (1, "* committed %s/%s\n", archive, revision);
	
	if (log_file)
	    safe_unlink (log_file);
      }
    else
	result = 2;

    lim_free (0, vsnspec);
    rel_free_table (file_list);
    lim_free (0, log);
    lim_free (0, tree_root);
    lim_free (0, archive);
    lim_free (0, version);
    lim_free (0, last_level);
    lim_free (0, revision);

  }

  lim_free (0, default_archive);
  lim_free (0, dir);

  return result;
}




static void
arch_make_edit_log (t_uchar *tree_root, t_uchar *archive, t_uchar *version)
{
  int out_fd;
  int ecode;
  t_uchar *log_path = arch_make_log_file (tree_root, archive, version);
  t_uchar *text = arch_auto_log_message (tree_root, archive, version, "", "");
  t_uchar *new_text = 0;

  safe_printfmt (1, "* no log found, creating one automatically\n");
  safe_printfmt (1, "* (Use \"tla make-log\" to create a log file.)\n");
  safe_flush (1);

  out_fd = safe_open (log_path, O_WRONLY | O_CREAT | O_EXCL, 0666);

  safe_printfmt (out_fd, "%s", text);
  safe_close (out_fd);

  ecode = arch_run_editor (log_path);
  if (ecode == -2)
    {
      /* no editor set */
      safe_unlink (log_path);
      exit (2);
    }
  if (ecode != 0)
    {
      safe_printfmt (2, "arch_make_edit_log: editor exited with error code %d\n", ecode);
      exit (2);
    }
  
  new_text = file_contents (log_path);
  if (!str_cmp (text, new_text))
    {
      safe_printfmt (2, "arch_make_edit_log: log file is unmodified, aborting\n");
      safe_unlink (log_path);
      exit (2);
    }
}


static t_uchar * last_tree_level (t_uchar * tree_root, t_uchar * archive, t_uchar *version)
{
  t_uchar * last_level = 0;
  rel_table log_entries = arch_logs (tree_root, archive, version, 0);
  if (log_entries)
    {
      last_level = str_save (0, log_entries[rel_n_records (log_entries) - 1][0]);
    }
  rel_free_table (log_entries);
  return last_level;
}




/* tag: Tom Lord Mon May 26 21:33:18 2003 (revcmt.c)
 */
