/*	$eterna: bozohttpd.c,v 1.70 2003/03/13 11:56:06 mrg Exp $	*/

/*
 * Copyright (c) 1997-2002 Matthew R. Green
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer and
 *    dedication in the documentation and/or other materials provided
 *    with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED
 * AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/* this program is dedicated to the Great God of Processed Cheese */

/*
 * bozohttpd.c:  minimal httpd; provides only these features:
 *	- HTTP/0.9 (by virtue of ..)
 *	- HTTP/1.0
 *	- HTTP/1.1
 *	- CGI/1.1 this will only be provided for "system" scripts
 *	- automatic "missing trailing slash" redirections
 *	- configurable translation of /~user/ to ~user/public_html,
 *	  however, this does not include cgi-bin support
 *	- access lists via libwrap
 *	- virtual hosting
 *	- not that we do not even pretend to understand MIME, but
 *	  rely only on the HTTP specification
 *	- ipv6 support
 *	- automatic `index.html' generation
 *	- configurable server name
 *	- directory index generation
 */

/*
 * requirements for minimal http/1.1 (at least, as documented in
 * <draft-ietf-http-v11-spec-rev-06> which expired may 18, 1999):
 *
 *	- 14.15: content-encoding handling. [1]
 *
 *	- 14.16: content-length handling.  this is only a SHOULD header
 *	  thus we could just not send it ever.  [1]
 *
 *	- 14.17: content-type handling. [1]
 *
 *	- 14.25/28: if-{,un}modified-since handling.  maybe do this, but
 *	  i really don't want to have to parse 3 differnet date formats
 *
 * [1] need to revisit to ensure proper behaviour
 *
 * and the following is a list of features that we do not need
 * to have due to other limits, or are too lazy.  there are more
 * of these than are listed, but these are of particular note,
 * and could perhaps be implemented.
 *
 *	- 3.5/3.6: content/transfer codings.  probably can ignore
 *	  this?  we "SHOULD"n't.  but 4.4 says we should ignore a
 *	  `content-length' header upon reciept of a `transfer-encoding'
 *	  header.
 *
 *	- 5.1.1: request methods.  only MUST support GET and HEAD,
 *	  but there are new ones besides POST that are currently
 *	  supported: OPTIONS PUT DELETE TRACE and CONNECT, plus
 *	  extensions not yet known?
 *
 * 	- 10.1: we can ignore informational status codes
 *
 *	- 10.3.3/10.3.4/10.3.8:  just use '302' codes always.
 *
 *	- 14.1/14.2/14.3/14.27: we do not support Accept: headers..
 *	  just ignore them and send the request anyway.  they are
 *	  only SHOULD.
 *
 *	- 14.5/14.16/14.35: we don't do ranges.  from section 14.35.2
 *	  `A server MAY ignore the Range header'.  but it might be nice.
 *
 *	- 14.9: we aren't a cache.
 *
 *	- 14.15: content-md5 would be nice...
 *	
 *	- 14.24/14.26/14.27: be nice to support this...
 *
 *	- 14.44: not sure about this Vary: header.  ignore it for now.
 */

/*
 * optional features on a compile-time basis.
 */
#if 0
#undef NO_USER_SUPPORT		/* don't support /~user requests */
#undef NO_CGIBIN_SUPPORT	/* don't support cgi-bin requests */
#undef NO_DIRINDEX_SUPPORT	/* don't support directory indexing */
#undef NO_DAEMON_MODE		/* don't support daemon mode */
#undef NO_DYNAMIC_CONTENT	/* don't support dynamic content updates */
#undef DEBUG			/* include debugging support */
#endif

#ifndef INDEX_HTML
#define INDEX_HTML	"index.html"
#endif
#ifndef PUBLIC_HTML
#define PUBLIC_HTML	"public_html"
#endif

/*
 * And so it begins ..
 */

#include <sys/param.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <sys/wait.h>

#include <netinet/in.h>

#include <arpa/inet.h>

#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#ifndef NO_CGIBIN_SUPPORT
#include <paths.h>
#endif /* NO_CGIBIN_SUPPORT */
#ifndef NO_DAEMON_MODE
#include <poll.h>
#endif /* NO_DAEMON_MODE */
#include <pwd.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <time.h>
#include <unistd.h>

#include "queue.h"

#ifndef MAX_WAIT_TIME
#define	MAX_WAIT_TIME	60	/* hang around for 60 seconds max */
#endif

/* headers */
struct headers {
	const char *header;
	const char *value;
	SIMPLEQ_ENTRY(headers)	next;
};

/* http_req */
typedef struct {
	int	method;
#define	HTTP_GET	0x01
#define HTTP_POST	0x02
#define HTTP_HEAD	0x03
#define HTTP_OPTIONS	0x04	/* not supported */
#define HTTP_PUT	0x05	/* not supported */
#define HTTP_DELETE	0x06	/* not supported */
#define HTTP_TRACE	0x07	/* not supported */
#define HTTP_CONNECT	0x08	/* not supported */
	const char *methodstr;
	char	*url;
	const char *proto;
	const char *content_type;
	const char *content_length;
	const char *allow;
	const char *host;		/* HTTP/1.1 Host: */
	const char *remotehost;
	const char *remoteaddr;
	const char *serverport;
	SIMPLEQ_HEAD(, headers)	headers;
	int	nheaders;
} http_req;

/* debug flags */
#define DEBUG_LIGHT	0
#define DEBUG_NORMAL	1
#define DEBUG_FAT	2
#define DEBUG_OBESE	3

/* variables and functions */

#ifndef NO_CGIBIN_SUPPORT
static	char	*cgibin;	/* cgi-bin directory */

#define CGIBIN_PREFIX		"cgi-bin/"
#define CGIBIN_PREFIX_LEN	(sizeof(CGIBIN_PREFIX)-1)

static	void	process_cgi(http_req *);
#endif /* NO_CGIBIN_SUPPORT */


#ifndef NO_USER_SUPPORT
static	int	uflag;		/* allow /~user/ translation */
static	const char *public_html	= PUBLIC_HTML;

static	char	*transform_user(http_req *, int *);
#endif /* NO_USER_SUPPORT */


#ifndef NO_DIRINDEX_SUPPORT
static	int	Xflag;		/* do directory indexing */
static	int	Hflag;		/* hide .* */

static	void	directory_index(http_req *, const char *);
#endif /* NO_DIRINDEX_SUPPORT */


static	int	bflag;		/* background; drop into daemon mode */
static	int	eflag;		/* don't clean environ; -t/-U only */
#ifndef NO_DAEMON_MODE
static	char	*iflag;		/* bind address; default INADDR_ANY */
static	const char *Iflag = "http";/* bind port; default "http" */
static	int	*sock;		/* bound sockets */
static	int	nsock;		/* number of above */

static	void	sigchild(int);	/* SIGCHLD handler */
#endif /* NO_DAEMON_MODE */


#ifdef DEBUG
static	int	dflag = 0;	/* debugging level */

static	void	debug__(int, const char *, ...);
#define debug(x)	debug__ x
#else
#define	debug(x)	
#endif /* DEBUG */

#define	strornull(x)	((x) ? (x) : "<null>")

#ifndef LOG_FTP
#define LOG_FTP LOG_DAEMON
#endif

static	char	*tflag;		/* root directory */
static	char 	*Uflag;		/* user name to switch to */
static	int	nflag;		/* avoid gethostby*() */
static	int	Hflag;		/* avoid gethostby*() */
static	int	Cflag;		/* added a cgi handler, always process_cgi() */
static	char	*myname;	/* my name */
static	char	*slashdir;	/* www slash directory */
static	char	*vpath;		/* virtual directory base */

static	char	date[40];	/* text date string */

static	const char *server_software = "bozohttpd/20030313";
static	const char *index_html = INDEX_HTML;
static	const char http_09[] = "HTTP/0.9";
static	const char http_10[] = "HTTP/1.0";
static	const char http_11[] = "HTTP/1.1";
static	const char text_plain[]	= "text/plain";

static	void	usage(void);
static	void	alarmer(int);
volatile sig_atomic_t	alarmhit;

static	void	parse_request(char *, char **, char **, char **);
static	http_req *read_request(void);
static	void	process_request(http_req *);
static	char	*transform_request(http_req *, int *);
static	void	handle_redirect(http_req *, const char *);
static	void	print_header(http_req *, struct stat *, const char *,
			     const char *);

static	void	check_virtual(http_req *);
static	void	fix_url_percent(http_req *);
static	void	process_method(http_req *, const char *);
static	void	process_proto(http_req *, const char *);
static	struct content_map *match_content_map(const char *, int);
static	struct content_map *get_content_map(const char *);
static	const char *content_type(http_req *, const char *);
static	const char *content_encoding(http_req *, const char *);
static	const char *content_cgihandler(http_req *, const char *);
static	void	escape_html(http_req *);
#ifndef NO_DYNAMIC_CONTENT
#ifndef NO_CGIBIN_SUPPORT
static	void	add_content_map_cgi(char *arg);
#endif
static	void	add_content_map_mime(char *arg);
#endif

static	void	set_http_date(void);

static	void	warning(const char *, ...);
static	void	error(int, const char *, ...);

static	void	http_error(int, http_req *, const char *);
static	const char *http_errors_short(int);
static	const char *http_errors_long(int);

static	char	*dgetln(int, ssize_t *);
static	char	*strnsep(char **, const char *, ssize_t *);

static	void	*bozomalloc(size_t);
static	void	*bozorealloc(void *, size_t);
static	char	*bozostrdup(const char *);

	char	*progname;

	int	main(int, char *[]);

static void
usage()
{

	warning("usage: %s [options] slashdir [myname]", progname);
	warning("options:");
#ifdef DEBUG
	warning("   -d\t\t\tenable debug support");
#endif
#ifndef NO_USER_SUPPORT
	warning("   -u\t\t\tenable ~user/public_html support");
	warning("   -p dir\t\tchange `public_html' directory name]");
#endif
#ifndef NO_DYNAMIC_CONTENT
	warning("   -M \"arg t c c11\"\tadd this mime extenstion");
#endif
#ifndef NO_CGIBIN_SUPPORT
#ifndef NO_DYNAMIC_CONTENT
	warning("   -C \"arg prog\"\tadd this CGI handler");
#endif
	warning("   -c cgibin\t\tenable cgi-bin support in this directory");
#endif
#ifndef NO_DAEMON_MODE
	warning("   -b\t\t\tbackground and go into daemon mode");
	warning("   -i address\t\tbind on this address (daemon mode only)");
	warning("   -I port\t\tbind on this port (daemon mode only)");
#endif
	warning("   -S version\t\tset server version string");
	warning("   -t dir\t\tchroot to `dir'");
	warning("   -U username\t\tchange user to `user'");
	warning("   -e\t\t\tdon't clean the environment (-t and -U only)");
	warning("   -v virtualroot\tenable virtual host support in this directory");
#ifndef NO_DIRINDEX_SUPPORT
	warning("   -X\t\t\tenable automatic directory index support");
	warning("   -H\t\t\thide files starting with a period (.) in index mode");
#endif
	warning("   -x index\t\tchange default `index.html' file name");
	error(1, "%s failed to start", server_software);
}

int
main(argc, argv)
	int	argc;
	char	*argv[];
{
	http_req *http_request;
	extern	char **environ;
	char	*cleanenv[1];
	uid_t	uid;
	int	c;

	openlog(progname = argv[0], LOG_PID|LOG_NDELAY, LOG_FTP);
	set_http_date();

	while ((c = getopt(argc, argv, "C:HI:M:S:U:Xbc:dehi:np:rt:uv:x:")) != -1) {
		switch(c) {

		case 'M':
#ifndef NO_DYNAMIC_CONTENT
			add_content_map_mime(optarg);
			break;
#else
			error(1, "dynmic mime content support is not enabled");
			/* NOTREACHED */
#endif /* NO_DYNAMIC_CONTENT */

		case 'n':
			nflag = 1;
			break;

		case 'r':
			/* nothing; silently ignore */
			break;

		case 'S':
			server_software = optarg;
			break;

		case 'U':
			Uflag = optarg;
			break;

		case 'v':
			vpath = optarg;
			break;

		case 'x':
			index_html = optarg;
			break;

#ifndef NO_DAEMON_MODE
		case 'b':
			bflag = 1;
			break;

		case 'e':
			eflag = 1;
			break;

		case 'i':
			iflag = optarg;
			break;

		case 'I':
			Iflag = optarg;
			break;
#else /* NO_DAEMON_MODE */
		case 'b':
		case 'i':
		case 'I':
			error(1, "Daemon mode is not enabled");
			/* NOTREACHED */
#endif /* NO_DAEMON_MODE */

#ifndef NO_CGIBIN_SUPPORT
		case 'c':
			cgibin = optarg;
			debug((DEBUG_OBESE, "cgibin (cgi-bin directory) is %s", cgibin));
			break;

		case 'C':
#ifndef NO_DYNAMIC_CONTENT
			Cflag = 1;
			add_content_map_cgi(optarg);
			break;
#else
			error(1, "dynmic CGI handler support is not enabled");
			/* NOTREACHED */
#endif /* NO_DYNAMIC_CONTENT */

#else
		case 'c':
		case 'C':
			error(1, "CGI is not enabled");
			/* NOTREACHED */
#endif /* NO_CGIBIN_SUPPORT */

		case 'd':
#ifdef DEBUG
			dflag++;
			break;
#else
			error(1, "Debugging is not enabled");
			/* NOTREACHED */
#endif /* DEBUG */

#ifndef NO_USER_SUPPORT
		case 'p':
			public_html = optarg;
			break;

		case 't':
			tflag = optarg;
			break;

		case 'u':
			uflag = 1;
			break;
#else
		case 'u':
		case 'p':
			error(1, "User support is not enabled");
			/* NOTREACHED */
#endif /* NO_USER_SUPPORT */

#ifndef NO_DIRINDEX_SUPPORT
		case 'H':
			Hflag = 1;
			break;

		case 'X':
			Xflag = 1;
			break;

#else
		case 'H':
		case 'X':
			error(1, "directory indexing is not enabled");
			/* NOTREACHED */
#endif /* NO_DIRINDEX_SUPPORT */

		default:
			usage();
			/* NOTREACHED */
		}
	}
	argc -= optind;
	argv += optind;

	if (argc == 1) {
		myname = malloc(MAXHOSTNAMELEN+1);
		if (myname == NULL)
			error(1, "malloc");
		/* XXX we do not check for FQDN here */
		if (gethostname(myname, MAXHOSTNAMELEN+1) < 0)
			error(1, "gethostname");
		myname[MAXHOSTNAMELEN] = '\0';
	} else if (argc == 2)
		myname = argv[1];
	else
		usage();
	
	slashdir = argv[0];
	debug((DEBUG_OBESE, "myname is %s, slashdir is %s", myname, slashdir));

#ifndef NO_DAEMON_MODE
	/*
	 * deal with daemon mode.
	 */
	if (bflag) {
		struct addrinfo h, *r, *r0;
		int e, i, on = 1;

		daemon(1, 0);

		warning("started in daemon mode as `%s' port `%s' root `%s'",
		    myname, Iflag, slashdir);
		
#if 0
		if (iflag == NULL)
			iflag = myname;
#endif
		memset(&h, 0, sizeof h);
		h.ai_family = PF_UNSPEC;
		h.ai_socktype = SOCK_STREAM;
		h.ai_flags = AI_PASSIVE;
		e = getaddrinfo(iflag, Iflag, &h, &r0);
		if (e)
			error(1, "getaddrinfo([%s]:%s): %s",
			    iflag ? iflag : "*", Iflag, gai_strerror(e));
		for (r = r0; r != NULL; r = r->ai_next)
			nsock++;
		sock = bozomalloc(nsock * sizeof *sock);
		for (i = 0, r = r0; r != NULL; r = r->ai_next) {
			sock[i] = socket(r->ai_family, SOCK_STREAM, 0);
			if (sock[i] == -1)
				continue;
			if (setsockopt(sock[i], SOL_SOCKET, SO_REUSEADDR, &on,
			    sizeof(on)) == -1)
				warning("setsockopt SO_REUSEADDR: %s",
				    strerror(errno));
			if (bind(sock[i], r->ai_addr, r->ai_addrlen) == -1)
				continue;
			if (listen(sock[i], SOMAXCONN) == -1)
				continue;
			i++;
		}
		if (i == 0)
			error(1, "could not find any addresses to bind");
		nsock = i;
		freeaddrinfo(r0);

		signal(SIGCHLD, sigchild);
	}
#endif /* NO_DAEMON_MODE */

	/*
	 * prevent info leakage between different compartments.
	 * some PATH values in the environment would be invalided
	 * by chroot. cross-user settings might result in undesirable
	 * effects.
	 */
	if ((tflag != NULL || Uflag != NULL) && !eflag) {
		cleanenv[0] = NULL;
		environ = cleanenv;
	}
	
	/*
	 * look up user/group information.
	 */
	if (Uflag != NULL) {
		struct passwd *pw;

		if ((pw = getpwnam(Uflag)) == NULL)
			error(1, "getpwnam(%s): %s", Uflag, strerror(errno));
		if (initgroups(pw->pw_name, pw->pw_gid) == -1)
			error(1, "initgroups: %s", strerror(errno));
		if (setgid(pw->pw_gid) == -1)
			error(1, "setgid(%u): %s", pw->pw_gid, strerror(errno));
		uid = pw->pw_uid;
	}

	/*
	 * handle chroot.
	 */
	if (tflag != NULL) {
		if (chdir(tflag) == -1)
			error(1, "chdir(%s): %s", tflag, strerror(errno));
		if (chroot(tflag) == -1)
			error(1, "chroot(%s): %s", tflag, strerror(errno));
	}

	if (Uflag != NULL)
		if (setuid(uid) == -1)
			error(1, "setuid(%d): %s", uid, strerror(errno));

	/*
	 * be sane, don't start serving up files from a
	 * hierarchy we don't have permission to get to.
	 */
	if (tflag != NULL)
		if (chdir("/") == -1)
			error(1, "chdir /: %s", strerror(errno));

	/*
	 * read and process the HTTP request.
	 */
	do {
		http_request = read_request();
		if (http_request) {
			process_request(http_request);
			return (0);
		}
	} while (bflag);

	return (0);
}

static void
set_http_date()
{
	struct	tm *tm;
	time_t	now;

	/* Sun, 06 Nov 1994 08:49:37 GMT */
	now = time(NULL);
	tm = gmtime(&now);	/* HTTP/1.1 spec rev 06 sez GMT only */
	strftime(date, sizeof date, "%a, %d %b %Y %H:%M:%S GMT", tm);
}

/*
 * convert "in" into the three parts of a request (first line)
 */
static void
parse_request(in, method, url, proto)
	char	*in;
	char	**method;
	char	**url;
	char	**proto;
{
	ssize_t	len;
	char	*val;
	
	*method = *url = *proto = NULL;		/* set them up */

	len = (ssize_t)strlen(in);
	val = strnsep(&in, " \t\n\r", &len);
	if (len < 1 || val == NULL)
		return;
	*method = val;
	while (*in == ' ' || *in == '\t')
		in++;
	val = strnsep(&in, " \t\n\r", &len);
	if (len < 1) {
		if (len == 0)
			*url = val;
		else
			*url = in;
		return;
	}
	*url = val;
	if (in) {
		while (*in && (*in == ' ' || *in == '\t'))
			in++;
		if (*in)
			*proto = in;
	}
}

/*
 * send a HTTP/1.1 408 response if we timeout.
 */
/* ARGSUSED */
static void
alarmer(sig)
	int sig;
{

	alarmhit = 1;
}

/*
 * This function reads a http request from stdin, returning a pointer to a
 * http_req structure, describing the request.
 */
static http_req *
read_request()
{
	struct	sigaction	sa;
	char	*str, *val, *method, *url, *proto;
	char	*host, *addr, *port;
	char	bufport[10];
	char	hbuf[NI_MAXHOST], abuf[NI_MAXSERV];
	struct	sockaddr_storage ss;
	ssize_t	len;
	int	line = 0;
	socklen_t slen;
	http_req *request;
#ifndef NO_DAEMON_MODE
	static struct pollfd *fds;

	while (bflag) {
		int fd;
		int i;

		if (fds == NULL) {
			fds = bozomalloc(nsock * sizeof *fds);
			for (i = 0; i < nsock; i++) {
				fds[i].events = POLLIN | POLLPRI | POLLRDNORM |
						POLLRDBAND | POLLERR;
				fds[i].fd = sock[i];
			}
		}

		/*
		 * wait for a connection, then fork() and return NULL in
		 * the parent, who will come back here waiting for another
		 * connection.  read the request in in the child, and return
		 * it, for processing.
		 */
again:
		if (poll(fds, nsock, INFTIM) == -1) {
			if (errno != EINTR)
				error(1, "poll: %s", strerror(errno));
			goto again;
		}

		for (i = 0; i < nsock; i++) {
			if (fds[i].revents & (POLLNVAL|POLLERR|POLLHUP))
				warning("poll on fd %d: %s", fds[i].fd,
				    strerror(errno));
			if (fds[i].revents == 0)
				continue;

			fd = accept(sock[i], (struct sockaddr *)&ss, &slen);
			set_http_date();	/* we were sleeping in accept() */
			if (fd == -1) {
				if (errno != EAGAIN)
					error(1, "accept: %s", strerror(errno));
				return NULL;
			}
			switch (fork()) {
			case -1: /* eep, failure */
				warning("fork() failed, sleeping for 10 seconds: %s",
				    strerror(errno));
				sleep(10);
				return NULL;

			case 0: /* child */
				/* setup stdin/stdout/stderr */
				dup2(fd, 0);
				dup2(fd, 1);
				dup2(fd, 2);
				close(fd);
				goto child;

			default: /* parent */
				close(fd);
				continue;
			}
		}
	}
child:
#endif /* NO_DAEMON_MODE */

	request = calloc(1, sizeof *request);
	if (request == NULL)
		error(1, "malloc of request descriptor");
	request->allow = request->host = request->content_type =
	    request->content_length = NULL;

	slen = sizeof(ss);
	if (getpeername(0, (struct sockaddr *)&ss, &slen) < 0)
		host = addr = NULL;
	else {
		addr = host = NULL;
		if (getnameinfo((struct sockaddr *)&ss, slen,
		    abuf, sizeof abuf, NULL, 0, NI_NUMERICHOST) == 0)
			addr = abuf;
		if (nflag == 0 && getnameinfo((struct sockaddr *)&ss, slen,
		    hbuf, sizeof hbuf, NULL, 0, 0) == 0)
			host = hbuf;
	}
	if (host != NULL)
		request->remotehost = bozostrdup(host);
	if (addr != NULL)
		request->remoteaddr = bozostrdup(addr);
	slen = sizeof(ss);
	if (getsockname(0, (struct sockaddr *)&ss, &slen) < 0)
		port = NULL;
	else {
		if (getnameinfo((struct sockaddr *)&ss, slen, NULL, 0,
		    bufport, sizeof bufport, NI_NUMERICSERV) == 0)
			port = bufport;
		else
			port = NULL;
	}
	if (port != NULL)
		request->serverport = bozostrdup(port);

	/*
	 * setup a timer to make sure the request is not hung
	 */
	sa.sa_handler = alarmer;
	sigemptyset(&sa.sa_mask);
	sigaddset(&sa.sa_mask, SIGALRM);
	sa.sa_flags = 0;
	(void)sigaction(SIGALRM, &sa, NULL);	/* XXX */

	alarm(MAX_WAIT_TIME);
	while ((str = dgetln(STDIN_FILENO, &len)) != NULL) {
		alarm(0);
		if (alarmhit)
			http_error(408, NULL, "request timed out");
		line++;
		str = bozostrdup(str);	/* we use this copy */

		if (line == 1) {
			if (len < 1)
				http_error(404, NULL, "null method");
			warning("got request ``%s'' from host %s to port %s",
			    str,
			    host ? host : addr ? addr : "<local>",
			    port ? port : "<stdin>");
			debug((DEBUG_FAT, "read_req, getting request: ``%s''",
			    str));

			parse_request(str, &method, &url, &proto);
			if (method == NULL)
				http_error(404, NULL, "null method");
			if (url == NULL)
				http_error(404, NULL, "null url");

			/*
			 * note that we parse the proto first, so that we
			 * can more properly parse the method and the url.
			 */
			request->url = url;
			process_proto(request, proto);
			process_method(request, method);

			/* http/0.9 has no header processing */
			if (request->proto == http_09)
				break;
		} else {		/* incoming headers */
			struct	headers *hdr;

			if (*str == '\0')
				break;

			val = strnsep(&str, ":", &len);
			debug((DEBUG_OBESE,
			    "read_req2: after strnsep: str ``%s'' val ``%s''",
			    str, val));
			if (val == NULL || len == -1)
				http_error(404, request, "no header");
			while (*str == ' ' || *str == '\t')
				len--, str++;
			
			hdr = bozomalloc(sizeof *hdr);
			hdr->header = val;
			if (str && *str)
				hdr->value = str;
			else
				hdr->value = " ";

			SIMPLEQ_INSERT_TAIL(&request->headers, hdr, next);
			request->nheaders++;

			if (strcasecmp(hdr->header, "content-type") == 0)
				request->content_type = hdr->value;
			else if (strcasecmp(hdr->header, "content-length") == 0)
				request->content_length = hdr->value;
			else if (strcasecmp(hdr->header, "host") == 0)
				request->host = hdr->value;
			/* HTTP/1.1 rev06 draft spec: 14.20 */
			else if (strcasecmp(hdr->header, "expect") == 0)
				http_error(417, request, "we don't support Expect:");

			debug((DEBUG_FAT, "adding header %s: %s",
			    hdr->header, hdr->value));
		}
		alarm(MAX_WAIT_TIME);
	}

	/* now, clear it all out */
	alarm(0);
	(void)signal(SIGALRM, SIG_DFL);

	/* RFC1945, 8.3 */
	if (request->method == HTTP_POST && request->content_length == NULL)
		http_error(400, request, "missing content length");

	/* HTTP/1.1 draft rev-06, 14.23 & 19.6.1.1 */
	if (request->proto == http_11 && request->host == NULL)
		http_error(400, request, "missing Host header");

	debug((DEBUG_FAT, "read_request returns url %s in request", request->url));
	return (request);
}

/*
 * process_request does the following:
 *	- check the request is valid
 *	- process cgi-bin if necessary
 *	- transform a filename if necesarry
 *	- return the HTTP request
 */
static void
process_request(request)
	http_req *request;
{
	struct	stat sb;
	char	*file;
	const char *type, *encoding;
	int	fd, isindex;

	/* note that transform_request chdir()'s if required */
	file = transform_request(request, &isindex);
	if (file == NULL)
		http_error(404, request, "empty file after transform");

	fd = open(file, O_RDONLY);
	if (fd < 0) {
		debug((DEBUG_FAT, "open failed: %s", strerror(errno)));
		if (errno == EPERM)
			http_error(403, request, "no permission to open file");
		else if (errno == ENOENT) {
#ifndef NO_DIRINDEX_SUPPORT
			if (isindex && Xflag) {
				const char *name;

				if (strlen(file) <= strlen(index_html))
					name = ".";
				else {
					file[strlen(file) - strlen(index_html)] = '\0';
					name = file;
				}
				debug((DEBUG_FAT, "trying dir index (%s)",
				    name));
				directory_index(request, name);
				return;
			}
#endif /* NO_DIRINDEX_SUPPORT */
			http_error(404, request, "no file");
		} else
			http_error(500, request, "open file");
	}
	if (fstat(fd, &sb) < 0)
		http_error(500, request, "can't fstat");
	if (S_ISDIR(sb.st_mode))
		handle_redirect(request, file);
	/* XXX RFC1945 10.9 If-Modified-Since (http code 304) */

	(void)printf("%s 200 OK\r\n", request->proto);

	if (request->proto != http_09) {
		type = content_type(request, file);
		encoding = content_encoding(request, file);

		print_header(request, &sb, type, encoding);
		(void)putchar('\r');
		(void)putchar('\n');
	}
	(void)fflush(stdout);

	if (request->method != HTTP_HEAD) {
		char *addr;
		void *oaddr;
		off_t sz = sb.st_size;

		oaddr = addr = mmap(0, (size_t)sz, PROT_READ,
		    MAP_SHARED, fd, 0);
		if (addr == (char *)-1)
			error(1, "mmap failed: %s", strerror(errno));

#ifdef MADV_SEQUENTIAL
		if (madvise(addr, sz, MADV_SEQUENTIAL) < 0)
			warning("madvise failed, ignoring");
#endif

#define WRSZ	(64 * 1024)
		while (sz > WRSZ) {
			if (write(STDOUT_FILENO, addr, WRSZ) != WRSZ)
				error(1, "write failed: %s", strerror(errno));
			sz -= WRSZ;
			addr += WRSZ;
		}
		if (sz && write(STDOUT_FILENO, addr, sz) != sz)
			error(1, "final write failed: %s", strerror(errno));
		if (munmap(oaddr, (size_t)sb.st_size) < 0)
			warning("munmap failed");
	}
	(void)close(fd);
}

#ifndef NO_DIRINDEX_SUPPORT
/*
 * output a directory index.
 */
static void
directory_index(request, dirname)
	http_req *request;
	const char *dirname;
{
	struct stat sb;
	struct dirent *de;
	struct tm *tm;
	DIR *dp;
	char buf[MAXPATHLEN];
	int l;

	debug((DEBUG_FAT, "directory_index: dirname ``%s''", dirname));
	if (stat(dirname, &sb) < 0 ||
	    (dp = opendir(dirname)) == NULL) {
		if (errno == EPERM)
			http_error(403, request,
			    "no permission to open directory");
		else if (errno == ENOENT)
			http_error(404, request, "no file");
		else
			http_error(500, request, "open directory");
		return;
	}

	(void)printf("%s 200 OK\r\n", request->proto);

	if (request->proto != http_09) {
		print_header(request, NULL, "text/html", "");
		(void)putchar('\r');
		(void)putchar('\n');
	}
	(void)fflush(stdout);

	if (request->method == HTTP_HEAD) {
		closedir(dp);
		return;
	}

	(void)printf("<html><head><title>Index of %s</title></head>\r\n",
	    request->url);
	(void)printf("<body><h1>Index of %s</h1>\r\n", request->url);
	(void)printf("<pre>\r\n");
	(void)printf("Name                             "
	    "Last modified          "
	    "Size\n");
	(void)printf("<hr noshade align=\"left\" width=\"80%%\">\r\n\r\n");

	while ((de = readdir(dp)) != NULL) {
		int nostat = 0;
		char *name = de->d_name;

		if (strcmp(name, ".") == 0 || 
		    (strcmp(name, "..") != 0 && Hflag && name[0] == '.'))
			continue;

		(void)snprintf(buf, sizeof buf, "%s/%s", dirname, name);
		if (stat(buf, &sb))
			nostat = 1;

		l = 0;

		if (strcmp(name, "..") == 0) {
			(void)printf("<a href=\"../\">");
			l += printf("Parent Directory");
		} else if (S_ISDIR(sb.st_mode)) {
			(void)printf("<a href=\"%s/\">", name);
			l += printf("%s/", name);
		} else {
			(void)printf("<a href=\"%s\">", name);
			l += printf("%s", name);
		}
		(void)printf("</a>");

		(void)putchar(' ');
		for (; l < 30; l++)
			(void)putchar(' ');

		if (nostat)
			(void)printf("?                         ?");
		else {
			tm = gmtime(&sb.st_mtime);
			(void)strftime(buf, sizeof buf, "%d-%b-%Y %R", tm);
			l += printf("%s", buf);

			(void)putchar(' ');
			for (; l < 44; l++)
				(void)putchar(' ');

			if (S_ISDIR(sb.st_mode))
				(void)printf("%7ukB",
				    ((unsigned int)sb.st_size >> 10));
			else
				(void)printf("        -");
		}
		(void)printf("\r\n");
	}

	closedir(dp);
	(void)printf("</pre><hr></body></html>\r\n");
	(void)fflush(stdout);
}
#endif /* NO_DIRINDEX_SUPPORT */

#ifndef NO_DAEMON_MODE 
/* ARGSUSED */
static void
sigchild(signo)
	int signo;
{
	int status;

	while (waitpid(-1, &status, WNOHANG) != -1)
		;
}
#endif /* NO_DAEMON_MODE */

/*
 * deal with virtual host names; we do this:
 *	if we have a virtual path root (vpath), and we are given a
 *	virtual host spec (Host: ho.st or http://ho.st/), see if this
 *	directory exists under vpath.  if it does, use this as the
 #	new slashdir.
 */
static void
check_virtual(request)
	http_req *request;
{
	char *url = request->url, *s;
	struct dirent **list;
	size_t len;
	int i;

	if (!vpath)
		goto out;

	/*
	 * convert http://virtual.host/ to request->host
	 */
	debug((DEBUG_OBESE, "checking for http:// virtual host in ``%s''", url));
	if (strncasecmp(url, "http://", 7) == 0) {
		/* we would do virtual hosting here? */
		url += 7;
		s = strchr(url, '/');
		/* HTTP/1.1 draft rev-06, 5.2: URI takes precedence over Host: */
		request->host = url;
		request->url = bozostrdup(s ? s : "/");
		debug((DEBUG_OBESE, "got host ``%s'' url is now ``%s''",
		    request->host, request->url));
	} else if (!request->host)
		goto out;


	/*
	 * ok, we have a virtual host, use scandir(3) to find a case
	 * insensitive match for the virtual host we are asked for.
	 * note that if the virtual host is the same as the master,
	 * we don't need to do anything special.
	 */
	len = strlen(request->host);
	debug((DEBUG_OBESE,
	    "check_virtual: checking host `%s' under vpath `%s' for url `%s'",
	    request->host, vpath, request->url));
	if (strncasecmp(myname, request->host, len) != 0) {
		s = 0;
		for (i = scandir(vpath, &list, 0, 0); i--; list++) {
			debug((DEBUG_OBESE, "looking at dir``%s''",
			    (*list)->d_name));
			if (strncasecmp((*list)->d_name, request->host,
			    len) == 0) {
				/* found it, punch it */
				myname = (*list)->d_name;
				if (asprintf(&s, "%s/%s", vpath, myname) < 0)
					error(1, "asprintf");
				break;
			}
		}
		if (s == 0)
			http_error(404, request, "unknown URL");
	} else
out:
		s = slashdir;

	/*
	 * ok, nailed the correct slashdir, chdir to it
	 */
	if (chdir(s) < 0)
		error(1, "can't chdir %s: %s", s, strerror(errno));
}

/*
 * transform_request does this:
 *	- ``expand'' %20 crapola
 *	- punt if it doesn't start with /
 *	- look for "http://myname/" and deal with it.
 *	- maybe call process_cgi() 
 *	- check for ~user and call transform_user() if so
 *	- if the length > 1, check for trailing slash.  if so,
 *	  add the index.html file
 *	- if the length is 1, return the index.html file
 *	- disallow anything ending up with a file starting
 *	  at "/" or having ".." in it.
 *	- anything else is a really weird internal error
 */
static char *
transform_request(request, isindex)
	http_req *request;
	int *isindex;
{
	char	*file;
	char	*url;
	size_t	len;

	file = NULL;	/* XXX gcc */
	*isindex = 0;
	debug((DEBUG_FAT, "tf_req: url %s", request->url));
	fix_url_percent(request);
	check_virtual(request);
	url = request->url;

	if (url[0] != '/')
		http_error(404, request, "unknown URL");

#ifndef NO_CGIBIN_SUPPORT
	if (cgibin || Cflag)
		process_cgi(request);
#endif /* NO_CGIBIN_SUPPORT */

	if (0) {
#ifndef NO_USER_SUPPORT
	} else if ((len = strlen(url)) > 1 && uflag && url[1] == '~') {
		if (url[2] == '\0')
			http_error(404, request, "missing username");
		if (strchr(url + 2, '/') == NULL)
			handle_redirect(request, url);
		debug((DEBUG_FAT, "calling transform_user"));
		return (transform_user(request, isindex));
	} else if (len > 1) {
#else
	} else if ((len = strlen(url)) > 1) {
#endif /* NO_USER_SUPPORT */
		debug((DEBUG_FAT, "url[len-1] == %c", url[len-1]));
		if (url[len-1] == '/') {	/* append index.html */
			*isindex = 1;
			debug((DEBUG_FAT, "appending index.html"));
			file = bozomalloc(len + strlen(index_html) + 1);
			strcpy(file, url + 1);
			strcat(file, index_html);
		} else
			file = bozostrdup(url + 1);
	} else if (len == 1) {
		debug((DEBUG_OBESE, "tf_req: len == 1"));
		file = bozostrdup(index_html);
		*isindex = 1;
	} else		/* len == 0 ? */
		http_error(500, request, "request->url is nul?");

	if (file == NULL)
		http_error(500, request, "internal failure");

	/*
	 * look for "http://myname/" and deal with it as necessary.
	 */

	/*
	 * stop traversing outside our domain 
	 *
	 * XXX true security only comes from our parent using chroot(2)
	 * before execve(2)'ing us.  or our own built in chroot(2) support.
	 */
	if (*file == '/' || strstr(file, ".."))
		http_error(403, request, "illegal request");

	debug((DEBUG_FAT, "transform_request returned: %s", file));
	return (file);
}

#ifndef NO_USER_SUPPORT
/*
 * transform_user does this:
 *	- chdir's /~user/public_html
 *	- returns the rest of the file, index.html appended if required
 *
 * transform_request() is supposed to check that we have user support
 * enabled.
 */
static char *
transform_user(request, isindex)
	http_req *request;
	int *isindex;
{
	char	c, *s, *file = NULL;
	struct	passwd *pw;

	*isindex = 0;

	if ((s = strchr(request->url + 2, '/')) != NULL) {
		*s++ = '\0';
		c = s[strlen(s)-1];
		*isindex = (c == '/' || c == '\0');
	}

	debug((DEBUG_FAT, "looking for user %s", request->url + 2));
	pw = getpwnam(request->url + 2);
	/* fix this up immediately */
	if (s)
		s[-1] = '/';
	if (pw == NULL)
		http_error(404, request, "no such user");

	debug((DEBUG_OBESE, "user %s home dir %s uid %d gid %d", pw->pw_name,
	    pw->pw_dir, pw->pw_uid, pw->pw_gid));

	if (chdir(pw->pw_dir) < 0) {
		warning("chdir1 error: %s: %s", pw->pw_dir, strerror(errno));
		http_error(403, request, "can't chdir to homedir");
	}
	if (chdir(public_html) < 0) {
		warning("chdir2 error: %s: %s", public_html, strerror(errno));
		http_error(403, request, "can't chdir to public_html");
	}
	if (s == NULL || *s == '\0') {
		file = bozostrdup(index_html);
	} else {
		file = bozomalloc(strlen(s) +
		    (*isindex ? strlen(index_html) + 1 : 1));
		strcpy(file, s);
		if (*isindex)
			strcat(file, index_html);
	}

	/* see above */
	if (*file == '/' || strstr(file, ".."))
		http_error(403, request, "illegal request");

	debug((DEBUG_FAT, "transform_user returning %s under %s", file,
	    pw->pw_dir));
	return (file);
}
#endif /* NO_USER_SUPPORT */

/*
 * If we are allowing automatic redirection, just do it, else return the
 * canned message, or the provided one.
 */
static void
handle_redirect(request, file)
	http_req *request;
	const char *file;
{
	
	warning("redirecting %s", request->url);
	debug((DEBUG_FAT, "redirecting %s", request->url));
	(void)printf("%s 301 Document Moved\r\n", request->proto);
	if (request->proto != http_09) 
		print_header(request, NULL, "text/html", NULL);
	if (request->proto != http_09)
		(void)printf("Location: http://%s%s/\r\n", myname,
		    request->url);
	(void)putchar('\r');
	(void)putchar('\n');
	if (request->method == HTTP_HEAD)
		goto head;
	(void)printf("<html><head><title>Document Moved</title></head>\n");
	(void)printf("<body><h1>Document Moved</h1>\n");
	(void)printf("This document had moved <a href=\"http://%s%s/\">here</a>\n",
	    myname, request->url);
	(void)printf("</body></html>\n");
head:
	fflush(stdout);
	exit(0);
}

/* generic header printing routine */
static void
print_header(request, sbp, type, encoding)
	http_req *request;
	struct stat *sbp;
	const char *type;
	const char *encoding;
{

	(void)printf("Date: %s\r\n", date);
	(void)printf("Server: %s\r\n", server_software);
	if (sbp) {
		char filedate[40];
		struct	tm *tm;

		tm = gmtime(&sbp->st_mtime);
		strftime(filedate, sizeof filedate,
		    "%a, %d %b %Y %H:%M:%S GMT", tm);
		(void)printf("Last-Modified: %s\r\n", filedate);
	}
	if (type && *type)
		(void)printf("Content-Type: %s\r\n", type);
	if (encoding && *encoding)
		(void)printf("Content-Encoding: %s\r\n", encoding);
	if (sbp)
		(void)printf("Content-Length: %qd\r\n", (long long)sbp->st_size);
	if (request && request->proto == http_11)
		(void)printf("Connection: close\r\n");
}

#ifndef NO_CGIBIN_SUPPORT
/*
 * Checks if the request has asked for a cgi-bin.  Should only be called if
 * cgibin is set.  If it starts CGIBIN_PREFIX or has a ncontent handler,
 * process the cgi, otherwise just return.
 */
static void
process_cgi(request)
	http_req *request;
{
	struct	headers *headp;
	const char *type, *clen, *info, *cgihandler;
	char	*query, *s, *t, *path, *env, *command, *url;
	char	**envp, *argv[4];
	size_t	len;
	int	envpsize, ix;

	debug((DEBUG_LIGHT, "process_cgi: url `%s'", request->url));

	url = bozostrdup(request->url + 1);
	if ((s = strchr(url, '?')) != NULL) {
		*s++ = '\0';
		query = s;
	} else
		query = NULL;
	path = NULL;
	envp = NULL;
	cgihandler = NULL;
	command = NULL;
	info = NULL;

	len = strlen(url);
	if (len == 0 || url[len - 1] == '/') {	/* append index.html */
		debug((DEBUG_FAT, "appending index.html"));
		url = bozorealloc(url, len + strlen(index_html) + 1);
		strcat(url, index_html);
		debug((DEBUG_LIGHT, "process_cgi: url adjusted to `%s'", url));
	}

	if (!cgibin || strncmp(url, CGIBIN_PREFIX, CGIBIN_PREFIX_LEN) != 0) {
		cgihandler = content_cgihandler(request, url);
		if (cgihandler == NULL) {
			free(url);
			return;
		}
		debug((DEBUG_LIGHT, "process_cgi: cgihandler `%s'",
		    cgihandler));
	}

	if (request->method == HTTP_HEAD)
		goto print_cgi_header;

	ix = 0;
	if (cgihandler) {
		command = url;
		path = bozostrdup(cgihandler);
		argv[ix++] = path;
			/* argv[] = [ path, command, query, NULL ] */
	} else {
		command = url + CGIBIN_PREFIX_LEN;
		if ((s = strchr(command, '/')) != NULL) {
			info = bozostrdup(s);
			*s = '\0';
		}
		path = bozomalloc(strlen(cgibin) + 1 + strlen(command) + 1);
		strcpy(path, cgibin);
		strcat(path, "/");
		strcat(path, command);
			/* argv[] = [ command, query, NULL ] */
	}
	argv[ix++] = command;
	argv[ix++] = query;
	argv[ix++] = NULL;

	debug((DEBUG_FAT,
	    "process_cgi: path `%s' cmd `%s' info `%s' query `%s'",
	    path, command, strornull(info), strornull(query)));

	type = request->content_type;
	clen = request->content_length;

	envpsize = 10 + request->nheaders + 
	    (info && *info ? 1 : 0) +
	    (query && query ? 1 : 0) +
	    (type && *type ? 1 : 0) +
	    (clen && *clen ? 1 : 0) +
	    (request->remotehost && *request->remotehost ? 1 : 0) +
	    (request->remoteaddr && *request->remoteaddr ? 1 : 0) +
	    (request->serverport && *request->serverport ? 1 : 0);

	envp = bozomalloc(sizeof(*envp) * envpsize);
	for (ix = 0; ix < envpsize; ix++)
		envp[ix] = NULL;
	ix = 0;

/* NOTE: not "side effects" safe! */
#define spsetenv(env, val) 					\
do {								\
	char *s1 = bozomalloc(strlen(env) + strlen(val) + 2);	\
								\
	strcpy(s1, env); strcat(s1, "="); strcat(s1, val);	\
	debug((DEBUG_OBESE, "spsetenv: %s", s1));		\
	envp[ix++] = s1;					\
} while (/* CONSTCOND */ 0)

	for (headp = SIMPLEQ_FIRST(&request->headers); headp;
	    headp = SIMPLEQ_NEXT(headp, next)) {
		const char *s2;
		env = bozomalloc(6 + strlen(headp->header) + 1 +
		    strlen(headp->value));

		t = env;
		strcpy(t, "HTTP_");
		t += strlen(t);
		for (s2 = headp->header; *s2; t++, s2++)
			if (islower(*s2))
				*t = toupper(*s2);
			else if (*s2 == '-')
				*t = '_';
			else
				*t = *s2;
		*t = '\0';
		debug((DEBUG_OBESE, "setting header %s as %s = %s",
		    headp->header, env, headp->value));
		spsetenv(env, headp->value);
		free(env);
	}
		
#ifndef _PATH_DEFPATH
#define _PATH_DEFPATH "/usr/bin:/bin"
#endif

	spsetenv("PATH", _PATH_DEFPATH);
	spsetenv("IFS", " \t\n");
	spsetenv("SERVER_NAME", myname);
	spsetenv("GATEWAY_INTERFACE", "CGI/1.1");
	spsetenv("SERVER_PROTOCOL", request->proto);
	spsetenv("REQUEST_METHOD", request->methodstr);
	spsetenv("SCRIPT_NAME", url);
	spsetenv("SERVER_SOFTWARE", server_software);
	if (query && *query)
		spsetenv("QUERY_STRING", query);
	if (info && *info)
		spsetenv("PATH_INFO", info);
	if (type && *type)
		spsetenv("CONTENT_TYPE", type);
	if (clen && *clen)
		spsetenv("CONTENT_LENGTH", clen);
	if (request->serverport && *request->serverport)
		spsetenv("SERVER_PORT", request->serverport);
	if (request->remotehost && *request->remotehost)
		spsetenv("REMOTE_HOST", request->remotehost);
	if (request->remoteaddr && *request->remoteaddr)
		spsetenv("REMOTE_ADDR", request->remoteaddr);
#undef spsetenv

	/* may as well wait as long as possible */
print_cgi_header:
	debug((DEBUG_OBESE, "process_cgi:  writing HTTP header .."));
	(void)printf("%s 200 Here it is\r\n", request->proto);
	if (request->proto != http_09)
		print_header(NULL, NULL, NULL, NULL);
	(void)fflush(stdout);
	if (request->method == HTTP_HEAD)
		exit(0);

	debug((DEBUG_FAT, "process_cgi: going exec %s, %s %s %s",
	    path, argv[0], strornull(argv[1]), strornull(argv[2])));
	if (execve(path, argv, envp) < 0)
		error(1, "child exec failed");
}
#endif /* NO_CGIBIN_SUPPORT */

/* this escape HTML tags */
static void
escape_html(request)
	http_req *request;
{
	int	i, j;
	char	*url = request->url, *tmp;

	for (i = 0, j = 0; url[i]; i++) {
		switch (url[i]) {
		case '<':
		case '>':
			j += 4;
			break;
		case '&':
			j += 5;
			break;
		}
	}

	if (j == 0)
		return;

	if ((tmp = (char *) malloc(strlen(url) + j)) == 0)
		/*
		 * ouch, but we are only called from an error context, and
		 * most paths here come from malloc(3) failures anyway...
		 * we could completely punt and just exit, but isn't returning
		 * an not-quite-correct error better than nothing at all?
		 */
		return;

	for (i = 0, j = 0; url[i]; i++) {
		switch (url[i]) {
		case '<':
			memcpy(tmp + j, "&lt;", 4);
			j += 4;
			break;
		case '>':
			memcpy(tmp + j, "&gt;", 4);
			j += 4;
			break;
		case '&':
			memcpy(tmp + j, "&amp;", 5);
			j += 5;
			break;
		default:
			tmp[j++] = url[i];
		}
	}
	tmp[j] = 0;

	/*
	 * XXX original "url" is a substring of an allocation, so we
	 * can't touch it.  so, ignore it and replace the request.
	 */
	request->url = tmp;
}

/* this fixes the %HH hack that RFC1738 requires.  */
static void
fix_url_percent(request)
	http_req *request;
{
	char	*s, *t, buf[3], *url;
	char	*end;	/* if end is not-zero, we don't translate beyond that */

	url = request->url;

	/* make sure we don't translate *too* much */
	/*
	 * XXX somewhat duplicate cgibin check here, but this one lets
	 * and encoded "/cgi-bin" work, but not to stop everything past
	 * the '?' being not fixed.
	 *
	 * XXX cgihandlers support
	 */
	if (cgibin && strncmp(url + 1, CGIBIN_PREFIX, CGIBIN_PREFIX_LEN) == 0) {
		end = strchr(request->url + CGIBIN_PREFIX_LEN, '?');
	} else
		end = 0;

	/* fast forward to the first % */
	if ((s = strchr(url, '%')) == NULL)
		return;

	t = s;
	do {
		if (end && s >= end) {
			debug((DEBUG_OBESE, "fu_%%: past end, filling out.."));
			while (*s)
				*t++ = *s++;
			break;
		}
		debug((DEBUG_OBESE, "fu_%%: got s == %%, s[1]s[2] == %c%c",
		    s[1], s[2]));
		if (s[1] == '\0' || s[2] == '\0')
			http_error(400, request,
			    "percent hack missing two chars afterwards");
		buf[0] = *++s;
		buf[1] = *++s;
		buf[2] = '\0';
		s++;
		*t = (char)strtol(buf, NULL, 16);
		debug((DEBUG_OBESE, "fu_%%: strtol put %c into *t", *t));
		if (*t++ == '\0')
			http_error(400, request, "percent hack got a 0 back");

		while (*s && *s != '%')
			*t++ = *s++;
	} while (*s);
	*t = '\0';
	debug((DEBUG_FAT, "fix_url_percent returns %s in url", request->url));
}

/*
 * process each type of HTTP method, setting this HTTP requests
 # method type.
 */
static struct method_map {
	const char *name;
	int	type;
} method_map[] = {
	{ "GET", 	HTTP_GET, },
	{ "POST",	HTTP_POST, },
	{ "HEAD",	HTTP_HEAD, },
#if 0	/* other non-required http/1.1 methods */
	{ "OPTIONS",	HTTP_OPTIONS, },
	{ "PUT",	HTTP_PUT, },
	{ "DELETE",	HTTP_DELETE, },
	{ "TRACE",	HTTP_TRACE, },
	{ "CONNECT",	HTTP_CONNECT, },
#endif
	{ NULL,		0, },
};

static void
process_method(request, method)
	http_req *request;
	const char *method;
{
	struct	method_map *mmp;

	for (mmp = method_map; mmp->name; mmp++)
		if (strcasecmp(method, mmp->name) == 0) {
			request->method = mmp->type;
			request->methodstr = mmp->name;
			return;
		}

	if (request->proto == http_11)
		request->allow = "GET, HEAD, POST";
	http_error(404, request, "unknown method");
}

/*
 * as the prototype string is not constant (eg, "HTTP/1.1" is equivalent
 * to "HTTP/001.01"), we MUST parse this.
 */
static void
process_proto(request, proto)
	http_req *request;
	const char *proto;
{
	char	majorstr[16], *minorstr;
	int	majorint, minorint;

	if (proto == NULL) {
got_proto_09:
		request->proto = http_09;
		debug((DEBUG_OBESE, "request %s is http/0.9", request->url));
		return;
	}

	if (strncasecmp(proto, "HTTP/", 5) != 0)
		goto bad;
	strncpy(majorstr, proto + 5, sizeof majorstr);
	majorstr[sizeof(majorstr)-1] = 0;
	minorstr = strchr(majorstr, '.');
	if (minorstr == NULL)
		goto bad;
	*minorstr++ = 0;

	majorint = atoi(majorstr);
	minorint = atoi(minorstr);

	switch (majorint) {
	case 0:
		if (minorint != 9)
			break;
		goto got_proto_09;
	case 1:
		if (minorint == 0)
			request->proto = http_10;
		else if (minorint == 1)
			request->proto = http_11;
		else
			break;

		debug((DEBUG_OBESE, "request %s is %s", request->url,
		    request->proto));
		SIMPLEQ_INIT(&request->headers);
		request->nheaders = 0;
		return;
	}
bad:
	http_error(404, NULL, "unknown prototype");
}

/*
 * this map and the functions below map between filenames and the
 * content type and content encoding definitions.  this should become
 * a configuration file, perhaps like apache's mime.types (but that
 * has less info per-entry).
 */
struct content_map {
	const char *name;	/* postfix of file */
	const char *type;	/* matching content-type */
	const char *encoding;	/* matching content-encoding */
	const char *encoding11;	/* matching content-encoding (HTTP/1.1) */
	const char *cgihandler;	/* optional CGI handler */
};

static struct content_map content_map[] = {
	{ ".html",	"text/html",			"",		"", },
	{ ".htm",	"text/html",			"",		"", },
	{ ".gif",	"image/gif",			"",		"", },
	{ ".jpeg",	"image/jpeg",			"",		"", },
	{ ".jpg",	"image/jpeg",			"",		"", },
	{ ".jpe",	"image/jpeg",			"",		"", },
	{ ".png",	"image/png",			"",		"", },
	{ ".mp3",	"audio/mpeg",			"",		"", },
	{ ".swf",	"application/x-shockwave-flash","",		"", },
	{ ".dcr",	"application/x-director",	"",		"", },
	{ ".pac",	"application/x-ns-proxy-autoconfig", "",	"", },
	{ ".pa",	"application/x-ns-proxy-autoconfig", "",	"", },
	{ ".tar",	"multipart/x-tar",		"",		"", },
	{ ".gtar",	"multipart/x-gtar",		"",		"", },
	{ ".tar.Z",	"multipart/x-tar",		"x-compress",	"compress", },
	{ ".tar.gz",	"multipart/x-tar",		"x-gzip",	"gzip", },
	{ ".taz",	"multipart/x-tar",		"x-gzip",	"gzip", },
	{ ".tgz",	"multipart/x-tar",		"x-gzip",	"gzip", },
	{ ".tar.z",	"multipart/x-tar",		"x-pack",	"x-pack", },
	{ ".Z",		"application/x-compress",	"x-compress",	"compress", },
	{ ".gz",	"application/x-gzip",		"x-gzip",	"gzip", },
	{ ".z",		"unknown",			"x-pack",	"x-pack", },
	{ ".bz2",	"application/x-bzip2",		"x-bzip2",	"x-bzip2", },
	{ ".ogg",	"application/x-ogg",		"",		"", },
	{ ".hqx",	"application/mac-binhex40",	"",		"", },
	{ ".cpt",	"application/mac-compactpro",	"",		"", },
	{ ".doc",	"application/msword",		"",		"", },
	{ ".bin",	"application/octet-stream",	"",		"", },
	{ ".dms",	"application/octet-stream",	"",		"", },
	{ ".lha",	"application/octet-stream",	"",		"", },
	{ ".lzh",	"application/octet-stream",	"",		"", },
	{ ".exe",	"application/octet-stream",	"",		"", },
	{ ".class",	"application/octet-stream",	"",		"", },
	{ ".oda",	"application/oda",		"",		"", },
	{ ".pdf",	"application/pdf",		"",		"", },
	{ ".ai",	"application/postscript",	"",		"", },
	{ ".eps",	"application/postscript",	"",		"", },
	{ ".ps",	"application/postscript",	"",		"", },
	{ ".ppt",	"application/powerpoint",	"",		"", },
	{ ".rtf",	"application/rtf",		"",		"", },
	{ ".bcpio",	"application/x-bcpio",		"",		"", },
	{ ".vcd",	"application/x-cdlink",		"",		"", },
	{ ".cpio",	"application/x-cpio",		"",		"", },
	{ ".csh",	"application/x-csh",		"",		"", },
	{ ".dir",	"application/x-director",	"",		"", },
	{ ".dxr",	"application/x-director",	"",		"", },
	{ ".dvi",	"application/x-dvi",		"",		"", },
	{ ".hdf",	"application/x-hdf",		"",		"", },
	{ ".cgi",	"application/x-httpd-cgi",	"",		"", },
	{ ".skp",	"application/x-koan",		"",		"", },
	{ ".skd",	"application/x-koan",		"",		"", },
	{ ".skt",	"application/x-koan",		"",		"", },
	{ ".skm",	"application/x-koan",		"",		"", },
	{ ".latex",	"application/x-latex",		"",		"", },
	{ ".mif",	"application/x-mif",		"",		"", },
	{ ".nc",	"application/x-netcdf",		"",		"", },
	{ ".cdf",	"application/x-netcdf",		"",		"", },
	{ ".patch",	"application/x-patch",		"",		"", },
	{ ".sh",	"application/x-sh",		"",		"", },
	{ ".shar",	"application/x-shar",		"",		"", },
	{ ".sit",	"application/x-stuffit",	"",		"", },
	{ ".sv4cpio",	"application/x-sv4cpio",	"",		"", },
	{ ".sv4crc",	"application/x-sv4crc",		"",		"", },
	{ ".tar",	"application/x-tar",		"",		"", },
	{ ".tcl",	"application/x-tcl",		"",		"", },
	{ ".tex",	"application/x-tex",		"",		"", },
	{ ".texinfo",	"application/x-texinfo",	"",		"", },
	{ ".texi",	"application/x-texinfo",	"",		"", },
	{ ".t",		"application/x-troff",		"",		"", },
	{ ".tr",	"application/x-troff",		"",		"", },
	{ ".roff",	"application/x-troff",		"",		"", },
	{ ".man",	"application/x-troff-man",	"",		"", },
	{ ".me",	"application/x-troff-me",	"",		"", },
	{ ".ms",	"application/x-troff-ms",	"",		"", },
	{ ".ustar",	"application/x-ustar",		"",		"", },
	{ ".src",	"application/x-wais-source",	"",		"", },
	{ ".zip",	"application/zip",		"",		"", },
	{ ".au",	"audio/basic",			"",		"", },
	{ ".snd",	"audio/basic",			"",		"", },
	{ ".mpga",	"audio/mpeg",			"",		"", },
	{ ".mp2",	"audio/mpeg",			"",		"", },
	{ ".aif",	"audio/x-aiff",			"",		"", },
	{ ".aiff",	"audio/x-aiff",			"",		"", },
	{ ".aifc",	"audio/x-aiff",			"",		"", },
	{ ".ram",	"audio/x-pn-realaudio",		"",		"", },
	{ ".rpm",	"audio/x-pn-realaudio-plugin",	"",		"", },
	{ ".ra",	"audio/x-realaudio",		"",		"", },
	{ ".wav",	"audio/x-wav",			"",		"", },
	{ ".pdb",	"chemical/x-pdb",		"",		"", },
	{ ".xyz",	"chemical/x-pdb",		"",		"", },
	{ ".ief",	"image/ief",			"",		"", },
	{ ".tiff",	"image/tiff",			"",		"", },
	{ ".tif",	"image/tiff",			"",		"", },
	{ ".ras",	"image/x-cmu-raster",		"",		"", },
	{ ".pnm",	"image/x-portable-anymap",	"",		"", },
	{ ".pbm",	"image/x-portable-bitmap",	"",		"", },
	{ ".pgm",	"image/x-portable-graymap",	"",		"", },
	{ ".ppm",	"image/x-portable-pixmap",	"",		"", },
	{ ".rgb",	"image/x-rgb",			"",		"", },
	{ ".xbm",	"image/x-xbitmap",		"",		"", },
	{ ".xpm",	"image/x-xpixmap",		"",		"", },
	{ ".xwd",	"image/x-xwindowdump",		"",		"", },
	{ ".txt",	"text/plain",			"",		"", },
	{ ".rtx",	"text/richtext",		"",		"", },
	{ ".tsv",	"text/tab-separated-values",	"",		"", },
	{ ".etx",	"text/x-setext",		"",		"", },
	{ ".sgml",	"text/x-sgml",			"",		"", },
	{ ".sgm",	"text/x-sgml",			"",		"", },
	{ ".mpeg",	"video/mpeg",			"",		"", },
	{ ".mpg",	"video/mpeg",			"",		"", },
	{ ".mpe",	"video/mpeg",			"",		"", },
	{ ".qt",	"video/quicktime",		"",		"", },
	{ ".mov",	"video/quicktime",		"",		"", },
	{ ".avi",	"video/x-msvideo",		"",		"", },
	{ ".movie",	"video/x-sgi-movie",		"",		"", },
	{ ".ice",	"x-conference/x-cooltalk",	"",		"", },
	{ ".wrl",	"x-world/x-vrml",		"",		"", },
	{ ".vrml",	"x-world/x-vrml",		"",		"", },
#if 0
	{ ".php",	"",				"",		"", "/usr/pkg/bin/php", },
	{ ".pl",	"",				"",		"", "/usr/pkg/bin/perl", },
#endif
	{ NULL,		NULL,				NULL,		NULL, },
};

static struct content_map *dynamic_content_map;
static int dynamic_content_map_size;

/* call with stage == 0 for full search, stage == 1 for dynamic only */
static struct content_map *
match_content_map(name, stage)
	const char	*name;
	int	stage;
{
	size_t	len = strlen(name), nlen;
	struct	content_map	*map;

	map = dynamic_content_map;
again:
	for (; map && map->name; map++) {
		nlen = strlen(map->name);
		if (nlen > len || strcmp(map->name, name + (len - nlen)) != 0)
			continue;
		return (map);
	}
	if (stage++ == 0) {
		map = content_map;
		goto again;
	}
		
	return (NULL);
}

#ifndef NO_DYNAMIC_CONTENT
static struct content_map *
get_content_map(name)
	const char	*name;
{
	struct	content_map	*map;

	if ((map = match_content_map(name, 1)))
		/* just overwrite the original */;
	else if (dynamic_content_map_size == 0) {
		map = bozomalloc(sizeof *map);
		dynamic_content_map = map;
		dynamic_content_map_size = 1;
	} else {
		dynamic_content_map_size++;
		dynamic_content_map = bozorealloc(dynamic_content_map, dynamic_content_map_size * sizeof *map);
		map = &dynamic_content_map[dynamic_content_map_size-1];
	}
	
	return (map);
}

#ifndef NO_CGIBIN_SUPPORT
/* cgi maps are simple ".postfix /path/to/prog" */
static void
add_content_map_cgi(arg)
	char	*arg;
{
	struct content_map *map;
	char *s;

	for (s = arg; *s; s++)
		if (isspace(*s))
			break;
	if (*s)
		*s++ = '\0';
	for (; *s; s++)
		if (!isspace(*s))
			break;

	debug((DEBUG_FAT, "add_content_map_cgi: name %s cgi %s", arg, s));

	map = get_content_map(arg);
	map->name = arg;
	map->type = map->encoding = map->encoding11 = NULL;
	map->cgihandler = s;
}
#endif /* NO_CGIBIN_SUPPORT */

/*
 * mime content maps look like:
 *	".name type encoding encoding11"
 * where any of type, encoding or encoding11 a dash "-" means "".
 * eg the .gtar, .tar.Z from above  could be written like:
 *	".gtar multipart/x-gtar - -"
 *	".tar.Z multipart/x-tar x-compress compress"
 * or
 *	".gtar multipart/x-gtar"
 *	".tar.Z multipart/x-tar x-compress compress"
 * NOTE: we destroy 'arg'
 */
static void
add_content_map_mime(arg)
	char	*arg;
{
	struct content_map *map;
	char *cmap[4], *s;
	int i;

	/* yuck, a parser; at least it's only command line. */
	cmap[0] = cmap[1] = cmap[2] = cmap[3] = NULL;
	for (i = 0, s = arg; *s; s++) {
		cmap[i++] = s;
		if (i == 4)
			break;
		for (; *s; s++)
			if (isspace(*s))
				break;
		if (*s)
			*s = '\0';
		for (; *s; s++)
			if (!isspace(*s))
				break;
	}

	debug((DEBUG_FAT, "add_content_map: name %s type %s enc %s enc11 %s ",
		cmap[0], cmap[1], cmap[2], cmap[3]));

	map = get_content_map(cmap[0]);
#define CHECKMAP(s)	(!s || ((s)[0] == '-' && (s)[1] == '\0') ? "" : (s))
	map->name = CHECKMAP(cmap[0]);
	map->type = CHECKMAP(cmap[1]);
	map->encoding = CHECKMAP(cmap[2]);
	map->encoding11 = CHECKMAP(cmap[3]);
#undef CHECKMAP
	map->cgihandler = NULL;
}
#endif /* NO_DYNAMIC_CONTENT */

/*
 * given the file name, return a valid Content-Type: value.
 */
/* ARGSUSED */
static const char *
content_type(request, file)
	http_req *request;
	const char	*file;
{
	struct	content_map	*map;

	map = match_content_map(file, 0);
	if (map)
		return (map->type);
	return (text_plain);
}

/*
 * given the file name, return a valid Content-Encoding: value.
 */
static const char *
content_encoding(request, file)
	http_req *request;
	const char	*file;
{
	struct	content_map	*map;

	map = match_content_map(file, 0);
	if (map)
		return (request->proto == http_11 ?
		    map->encoding11 : map->encoding);
	return (NULL);
}

/*
 * given the file name, return a CGI interpreter
 */
static const char *
content_cgihandler(request, file)
	http_req *request;
	const char	*file;
{
	struct	content_map	*map;

	map = match_content_map(file, 0);
	if (map)
		return (map->cgihandler);
	return (NULL);
}

/*
 * XXX
 */
#ifdef DEBUG
static void
debug__(int level, const char *fmt, ...)
{
	va_list	ap;
	int savederrno;
	
	/* only log if the level is low enough */
	if (dflag < level)
		return;

	savederrno = errno;
	va_start(ap, fmt);
	vsyslog(LOG_DEBUG, fmt, ap);
	va_end(ap);
	errno = savederrno;
}
#endif /* DEBUG */

/* these are like warn() and err(), except for syslog not stderr */
static void
warning(const char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	if (isatty(STDERR_FILENO)) {
		vfprintf(stderr, fmt, ap);
		fputs("\n", stderr);
	} else
		vsyslog(LOG_INFO, fmt, ap);
	va_end(ap);
}

static void
error(int code, const char *fmt, ...)
{
	va_list ap;

	va_start(ap, fmt);
	if (isatty(STDERR_FILENO)) {
		vfprintf(stderr, fmt, ap);
		fputs("\n", stderr);
	} else
		vsyslog(LOG_ERR, fmt, ap);
	va_end(ap);
	exit(code);
}

/* the follow functions and variables are used in handling HTTP errors */
/* ARGSUSED */
static void
http_error(code, request, log)
	int	code;
	http_req *request;
	const char *log;
{
	static	char buf[BUFSIZ];	/* static so we don't need to malloc? */
	const char *header = http_errors_short(code);
	const char *reason = http_errors_long(code);
	const char *proto = (request && request->proto) ? request->proto : http_11;
	int	size;

	debug((DEBUG_FAT, "http_error %d: %s", code, log));
	if (header == NULL || reason == NULL)
		error(1, "http_errors*() failed (short = %p, long = %p)",
		    header, reason);

	if (request && request->url) {
		escape_html(request);
		size = snprintf(buf, sizeof buf,
		    "<html><head><title>%s</title></head>\n"
		    "<body><h1>%s</h1>\n"
		    "%s: <pre>%s</pre>\n"
		    "<hr><address><a href=\"http://%s/\">%s</a></address>\n"
		    "</body></html>\n",
		    header, header, request->url, reason, myname, myname);
		if (size >= sizeof buf)
			warning("http_error buffer too small, truncated");
	} else
		size = 0;

	(void)printf("%s %s\r\n", proto, header);
	(void)printf("Content-Type: text/html\r\n");
	(void)printf("Content-Length: %d\r\n", size);
	(void)printf("Server: %s\r\n", server_software);
	if (request && request->allow)
		(void)printf("Allow: %s\r\n", request->allow);
	(void)putchar('\r');
	(void)putchar('\n');
	if (size)
		(void)puts(buf);
	fflush(stdout);

	exit(1);
}

/* short map between error code, and short/long messages */
static struct errors_map {
	int	code;			/* HTTP return code */
	const char *shortmsg;		/* short version of message */
	const char *longmsg;		/* long version of message */
} errors_map[] = {
	{ 400,	"400 Bad Request",	"The request was not valid", },
	{ 403,	"403 Forbidden",	"Access to this item has been denied", },
	{ 404, 	"404 Not Found",	"This item has not been found", },
	{ 408, 	"408 Request Timeout",	"This request took too long", },
	{ 417,	"417 Expectation Failed","Expectations not available", },
	{ 500,	"500 Internal Error",	"An error occured on the server", },
	{ 501,	"501 Not Implemented",	"This request is not available", },
	{ 0,	NULL,			NULL, },
};

static const char *help = "DANGER! WILL ROBINSON! DANGER!";

static const char *
http_errors_short(code)
	int	code;
{
	struct errors_map *ep;

	for (ep = errors_map; ep->code; ep++)
		if (ep->code == code)
			return (ep->shortmsg);
	return (help);
}

static const char *
http_errors_long(code)
	int	code;
{
	struct errors_map *ep;

	for (ep = errors_map; ep->code; ep++)
		if (ep->code == code)
			return (ep->longmsg);
	return (help);
}

/* Below are various modified libc functions */

/*
 * returns -1 in lenp if the string ran out before finding a delimiter,
 * but is otherwise the same as strsep.  Note that the length must be
 * correctly passed in.
 */
static char *
strnsep(strp, delim, lenp)
	char	**strp;
	const char *delim;
	ssize_t	*lenp;
{
	char	*s;
	const	char *spanp;
	int	c, sc;
	char	*tok;

	if ((s = *strp) == NULL)
		return (NULL);
	for (tok = s;;) {
		if (lenp && --(*lenp) == -1)
			return (NULL);
		c = *s++;
		spanp = delim;
		do {
			if ((sc = *spanp++) == c) {
				if (c == 0)
					s = NULL;
				else
					s[-1] = '\0';
				*strp = s;
				return (tok);
			}
		} while (sc != 0);
	}
	/* NOTREACHED */
}

/*
 * inspired by fgetln(3), but works for fd's.  should work identically
 * except it, however, does *not* return the newline, and it does nul
 * terminate the string.
 */
static char *
dgetln(fd, lenp)
	int	fd;
	ssize_t	*lenp;
{
	static	char *buffer;
	static	ssize_t buflen = 0;
	ssize_t	len;
	int	got_cr = 0;
	char	c, *nbuffer;

	/* initialise */
	if (buflen == 0) {
		buflen = 128;	/* should be plenty for most requests */
		buffer = malloc(buflen);
		if (buffer == NULL) {
			buflen = 0;
			return NULL;
		}
	}
	len = 0;

	/*
	 * we *have* to read one byte at a time, to not break cgi
	 * programs (for we pass stdin off to them).  could fix this
	 * by becoming a fd-passing program instead of just exec'ing
	 * the program
	 */
	for (; read(fd, &c, 1) == 1; ) {
		debug((DEBUG_OBESE, "dgetln read %c", c));

		if (len == buflen) {
			buflen *= 2;
			debug((DEBUG_OBESE,
			    "dgetln: reallocating buffer to buflen %d", buflen));
			nbuffer = realloc(buffer, buflen);
			if (nbuffer == NULL) {
				free(buffer);
				buflen = 0;
				buffer = NULL;
				return NULL;
			}
			buffer = nbuffer;
		}

		buffer[len++] = c;
		if (c == '\r') {
			got_cr = 1;
			continue;
		} else if (c == '\n') {
			/*
			 * HTTP/1.1 spec says to ignore CR and treat
			 * LF as the real line terminator.  even though
			 * the same spec defines CRLF as the line
			 * terminator, it is recommended in section 19.3
			 * to do the LF trick for tolerance.
			 */
			if (got_cr)
				len -= 2;
			else
				len -= 1;
			break;
		}

	}
	buffer[len] = '\0';
	debug((DEBUG_OBESE, "dgetln returns: ``%s'' with len %d", buffer, len));
	*lenp = len;
	return (buffer);
}

static void *
bozorealloc(ptr, size)
	void	*ptr;
	size_t	size;
{
	void	*p;

	p = realloc(ptr, size);
	if (p == NULL)
		http_error(500, NULL, "memory allocation failure");
	return (p);
}

static void *
bozomalloc(size)
	size_t	size;
{
	void	*p;

	p = malloc(size);
	if (p == NULL)
		http_error(500, NULL, "memory allocation failure");
	return (p);
}

static char *
bozostrdup(str)
	const char	*str;
{
	char	*p;

	p = strdup(str);
	if (p == NULL)
		http_error(500, NULL, "memory allocation failure");
	return (p);
}
