#ifndef _ELEMENT_H_
#define _ELEMENT_H_

//
// Do not change the order of the first five types in order to be compatible with Brahms-0.97!
// ___________________________________________________________________________________________

enum Type { SCORETRACK, DRUMTRACK, MASTERTRACK, AUDIOTRACK, COMMENTTRACK, TRACK,
	    ABSTRACT, ATOM, STRING, POSITION, REFERENCE,
	    EVENT, MASTEREVENT, NOTE, MIDIEVENT, SYMBOL, AUDIOEVENT,
	    ORNAMENT, LYRICS, STEM, EXPRESSION, BOW, RIEMANNFUNCTION,
	    COMPOUND, SONG, PART, SELECTION, VECTOR, TABLE,
	    OPERATION, ADDON, ADDELEMENT, REMOVEELEMENT, CONVERTTRACK, MOVEPART, COPYPART, COPYGHOSTPART, MOVEEVENT, COPYEVENT, CHANGENOTE,
	    GLUENOTE, SPLITNOTE, ADDORNAMENT, REMOVEORNAMENT, ADDSYMBOL,
	    ADDTOSELECTION, REMOVEFROMSELECTION, NEWSELECTION, SELECTLEFT, SELECTRIGHT, UNSELECT, SPLITPART, GLUEPARTS,
	    COPYSELECTION, CUTSELECTION, PASTESELECTION, DELETESELECTION };

enum { DOWN, UP };

#define STEM_UP   1
#define STEM_AUTO 0
#define STEM_DOWN -1

enum GuiEvents { UNDO, REDO, CHANGES, SELECTIONS, MEMORY };

class ostream;
class ifstream;
class Compound;
class Presentation;

class Vector; // for debugging only
class Table;
class PrProgress;

/**
 * The class Element is the most essentail class.
 * All objects related to a song in any way (tracks, parts, notes, events, etc) are Elements. Each Element is of a certain Type, which is
 * returned by the isA() method. E.g. to treat an event ev properly, one may type:
 *
 * <pre>
 *
 * if (ev->isA()==NOTE) {
 *    ...
 * } else if (ev->isA()==MASTEREVENT) {
 *    ...
 * }
 * </pre>
 *
 * The Types used by the API are:
 *
 * Events: AUDIOEVENT, MASTEREVENT, MIDIEVENT, NOTE, SYMBOL,
 *
 * Compounds: SONG, TRACK, PART, TABLE, VECTOR
 * 
 * Operations: ADDELEMENT, REMOVEELEMENT, CONVERTTRACK, MOVEPART, COPYPART, COPYGHOSTPART, MOVEEVENT, COPYEVENT, CHANGENOTE,
 * GLUENOTE, SPLITNOTE, ADDORNAMENT, REMOVEORNAMENT, ADDSYMBOL,
 * ADDTOSELECTION, REMOVEFROMSELECTION, NEWSELECTION, SELECTLEFT, SELECTRIGHT, UNSELECT, SPLITPART, GLUEPARTS,
 * COPYSELECTION, CUTSELECTION, PASTESELECTION, DELETESELECTION
 *
 * Furthermore the Element is the base class for a composite pattern: An Element can either be a compound (which is - more or less - a list!)
 * containing further elements, or an Atom. Examples for Atoms are Strings, Operations, Positions and Events (like Notes). Examples for
 * Compounds are Songs, Tracks, Parts, Tables and Vectors.
 *
 * Thus Elements may have a successor and a predecessor when they are arranged within a compound. All this may become totally clear by taking
 * a look at the API class hierarchy.
 *
 **/
class Element
{
 private:
  Element * _next;
  Element * _prev;
  Presentation * _pr;

  /* commented out, because kdoc ignores the ifdef directives...
     #ifdef DEBUGIT
     static int _total;      // for debugging only
     static Element * _list; // for debugging only
     Element * _nnnn;        // for debugging only
     #endif
  */

 public:
  /**
   * Element is an abstract class. Do not initiate elements youself!
   *
   **/
  Element();

  /**
   * Copy constructor
   **/
  Element(const Element & el);

  /**
   * The destructor contains code only in debugging mode
   **/
  virtual ~Element();

  /* commented out, because kdoc ignores the ifdef directives...
     #ifdef DEBUGIT
     int total() const { return _total; } // for debugging purposes only
     void dump(int i=-1); // for debugging purposes only
     #endif
  */

 protected:
  /**
   * For internal use only!
   **/
  Type _type;

  /**
   * For internal use only!
   **/
  char * spc(int) const;

  /**
   * For internal use only!
   **/
  static Element * next(Element *);

  /**
   * For internal use only!
   **/
  static Element * prev(Element *);

  /**
   * For internal use only!
   **/
  static void splitBefore(Element *);

  /**
   * For internal use only!
   **/
  static Element * first(Element *);

  /**
   * For internal use only!
   **/
  static Element * last(Element *);

  /**
   * For internal use only!
   **/
  static Element * get(int,Element*);

  /**
   * For internal use only!
   **/
  static void append(Element *, Element *);

  /**
   * For internal use only!
   **/
  static Element * preput(Element *, Element *);

  /**
   * For internal use only!
   **/
  static Element * insertBefore(Element *, Element *);

  /**
   * For internal use only!
   **/
  static void insertAfter(Element *, Element *);

  /**
   * For internal use only!
   **/
  static Element * remove(Element *);


  /* This method exchanges the contexts of e1 and e2:
   * before: a1 <=> e1 <=> b1     a2 <=> e2 <=> b2
   * after:  a1 <=> e2 <=> b1     a2 <=> e1 <=> b2
   */
  /**
   * For internal use only!
   **/
  static void exchange(Element * e1, Element * e2);

 public:

  /**
   * Checks whether an element is member of a compound c
   **/
  bool member(Compound * c);

  /**
   * If member of a compound, this method returns the element position within the compound, and -1 otherwise
   **/
  int ord(Compound*);

  /**
   * Returns the type of an element as char*. This has to be reimplemented when new Track types are declared as addons
   **/
  virtual const char * ctype() const;

  /**
   * Returns the type of an element as a Type (integer)
   **/
  Type isA() const;

  /**
   * Returns true if the element is an event (Note, etc.)
   **/
  virtual bool isEvent() const = 0;

  /**
   * Returns true if the element is a track
   **/
  virtual bool isTrack() const = 0;

  /**
   * Overload this, if this element has a presentation, and delegate to it
   **/
  virtual void show();

  /**
   * Overload this, if this element has a presentation, and delegate to it
   **/
  virtual void hide();

  /**
   * Implement this to define output for each element
   **/
  virtual ostream & print(int,ostream&) const = 0;

  /**
   * Implement this to define short output for each element
   **/
  virtual void flush(const char*) const = 0;

  /**
   * Implement this to define the copy process for each element
   **/
  virtual Element * copy() const = 0;

  /**
   * For internal use only!
   **/
  static void loadContent(Compound * list, const char * term, const char * name, Element*(*loadme)(char*,ifstream*&,Element*), ifstream * inPtr, PrProgress * progress=0, int num=0);


  /**
   * For internal use only!
   **/
  static void loadContent(Compound * list, const char * term, int num, const char * name[], Element*(*loadme[])(char*,ifstream*&,Element*), ifstream * inPtr, PrProgress * progress=0, int num=0);

};

ostream & operator<<(ostream&,Element&);

#endif
