/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_auth_ldap.pike,v 1.45.2.2 2004/02/13 16:15:26 vida Exp $
 */

//
//! module: CAMAS: LDAP Auth
//!  LDAP Auth Module for CAMAS.<br />
//!  This module will give the interface between LDAP and CAMAS
//!  to get names, login etc...
//! type: MODULE_PROVIDER
//! cvs_version: $Id: camas_auth_ldap.pike,v 1.45.2.2 2004/02/13 16:15:26 vida Exp $
//

#if constant(Protocols.LDAP) && constant(Protocols.LDAP.client)

#include <module.h>
#include <camas/globals.h>
inherit "module";

#if constant (thread_create)
object global_lock = Thread.Mutex ();
#endif

constant cvs_version   ="$Id: camas_auth_ldap.pike,v 1.45.2.2 2004/02/13 16:15:26 vida Exp $";
constant module_type   = MODULE_PROVIDER;
constant module_name   = "CAMAS: Auth LDAP";
constant module_doc    = "LDAP Auth Module for CAMAS.<br />"
                         "This module will give the interface between LDAP "
                         "and CAMAS to get names, login, etc..."
                         "<br />You can have several Auth modules, the internal priority of these "
                         "modules can be used to specify the order in which the user are "
                         "checked against each module.";
constant module_unique = 1;
constant thread_safe   = 1;

int auth_requests = 0;
int auth_requests_cached = 0;

int hide_imapserver_from_ldap()
{
  return !QUERY(ldapimap);
}

void create() {
#ifdef CAMAS_DEBUG
  defvar("debug", 0, "Debug", TYPE_FLAG,
         "When on, debug messages will be logged in Caudium's debug logfile. "
         "This information is very useful to the developers when fixing bugs.");
#endif
  
  defvar("lotus", 0, "LDAP Server is Lotus Notes", TYPE_FLAG,
         "Klundge for brain damaged Lotus Notes LDAP Server. Set this to on "
         "if Camas Auth LDAP cannot get correctly infos from Lotus Notes LDAP "
         "server.");

  defvar("dnsname","","The sitewide DNS domain name",TYPE_STRING,
         "The DNS domain name used for CAMAS webmail");

  defvar("emaillogin",1,"Login using email address",TYPE_FLAG,
         "Use the email address to login instead of the imap login");

  defvar("showdns",1,"Set the dns domain on login", TYPE_FLAG,
         "Set the right part of the DNS domain on login. This make less "
         "to type for the users, but in the other hand avoid users to "
         "login under another email if the imap server can handle lots of "
         "different domains...");

  defvar("ldaphosts","","LDAP Auth servers",TYPE_STRING,
         "Comma separated of LDAP auth servers to use. All hosts will be "
         "tested one by one in the order written here.");

  defvar("usecache", 1, "Cache entries", TYPE_FLAG,
         "This flag defines whether the module will cache the "
         "database entries. Makes accesses faster, but changes "
         "in the database will not show immediately. "
         "<b>Recommended</b>.");

  defvar("ldapimap",0,"IMAP Server:Get from LDAP", TYPE_FLAG,
         "If set then the module will search in the IMAP server in the "
         "LDAP server.");
  
  defvar("ldap_imapserver", "localhost", "IMAP Server:IMAP server", TYPE_STRING,
  	     "The default IMAP server if Get IMAP from LDAP is not set or if no "
         "IMAP server is found");
  
  defvar("ldap_imapport", 143, "IMAP Server:IMAP server port", TYPE_INT,
         "The port of the IMAP server we connect to");

  defvar("basednimap","ou=mail,dc=domain,dc=com",
         "IMAP login:Base DN",TYPE_STRING,
         "The base DN to use to get the IMAP login.");

  defvar("basednimapsrv","ou=mail,dc=domain,dc=com",
         "IMAP Server:Base DN",TYPE_STRING,
         "The base DN to use to get the IMAP server.",0,hide_imapserver_from_ldap);

  defvar("filterimap","mail=","IMAP login:Filter",TYPE_STRING,
         "The filter to use to get the IMAP login.");

  defvar("filterimapsrv","uid=","IMAP Server:Filter",TYPE_STRING,
         "The filter to use to get the IMAP server.",0,hide_imapserver_from_ldap);
  defvar("basednname","ou=mail,dc=domain,dc=com",    
         "Webmail names:Base DN",TYPE_STRING,   
         "The base DN to use to get user's name.");

  defvar("filtername","uid=","Webmail names:Filter",TYPE_STRING,   
         "The filter to use to get user's name.");

  defvar("ldapversion", 2, "LDAP Version", TYPE_INT_LIST,
         "The LDAP protocol version to use.", ({ 2, 3 }));

  defvar("imaplogin","uid","IMAP login:LDAP attribute",TYPE_STRING,
         "The LDAP IMAP login attribute to use. <b>MUST</b> be lower case.");

  defvar("imapcn","cn","IMAP login:LDAP Name attribute",TYPE_STRING,
         "The LDAP attribute to find the Name");

  defvar("imapsn","sn","IMAP login:LDAP Surname attribute",TYPE_STRING,
         "The LDAP attribute to find the Surname");

  defvar("imapmail","mail","IMAP login:LDAP Mail attribute",TYPE_STRING,
         "The LDAP attribute where is stored the Email address");

  defvar("ldapuser","","LDAP user",TYPE_STRING,
         "The username used to connect on the server. NOTE ! This is a LDAP "
         "user format and can be empty depending on the configuration of your "
         "LDAP server.");

  defvar("ldappasswd","","LDAP Password", TYPE_STRING,
         "The password you should you to connect. Could be empty.");

  defvar("imapsrvattr","mailhost","IMAP Server:LDAP attribute",TYPE_STRING,
         "The LDAP attribute where is stored the IMAP server name.",
         0,hide_imapserver_from_ldap);
}

string status () {
  string ret = "LDAP Authentication module.";
  ret += "<br /><br />Cache statistics<br />";
  ret += "Authentications: " + auth_requests + "<br />";
  ret += "Cache lookups (success): " + auth_requests_cached + "<br />";

  if (auth_requests) {
    float cache_pct = ((float)auth_requests_cached / (float)auth_requests) * 100.0;
    ret += sprintf ("Cache use: %3.2f%%<br />", cache_pct);
  }

  return ret;
}

string query_provides () {
  return ("camas_auth");
}

/*
 * What we provide here
 */

private mapping ldap_auth (string login, int auth) {
  /*
     auth= 0: IMAP login query
     auth= 1: webmail names query
     auth= 2: IMAP server name query
  */
  string basedn, filter;
  switch (auth) {
  case 0: basedn = QUERY(basednimap);
    filter = QUERY(filterimap);
    break;
  case 1: basedn = QUERY(basednname);
    filter = QUERY(filtername);
    break;
  case 2: basedn = QUERY(basednimapsrv);
    filter = QUERY(filterimapsrv);
    break;
  default: throw(({ "Unknown auth action : "+(string)auth, backtrace()}));
  }

  if (QUERY(emaillogin)) {
    if (QUERY(showdns)) {
      if (has_value (login, "@"))
        login = (login / "@")[0];

      string|int dnsname = getdnsname ();
      if (stringp (dnsname) && !auth)
        login += "@" + dnsname;
    }
  }

  array hosts = QUERY (ldaphosts) / ","; // FIXME: compute in module start() function ?
  filter = filter + login;

  foreach (hosts, string host) {
    CDEBUG("querying "+login+" on host " + host + ".");

#if constant (thread_create)
    object lock = global_lock->lock ();
#endif
    mixed err = catch {
      object con = Protocols.LDAP.client ("ldap://" + host);
      if (objectp (con)) {
        con->bind (QUERY(ldapuser), QUERY(ldappasswd), QUERY (ldapversion));

        if (objectp (con)) { // oliv3 FIXME wrong! catch this
          if(QUERY(lotus))
            con->set_basedn (filter);
          else
            con->set_basedn (basedn);
          con->set_scope (2);

          object en = con->search (filter);
          if (objectp (en)) {
            con->unbind ();
            if (en->num_entries () == 1) {
#if constant (thread_create)
              destruct (lock);
#endif
              mapping answer = en->fetch (1);
              mapping answer2 = ([ ]);
              foreach (indices (answer), string i)
                answer2 += ([ lower_case (i) : answer[i] ]);
              destruct (con);
              return answer2;
            }
            else
              CDEBUG("0 entries found for filter " + filter);
          }
          else
            CDEBUG("Can't search with filter " + filter);
          destruct (con);
        }
        else
          CDEBUG("Fail: can't connect");
      }
    };
#if constant (thread_create)
    destruct (lock);
#endif
    if(err)
      report_error("error in camas_auth_ldap.pike: %s\n", describe_backtrace(err));
  }

  return 0;
}

//
//! method: int version(void)
//!  Give the CAMAS_AUTH api version
//!  supported by the module
//! returns:
//!  the version of the API
//! note:
//!  The base API is 1. But if we provide v2, we
//!  *MUST* be backward compatible with v1.
//
int version() {
  return 2;
}

//
//! method: string|int getlogin(string login)
//!  Return the imap login to be used to connect into IMAP
//!  server. Or a int to throw with an error
//! arg: string login
//!  The login typed/entered on the CAMAS login string.
//!  This can be an email, a part of an email or an imap id.
//! returns:
//!  a string : the imap login to be used.<br />
//!  an int : an error code if there is an error, see notes.
//! note:
//!  error codes :<br />
//!  0 : Error (standart code or for unknown error type)<br />
//!  1 : Bad password<br />
//!  2 : Access denied<br />
//!  3 : Account is locked<br />
//!  4 : Change your password<br />
//
string|int getlogin(string login) {
  ++auth_requests;

  CDEBUG("getlogin("+login+");");

  mapping|int auth_result = 0;

  if (QUERY (usecache)) {
    auth_result = cache_lookup ("camas_auth_ldap", login);
    if (auth_result) {
      CDEBUG(sprintf ("getlogin: cache found %O", auth_result));
      auth_requests_cached++;
      if (auth_result[QUERY (imaplogin)])
        return auth_result[QUERY (imaplogin)][0];
      else
        return 1;
    }
  }

  auth_result = ldap_auth (login, 0);

  CDEBUG(sprintf("auth_result: %O\n", auth_result));

  if (auth_result && auth_result[QUERY (imaplogin)] && QUERY (usecache)) {
    CDEBUG(sprintf ("getlogin: cache set %s => %O", login, auth_result));
    cache_set ("camas_auth_ldap", login, auth_result);
  }

  return (auth_result && auth_result[QUERY (imaplogin)]) ? auth_result[QUERY (imaplogin)][0] : 1;
}

//
//! method: string|int getdnsname(void)
//!  Gets the sitewide email domain name that can be used
//!  for email login or to complete unqualified emails
//! returns:
//!  a string : the dns domain name<br />
//!  an int : an error code if there is an error, see notes.
//! note:
//!  error codes :<br />
//!  0 : Error (standart code or unknown error type)<br />
//!  1 : There is no domain name set, use whole email or imap login
//
string|int getdnsname() {
  if(QUERY(dnsname)=="")
    return 1;
  else
    return QUERY(dnsname);
}

//
//! method: int getemaillogin(void)
//!  Do the email is used instead of the imap login as
//!  login method ?
//! returns:
//!  a int : 1 = Yes, 0 = No
//
int getemaillogin() {
  return (int)QUERY(emaillogin);
}

//
//! method: array|int getfullnames(string login)
//!  Get the name and the surname of the specified imap login
//! arg: string login
//!  The imap login
//! returns:
//!  an array : the name and surname in the form ({ "surname", "name", "address" })<br />
//!  an int : an error code if there is an error, see notes.
//! note:
//!  error codes :<br />
//!  0 : Error (standard error code or unknown error type)<br />
//!  1 : Empty set
//
array|int getfullnames(string login) {
  ++auth_requests;
  CDEBUG("getfullnames("+login+");");

  mapping|int auth_result = 0;
  string name = "";
  string surname = "";
  string email = login;

  if(QUERY(usecache)) {
    auth_result = cache_lookup("camas_auth_ldap",login);
    if(auth_result) {
      CDEBUG(sprintf("getfullnames: cache found %O", auth_result));
      auth_requests_cached++;
      if(auth_result[QUERY(imapcn)])
        name = utf8_to_string(auth_result[QUERY(imapcn)][0]);
      if(auth_result[QUERY(imapsn)])
        surname = utf8_to_string(auth_result[QUERY(imapsn)][0]);
      if(auth_result[QUERY(imapmail)])
        email = utf8_to_string(auth_result[QUERY(imapmail)][0]);
      return ({ surname, name, email });
    }
  }
  auth_result = ldap_auth(login, 1);
  if(auth_result) {
    if (QUERY (usecache)) {
      CDEBUG(sprintf ("getfullnames: cache set %s => %O", login, auth_result));
      cache_set ("camas_auth_ldap", login, auth_result);
    }
    mixed err;
    if(auth_result[QUERY(imapcn)])
    {
      err = catch {
        name = utf8_to_string(auth_result[QUERY(imapcn)][0]);
      };
      if(err)
        name = auth_result[QUERY(imapcn)][0];
    }
    if(auth_result[QUERY(imapsn)])
    {
      err = catch {
        surname = utf8_to_string(auth_result[QUERY(imapsn)][0]);
      };
      if(err)
        surname = auth_result[QUERY(imapsn)][0];
    }
    if(auth_result[QUERY(imapmail)])
    {
      err = catch {
        email = utf8_to_string(auth_result[QUERY(imapmail)][0]);
      };
      if(err)
        email = auth_result[QUERY(imapmail)][0];
    }
  }
  return ({ surname, name, email });
}

//
//! method: string auth_error(int error)
//!  To be documented !
//
string auth_error (int error) {
  switch (error) {
  case 1 : return "Unknown user";
  default : return "Unknown user";
  }
}

//
// Auth V2 API
//

//
//! method: mapping camas_provide()
//!  Return the capabilities of this CAMAS auth module.
//!  Used when multiple auth modules are installed in a Camas enviroment.
//! returns:
//!  A mapping with the following format :
//!  (["function":"auth" ])
//
mapping camas_provides() {
  return ([ "function":"auth" ]);
}

//
//! method: string|int getimapsrv(string login)
//!  Return imap server name for the IMAP login given
//! returns:
//!  a string : the imap server name for the IMAP login given<br />
//!  an int : error code, usualy 0 = function desactived or IMAP server
//!  not found.
//
string|int getimapsrv(string login)
{
  ++auth_requests;
  CDEBUG("getimapserver("+login+");");
  if(QUERY(ldapimap)) {

    mapping|int auth_result = 0;

    if(QUERY(usecache)) {
      auth_result = cache_lookup("camas_auth_ldap",login);
      if(auth_result) {
        CDEBUG(sprintf("getimapserver: cache found %O", auth_result));
        auth_requests_cached++;
        if(auth_result[QUERY(imapsrvattr)])
          return auth_result[QUERY(imapsrvattr)][0];
        return 0;
      }
    }
    auth_result = ldap_auth(login, 2);
    if(auth_result) {
      if (QUERY (usecache)) {
        CDEBUG(sprintf ("getiampserver: cache set %s => %O", login, auth_result));
        cache_set ("camas_auth_ldap", login, auth_result);
      }
      if(auth_result[QUERY(imapsrvattr)])
        return auth_result[QUERY(imapsrvattr)][0];
      return 0;
    }
  }
  else
    return 0;
}

//
//! method: mapping(string:string|int) getimapserver(string login)
//!  Return the imap informations to be used to connect into IMAP
//!  server. Or a int to throw with an error
//! arg: string login
//!  The login typed/entered on the CAMAS login string.
//!  This can be an email, a part of an email or an imap id.
//! returns:
//!  a mapping : maps configuration name to their values<br />
//!   usual names include domain, imapport, imapserver.
//!  an int : an error code if there is an error.
//
mapping(string:string|int)|int getimapserver(string login)
{
  CDEBUG("getimapserver from AUTH LDAP\n");
  string|int imapsrv = getimapsrv(login);
  if(!imapsrv)
    imapsrv = QUERY(ldap_imapserver);
  mapping res = ([ 
                 "imapserver":   imapsrv,
                 "imapport": QUERY(ldap_imapport),
                ]);
  array loginsplit = login / "@";
  if(sizeof(loginsplit) == 2 && loginsplit[1])
    res += ([ "domain": loginsplit[1] ]);
  CDEBUG(sprintf("result=%O\n", res));
  return res;
}

#endif

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: debug
//! When on, debug messages will be logged in Caudium's debug logfile. This information is very useful to the developers when fixing bugs.
//!  type: TYPE_FLAG
//!  name: Debug
//
//! defvar: lotus
//! Klundge for brain damaged Lotus Notes LDAP Server. Set this to on if Camas Auth LDAP cannot get correctly infos from Lotus Notes LDAP server.
//!  type: TYPE_FLAG
//!  name: LDAP Server is Lotus Notes
//
//! defvar: dnsname
//! The DNS domain name used for CAMAS webmail
//!  type: TYPE_STRING
//!  name: The sitewide DNS domain name
//
//! defvar: emaillogin
//! Use the email address to login instead of the imap login
//!  type: TYPE_FLAG
//!  name: Login using email address
//
//! defvar: showdns
//! Set the right part of the DNS domain on login. This make less to type for the users, but in the other hand avoid users to login under another email if the imap server can handle lots of different domains...
//!  type: TYPE_FLAG
//!  name: Set the dns domain on login
//
//! defvar: ldaphosts
//! Comma separated of LDAP auth servers to use. All hosts will be tested one by one in the order written here.
//!  type: TYPE_STRING
//!  name: LDAP Auth servers
//
//! defvar: usecache
//! This flag defines whether the module will cache the database entries. Makes accesses faster, but changes in the database will not show immediately. <b>Recommended</b>.
//!  type: TYPE_FLAG
//!  name: Cache entries
//
//! defvar: ldapimap
//! If set then the module will search in the IMAP server in the LDAP server.
//!  type: TYPE_FLAG
//!  name: IMAP Server:Get from LDAP
//
//! defvar: ldap_imapserver
//! The default IMAP server if Get IMAP from LDAP is not set or if no IMAP server is found
//!  type: TYPE_STRING
//!  name: IMAP Server:IMAP server
//
//! defvar: ldap_imapport
//! The port of the IMAP server we connect to
//!  type: TYPE_INT
//!  name: IMAP Server:IMAP server port
//
//! defvar: basednimap
//! The base DN to use to get the IMAP login.
//!  type: TYPE_STRING
//!  name: IMAP login:Base DN
//
//! defvar: basednimapsrv
//! The base DN to use to get the IMAP server.
//!  type: TYPE_STRING
//!  name: IMAP Server:Base DN
//
//! defvar: filterimap
//! The filter to use to get the IMAP login.
//!  type: TYPE_STRING
//!  name: IMAP login:Filter
//
//! defvar: filterimapsrv
//! The filter to use to get the IMAP server.
//!  type: TYPE_STRING
//!  name: IMAP Server:Filter
//
//! defvar: ldapversion
//! The LDAP protocol version to use.
//!  type: TYPE_INT_LIST
//!  name: LDAP Version
//
//! defvar: imaplogin
//! The LDAP IMAP login attribute to use. <b>MUST</b> be lower case.
//!  type: TYPE_STRING
//!  name: IMAP login:LDAP attribute
//
//! defvar: imapcn
//! The LDAP attribute to find the Name
//!  type: TYPE_STRING
//!  name: IMAP login:LDAP Name attribute
//
//! defvar: imapsn
//! The LDAP attribute to find the Surname
//!  type: TYPE_STRING
//!  name: IMAP login:LDAP Surname attribute
//
//! defvar: imapmail
//! The LDAP attribute where is stored the Email address
//!  type: TYPE_STRING
//!  name: IMAP login:LDAP Mail attribute
//
//! defvar: ldapuser
//! The username used to connect on the server. NOTE ! This is a LDAP user format and can be empty depending on the configuration of your LDAP server.
//!  type: TYPE_STRING
//!  name: LDAP user
//
//! defvar: ldappasswd
//! The password you should you to connect. Could be empty.
//!  type: TYPE_STRING
//!  name: LDAP Password
//
//! defvar: imapsrvattr
//! The LDAP attribute where is stored the IMAP server name.
//!  type: TYPE_STRING
//!  name: IMAP Server:LDAP attribute
//

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */

