(* $Id: rpc_over_http_client.ml 182 2004-05-25 16:49:11Z gerd $
 * ----------------------------------------------------------------------
 *
 *)

open Unixqueue
open Rpc

type gateway =
    { esys   : Unixqueue.event_system;
      group  : Unixqueue.group;
      url    : string;
      http   : Http_client.pipeline;
      socket : Unix.file_descr;
      trans  : Rpc_transport.t;
      mutable exn : exn -> unit;
      mutable has_output_resource : bool;
      mutable replies : Rpc_packer.packed_value list;
    }
;;


let debug = ref false;;



(* The connection_event_handler is almost the same as the handler for
 * Rpc_server.
 *)

let connection_event_handler gw esys esys' ev =
    let conn_d = Rpc_transport.descriptor gw.trans in
    match ev with

      (*** event: input data have been arrived ***)

      Input_arrived(_,d) ->
	if d <> conn_d then raise (Equeue.Reject);

	if Rpc_transport.at_eof gw.trans then begin
	  if !debug then prerr_endline "EH: eof";

	  remove_resource gw.esys gw.group (Wait_in conn_d);
	  if gw.has_output_resource then
	    remove_resource gw.esys gw.group (Wait_out conn_d);

          (* TODO: raise error if incomplete message received *)

	  raise Equeue.Terminate
	end
	else begin

	  (* process all what is buffered: *)
	  let again = ref true in
	  while !again do

      	    (* Receive next part of the message *)

	    if !debug then prerr_endline "EH: receive_part";
	    ignore(Rpc_transport.receive_part gw.trans);

	    (* Is the message complete? Yes: process it *)

	    if Rpc_transport.is_message_complete gw.trans then begin
	      if !debug then prerr_endline "EH: got message";
	      let message = Rpc_packer.string_of_packed_value
			      (Rpc_transport.get gw.trans) in
	      Rpc_transport.clean_input gw.trans;

	      (* A message has arrived: Make a HTTP POST request, and add
	       * it to the HTTP pipeline:
	       *)
	      let post_msg = new Http_client.post_raw gw.url message in
	      gw.http # add_with_callback
		post_msg
		(fun _ ->
		   if not gw.has_output_resource then begin
	      	     add_resource gw.esys gw.group (Wait_out d, (-1.0));
		     gw.has_output_resource <- true;
		   end;
		   try
		     let raw_reply = post_msg # get_resp_body() in
		     let reply = Rpc_packer.packed_value_of_string raw_reply in
		     gw.replies <- gw.replies @ [reply];
		   with
		       err -> gw.exn err
		);
	    end;

	    again := not (Rpc_transport.is_buffer_empty gw.trans);
	  done;
	end

      (*** event: ready to output data ***)

    | Output_readiness(_,d) ->
	if d <> conn_d then raise (Equeue.Reject);

	if Rpc_transport.is_sending_complete gw.trans then begin
	  (* send next reply *)
	  match gw.replies with
	    next_reply :: other_replies ->
	      if !debug then prerr_endline "EH: next reply";
	      Rpc_transport.put gw.trans next_reply;
	      gw.replies <- other_replies
	  | [] ->
	    (* this was the last reply in the queue *)
	      if !debug then prerr_endline "EH: last reply";
	      Rpc_transport.clean_output gw.trans;
	      remove_resource gw.esys gw.group (Wait_out d);
	      gw.has_output_resource <- false;
	end;

	if not (Rpc_transport.is_sending_complete gw.trans) then begin
	  (* next part of call message *)
	  if !debug then prerr_endline "EH: send_part";
	  ignore(Rpc_transport.send_part gw.trans);
	  ()
	    (* TODO: catch EPIPE errors...
	     *)
	end

    | _ ->
	raise (Equeue.Reject)
;;


let log_to_stderr exn =
  match exn with
    | Http_client.Http_error(code,e) ->
	prerr_endline("Uncaught exception (HTTP): " ^ string_of_int code)
    | _ ->
	prerr_endline("Uncaught exception: " ^ Printexc.to_string exn)
;;


let close_endpoint gw d =
  Unix.close d
;;


let create esys url =
  let endpoint1, endpoint2 = Unix.socketpair Unix.PF_UNIX Unix.SOCK_STREAM 0 in
  let gw =
    { esys = esys;
      group = Unixqueue.new_group esys;
      url = url;
      http = new Http_client.pipeline;
      socket = endpoint1;
      trans = Rpc_transport.create endpoint2 Tcp BiPipe;
      exn = log_to_stderr;
      has_output_resource = false;
      replies = []
    }
  in
  gw.http # set_event_system esys;
  add_handler esys gw.group (connection_event_handler gw);
  add_resource esys gw.group (Wait_in endpoint2, -1.0);
  add_close_action esys gw.group (endpoint2, (close_endpoint gw));
  gw
;;


let http_pipeline gw = gw.http;;

let rpc_client_connector gw = Rpc_client.Descriptor gw.socket;;

let set_exception_handler gw handler =
  gw.exn <- handler
;;

let close gw =
  let d = Rpc_transport.descriptor gw.trans in
  remove_resource gw.esys gw.group (Wait_in d)
;;


let verbose b = debug := b;;
