;;;
;;; Example OpenGL program used to test the Lisp OpenGL and GLUT bindings
;;;
;;; Mark Owen Riedl
;;; 02 May 2001
;;;

(use-package :gl)

;;; Global reference to the window object.  Used to manipulate the graphics window.
(defvar *window* nil)

;;; This is a callback function that will be called by (glutMainLoop) every time the window 
;;; must be refreshed.
(ff:defun-foreign-callable display-callback ()
 (display))

;;; The display function tells OpenGL how to draw the scene.
;;; This simple example draws a red sphere.
(defun display ()
   (glMatrixMode GL_MODELVIEW)
   (glClear (+ GL_COLOR_BUFFER_BIT GL_DEPTH_BUFFER_BIT))
   (glColor3f 1f0 1f0 1f0)
   (glPushMatrix)
   ; set the diffuse color of the sphere
   (let ((diffuse (make-array 4 :element-type 'single-float :initial-contents '(0.0 0.0 1.0 1.0))))
      (glMaterialfv GL_FRONT  GL_DIFFUSE  diffuse)
      ; create the sphere
      (glutSolidSphere 1d0 50 50))
   (glPopMatrix)
   (glFlush)
   (glutSwapBuffers))

;;; This is a callback function that is called by OpenGL every time a key is pressed on the
;;; keyboard. k is the key (character).  x & y are cursor positions in the window when the key
;;; is pressed.
(ff:defun-foreign-callable keyboard-callback (k x y)
 ; Handle the escape key
 (case (character k)
   (#\Escape
    (glutDestroyWindow *window*)
    (break))))

;;; Register the callback functions.  The values returned by (ff:register-foreign-callable)
;;; are needed to initialize the callbacks in OpenGL.
(setq display-callback (ff:register-foreign-callable 'display-callback))
(setq keyboard-callback (ff:register-foreign-callable 'keyboard-callback))


;;; The main function initializes the OpenGL/GLUT system, registers the callback functions
;;; required for the system to work, and sets the graphics rendering environment.
;;; The main function's last responsibility is to hand control over to the OpenGL/GLUT
;;; system by calling (glutmainloop) which never returns.  (glutmainloop) handles
;;; keyboard and mouse events and is responsible for calling the display callback function
;;; every time the graphics window needs to be refreshed.
(defun main ()
   ; Set up the graphics window
   (glutInitDisplayMode (+ GLUT_DOUBLE GLUT_RGB GLUT_DEPTH))
   (glutInitWindowPosition 0 0)
   (glutInitWindowSize 400 400)
   (setq *window* (glutCreateWindow "sphere"))
   
   ; Set up the callbacks in OpenGL/GLUT
   (glutDisplayFunc display-callback)
   (glutKeyboardFunc keyboard-callback)
   
   ; Set the initial rendering parameters
   (glMatrixMode GL_PROJECTION)
   (glLoadIdentity)
   (gluPerspective 70d0 1d0 1d0 1000d0)
   (gluLookAt 0d0 0d0 5d0 0d0 0d0 0d0 0d0 1d0 0d0)

   (glMatrixMode GL_MODELVIEW)
   (glLoadIdentity)
   (glClearDepth 1d0)
   (glClearColor 0f0 0f0 0f0 0f0)
   (glEnable GL_DEPTH_TEST)

   ; Material use and lighting is tricky since it requires the passing of arrays
   (let ((specular (make-array 4 :element-type 'single-float 
                     :initial-contents '(0.75f0 0.75f0 0.75f0 1.0f0)))
         (shininess (make-array 1 :element-type 'single-float 
                      :initial-contents '(25.0f0)))
         (light (make-array 4 :element-type 'single-float 
                  :initial-contents '(5.0f0 5.0f0 5.0f0 1.0f0)))
         (white (make-array 4 :element-type 'single-float 
                  :initial-contents '(1.0f0 1.0f0 1.0f0 1.0f0))))
      (glShadeModel GL_SMOOTH)
      (glMaterialfv GL_FRONT GL_SPECULAR specular)
      (glMaterialfv GL_FRONT GL_SHININESS shininess)
      (glLightfv GL_LIGHT0 GL_POSITION light)
      (glLightfv GL_LIGHT0 GL_DIFFUSE white)
      (glLightfv GL_LIGHT0 GL_SPECULAR white)
      (glEnable GL_LIGHTING)
      (glEnable GL_LIGHT0)
      (glEnable GL_NORMALIZE))
   
   ; Start the OpenGL/GLUT system processing.  This call does not return.
   (glutMainLoop))

