;;;
;;; FOREIGN-FUNCTION INTERFACE
;;;
;;;
;;;
;;; The format of the function definitions is as follows:
;;;
;;; (FOREIGN-FUNCTION name ([param1 [param2 ...]]) param "foreign-name")
;;;
;;; param = {POINTER, STRING, CHAR, UNSIGNED-CHAR, SHORT, UNSIGNED-SHORT,
;;;          INT, UNSIGNED-INT, LONG, UNSIGNED-LONG, FLOAT, DOUBLE,
;;;          (POINTER composite)}
;;;
;;; composite = {VOID, CHAR, UNSIGNED-CHAR, SHORT, UNSIGNED-SHORT,
;;;              INT, UNSIGNED-INT, LONG, UNSIGNED-LONG, FLOAT, DOUBLE,
;;;              STRUCT, FUNCTION}.
;;; 
;;; Right now all composite types are treated as pointers (void* in C;
;;; INT in GCL).
;;;
;;; Composite types are usally allocated on the C side (and treated
;;; as a pointer in LISP).  (See the functions in misc-bindings.lsp.)
;;;
;;; It is possible to allocate composite types (like arrays) in LISP and 
;;; pass a pointer to a function.  To do this "wrap" the argument (ie., 
;;; like (int-ptr x), (floata-ptr x), etc.) before calling the C function.
;;; (See types below.)
;;;
;;;
;;;
;;; The macro definitions are as follows:
;;;
;;; (FOREIGN-DEFINE name value)
;;;
;;; value is any LISP expression (string, symbol or number).
;;;
;;;
;;;
;;; Finally, the function (EXPORT-ALL-FOREIGN-SYMBOLS) exports all symbols.
;;; (Valid only if we are in a package already.)
;;;
;;;
;;;
;;; This code is based on "foreign.lsp" by Paul Viola.
;;;
;;; Richard Mann
;;; 31 October 1996
;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;
;;; Low-level accessors for GCL data:
;;;  {INT,FLOAT,DOUBLE}-ref access simple types,
;;;  {CHARA,INTA,FLOATA,DOUBLEA}-ref access arrays of simple types,
;;;  STRUCT-REF accesses the data field of a struct.
;;;

(defCfun "static void* chara_ptr(object s)" 0 
" return(s->st.st_self);")
(defentry chara-ptr (object) (int "chara_ptr"))

(defCfun "static void* int_ptr(object s)" 0 
" return(&fix(s));")
(defentry int-ptr (object) (int "int_ptr"))

(defCfun "static void* inta_ptr(object s)" 0 
" return(s->fixa.fixa_self);")
(defentry inta-ptr (object) (int "inta_ptr"))

(defCfun "static void* float_ptr(object s)" 0 
" return(&sf(s));")
(defentry float-ptr (object) (int "float_ptr"))

(defCfun "static void* floata_ptr(object s)" 0 
" return(s->sfa.sfa_self);")
(defentry floata-ptr (object) (int "floata_ptr"))

(defCfun "static void* double_ptr(object s)" 0 
" return(&lf(s));")
(defentry double-ptr (object) (int "double_ptr"))

(defCfun "static void* doublea_ptr(object s)" 0 
" return(s->lfa.lfa_self);")
(defentry doublea-ptr (object) (int "doublea_ptr"))

(defCfun "static void* struct_ptr(object s)" 0 
" return(s->str.str_self);")
(defentry struct-ptr (object) (int "struct_ptr"))

(eval-when (load eval)
   
 ;;
 ;; Map foreign types to types used by GCL
 ;;

 (defvar foreign-functions '())
 (defvar foreign-defines '())

 (defun parameter-type (p)
   (case p
	 (void 'void)
	 (string 'string)
	 ((short unsigned-short int unsigned-int long unsigned-long) 'int)
	 (float 'float)
	 (double 'double)
	 ((char unsigned-char) 'char)
	 (otherwise 'int)))

 (defun extern-type (p)
   (case p
	 (void "void")
	 (string "char*")
	 (short "short")
	 (unsigned-short "unsigned short")
	 (int "int")
	 (unsigned-int "unsigned")
	 (long "long")
	 (unsigned-long "unsigned long")
	 (float "float")
	 (double "double")
	 (char "char")
	 (unsigned-char "unsigned char")
	 (otherwise "void*")))

 ;;
 ;; Macro to do translation.
 ;;
 ;; All calls translated to "defentry" form.  (Pointer types just treated
 ;; as int.)
 ;;
 ;; Note! In addition to the LISP "defentry" form, you need to put
 ;; a C "extern" definition before each function so that the types are
 ;; properly converted to C.
 ;;

 (defmacro foreign-function (lisp-name parameters return-parameter c-name)
   (push lisp-name foreign-functions)
   (let* ((parameter-types
	   (mapcar #'parameter-type parameters))
	  (return-parameter-type (parameter-type return-parameter))
	  (extern-types
	   (mapcar #'extern-type parameters))
	  (return-extern-type (extern-type return-parameter))
	  ;; extern <ret-type> <c-name>(<type1>, <type2>, ... ,<typen>);
	  (extern-declaration
	   (concatenate
	    'string "extern " return-extern-type " " c-name "("
	    ;; Concatenate strings with commas between terms.  Messy!
	    (let ((l extern-types)
		  (s ""))
	      (loop (when (null l) (return s))
		    (setf s (concatenate 'string s (first l)))
		    (setf l (rest l))
		    (unless (null l) (setf s (concatenate 'string s ", ")))))
	    ;; Add final bracket
	    ");")))
     ;;
     `(progn
	(clines ,extern-declaration)
	(defentry ,lisp-name ,parameter-types (,return-parameter-type
					       ,c-name)))))
 
 ;;
 ;; Simple treatment of CPP "define" macro for now; may expand later.
 ;;
 
 (defmacro foreign-define (a b)
   (push a foreign-defines)
   `(defconstant ,a ,b))
 
 ;;
 ;; Generate externs for symbols
 ;;

 (defmacro export-all-foreign-symbols ()
   `(progn
      (export ',(reverse foreign-defines))
      (export ',(reverse foreign-functions))))

 ) ;; eval-when

