;;; -*- Syntax: Ansi-Common-Lisp; Base: 10; Mode: lisp; Package: odcl -*-
;;; $Id: profile.lisp,v 1.15 2003/03/24 21:51:31 adam Exp $
;;;
;;; Copyright (c) 2000 - 2003 onShore Development, Inc.

(in-package :profile)

(defun profile-pkg (package)
  (let ((package (if (packagep package)
		     package
		     (find-package package))))
    (do-symbols (symbol package (values))
      (when (and (eq (symbol-package symbol) package)
		 (fboundp symbol)
                 (not (macro-function symbol))
                 (not (equal #\( (aref (symbol-name symbol) 0))))
	(profile-1-function symbol nil)))))

(defun %report-times-2 (count &key sorter)
  (declare (optimize (speed 0)))
  (unless (boundp '*call-overhead*)
    (compute-time-overhead))
  (let ((info ()))
    (dolist (name *timed-functions*)
      (let ((pinfo (profile-info-or-lose name)))
	(unless (eq (fdefinition name)
		    (profile-info-new-definition pinfo))
	  (warn "Function ~S has been redefined, so times may be inaccurate.~@
	         PROFILE it again to record calls to the new definition."
		name))
	(multiple-value-bind
              (calls time consing profile callers)
	    (funcall (profile-info-read-time pinfo))
	  (if (not (zerop calls))
	      (push (make-time-info name calls
				    (compensate-time calls time profile)
				    consing
				    (sort (copy-seq callers)
					  #'>= :key #'cdr))
		    info)))))
    (if sorter
        (setq info (subseq (sort info sorter) 0 count))
        (setq info (subseq (sort info #'>= :key #'time-info-time) 0 count)))
    (format *trace-output*
	    "~&  Seconds  |  Consed   |  Calls  |  Sec/Call  |  Name:~@
	       ------------------------------------------------------~%")
    (let ((total-time 0.0)
	  (total-consed 0)
	  (total-calls 0))
      (dolist (time info)
	(incf total-time (time-info-time time))
	(incf total-calls (time-info-calls time))
	(incf total-consed (time-info-consing time))
	(format *trace-output*
		"~10,3F | ~9:D | ~7:D | ~10,5F | ~S~%"
		(time-info-time time)
		(time-info-consing time)
		(time-info-calls time)
		(/ (time-info-time time) (float (time-info-calls time)))
		(time-info-name time))
	(let ((callers (time-info-callers time)))
	  (when callers
	    (dolist (x (subseq callers 0 (min (length callers) 5)))
	      (format *trace-output* "~10:D: " (cdr x))
	      (print-caller-info (car x) *trace-output*)
	      (terpri *trace-output*))
	    (terpri *trace-output*))))
      (format *trace-output*
	      "------------------------------------------------------~@
	      ~10,3F | ~9:D | ~7:D |            | Total~%"
	      total-time total-consed total-calls)
      (format *trace-output*
	      "~%Estimated total profiling overhead: ~4,2F seconds~%"
	      (* *total-profile-overhead* (float total-calls))))
    (values)))

(defun report-not-called (&aux no-call)
  (declare (optimize (speed 0)))
  (dolist (name *timed-functions*)
    (let* ((pinfo (profile-info-or-lose name))
           (calls (funcall (profile-info-read-time pinfo))))
      (if (zerop calls)
          (push name no-call))))
  (sort no-call #'string<
        :key #'(lambda (n)
                 (cond ((symbolp n)
                        (symbol-name n))
                       ((and (listp n)
                             (eq (car n) 'setf)
                             (consp (cdr n))
                             (symbolp (cadr n)))
                        (symbol-name (cadr n)))
                       (t
                        (princ-to-string n))))))

(defun record-package (package)
  (do-external-symbols (s package)
    (when (fboundp s)
      (make-function-recordable s))))

(defun unrecord-package (package)
  (do-external-symbols (s package)
    (when (fboundp s)
      (make-function-nonrecordable s))))

(defun call-record (package)
  (let ((calls nil))
    (do-external-symbols (s package)
      (when (fboundp s)
        (let ((record (get s :call-record)))
          (dolist (call record)
            (push (cons (car call) (cons s (cdr call)))
                  calls)))))
    (sort calls #'(lambda (x y) (< (car x) (car y))))))

(defun get-utime ()
  (declare (optimize (speed 0)))
  (multiple-value-bind (x sec usec)
      (unix:unix-gettimeofday)
    (declare (ignore x))
    (+ (* 1000000 sec) usec)))

(defun recordable-function (function-name)
  "Takes a function NAME and returns a function OBJECT that does what #'NAME
   did, except also keeps track of the number of times it has been called"
  (let ((function (symbol-function function-name)))
    (setf (get function-name :call-record) nil)
    (setf (get function-name :non-recording-function) function)
    #'(lambda (&rest args)
        (let ((start-time (get-utime))
              (res nil))
          (setq res (multiple-value-list (apply function args)))
          (push (list (- (get-utime) start-time) res args)
                (get function-name :call-record))
          (values-list res)))))

(defun make-function-recordable (function-name)
  "given a function name changes it into equivalent version that records
   function calls"
  (setf (symbol-function function-name)
        (recordable-function function-name)) )

(defun report-calls (s &key (return nil))
  (let (calls)
    (when (fboundp s)
      (let ((record (get s :call-record)))
        (dolist (call record)
          (push (if return
                    call
                    (third call))
                calls))))
    calls))

(defun report-calls-2 (s)
  (let ((record (get s :call-record)))
    (sort record #'(lambda (x y) (> (car x) (car y))))))

(defun make-function-nonrecordable (function-name)
  "Returns the function to its original (non-recording) state"
  (let ((original (get function-name :non-recording-function)))
    (cond      
      (original
       (setf (symbol-function function-name) original)
       (remf (symbol-plist function-name) :non-recording-function)
       (remf (symbol-plist function-name) :call-record)
       original)
      (t
       (format nil "~%function ~s wasn't recordable to begin with: unchanged."
                   function-name)))))

(in-package :odcl)

(defun iterate-plot (fn &key (init-fn nil) (times 100) &aux data)
  (when init-fn
    (funcall init-fn))
  (dotimes (x times)
    (let (start end)
      (setq start (get-utime))
      (funcall fn)
      (setq end (get-utime))
      (push (cons x (- end start)) data)))
  (let ((temp (temp-file "plot"))
        (ctrl (temp-file "control"))
        (ps (temp-file "psdata")))
    (with-open-file (stream temp :direction :output :if-does-not-exist :create :if-exists :error)
      (dolist (pair (nreverse data))
        (format stream "~&~d ~d~%" (car pair) (cdr pair))))
    (with-open-file (stream ctrl :direction :output :if-does-not-exist :create :if-exists :error)
      (format stream "set terminal postscript~%")
      (format stream "set output \"~a\"~%" ps)
      (format stream "plot [0:~d] '~a' with lines~%" times temp))
    (ext:run-program "gnuplot" (list ctrl))
    (ext:run-program "gv" (list ps))))

#+disabled
(defun iterate-profile (fn &key (init-fn nil) (times 100) &aux data max)
  (when init-fn
    (funcall init-fn))
  (profile:reset-time)
  (dotimes (x times)
    (funcall fn)
    (push (profile::%report-time-list) data))
  (setq max (car data))
  (setq data (nreverse data))
  (let ((ctrl (temp-file "control")))
    (with-open-file (stream ctrl :direction :output :if-does-not-exist :create :if-exists :error)
      (dotimes (x (length data))
        (dolist (fun max)
          (let ((function (car fun)))
            (cmsg "Dumping ~s" function)
            (let ((calls (temp-file "plot"))
                  (bytes (temp-file "plot"))
                  (timing (temp-file "plot")))
              (with-open-file (calls calls :direction :output :if-does-not-exist :create :if-exists :error)
                (with-open-file (bytes bytes :direction :output :if-does-not-exist :create :if-exists :error)
                  (with-open-file (timing timing :direction :output :if-does-not-exist :create :if-exists :error)
                    (dotimes (x times)
                      (destructuring-bind (c b tt)
                          (get-alist function (nth x data))
                        (format calls "~&~d ~d~%" x c)
                        (format bytes "~&~d ~d~%" x b)
                        (format timing "~&~d ~d~%" x tt))))))
              (format stream "plot [0:~d] '~a' t \"~a: number of calls\" with lines~%" times calls (symbol-name function))
              (format stream "pause -1 \"Hit return to continue\"~%")
              (format stream "plot [0:~d] '~a' t \"~a: bytes consed\" with lines~%" times bytes (symbol-name function))
              (format stream "pause -1 \"Hit return to continue\"~%")
              (format stream "plot [0:~d] '~a' t \"~a: timing\" with lines~%" times timing (symbol-name function))
              (format stream "pause -1 \"Hit return to continue\"~%"))))))
    ctrl))
