/*-------------------------------------------------------------------------
Compiler Generator Coco/R,
Copyright (c) 1990, 2004 Hanspeter Moessenboeck, University of Linz
extended by M. Loeberbauer & A. Woess, Univ. of Linz
ported from C# to Java by Wolfgang Ahorner
with improvements by Pat Terry, Rhodes University

This program is free software; you can redistribute it and/or modify it 
under the terms of the GNU General Public License as published by the 
Free Software Foundation; either version 2, or (at your option) any 
later version.

This program is distributed in the hope that it will be useful, but 
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License 
for more details.

You should have received a copy of the GNU General Public License along 
with this program; if not, write to the Free Software Foundation, Inc., 
59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

As an exception, it is allowed to write an extension of Coco/R that is
used as a plugin in non-free software.

If not otherwise stated, any source code generated by Coco/R (other than 
Coco/R itself) does not fall under the GNU General Public License.
-----------------------------------------------------------------------*/
package Coco;

import java.io.InputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.BitSet;

class Token {
	public int kind;    // token kind
	public int pos;     // token position in the source text (starting at 0)
	public int col;     // token column (starting at 0)
	public int line;    // token line (starting at 1)
	public String val;  // token value
	public Token next;  // ML 2005-03-11 Peek tokens are kept in linked list
}

class Buffer {
	public static final char EOF = (char) 256;
	static final int MAX_BUFFER_LENGTH = 64 * 1024; // 64KB
	static byte[] buf;   // input buffer
	static int bufStart; // position of first byte in buffer relative to input stream
	static int bufLen;   // length of buffer
	static int fileLen;  // length of input stream
	static int pos;      // current position in buffer
	static RandomAccessFile file; // input stream (seekable)

	public static void Fill(InputStream s) {
		try {
			fileLen = bufLen = s.available();
			buf = new byte[bufLen];
			s.read(buf, 0, bufLen);
			pos = 0;
			bufStart = 0;
		} catch (IOException e){
			System.out.println("--- error on filling the buffer ");
			System.exit(1);
		}
	}

	public static void Fill(String fileName) {
		try {
			file = new RandomAccessFile(fileName, "r");
			fileLen = bufLen = (int) file.length();
			if (bufLen > MAX_BUFFER_LENGTH) bufLen = MAX_BUFFER_LENGTH;
			buf = new byte[bufLen];
			bufStart = Integer.MAX_VALUE; // nothing in buffer so far
			setPos(0); // setup buffer to position 0 (start)
			if (bufLen == fileLen) Close();
		} catch (IOException e) {
			System.out.println("--- could not open file " + fileName);
			System.exit(1);
		}
	}

	// called at the end of Parser.Parse()
	public static void Close() {
		if (file != null) {
			try {
				file.close();
				file = null;
			} catch (IOException e) {
				e.printStackTrace(); System.exit(1);
			}
		}
	}

	public static int Read() {
		if (pos < bufLen) {
			return buf[pos++] & 0xff;  // mask out sign bits
		} else if (getPos() < fileLen) {
			setPos(getPos());         // shift buffer start to pos
			return buf[pos++] & 0xff; // mask out sign bits
		} else {
			return EOF;
		}
	}

	public static int Peek() {
		if (pos < bufLen) {
			return buf[pos] & 0xff;  // mask out sign bits
		} else if (getPos() < fileLen) {
			setPos(getPos());       // shift buffer start to pos
			return buf[pos] & 0xff; // mask out sign bits
		} else {
			return EOF;
		}
	}

	public static String GetString(int beg, int end) {
	    int len = end - beg;
	    char[] buf = new char[len];
	    int oldPos = getPos();
	    setPos(beg);
	    for (int i = 0; i < len; ++i) buf[i] = (char) Read();
	    setPos(oldPos);
	    return new String(buf);
	}

	public static int getPos() {
		return pos + bufStart;
	}

	public static void setPos(int value) {
		if (value < 0) value = 0;
		else if (value > fileLen) value = fileLen;
		if (value >= bufStart && value < bufStart + bufLen) { // already in buffer
			pos = value - bufStart;
		} else if (file != null) { // must be swapped in
			try {
				file.seek(value);
				bufLen = file.read(buf);
				bufStart = value; pos = 0;
			} catch(IOException e) {
				e.printStackTrace(); System.exit(1);
			}
		} else {
			pos = fileLen - bufStart; // make getPos() return fileLen
		}
	}

}

public class Scanner {
	static final char EOL = '\n';
	static final int  eofSym = 0;
	static final int charSetSize = 256;
	static final int maxT = 44;
	static final int noSym = 44;
	static short[] start = {
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0, 11,  0, 10,  0,  0,  5, 31, 24,  0, 14, 22, 15, 30,  0,
	  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  0, 29, 17, 13, 21,  0,
	  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 19,  0, 20, 18,  0,
	  0,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,
	  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1, 25, 23, 26,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,
	  -1};


	static Token t;         // current token
	static char ch;         // current input character
	static int pos;         // column number of current character
	static int line;        // line number of current character
	static int lineStart;   // start position of current line
	static int oldEols;     // EOLs that appeared in a comment;
	static BitSet ignore;  	// set of characters to be ignored by the scanner

	static Token tokens;  // list of tokens already peeked (first token is a dummy)
	static Token pt;      // current peek token
	
	static char[] tokenText = new char[16]; // token text used in NextToken(), dynamically enlarged
	
	public static void Init (String fileName) {
		Buffer.Fill(fileName);
		Init();
	}
	
	public static void Init(InputStream s) {
		Buffer.Fill(s);
		Init();
	}
	
	static void Init () {
		pos = -1; line = 1; lineStart = 0;
		oldEols = 0;
		NextCh();
		ignore = new BitSet(charSetSize+1);
		ignore.set(' '); // blanks are always white space
		ignore.set(9); ignore.set(10); ignore.set(13); 
		pt = tokens = new Token();  // first token is a dummy
	}
	
	static void NextCh() {
		if (oldEols > 0) { ch = EOL; oldEols--; } 
		else {
			ch = (char)Buffer.Read(); pos++;
			// replace isolated '\r' by '\n' in order to make
			// eol handling uniform across Windows, Unix and Mac
			if (ch == '\r' && Buffer.Peek() != '\n') ch = EOL;
			if (ch == EOL) { line++; lineStart = pos + 1; }
		}

	}
	

	static boolean Comment0() {
		int level = 1, line0 = line, lineStart0 = lineStart;
		NextCh();
		if (ch == '*') {
			NextCh();
			for(;;) {
				if (ch == '*') {
					NextCh();
					if (ch == '/') {
						level--;
						if (level == 0) { oldEols = line - line0; NextCh(); return true; }
						NextCh();
					}
				} else if (ch == '/') {
					NextCh();
					if (ch == '*') {
						level++; NextCh();
					}
				} else if (ch == Buffer.EOF) return false;
				else NextCh();
			}
		} else {
			if (ch==EOL) {line--; lineStart = lineStart0;}
			pos = pos - 2; Buffer.setPos(pos+1); NextCh();
		}
		return false;
	}

	
	static void CheckLiteral() {
		String lit = t.val;
		if (lit.compareTo("COMPILER") == 0) t.kind = 6;
		else if (lit.compareTo("IGNORECASE") == 0) t.kind = 7;
		else if (lit.compareTo("CHARACTERS") == 0) t.kind = 8;
		else if (lit.compareTo("TOKENS") == 0) t.kind = 9;
		else if (lit.compareTo("PRAGMAS") == 0) t.kind = 10;
		else if (lit.compareTo("COMMENTS") == 0) t.kind = 11;
		else if (lit.compareTo("FROM") == 0) t.kind = 12;
		else if (lit.compareTo("TO") == 0) t.kind = 13;
		else if (lit.compareTo("NESTED") == 0) t.kind = 14;
		else if (lit.compareTo("IGNORE") == 0) t.kind = 15;
		else if (lit.compareTo("PRODUCTIONS") == 0) t.kind = 16;
		else if (lit.compareTo("END") == 0) t.kind = 19;
		else if (lit.compareTo("ANY") == 0) t.kind = 23;
		else if (lit.compareTo("out") == 0) t.kind = 26;
		else if (lit.compareTo("WEAK") == 0) t.kind = 32;
		else if (lit.compareTo("SYNC") == 0) t.kind = 37;
		else if (lit.compareTo("IF") == 0) t.kind = 38;
		else if (lit.compareTo("CONTEXT") == 0) t.kind = 39;
		else if (lit.compareTo("import") == 0) t.kind = 42;
	}

	static Token NextToken() {
		while (ignore.get(ch)) NextCh();
		if (ch == '/' && Comment0()) return NextToken();
		t = new Token();
		t.pos = pos; t.col = pos - lineStart + 1; t.line = line; 
		int state = start[ch];
		char[] tval = tokenText; // local variables are more efficient
		int tlen = 0;
		tval[tlen++] = ch; NextCh();
		
		boolean done = false;
		while (!done) {
			if (tlen >= tval.length) {
				char[] newBuf = new char[2 * tval.length];
				System.arraycopy(tval, 0, newBuf, 0, tval.length);
				tokenText = tval = newBuf;
			}
			switch (state) {
				case -1: { t.kind = eofSym; done = true; break; }  // NextCh already done /* pdt */
				case 0: { t.kind = noSym; done = true; break; }  // NextCh already done
				case 1:
					if ((ch >= '0' && ch <= '9' || ch >= 'A' && ch <= 'Z' || ch >= 'a' && ch <= 'z')) {tval[tlen++] = ch; NextCh(); state = 1; break;}
					else {t.kind = 1; t.val = new String(tval, 0, tlen); CheckLiteral(); return t;}
				case 2:
					if ((ch >= '0' && ch <= '9')) {tval[tlen++] = ch; NextCh(); state = 2; break;}
					else {t.kind = 2; done = true; break;}
				case 3:
					{t.kind = 3; done = true; break;}
				case 4:
					{t.kind = 4; done = true; break;}
				case 5:
					if ((ch <= 9 || ch >= 11 && ch <= 12 || ch >= 14 && ch <= '&' || ch >= '(' && ch <= '[' || ch >= ']')) {tval[tlen++] = ch; NextCh(); state = 6; break;}
					else if (ch == 92) {tval[tlen++] = ch; NextCh(); state = 7; break;}
					else {t.kind = noSym; done = true; break;}
				case 6:
					if (ch == 39) {tval[tlen++] = ch; NextCh(); state = 9; break;}
					else {t.kind = noSym; done = true; break;}
				case 7:
					if ((ch >= ' ' && ch <= '~')) {tval[tlen++] = ch; NextCh(); state = 8; break;}
					else {t.kind = noSym; done = true; break;}
				case 8:
					if ((ch >= '0' && ch <= '9' || ch >= 'a' && ch <= 'f')) {tval[tlen++] = ch; NextCh(); state = 8; break;}
					else if (ch == 39) {tval[tlen++] = ch; NextCh(); state = 9; break;}
					else {t.kind = noSym; done = true; break;}
				case 9:
					{t.kind = 5; done = true; break;}
				case 10:
					if ((ch >= '0' && ch <= '9' || ch >= 'A' && ch <= 'Z' || ch >= 'a' && ch <= 'z')) {tval[tlen++] = ch; NextCh(); state = 10; break;}
					else {t.kind = 45; done = true; break;}
				case 11:
					if ((ch <= 9 || ch >= 11 && ch <= 12 || ch >= 14 && ch <= '!' || ch >= '#' && ch <= '[' || ch >= ']')) {tval[tlen++] = ch; NextCh(); state = 11; break;}
					else if ((ch == 10 || ch == 13)) {tval[tlen++] = ch; NextCh(); state = 4; break;}
					else if (ch == '"') {tval[tlen++] = ch; NextCh(); state = 3; break;}
					else if (ch == 92) {tval[tlen++] = ch; NextCh(); state = 12; break;}
					else {t.kind = noSym; done = true; break;}
				case 12:
					if ((ch >= ' ' && ch <= '~')) {tval[tlen++] = ch; NextCh(); state = 11; break;}
					else {t.kind = noSym; done = true; break;}
				case 13:
					{t.kind = 17; done = true; break;}
				case 14:
					{t.kind = 20; done = true; break;}
				case 15:
					{t.kind = 21; done = true; break;}
				case 16:
					{t.kind = 22; done = true; break;}
				case 17:
					{t.kind = 24; done = true; break;}
				case 18:
					{t.kind = 25; done = true; break;}
				case 19:
					{t.kind = 27; done = true; break;}
				case 20:
					{t.kind = 28; done = true; break;}
				case 21:
					{t.kind = 29; done = true; break;}
				case 22:
					{t.kind = 30; done = true; break;}
				case 23:
					{t.kind = 31; done = true; break;}
				case 24:
					{t.kind = 34; done = true; break;}
				case 25:
					{t.kind = 35; done = true; break;}
				case 26:
					{t.kind = 36; done = true; break;}
				case 27:
					{t.kind = 40; done = true; break;}
				case 28:
					{t.kind = 41; done = true; break;}
				case 29:
					{t.kind = 43; done = true; break;}
				case 30:
					if (ch == '.') {tval[tlen++] = ch; NextCh(); state = 16; break;}
					else if (ch == ')') {tval[tlen++] = ch; NextCh(); state = 28; break;}
					else {t.kind = 18; done = true; break;}
				case 31:
					if (ch == '.') {tval[tlen++] = ch; NextCh(); state = 27; break;}
					else {t.kind = 33; done = true; break;}

			}
		}
		t.val = new String(tval, 0, tlen);
		return t;
	}
	
	// get the next token (possibly a token already seen during peeking)
	public static Token Scan () {
		if (tokens.next == null) {
			return NextToken();
		} else {
			pt = tokens = tokens.next;
			return tokens;
		}
	}

	// get the next token, ignore pragmas
	public static Token Peek () {
		if (pt.next == null) {
			do {
				pt = pt.next = NextToken();
			} while (pt.kind > maxT); // skip pragmas
		} else {
			do {
				pt = pt.next;
			} while (pt.kind > maxT);
		}

		return pt;
	}

	// make sure that peeking starts at current scan position
	public static void ResetPeek () { pt = tokens; }

} // end Scanner

