/***************************************************************************
              rsbep.c  -  Reed-Solomon & Burst Error Protection
                             -------------------
    begin                : Mon Jul  30 20:27:09 MEST 2001
    copyright            : (C) 2000 by Guido Fiala
    email                : gfiala@s.netic.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <string.h>
#include <assert.h>
#include <malloc.h>

#ifdef ASM_I386
  //the hardcoded i386-asm version only for RS(255,223)  
  #include "rs32.h"     
#else
  //plain & slow C version (can also be used for anything different from RS(255,223))
  #include "rs.h"       
  #define rse32 encode_rs       
  #define rsd32 eras_dec_rs
#endif

#define ERR_BURST_LEN (255*3)   //default
#define RS_BSIZE 255            //default
#define RS_DSIZE 223            //default
#define RSBEP_VERSION "0.0.3"
#define RSBEP_HDR "rsbep"

//just a "large" random string because it's so unlikely to be in the data by chance - better idea?
#define RSBEP_MAGIC_NUM "Inʘ8R7S"

#define PHDR_LEN (40+strlen(RSBEP_MAGIC_NUM)) //maximum expected len of header line...


union long_byte_union
{
        long long i;
        char b[8];
};

//this function is used to copy src to dst while separating consecutive bytes
//equally distributed over the entire array, it tries to utilize several
//cpu-pipes and as high as possible memory efficiency without using asm.
//(but not several CPU's...anyway -it's only 15% of CPU-time, most is RS)

inline void distribute(dtype *src, dtype *dst, int rows, int cols, int reverse)
{
        register long i,j,rc,ri,rl;
        union long_byte_union lb[8];
        rc=rows*cols;
        ri=0;
        rl = reverse ? cols : rows;
        for(i=0;i<rc;i+=64) //we read 64 byte at once
        {
                lb[0].i=*((long long*)(src+i));    
                lb[1].i=*((long long*)(src+i+8));  
                lb[2].i=*((long long*)(src+i+16)); 
                lb[3].i=*((long long*)(src+i+24)); 
                lb[4].i=*((long long*)(src+i+32)); 
                lb[5].i=*((long long*)(src+i+40)); 
                lb[6].i=*((long long*)(src+i+48)); 
                lb[7].i=*((long long*)(src+i+56)); 
                for(j=0;j<64 && i+j<rc;j++) 
                {       //we let the compiler optimize this loop...
                        //branch prediction should kill most the overhead...
                        *(dst+ri)=lb[0].b[j];//write byte in dst
                        ri+=rl;//next position in dst
                        if(ri>=rc)ri-=rc-1;//go around and add one.
                }
        }
}

//decode buffer and write to stdout
inline void decode_and_write(char* ec_buf,int rs_bsize,int bep_size,int rs_dsize,long* sum_of_failures,
                             long* uncorrectable_failures,char* argv,int quiet,int verbose)
{
        register int i;
        int eras_pos[32]; //may once hold positions of erasures...
        for(i=0;i<bep_size;i++)
        {
                int failure=rsd32((ec_buf+i*rs_bsize),eras_pos,0);
                if (failure>0)
                {
                        *sum_of_failures+=failure;
                        if(!quiet && verbose) fprintf(stderr,"%s: errors corrected: %ld uncorrectable blocks: %ld\r",
                                                      argv,*sum_of_failures,*uncorrectable_failures);
                }
                else if (failure==-1)
                {       
                        *uncorrectable_failures++;
                        if(!quiet && verbose) fprintf(stderr,"%s: errors corrected: %ld uncorrectable blocks: %ld\r",
                                                      argv,*sum_of_failures,*uncorrectable_failures);
                }
                if (rs_dsize!=fwrite((ec_buf+i*rs_bsize), sizeof(dtype),rs_dsize, stdout))
                {       //write decoded data
                        perror("serious problem: ");
                        exit(-1);
                }
        }
}                       

//resync-function
//is searches tr_buf for the header-string, copies the data after into bk_buf
//zeroes tr_buf from header string, decodes tr_buf, writes tr_buf to stdout
//copies bk_buf to tr_buf and fills it up, checks if we are in synch again
//if yes decode and exit else loop.
inline void resync(char* tr_buf,char* ec_buf, char* bk_buf,int len,char* phdr,int bep_size,int rs_bsize, int rs_dsize, 
                   int quiet, int verbose, int override, long* sum_of_failures, long* uncorrectable_failures, char* argv)
{
        int i,j;
        char hdr[PHDR_LEN];//just to hold the header line
        int got, aa, bb;
        int rsb,rsd,bep;
        char nam[PHDR_LEN];
	char mag[PHDR_LEN];
	fprintf(stderr,"entering resynch\n");fflush(stderr);//debug only
        start_over://label for the ugly goto below...
        if(verbose && !quiet) fprintf(stderr,"%s: trying to resync on data...\n", argv);
        //search for the magic num (can be anywhere!)
        for(i=0;i<len-15;i++) //if the magic is broken over the buffer-end, we loose another frame...
        {
                if(0==strncmp(tr_buf+i,RSBEP_MAGIC_NUM,strlen(RSBEP_MAGIC_NUM)))
                { //found magic, check for string "rsbep"
			if(verbose && !quiet) fprintf(stderr,"magic found. \n");
                        for(j=i;j>i-PHDR_LEN;j--)
                        {
                                if(0==strncmp(tr_buf+j,RSBEP_HDR,strlen(RSBEP_HDR)))
                                {
					if(verbose && !quiet) fprintf(stderr,"rspeb-header found. \n");
                                        aa = len-i-9;
                                        bb = len-j;
                                        //copy data into bk_uf
                                        memcpy(bk_buf,tr_buf+i+9,aa);
                                        //zero tr_buf
                                        bzero(tr_buf+j,bb);
                                        //decode tr_buf
                                        distribute(tr_buf,ec_buf,bep_size,rs_bsize,1);
                                        //write to stdout
                                        decode_and_write(ec_buf,rs_bsize,bep_size,rs_dsize,sum_of_failures,
                                                         uncorrectable_failures,argv, quiet,verbose);
                                        //copy bk_buf
                                        memcpy(tr_buf,bk_buf,aa);
                                        //fillup tr_buf (until EOF)
                                        got=fread(tr_buf+aa,1,bep_size*rs_bsize-aa,stdin);//read the encoded data
                                        if (got < (bep_size*rs_bsize-aa))
                                        {       //can only happen, when data at end of stream are lost, zero missing data
						if(verbose && !quiet) fprintf(stderr,"eof-data decoded. \n");
                                                bzero((tr_buf+got),(bep_size*rs_bsize)-got);
                                                distribute(tr_buf,ec_buf,bep_size,rs_bsize,1);
                                                decode_and_write(ec_buf,rs_bsize,bep_size,rs_dsize,sum_of_failures,
                                                                 uncorrectable_failures,argv, quiet,verbose);
                                                return;//no more data
                                        }
                                        else
                                        {
                                                //Check if we now get an rsbep-buffer-stream at stdin with block-sync
                                                phdr=fgets(hdr,PHDR_LEN-1,stdin);
                                                if(phdr==NULL)
                                                {
							return;//no more data
                                                }
                                                else
                                                {       //on we go:
                                                        sscanf(phdr,"%9s %d %d %d %20s",&nam,&rsb,&rsd,&bep,&mag);
                                                        if( 0!=strncmp(RSBEP_HDR,nam,strlen(RSBEP_HDR)) ||
							    0!=strncmp(RSBEP_MAGIC_NUM,mag,strlen(RSBEP_MAGIC_NUM)) || 
                                                            (!override && (rs_bsize!=rsb || rs_dsize!=rsd || bep_size!=bep)) )
                                                        {       //again, the evil thing:
                                                                //would be recursive call of resync, we have to avoid it by a ugly:
                                                                //(the fgets above has probably read in the header into tr_buf)
								if(verbose && !quiet) fprintf(stderr,"\nstartover. ");
                                                                goto start_over;
                                                        }
                                                        else
                                                        {       //we are through the bad data for now:
                                                                //normal decode and write
								if(verbose && !quiet) fprintf(stderr,"successful resynched.\n");								
                                                                distribute(tr_buf,ec_buf,bep_size,rs_bsize,1);
                                                                decode_and_write(ec_buf,rs_bsize,bep_size,rs_dsize,sum_of_failures,
                                                                                 uncorrectable_failures,argv,quiet,verbose);
                                                        }
                                                }
                                        }
                                        continue;//for-j
                                }
                        }
                        continue;//for-i
                }               
        }
        //no magic found, just go on - what else can we do...
	fprintf(stderr,"leaving resynch.\n");
}

//main-program: reads commandline args and does anything else...

int main(int argc,char *argv[])
{
  dtype *ec_buf = NULL;//pointer to encoding/decoding buffer
  dtype *tr_buf = NULL;//pointer to transmit-buffer (fread/fwrite)
  dtype *bk_buf = NULL;//pointer to backup-buffer for resync
  char hdr[PHDR_LEN];//just to hold the header line
  char *phdr=NULL;//holds return value of fgets()
  char name[PHDR_LEN];//will hold the name in header
  char mag[PHDR_LEN];//for magic string
  long verbose=0,decode=0,quiet=0,temp=0,override=0;
  long rs_bsize=0,rs_dsize=0,bep_size=0;
  long i=0;
  long sum_of_failures=0;
  long uncorrectable_failures=0;
  extern char *optarg;

  assert(sizeof(dtype)==1);//anything works only with bytes
  
  while((i = getopt(argc,argv,"dvqB:D:R:")) != EOF)
  {
    switch(i)
    {
        case 'd':
                decode = 1;
                break;
        case 'v':       /* Be quite verbose */
                verbose = 1;
                break;
        case 'q':       /* Be strictly quiet */
                quiet = 1;
                break;
        case 'B':       /* take rs_bsize */
                temp=atoi(optarg);
                if((temp<=255) && (temp>=32))
                {
                        rs_bsize=temp;
                }
                else
                {
                        fprintf(stderr,"%s: illegal RS-block size (32..255), bailing.\n",argv[0]);
                        exit(-1);
                }                       
                break;
        case 'D':       /* take rs_dsize */
                temp=atoi(optarg);
                if((temp<=rs_bsize-2) && (temp>=rs_bsize-32))
                {
                        rs_dsize=temp;
                }
                else
                {
                        fprintf(stderr,"%s: illegal RS-data size (RS-block-2..-32), bailing.\n",argv[0]);
                        exit(-1);
                }
                break;
        case 'R':       /* take bep_size */
                temp=atoi(optarg);
                if((temp>rs_bsize) && (temp<=65535))//just a sane limit
                {
                        bep_size=temp;
                        override=1;//we don't take header/default below
                }
                else
                {
                        fprintf(stderr,"%s: illegal error burst len (has to be >=RS-block size, <65535), bailing.\n",argv[0]);
                        exit(-1);
                }
                break;
        default:
                printf("rsbep version %s\n",RSBEP_VERSION);
                printf("usage: %s [-d] [-v] [-q] [-B x -D y -R z ]\nd=decode,"\
                                "v=verbose, q=quiet\nOverride-Settings with:\n"\
                                "x=Reed-Solomon Blocksize, "\
                                "y=Reed-Solomon-Datasize, z=ERR_BURST_LEN\n"\
                                "(defaults to 255,223,765)\n",argv[0]);
                exit(1);
    }
  }
#ifdef ASM_I386
  init_rs();//Initialisiere RS(255)-GF(256)
#else
  //c-Version initialises itself...
#endif
  if(decode)
  {
        //Check if we get an rsbep-buffer-stream at stdin:
        phdr=fgets(hdr,PHDR_LEN-1,stdin);
        if(phdr==NULL)
        {
                if(!quiet) fprintf(stderr,"%s: got no rsbep-stream, bailing.\n",argv[0]);
                exit(-1);
        }
        else if(!override)
        {
                sscanf(phdr,"%9s %d %d %d %s",&name,&rs_bsize,&rs_dsize,&bep_size,&mag);
                if( 0!=strcmp(RSBEP_HDR,name) || rs_bsize>255 || rs_dsize>253 ||
                    bep_size<rs_bsize || rs_bsize<32 || rs_dsize<30 || 
		    0!=strncmp(RSBEP_MAGIC_NUM,mag,strlen(RSBEP_MAGIC_NUM)))
                {
                        if(!quiet) fprintf(stderr,"%s: got non rsbep-stream, bailing (rs_bsize=%d, rs_dsize=%d, bep_size=%d).\n",
                                  argv[0],rs_bsize,rs_dsize,bep_size);
                        exit(-1);
                }
        }
        else
        {
                if(!quiet) fprintf(stderr,"%s: overriding header-settings (rs_bsize=%d, rs_dsize=%d, bep_size=%d).\n",
                           argv[0],rs_bsize,rs_dsize,bep_size);
        }
        //get the buffers we need (here we know how large):
        ec_buf=malloc(rs_bsize*bep_size*sizeof(dtype));  
        tr_buf=malloc(rs_bsize*bep_size*sizeof(dtype));  
        bk_buf=malloc(rs_bsize*bep_size*sizeof(dtype));  
        if(ec_buf==NULL || tr_buf==NULL || bk_buf==NULL)
        {
              fprintf(stderr,"%s: out of memory, bailing.\n",argv[0]);
              exit(-1);
        }
        //go on:
        while (!feof(stdin))
        {
                long got=fread(tr_buf,1,bep_size*rs_bsize,stdin);//read the encoded data
                if (got < (bep_size*rs_bsize))
                {
                        bzero((tr_buf+got),(bep_size*rs_bsize)-got);//zero remaining data - should never happen!
                }
                //Check if we still get an rsbep-buffer-stream at stdin with block-sync (else we lost some data!):
                phdr=fgets(hdr,PHDR_LEN-1,stdin);
                if(phdr==NULL)
                {
                        //is EOF, normal decode and write last buffer
                        distribute(tr_buf,ec_buf,bep_size,rs_bsize,1);
                        decode_and_write(ec_buf,rs_bsize,bep_size,rs_dsize,&sum_of_failures,&uncorrectable_failures,argv[0],
                                         quiet,verbose);
                }
                else
                {
                        int rsb,rsd,bep;
                        char nam[PHDR_LEN];
                        sscanf(phdr,"%9s %d %d %d %20s",&nam,&rsb,&rsd,&bep,&mag);
                        if( 0!=strncmp(RSBEP_HDR,nam,strlen(RSBEP_HDR)) || 
			    0!=strncmp(RSBEP_MAGIC_NUM,mag,strlen(RSBEP_MAGIC_NUM)) || 
                            (!override && (rs_bsize!=rsb || rs_dsize!=rsd || bep_size!=bep)) )
                        {
                                //now something evil has happened, we do our very best...
                                resync(tr_buf,ec_buf,bk_buf,bep_size*rs_bsize,phdr,bep_size,rs_bsize, rs_dsize, quiet, verbose,
                                       override, &sum_of_failures, &uncorrectable_failures, argv[0]);
                        }
                        else
                        {
                                //normal decode and write
                                distribute(tr_buf,ec_buf,bep_size,rs_bsize,1);
                                decode_and_write(ec_buf,rs_bsize,bep_size,rs_dsize,&sum_of_failures,&uncorrectable_failures,argv[0],
                                                 quiet,verbose);
                        }
                }
        }
        if((sum_of_failures>0 || uncorrectable_failures>0 || verbose) && !quiet)
        {
                fprintf(stderr,"\n%s: number of corrected failures   : %ld\n",argv[0],sum_of_failures);
                fprintf(stderr,"%s: number of uncorrectable blocks : %ld\n",argv[0],uncorrectable_failures);
        }
  }
  else
  {     //encode
        if(!override) 
        {       //so we take defaults
                rs_bsize=RS_BSIZE;rs_dsize=RS_DSIZE;bep_size=ERR_BURST_LEN;
        }
        //get the buffers we need (here we know how large):
        ec_buf=malloc(rs_bsize*bep_size*sizeof(dtype));  
        tr_buf=malloc(rs_bsize*bep_size*sizeof(dtype));  
        bk_buf=malloc(rs_bsize*bep_size*sizeof(dtype));  
        if(ec_buf==NULL || tr_buf==NULL || bk_buf==NULL)
        {
              fprintf(stderr,"%s: out of memory, bailing.\n",argv[0]);
              exit(-1);
        }
        //no go on with data
        while (!feof(stdin))
        {
                //write header for every rsbep-buffer-block:
                fprintf(stdout,"%s %d %d %d %s\n",RSBEP_HDR,rs_bsize,rs_dsize,bep_size,RSBEP_MAGIC_NUM);
                //now encode and write the block:
                for(i=0;i<bep_size;i++)
                {
                        long got = fread((ec_buf+i*rs_bsize), 1,rs_dsize, stdin);
                        if (got < rs_dsize)
                        {
                                bzero((ec_buf+i*rs_bsize+got),rs_dsize-got);//zero remaining data
                        }
                        //encode rs_dsize bytes of data, append Parity in row
                        rse32((ec_buf+i*rs_bsize),(ec_buf+i*rs_bsize+rs_dsize));
                }
                distribute(ec_buf,tr_buf,bep_size,rs_bsize,0);
                if (bep_size*rs_bsize!=fwrite(tr_buf,sizeof(dtype),bep_size*rs_bsize,stdout))
                {       //write the encoded data
                        perror("serious problem: ");
                        if(ec_buf) free(ec_buf);
                        if(tr_buf) free(tr_buf);
                        exit(-1);
                }
        }
  }
  //clean up:
  if(ec_buf) free(ec_buf);
  if(tr_buf) free(tr_buf);
  if(bk_buf) free(bk_buf);
  return 0;
}
