#include "dxpcconf.h"
#include <string.h>
#include <stdlib.h>
#include <stddef.h>
#include "EncodeBuffer.H"
#include "IntCache.H"
#include "CharCache.H"
#include "PixelCache.H"
#include "HuffmanCoder.H"
#include "constants.H"


static const int INITIAL_BUFFER_SIZE = 256;
static const int PREFIX_SIZE = 16;

#define ADVANCE_DEST do { \
		         if (destShift_ == 0) \
			 { \
			   destShift_ = 7; nextDest_++; *nextDest_ = 0; \
			 } else \
			 { \
			   destShift_--; \
			 } \
			} while (0)
			 

EncodeBuffer::EncodeBuffer():
  size_(INITIAL_BUFFER_SIZE),
  buffer_(new unsigned char[size_ + PREFIX_SIZE] + PREFIX_SIZE),
  end_(buffer_ + size_),
  nextDest_(buffer_),
  destShift_(7), 
  cumulativeBits_(0)
{
  *nextDest_ = 0;
}


EncodeBuffer::~EncodeBuffer()
{
  delete[](buffer_ - PREFIX_SIZE);
}


void
EncodeBuffer::reset()
{
  nextDest_ = buffer_;
  destShift_ = 7;
  cumulativeBits_ = 0;
  *nextDest_ = 0;
}


void
EncodeBuffer::encodeValue(unsigned int value, unsigned int numBits,
			  unsigned int blockSize)
{
  unsigned int srcMask = 0x1;
  unsigned int bitsWritten = 0;
  if (blockSize == 0)
    blockSize = numBits;

  if (end_ - nextDest_ < 8)
  {
    growBuffer_();
  }
  unsigned int numBlocks = 1;
  do
  {
    if (numBlocks == 4)
      blockSize = numBits;
    unsigned int bitsToWrite = ((blockSize > numBits - bitsWritten) ?
				numBits - bitsWritten : blockSize);
    unsigned int count = 0;
    unsigned int lastBit;
    do
    {
      lastBit = (value & srcMask);
      if (lastBit)
	*nextDest_ |= (1 << destShift_);
      ADVANCE_DEST;
      srcMask <<= 1;
    }
    while (bitsToWrite > ++count);
    bitsWritten += bitsToWrite;
    if (bitsWritten < numBits)
    {
      unsigned int tmpMask = srcMask;
      unsigned int i = bitsWritten;
      if (lastBit)
      {
	do
	{
	  unsigned int nextBit = (value & tmpMask);
	  if (!nextBit)
	    break;
	  tmpMask <<= 1;
	}
	while (numBits > ++i);
      }
      else
      {
	do
	{
	  unsigned int nextBit = (value & tmpMask);
	  if (nextBit)
	    break;
	  tmpMask <<= 1;
	}
	while (numBits > ++i);
      }
      if (i < numBits)
	*nextDest_ |= (1 << destShift_);
      else
	bitsWritten = numBits;
      ADVANCE_DEST;
    }
    blockSize >>= 1;
    if (blockSize < 2)
      blockSize = 2;
    numBlocks++;
  }
  while (numBits > bitsWritten);
}


void
EncodeBuffer::encodeCachedValue(unsigned int value, unsigned int numBits,
				IntCache & cache, unsigned int blockSize)
{
  // The next line is to avoid a warning;
  blockSize = 0;

  if (end_ - nextDest_ < 8)
  {
    growBuffer_();
  }

  unsigned int newBlockSize = cache.getBlockSize(numBits);
  unsigned int index;
  unsigned int sameDiff;
  if (cache.lookup(value, index, PARTIAL_INT_MASK[numBits], sameDiff))
  {
    if (index > 1)
      index++;
    while (destShift_ < index)
    {
      index -= destShift_;
      index--;
      destShift_ = 7;
      nextDest_++;
      *nextDest_ = 0;
    }
    destShift_ -= index;
    *nextDest_ |= (1 << destShift_);
    ADVANCE_DEST;
  }
  else
  {
    ADVANCE_DEST;
    ADVANCE_DEST;
    *nextDest_ |= (1 << destShift_);
    ADVANCE_DEST;
    if (sameDiff)
      encodeValue(1, 1);
    else
    {
      encodeValue(0, 1);
      encodeValue(value, numBits, newBlockSize);
    }
  }
}


void
EncodeBuffer::encodeCachedValue(unsigned char value, unsigned int numBits,
				CharCache & cache, unsigned int blockSize)
{
  if (end_ - nextDest_ < 8)
  {
    growBuffer_();
  }

  unsigned int index;
  if (cache.lookup(value, index))
  {
    if (index > 1)
      index++;
    while (destShift_ < index)
    {
      index -= destShift_;
      index--;
      destShift_ = 7;
      nextDest_++;
      *nextDest_ = 0;
    }
    destShift_ -= index;
    *nextDest_ |= (1 << destShift_);
    ADVANCE_DEST;
  }
  else
  {
    ADVANCE_DEST;
    ADVANCE_DEST;
    *nextDest_ |= (1 << destShift_);
    ADVANCE_DEST;
    encodeValue(value, numBits, blockSize);
  }
}


void
EncodeBuffer::encodeCachedValue(unsigned int value, unsigned int numBits,
				PixelCache & cache,
				HuffmanCoder& escapeCoder0,
				HuffmanCoder& escapeCoder1)
{
  if (end_ - nextDest_ < 8)
  {
    growBuffer_();
  }

  unsigned int index;
  if (cache.lookup(value, index))
  {
    if (index > 1)
      index++;
    while (destShift_ < index)
    {
      index -= destShift_;
      index--;
      destShift_ = 7;
      nextDest_++;
      *nextDest_ = 0;
    }
    destShift_ -= index;
    *nextDest_ |= (1 << destShift_);
    ADVANCE_DEST;
  }
  else
  {
    // The value is not in the cache, so send an escape code, followed by
    // the value
    ADVANCE_DEST;
    ADVANCE_DEST;
    *nextDest_ |= (1 << destShift_);
    ADVANCE_DEST;
    // To transmit the value, use run-length coding with the static
    // Huffman code implemented by the supplied "escapeCoder" object
    //X encodeValue(value, numBits, numBits);
    unsigned int srcMask = 0x1;
    unsigned int pixelValue = ((value & srcMask) ? 1 : 0);
    encodeValue(pixelValue, 1, 1);
    for (unsigned int x = 0; x < numBits;)
    {
      unsigned int runStart = x;
      if (pixelValue)
      {
	while (x < numBits)
	{
	  if (!(value & srcMask))
	    break;
	  srcMask <<= 1;
	  x++;
	}
      }
      else
      {
	while (x < numBits)
	{
	  if (value & srcMask)
	    break;
	  srcMask <<= 1;
	  x++;
	}
      }
      unsigned int runLength = x - runStart;
      if (pixelValue)
      {
	escapeCoder1.encode(runLength - 1, *this);
	pixelValue = 0;
      }
      else
      {
	escapeCoder0.encode(runLength - 1, *this);
	pixelValue = 1;
      }
    }
  }
}

// This is currently unused.
#if 0
void
EncodeBuffer::encodeByte(unsigned char value)
{
  forceBufferToByteBoundary_();
  *nextDest_++ = value;
  if (nextDest_ == end_)
    growBuffer_();
  *nextDest = 0;
}
#endif

void 
EncodeBuffer::encodeRawMem(const unsigned char *buffer, unsigned int len)
{
    forceBufferToByteBoundary_();

    if (end_ - nextDest_ < (ptrdiff_t)len)
    {
    	growBuffer_(len);
    }
 
    memcpy(nextDest_, buffer, len);
    nextDest_ += len;
    
    if (nextDest_ == end_)
    {
	growBuffer_();
    }
    else if (nextDest_ > end_)
    {
        CERR << "EncodeBuffer::encodeRawMem overrun" << ENDL;
	abort();
    }
    *nextDest_ = 0; 
}

unsigned int
EncodeBuffer::getDataLength() const
{
  unsigned int length = nextDest_ - buffer_;
  if (destShift_ != 7)
    length++;
  return length;
}

unsigned int
EncodeBuffer::getDataLengthInBits() const
{
  unsigned int length = nextDest_ - buffer_;
  length <<= 3;
  length += (7 - destShift_);
  return length;
}


unsigned char *
EncodeBuffer::getData()
{
  return buffer_;
}


unsigned int
EncodeBuffer::getCumulativeBitsWritten()
{
  unsigned int bitsWritten = ((nextDest_ - buffer_) << 3);
  bitsWritten += (7 - destShift_);
  unsigned int diff = bitsWritten - cumulativeBits_;
  cumulativeBits_ = bitsWritten;
  return diff;
}


void
EncodeBuffer::growBuffer_(unsigned int minumumFreeSpaceAfterGrow)
{
  unsigned int nextDestOffset = nextDest_ - buffer_;
  unsigned int newSize = size_ + size_;

  // Make sure the new size will accomodate the required minumum free
  // space.
  if (minumumFreeSpaceAfterGrow < 2)
  {
       minumumFreeSpaceAfterGrow = 2;
  }
  if (newSize - nextDestOffset < minumumFreeSpaceAfterGrow)
  {
       newSize = nextDestOffset + minumumFreeSpaceAfterGrow;
  }

  unsigned char *newBuffer = new unsigned char[newSize + PREFIX_SIZE] +
    PREFIX_SIZE;
  memcpy(newBuffer, buffer_, nextDestOffset + 1);
  newBuffer[nextDestOffset + 1] = 0;
  delete[](buffer_ - PREFIX_SIZE);
  buffer_ = newBuffer;
  size_ = newSize;
  end_ = buffer_ + size_;
  nextDest_ = buffer_ + nextDestOffset;
}


void
EncodeBuffer::forceBufferToByteBoundary_()
{
  if (destShift_ != 7)
  {
    destShift_ = 7;
    nextDest_++;

    if (nextDest_ == end_)
    {
      growBuffer_();
    }
    
    *nextDest_ = 0;
  }
}
