/* This file is part of EdiTeX, an editor of mathematical
 * expressions based on TeX syntax.
 * 
 * Copyright (C) 2002-2003 Luca Padovani <lpadovan@cs.unibo.it>,
 *                    2003 Paolo Marinelli <pmarinel@cs.unibo.it>.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * For more information, please visit the project's home page
 * http://helm.cs.unibo.it/editex/
 * or send an email to <lpadovan@cs.unibo.it>
 */

#include "globals.hh"
#include "TNode.hh"
#include <cassert>

TNode
TNode::next() const
{
  assert(node);
  DOM::Node p = node.get_nextSibling();
  while (p && p.get_nodeType() != DOM::Node::ELEMENT_NODE) p = p.get_nextSibling();
  return p;
}

TNode
TNode::nextL() const
{
  assert(node);
  if (TNode n = next())
    if (n.isG()) return n.firstL();
    else return n;
  else return TNode();
}

TNode
TNode::prev() const
{
  assert(node);
  DOM::Node p = node.get_previousSibling();
  while (p && p.get_nodeType() != DOM::Node::ELEMENT_NODE) p = p.get_previousSibling();
  return p;
}

TNode
TNode::prevL() const
{
  assert(node);
  if (TNode n = prev())
    if (n.isG()) return n.lastL();
    else return n;
  else return TNode();
}

TNode
TNode::last() const
{
  assert(node);
  DOM::Node p = node.get_lastChild();
  while (p && p.get_nodeType() != DOM::Node::ELEMENT_NODE) p = p.get_previousSibling();
  return p;
}

TNode
TNode::lastL() const
{
  assert(node);
  if (TNode n = last())
    if (n.isG()) return n.lastL();
    else return n;
  else
    return TNode();
}

TNode
TNode::first() const
{
  assert(node);
  DOM::Node p = node.get_firstChild();
  while (p && p.get_nodeType() != DOM::Node::ELEMENT_NODE) p = p.get_nextSibling();
  return p;
}

TNode
TNode::firstL() const
{
  assert(node);
  if (TNode n = first())
    if (n.isG()) return n.firstL();
    else return n;
  else
    return TNode();
}

TNode
TNode::core() const
{
  assert(node);
  // check also if there is a macro embellishment (\not)
  if (isSb() || isSp()) return first().core();
  else return *this;
}

TNode
TNode::parent() const
{
  assert(node);
  DOM::Node p = node.get_parentNode();
  assert(!p || p.get_nodeType() == DOM::Node::ELEMENT_NODE);
  return p;
}

unsigned
TNode::size() const
{
  assert(node);
  unsigned size = 0;
  TNode p = first();
  while (p)
    {
      p = p.next();
      size++;
    }

  return size;
}

void
TNode::remove() const
{
  assert(node);
  DOM::Node parent = node.get_parentNode();
  parent.removeChild(node);
}

void
TNode::replace(const TNode& newNode) const
{
  assert(node);
  DOM::Node parent = node.get_parentNode();
  parent.replaceChild(newNode.node, node);
}

void
TNode::replace(const TNode& first, const TNode& last) const
{
  assert(node);
  assert(first);

  TNode p = first;
  while (p != last)
    {
      TNode next = p.next();
      insert(p);
      p = next;
    }
  remove();
}

void
TNode::insert(const TNode& newNode) const
{
  assert(node);
  DOM::Node parent = node.get_parentNode();
  parent.insertBefore(newNode.node, node);
}

void
TNode::append(const TNode& newNode) const
{
  assert(node);
  node.appendChild(newNode.node);
}

void
TNode::append(const TNode& first, const TNode& last) const
{
  assert(node);
  assert(first);
  assert(last);

  TNode p = first;
  while (p != last)
    {
      TNode next = p.next();
      append(p);
      p = next;
    }
}

void
TNode::prepend(const TNode& newNode) const
{
  assert(node);
  DOM::Node parent = node.get_parentNode();
  parent.insertBefore(newNode.node, parent.get_firstChild());
}

#if 0
#endif

TNode
TNode::child(unsigned pos) const
{
  assert(node);
  TNode p = first();
  while (p && pos-- > 0) p = p.next();
  return p;
}

std::string
TNode::get(const std::string& name) const
{
  assert(node);
  return node.getAttribute(name);
}

void
TNode::set(const std::string& name, const std::string& value) const
{
  assert(node);
  node.setAttribute(name, value);
}
