/* Emcast - Endhost multicast library
 * Copyright (C) 2001  The Regents of the University of Michigan
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the 
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>

#include "emcast-protocol.h"
#include "util.h"


#define MAXBUF USHRT_MAX
#define SHORTCPY(BUF, OFF, SHRT) do {    \
   unsigned short _shrt = htons(SHRT);  \
   BUF[OFF]   = ((char*) (&_shrt))[0];  \
   BUF[OFF+1] = ((char*) (&_shrt))[1];  \
   } while (0)

#define WRITE_BYTE(B) do {		\
   int _rv;				\
   char _b = B; 			\
   _rv = writen (fd_out, &_b, 1); 	\
   if (_rv != 1) return -1;		\
   } while (0)



#define READ_SHORT(S) do {		\
   int _rv;				\
   _rv = readn (fd_in, &S, 2);		\
   if (_rv == 0) return -1;		\
   if (_rv != 2) return 1;		\
   S = ntohs (S);			\
   } while (0)				

#define READ_BYTES(BUF, LEN) do {	\
   int _rv;				\
   if (LEN >= sizeof (BUF)) return -1;	\
   _rv = readn (fd_in, BUF, LEN);	\
   if (_rv == 0) return -1;		\
   if (_rv != LEN) return 1;		\
   BUF[LEN] = '\0';			\
   } while (0)



void
emcast_loop (Emfuncs* emfuncs, int fd_in, int fd_out, int* fd_fifo)
{
  while (1)
    {
      int rv;

      rv = emcast_loop_once (emfuncs, fd_in, fd_out, fd_fifo);
      if (rv != 0)
	break;
    }
}




 /**

    emcast_loop_once:
    @emfuncs: Handler functions to call on events
    @fd_in: Input file descriptor
    @fd_out: Output file descriptor

    Read a single message from fd_in, call a handler, and write the
    response.

    Returns: 0 on success, -1 on EOF, 1 on internal error (e.g., IO)

 */
int
emcast_loop_once (Emfuncs* emfuncs, int fd_in, int fd_out, int* fd_fifo)
{
  unsigned short id;

  if (!emfuncs || !fd_fifo) return 1;

  READ_SHORT (id);
  switch (id)
    {
      /* INIT */
    case EMCAST_INIT:
      {
	unsigned short version;
	unsigned short fifo_len;
	char fifonam[MAXBUF];
	unsigned short our_version;
	char res;
	char buf[3];
	int rv;

	/* Read VERSION, FIFOLEN, FIFO */
	READ_SHORT(version);
	READ_SHORT(fifo_len);
	READ_BYTES(fifonam, fifo_len);

	if (emfuncs->init)
	  res = (emfuncs->init) (version, &our_version);
	else
	  {
	    if (version > EMCAST_VERSION)
	      our_version = EMCAST_VERSION;
	    else
	      our_version = version;
	    res = 0;
	  }

	/* Write response */
	buf[0] = res;
 	SHORTCPY(buf, 1, our_version);
	rv = writen (fd_out, buf, 3);
	if (rv != 3) return 1;

	/* Open fifo */
	rv = open (fifonam, O_WRONLY);
	if (rv == -1) 
	  return 1;
	*fd_fifo = rv;

	break;
      }

    case EMCAST_JOIN:
      {
	unsigned short len;
	char buf[MAXBUF];
	char res = 0;

	READ_SHORT (len);
	READ_BYTES (buf, len);

	if (emfuncs->join)
	  res = (emfuncs->join) (buf);

	/* Write response */
	WRITE_BYTE(res);

	break;
      }

    case EMCAST_LEAVE:
      {
	char res = 0;

	if (emfuncs->leave)
	  res = (emfuncs->leave) ();

	/* Write response */
	WRITE_BYTE (res);

	return 0;
	break;
      }

    case EMCAST_SEND:
      {
	unsigned short len;
	char buf[MAXBUF];
	char res = 0;

	/* Read packet */
	READ_SHORT (len);
	READ_BYTES (buf, len);

	if (emfuncs->send)
	  res = (emfuncs->send) (buf, len);

	/* Write response */
	WRITE_BYTE (res);

	/* Handle loopback */
	if (emfuncs->handle_loopback && emfuncs->loopback)
	  emcast_handler_recv (*fd_fifo, buf, len, NULL, 0);

	break;
      }

    case EMCAST_GETOPT:
      {
	unsigned short len;
	char optname[MAXBUF];
	char optval[MAXBUF + 3];
	unsigned short optlen;
	char res;
	int rv;

	/* Read optname */
	READ_SHORT (len);
	READ_BYTES (optname, len);

	/* Check if loopback */
	if (emfuncs->handle_loopback && !strncmp(optname, "loopback", 8))
	  {
	    optval[0] = optval[1] = optval[2] = 0;
	    if (emfuncs->loopback)
	      optval[3] = 1;
	    else
	      optval[3] = 0;
	    optlen = 4;
	    res = 0;
	  }
	    
	/* Dispatch */
	else
	  {
	    optlen = MAXBUF;
	    if (emfuncs->getopt)
	      {
		res = (emfuncs->getopt)(optname, &optval[3], &optlen);
		if (res) optlen = 0;
	      }
	    else
	      {
		optlen = 0;
		res = 1;
	      }
	  }

	/* Write res and value */
	optval[0] = res;
 	SHORTCPY(optval, 1, optlen);
	if (res) optlen = 0;
	rv = writen (fd_out, optval, 3 + optlen);
	if (rv != (3 + optlen)) return 1;

	break;
      }

    case EMCAST_SETOPT:
      {
	unsigned short len;
	char optname[MAXBUF];
	unsigned short optlen;
	char optval[MAXBUF];
	char res;

	/* Read len, vallen, optname, val */
	READ_SHORT (len);
	READ_SHORT (optlen);
	READ_BYTES (optname, len);
	READ_BYTES (optval, optlen);

	/* Check if loopback */
	if (emfuncs->handle_loopback && !strncmp(optname, "loopback", 8))
	  {
	    if (optlen == 4)
	      {
		if (optval[0] || optval[1] || optval[2] || optval[3])
		  emfuncs->loopback = 1;
		else
		  emfuncs->loopback = 0;
		res = 0;
	      }
	    else
	      res = 2;	/* bad value */
	  }
	else
	  {
	    /* Dispatch */
	    if (emfuncs->setopt)
	      res = (emfuncs->setopt)(optname, optval, optlen);
	    else
	      res = 1;   /* bad option */
	  }

	/* Write res */
	WRITE_BYTE (res);

	break;
      }

    default:
      return 1;
    }

  return 0;
}


/* **************************************** */

int
emcast_handler_recv (int fd_fifo,
		     const void* buf, unsigned short len, 
		     const void* from, unsigned short fromlen)
{
  char hdr[6];
  int rv;

  /* Write RECV */
  SHORTCPY(hdr, 0, EMCAST_RECV);
  SHORTCPY(hdr, 2, len);
  SHORTCPY(hdr, 4, fromlen);
  rv = writen (fd_fifo, hdr, 6);
  if (rv != 6) return -1;
  rv = writen (fd_fifo, buf, len);
  if (rv != len) return -1;
  rv = writen (fd_fifo, from, fromlen);
  if (rv != fromlen) return -1;

  return len;
}



int
emcast_parse_url (const char* url, char** protocolp, char** hostnamep, 
		  int* portp, char** resourcep)
{
  const char* p;
  const char* temp;

  if (!url)
    return -1;

  if (protocolp) *protocolp = NULL;
  if (hostnamep) *hostnamep = NULL;
  if (portp)     *portp     = 0;
  if (resourcep) *resourcep = NULL;

  /* Skip initial whitespace */
  while (*url != 0 && isspace((int)*url))
    ++url;
  p = url;

  /* Read in protocol (protocol is optional) */
  temp = p;
  while (*p != 0 && *p != ':' && *p != '/')
    ++p;
  if (*p == ':' && p[1] == '/')
    {
      if (protocolp)
	*protocolp = strndup (url, p - url);
      while (*p != 0 && (*p == ':' || *p == '/')) ++p;
    }
  else	/* This is a \0 or /, so it must be the hostname */
    p = temp;

  /* Read in the hostname */
  temp = p;
  while (*p != 0 && *p != '/' && *p != ':') ++p;
  if ((p - temp) == 0) 
    {
      if (protocolp && *protocolp) free (*protocolp);
      return -1;
    }
  if (hostnamep)
    *hostnamep = strndup (temp, p - temp);

  /* Read in the port */
  if (*p == ':')
    {
      temp = ++p;
      while (*p != 0 && *p != '/') ++p;
      if ((p - temp) != 0 && portp)
	{
	  *portp = strtol (temp, NULL, 10);
	}
    }

  /* Read in the resource */
  if (resourcep && *p != 0)
    {
      *resourcep = strdup (p);
    }

  return 0;
}
