/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  Author: Stéphane Démurget  <stephane.demurget@enst-bretagne.fr>
 */

#include <glib.h> 
#include <glib/gi18n.h> 

#include <panel-applet-gconf.h>
#include <glade/glade.h>

#include "ef-prefs.h"
#include "ef-utils.h"

#define PATH_GLADE_XML GLADEDIR"emifreq-preferences.glade"

#define DEFAULT_REFRESH_TIME 1000

void
ef_preferences_load (EmiFreqApplet *applet)
{
	GError *error;

	panel_applet_add_preferences (PANEL_APPLET (applet),
				      "/schemas/apps/emifreq-applet/prefs", NULL);

	/* refresh_time */
	error = NULL;
	applet->refresh_time = panel_applet_gconf_get_int (PANEL_APPLET (applet), "refresh_time", &error);

	if (error) {
		display_error (_("Error while loading settings."), error->message);
		g_error_free (error);

		applet->refresh_time = DEFAULT_REFRESH_TIME;
	} else if (applet->refresh_time < 100 || applet->refresh_time >  60000)
		applet->refresh_time = DEFAULT_REFRESH_TIME;

	/* enable_metric */
	error = NULL;
	applet->enable_metric = panel_applet_gconf_get_bool (PANEL_APPLET (applet), "enable_metric", &error);

	if (error) {
		display_error (_("Error while loading settings."), error->message);
		g_error_free (error);
		applet->enable_metric = TRUE;
	}

	/* show_temp */
	error = NULL;
	applet->show_temp = panel_applet_gconf_get_bool (PANEL_APPLET (applet), "show_temp", &error);

	if (error) {
		display_error (_("Error while loading settings."), error->message);
		g_error_free (error);
		applet->show_temp = TRUE;
	}

	/* read_only */
	error = NULL;
	applet->read_only = panel_applet_gconf_get_bool (PANEL_APPLET (applet), "read_only", &error);

	if (error) {
		display_error (_("Error while loading settings."), error->message);
		g_error_free (error);
		applet->read_only = FALSE;
	}
}

static void
preferences_response_cb (GtkDialog *dialog, gint response, gpointer data)
{
	EmiFreqApplet *applet = data;

	if (response == GTK_RESPONSE_HELP){
		display_help("emifreq-applet-prefs");
		return;
	}

	gtk_widget_destroy (GTK_WIDGET (applet->prefs));
	applet->prefs = NULL;

	return;
}

static void
refresh_time_changed_cb (GtkSpinButton *sb, gpointer data)
{
	EmiFreqApplet *applet = (EmiFreqApplet *) data;

	applet->refresh_time = gtk_spin_button_get_value_as_int (sb);
	reset_timeout (applet);

	panel_applet_gconf_set_int (PANEL_APPLET (applet), "refresh_time", applet->refresh_time, NULL);
}

static void
enable_metric_toggled_cb (GtkToggleButton *tb, gpointer data)
{
	EmiFreqApplet *applet = (EmiFreqApplet *) data;

	applet->enable_metric = gtk_toggle_button_get_active (tb);
	force_refresh (applet, FALSE);

	panel_applet_gconf_set_bool (PANEL_APPLET (applet), "enable_metric", applet->enable_metric, NULL);
}

static void
show_temp_toggled_cb (GtkToggleButton *tb, gpointer data)
{
	EmiFreqApplet *applet = (EmiFreqApplet *) data;

	applet->show_temp = gtk_toggle_button_get_active (tb);
	force_refresh (applet, FALSE);

	panel_applet_gconf_set_bool (PANEL_APPLET (applet), "show_temp", applet->show_temp, NULL);
}

static void
toggle_sensitiveness (GtkToggleButton *tb, gpointer data)
{
	g_return_if_fail (tb != NULL);
	g_return_if_fail (data != NULL);

	gtk_widget_set_sensitive (GTK_WIDGET (data),
				  gtk_toggle_button_get_active (tb));
}

void
ef_preferences_dialog_show (EmiFreqApplet *applet)
{
	GladeXML *xml;
	GtkWidget *refresh_time_sb, *enable_metric_cb, *show_temp_cb;

	g_assert (applet);
	
	if (applet->prefs) {
		gtk_window_present (GTK_WINDOW (applet->prefs));
		return;
	}

	xml = glade_xml_new (PATH_GLADE_XML, NULL, NULL);
	applet->prefs = GTK_DIALOG (glade_xml_get_widget (xml, "preferences-dialog"));
	refresh_time_sb = glade_xml_get_widget (xml, "refresh_time_spinbutton");
	enable_metric_cb = glade_xml_get_widget (xml, "enable_metric_cb");
	show_temp_cb = glade_xml_get_widget (xml, "show_temp_cb");
	g_object_unref (G_OBJECT (xml));

	/* update preferences */
	gtk_spin_button_set_value (GTK_SPIN_BUTTON (refresh_time_sb),
				   applet->refresh_time);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (enable_metric_cb),
				      applet->enable_metric);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (show_temp_cb),
				      applet->show_temp);

	gtk_widget_set_sensitive (enable_metric_cb, applet->show_temp);

	gtk_window_set_screen (GTK_WINDOW (applet->prefs),
			       gtk_widget_get_screen (GTK_WIDGET (applet)));

	g_signal_connect (G_OBJECT (applet->prefs), "response",
			  G_CALLBACK (preferences_response_cb), (gpointer) applet);

	g_signal_connect (G_OBJECT (refresh_time_sb), "value-changed",
			  G_CALLBACK (refresh_time_changed_cb), (gpointer) applet);

	g_signal_connect (G_OBJECT (enable_metric_cb), "toggled",
			  G_CALLBACK (enable_metric_toggled_cb), (gpointer) applet);

	g_signal_connect (G_OBJECT (show_temp_cb), "toggled",
			  G_CALLBACK (show_temp_toggled_cb), (gpointer) applet);

	g_signal_connect (G_OBJECT (show_temp_cb), "toggled",
			  G_CALLBACK (toggle_sensitiveness), (gpointer) enable_metric_cb);

	gtk_widget_show (GTK_WIDGET (applet->prefs));
}
