#include <glib.h>
#include <stdio.h>
#include <string.h>
#include "entity.h"

static GHashTable *element_ht = NULL;

GSList *
element_list (void)
{
    return (eutils_hash_value_list (element_ht, NULL));
}

Element *
element_lookup_element (EBuf * element)
{
    if (ebuf_not_empty (element) && element_ht)
	return (g_hash_table_lookup (element_ht, element->str));
    else
	return (NULL);
}


void
element_register (Element * element)
{
    if (element_ht == NULL)
	element_ht = g_hash_table_new (x31_hash, g_str_equal);

    g_hash_table_insert (element_ht, element->tag, element);
}

/* 
 * this could majorly fuck up things if called in the wrong place, ie while
 * any nodes exist.  Currently only used from modulegen. - Jim
 */

void
element_unregister_all (void)
{
    if (element_ht)
	g_hash_table_destroy (element_ht);
    element_ht = NULL;
}

void
element_register_attrib (Element * element, ElementAttr * element_attrib)
{
    if (!element || !element_attrib || !element_attrib->attribute)
	return;

    if (element->attributes == NULL)
	element->attributes = g_hash_table_new (x31_hash, g_str_equal);

    g_hash_table_insert (element->attributes,
			 element_attrib->attribute, element_attrib);
    EDEBUG (("elements", "'%s' attribute '%s', accepts type '%s', '%s'",
	     element->tag, element_attrib->attribute ? element_attrib->attribute : "(NULL)",
	     element_attrib->value_desc ? element_attrib->value_desc : "(NULL)", 
	     element_attrib->possible_values ? element_attrib->possible_values : "(NULL)"));
}

void
element_register_child_attrib (Element * element, ElementAttr * element_attrib)
{

    ECHECK_RET (element != NULL);
    ECHECK_RET (element_attrib != NULL);
    ECHECK_RET (element_attrib->attribute != NULL);

    if (element->child_attributes == NULL)
	element->child_attributes = g_hash_table_new (x31_hash, g_str_equal);

    g_hash_table_insert (element->child_attributes,
			 (gpointer) element_attrib->attribute, element_attrib);

    EDEBUG (("elements",
	     "'%s' registered attribute on CHILD's behalf '%s', accepts type '%s', '%s'",
	     element->tag, element_attrib->attribute,
	     element_attrib->value_desc, element_attrib->possible_values));
}


ElementAttr *
element_attrib_info_for_node (ENode * node, gchar * attrib)
{
    ENode *parent;
    Element *element;

    element = element_lookup_element (node->element);

    if (element && element->attributes) {
	ElementAttr *element_attrib;

	element_attrib = g_hash_table_lookup (element->attributes, attrib);
	if (element_attrib)
	    return (element_attrib);
    }
    /* if the above didn't work, try parent */
    parent = enode_parent (node, NULL);

    if (!parent) {
        return (NULL);
    }
    
    /* find its element type */
    element = element_lookup_element (parent->element);

    /* see if it implements child attributes */
    if (element && element->child_attributes) {
	ElementAttr *element_attrib;

	element_attrib =
	    g_hash_table_lookup (element->child_attributes, attrib);
	if (element_attrib)
	    return (element_attrib);
    }
    /* give up */
    return (NULL);
}

GSList *
element_supported_attribs_for_node (ENode * node)
{
    ENode *parent;
    Element *element;
    GSList *attr_list = NULL;

    element = element_lookup_element (node->element);

    if (element && element->attributes)
	attr_list = eutils_hash_key_list (element->attributes, NULL);

    /* check parent */
    parent = enode_parent (node, NULL);

    if (parent) {
	/* find its element type */
	element = element_lookup_element (parent->element);

	/* see if it implements child attributes */
	if (element && element->child_attributes)
	    attr_list =
		eutils_hash_key_list (element->child_attributes, attr_list);
    }
    return (attr_list);
}


void
element_render_notify (ENode * node)
{
    Element *element;
    ENode *parent;

    element = element_lookup_element (node->element);

    parent = enode_parent (node, NULL);

    /* This is true if its the root node.. or a screwed node I guess */
    if (!parent)
	return;

    /* If parent not rendered neither should node. */
    if (!ENODE_FLAG_ISSET (parent, ENODE_RENDERED))
	return;

    /* If parent is a NO_RENDER_CHILD node, we shouldn't be renderering */
    if (ENODE_FLAG_ISSET (parent, ENODE_NO_RENDER_CHILDREN))
	return;

    /* Don't render if we are already rendered.. */
    ECHECK_RET (!ENODE_FLAG_ISSET (node, ENODE_RENDERED));

    ENODE_SET_FLAG (node, ENODE_RENDERED);

    if (element && element->render_func) {
	element->render_func (node);
    }
}


void
element_destroy_notify (ENode * node)
{
    Element *element;

    if (!ENODE_FLAG_ISSET (node, ENODE_RENDERED))
	return;

    element = element_lookup_element (node->element);

    if (element && element->destroy_func) {

        EDEBUG (("elements-destroy", "node type = %s", node->element->str));
	element->destroy_func (node);
	/* reset appropriate flags. */
	ENODE_UNSET_FLAG (node, ENODE_RENDERED);
	ENODE_UNSET_FLAG (node, ENODE_PARENTED);
    }
}


void
element_parent_notify (ENode * parent, ENode * child)
{
    Element *element;

    if (!ENODE_FLAG_ISSET (parent, ENODE_RENDERED) ||
	!ENODE_FLAG_ISSET (child, ENODE_RENDERED)) return;

    /* Parenting is done based on the parents type */
    element = element_lookup_element (parent->element);

    if (element && element->parent_func) {
	EDEBUG (("elements", "parenting child %s to parent %s\n",
		 child->element->str, parent->element->str));
	element->parent_func (parent, child);
    } else {
	/* If any of the above failed.. */
	erend_short_curcuit_parent (parent, child);
    }

    ENODE_SET_FLAG (child, ENODE_PARENTED);
}

void
element_set_attrib_notify (ENode * node, EBuf * attr, EBuf * value)
{
    ENode *parent;
    Element *element;
    gint ret = FALSE;
    ElementAttr *element_attrib;

    if (!ENODE_FLAG_ISSET (node, ENODE_RENDERED))
	return;

    element = element_lookup_element (node->element);

    if (!element || !element->attributes)
	return;

    element_attrib = g_hash_table_lookup (element->attributes, attr->str);

    if (element_attrib && element_attrib->set_attr_func) {
	ret = element_attrib->set_attr_func (node, attr, value);
    } else {
	element_attrib = g_hash_table_lookup (element->attributes, "*");
	if (element_attrib && element_attrib->set_attr_func)
	    ret = element_attrib->set_attr_func (node, attr, value);
    }

    /* 
     * If setting of this attribute fails, we see if the parent
     * implements a method for us to check.
     */
    if (ret == FALSE) {
	/* Get parent node */
	parent = enode_parent (node, NULL);
	/* find its element type */
	element = element_lookup_element (parent->element);

	/* see if it implements child attributes */
	if (element && element->child_attributes) {
	    ElementAttr *element_attrib;

	    element_attrib =
		g_hash_table_lookup (element->child_attributes, attr->str);

	    /* if it does, call it */
	    if (element_attrib && element_attrib->set_child_attr_func)
		element_attrib->set_child_attr_func (parent, node, attr, value);
	}
    }
}

void
element_set_attrib_notify_all (ENode * node)
{
    GSList *tmp;
    EBuf *attrib;
    EBuf *value;

    if (!node || !ENODE_FLAG_ISSET (node, ENODE_RENDERED))
	return;

    tmp = node->attribs;

    while (tmp) {
	attrib = tmp->data;
	tmp = tmp->next;
	value = tmp->data;
	tmp = tmp->next;

	if (value && attrib)
	    element_set_attrib_notify (node, attrib, value);
	else
	    g_warning
		("hrrmpf, somehow value and attrib for node %s has %s => %s",
		 node->element->str, attrib ? attrib->str : "NULL",
		 value ? value->str : "NULL");

    }
}

void
element_get_attrib_notify (ENode * node, gchar * attr)
{
    Element *element;

    ECHECK_RET (node != NULL);
    ECHECK_RET (attr != NULL);

    if (!ENODE_FLAG_ISSET (node, ENODE_RENDERED))
	return;

    /* get element type for this node */
    element = element_lookup_element (node->element);

    if (element && element->attributes) {
	ElementAttr *element_attrib;

	element_attrib = g_hash_table_lookup (element->attributes, attr);

	if (element_attrib && element_attrib->get_attr_func)
	    element_attrib->get_attr_func (node, attr);
    }
}

void
element_get_data_notify (ENode * node)
{
    Element *element;

    if (!node || !ENODE_FLAG_ISSET (node, ENODE_RENDERED))
	return;

    /* Parenting is done based on the parents type */
    element = element_lookup_element (node->element);

    if (element && element->get_data_func)
	element->get_data_func (node);
}

/* If they have set_data method implemented, use that, else set it in struct */
void
element_set_data_notify (ENode * node, EBuf * data)
{
    Element *element;

    if (!node || !ENODE_FLAG_ISSET (node, ENODE_RENDERED))
	return;

    element = element_lookup_element (node->element);

    if (element && element->set_data_func)
	element->set_data_func (node, data);
}

void
element_insert_data_notify (ENode * node, unsigned long offset, EBuf * data)
{
    Element *element;

    if (!node || !ENODE_FLAG_ISSET (node, ENODE_RENDERED))
	return;

    /* Parenting is done based on the parents type */
    element = element_lookup_element (node->element);

    if (element && element->insert_data_func)
	element->insert_data_func (node, offset, data);
}


void
element_delete_data_notify (ENode * node, unsigned long offset,
			    unsigned long count)
{
    Element *element;

    if (!node || !ENODE_FLAG_ISSET (node, ENODE_RENDERED))
	return;

    /* Parenting is done based on the parents type */
    element = element_lookup_element (node->element);

    if (element && element->delete_data_func)
	element->delete_data_func (node, offset, count);
}


void
element_append_data_notify (ENode * node, EBuf * data)
{
    Element *element;

    if (!node || !ENODE_FLAG_ISSET (node, ENODE_RENDERED))
	return;

    /* Parenting is done based on the parents type */
    element = element_lookup_element (node->element);

    if (element && element->append_data_func)
	element->append_data_func (node, data);
}


