/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdrivelink.so
 *
 *   File: dl_commit.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "dl_common.h"

/*
 *  Function: dl_build_metadata
 *
 *  Called to build a drivelink metadata for the specified
 *  child object link.
 */
static int dl_build_metadata( storage_object_t      *drivelink,
                              drive_link_t          *link,
                              drivelink_metadata_t  *md    )
{
        int rc=0;
        drivelink_private_data_t *pdata;

        LOG_ENTRY();

        REQUIRE(dl_isa_drivelink(drivelink) == TRUE);
        REQUIRE(link != NULL);
        REQUIRE(md != NULL);

        LOG_DEBUG("building metadata for child object %s\n", link->object->name );

        pdata = (drivelink_private_data_t *)drivelink->private_data;

        memset(md, 0, sizeof(drivelink_metadata_t));

        md->signature            = EVMS_DRIVELINK_SIGNATURE;
        md->parent_serial_number = pdata->parent_serial_number;
        md->child_serial_number  = link->serial_number;
        md->child_count          = pdata->drive_link_count;
        md->version.major        = MAJOR_VERSION;
        md->version.minor        = MINOR_VERSION;
        md->version.patchlevel   = PATCH_LEVEL;
        md->sequence_number      = pdata->sequence_number;

        memcpy( &md->ordering_table[0],
                &pdata->ordering_table[0],
                sizeof(dot_entry_t)*pdata->drive_link_count );

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: dl_write_metadata
 *
 *  Called to write drivelink metadata to the specified child
 *  object.
 */
static int dl_write_metadata( storage_object_t          * object,
                              drivelink_metadata_t      * metadata,
                              evms_feature_header_t     * feature_header,
                              uint                        commit_phase,
                              boolean                     backup )
{
        storage_object_t *parent;
        int       rc=EINVAL;
        u_int32_t crc;
        lsn_t     lsn1=0;
        lsn_t     lsn2=0;

        LOG_ENTRY();

        parent = EngFncs->first_thing(object->parent_objects, NULL);
        lsn1 = feature_header->feature_data1_start_lsn;
        lsn2 = feature_header->feature_data2_start_lsn;

        // convert metadata to disk format
        dl_cpu_metadata_to_disk( metadata );

        // crc the data
        metadata->crc = 0;
        crc = EngFncs->calculate_CRC( EVMS_INITIAL_CRC,
                                             (void *)metadata,
                                             sizeof(drivelink_metadata_t) );
        metadata->crc = CPU_TO_DISK32(crc);

        // now write out the metadata
        if (commit_phase == FIRST_METADATA_WRITE) {
                LOG_DEBUG("writing metadata for commit phase 1 @ lsn %"PRIu64"\n", lsn1);
                if (backup) {
                        rc = EngFncs->save_metadata(parent->name, object->name, lsn1,
                                                    DRIVELINK_METADATA_SECTOR_COUNT, metadata);
                } else {
                        rc = WRITE( object, lsn1, DRIVELINK_METADATA_SECTOR_COUNT, metadata );
                }
        }
        else if (commit_phase == SECOND_METADATA_WRITE) {
                // only write 2nd copy if feature header says to do so
                if ( feature_header->feature_data2_size > 0)  {
                        LOG_DEBUG("writing metadata for commit phase 2 @ lsn %"PRIu64"\n", lsn2);
                        if (backup) {
                                rc = EngFncs->save_metadata(parent->name, object->name, lsn2,
                                                            DRIVELINK_METADATA_SECTOR_COUNT, metadata);
                        } else {
                                rc = WRITE( object, lsn2, DRIVELINK_METADATA_SECTOR_COUNT, metadata );
                        }
                }
                else {
                        LOG_DEBUG("not writing metadata cuz feature header has data2 size=0\n");
                        rc = 0;
                }
        }

        if (rc) {
                LOG_ERROR("error, failed to write metadata ... rc= %d\n", rc);
        }

        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Function: dl_commit_child
 *
 *  Called to commmit metadata for the specified drive link child object.
 */
static int dl_commit_child( storage_object_t   *drivelink,
                            drive_link_t       *link,
                            uint                commit_phase,
                            boolean             backup )
{
        int rc=0;
        drivelink_private_data_t  *pdata;
        drivelink_metadata_t       metadata;

        LOG_ENTRY();

        REQUIRE(link->object != NULL);
        REQUIRE(link->object->feature_header != NULL);

        pdata = (drivelink_private_data_t *) drivelink->private_data;

        if (commit_phase == 1 && !backup) {
                rc = dl_build_feature_header( drivelink, link, link->object->feature_header );
        }

        if (rc==0) {
                rc = dl_build_metadata( drivelink, link, &metadata );
        }

        if (rc==0) {
                rc = dl_write_metadata( link->object, &metadata, link->object->feature_header, commit_phase, backup );
        }

        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Function:  dl_commit_changes
 *
 *  Called to commit changes to a drivelink storage object.
 *
 *  This is done by committing each of the child storage objects
 *  that are aggregated by the single drivelink storage object.
 *
 *  I do it by walking the drivelink ordering table and calling
 *  commit for every object in the table.
 */
int dl_commit_changes( storage_object_t * object, uint commit_phase )
{
        int i,rc=0;
        drivelink_private_data_t  *pdata;

        LOG_ENTRY();

        LOG_DEBUG("object->name= %s  commit_phase= %d\n", object->name, commit_phase );

        REQUIRE(dl_isa_drivelink(object)==TRUE);

        if ( commit_phase ==  FIRST_METADATA_WRITE  ||
             commit_phase == SECOND_METADATA_WRITE ) {

                pdata = (drivelink_private_data_t *)object->private_data;

		if (commit_phase == FIRST_METADATA_WRITE) {
			pdata->sequence_number++;
		}

                for (i=0,rc=0; i<pdata->drive_link_count; i++) {
                        if (dl_isa_missing_child(pdata->drive_link[i].object)==TRUE)
                                continue;
                        rc += dl_commit_child( object, &pdata->drive_link[i], commit_phase, FALSE );
                }

                if (rc==0){
                        if (commit_phase == SECOND_METADATA_WRITE) {
                                object->flags &= ~SOFLAG_DIRTY;
                        }
                }
                else {
                        LOG_ERROR("error, failed to commit all child objects in this drivelink\n");
                        rc = ENOMSG;
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Function:  dl_backup_metadata
 *
 *  Called to write metadata backup entries for a drivelink storage object.
 */
int dl_backup_metadata(storage_object_t *object)
{
	drivelink_private_data_t *pdata;
	int i, rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("object->name = %s\n", object->name);

	REQUIRE(dl_isa_drivelink(object) == TRUE);

	pdata = (drivelink_private_data_t *)object->private_data;

	for (i = 0; i < pdata->drive_link_count; i++) {
		if (dl_isa_missing_child(pdata->drive_link[i].object)) {
			continue;
		}

		rc = dl_commit_child(object, &pdata->drive_link[i],
				     FIRST_METADATA_WRITE, TRUE);
		if (rc) {
			break;
		}
		rc = dl_commit_child(object, &pdata->drive_link[i],
				     SECOND_METADATA_WRITE, TRUE);
		if (rc) {
			break;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

