/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * 	Module: ecemisc.c
 */

#include "eceinternal.h"

static void
ipc_msg_done(struct IPC_MESSAGE *ipcmsg)
{
	LOG_ENTRY();
	ha_free(ipcmsg->msg_body);
	g_free(ipcmsg);
	LOG_EXIT_VOID();
}

static int
ipc_send(IPC_Channel *ch ,IPC_Message *msg)
{
	int count=0;
	int ret=IPC_FAIL;
	LOG_ENTRY();

	if(ch==NULL) {
		LOG_WARNING("peer_send failed to send\n");
		goto end;
	}

	while ((ret = ch->ops->waitout(ch)) != IPC_OK && count < 10) {
		if(ret==IPC_BROKEN) {
			LOG_WARNING("ipc channel broken");
			break;
		}
		if(ret!=IPC_INTR){
			count++;
		}
		cl_shortsleep();
	}
	if(ret==IPC_OK) {
		ret =  ch->ops->send(ch, msg);
	}

end:
	if(ret!=IPC_OK && ret!= IPC_BROKEN) {
		LOG_EXTRA("Enter calling ipc_msg_done() ret=%d\n",ret);
		ipc_msg_done(msg);
	}

	LOG_EXIT_INT(ret);
	return ret;
}


/*
 * send a message to the peer. NOTE: msg will be freed on the
 * caller behalf.
 */
int
peer_send(void *msg, const int size, IPC_Channel *ch)
{
	IPC_Message *ipcmsg;
	int   ret=IPC_FAIL;

	LOG_ENTRY();
	if(!msg || !ch) {
		ha_free(msg);
	} else {
		ipcmsg = (IPC_Message *)g_malloc(sizeof(IPC_Message));
		ipcmsg->msg_len = size;
		ipcmsg->msg_done = ipc_msg_done;
		ipcmsg->msg_private = NULL;
		ipcmsg->msg_body = msg;
		ret = ipc_send(ch, ipcmsg);
	}
	switch (ret) {
		case IPC_OK:
			ret = 0;
			break;
		case IPC_FAIL:
			LOG_WARNING("ipc_send failed with error IPC_FAIL.\n");
			ret = ECOMM;
                        break;
		case IPC_BROKEN:
                        LOG_WARNING("ipc_send failed with error IPC_BROKEN.\n");
			ret = ENOLINK;
			break;
		case IPC_INTR:
                        LOG_WARNING("ipc_send failed with error IPC_INTR.\n");
			ret = EINTR;
			break;
		default:
                        LOG_WARNING("ipc_send failed with unknown error %d.\n", ret);
			break;

	}
	LOG_EXIT_INT(ret);
	return ret;
}

int
peer_recv_done(IPC_Message *ipcmsg)
{
	LOG_ENTRY();
	ipcmsg->msg_done(ipcmsg);
	LOG_EXIT_INT(0);
	return 0;
}

/* receive a message from the peer */
int
peer_recv(IPC_Message **ipcmsg, IPC_Channel *ch, gboolean block)
{
	int ret=0;

	LOG_ENTRY();
	if(ch==NULL) {
		LOG_EXIT_INT(IPC_FAIL);
		return IPC_FAIL;
	}
	if(!block) {
		if(!ch->ops->is_message_pending(ch)) {
			LOG_EXIT_INT(IPC_FAIL);
			return IPC_FAIL;
		}
	}

	do {
		if(ret) cl_shortsleep();
		ret = ch->ops->recv(ch, ipcmsg);
	} while(block && ret==IPC_FAIL);


	if(ret==IPC_OK && *ipcmsg==NULL) {
		LOG_SERIOUS("IPC subsystem misbehaving");
		ECE_ASSERT(0);
	}

	LOG_EXIT_INT(ret);
	return ret;
}

/* given the membership information, creates a string */
char *
oc2membstr(gboolean quorum, const oc_ev_membership_t *oc, const int size)
{
	int 	strsize;
	char 	*data_str;
	gchar 	*quorum_str;
	char 	*str;
	struct ha_msg	*h_msg;

	LOG_ENTRY();

	if ((h_msg=ha_msg_new(0)) == NULL) {
		LOG_WARNING("Cannot create msg");
		LOG_EXIT_PTR(NULL);
		return NULL;
	}

	strsize = B64_stringlen(size)+1;
	data_str = (char *)g_malloc(strsize);
	binary_to_base64(oc, size, data_str, strsize);

	quorum_str = g_strdup_printf("%d", quorum);

	if ((ha_msg_add(h_msg, F_TYPE, ECEi_MEM) == HA_FAIL)
	||(ha_msg_add(h_msg, ECEi_QUORUM, quorum_str) == HA_FAIL)
	||(ha_msg_add(h_msg, ECEi_MEMBERSHIP, data_str) == HA_FAIL)) {
		LOG_WARNING("Cannot fill the heartbeat message\n");
		str = NULL;
	} else {
		str = msg2string(h_msg);
	}

	ha_msg_del(h_msg);
	g_free(data_str);
	g_free(quorum_str);
	LOG_EXIT_PTR(str);
	return str;
}


/* given a str converts it to membership datastructure*/
int
membstr2oc(struct ha_msg *h_msg, oc_ev_membership_t **oc, gboolean *quorum)
{
	const char *data_str, *quorum_str;
	int   len, outbytes, size;

	LOG_ENTRY();
	ECE_ASSERT(strcmp(ha_msg_value(h_msg, F_TYPE), ECEi_MEM)==0);
	ECE_ASSERT(data_str = ha_msg_value(h_msg,
				ECEi_MEMBERSHIP));
	ECE_ASSERT(quorum_str = ha_msg_value(h_msg,
				ECEi_QUORUM));

	len    = strlen(data_str);
	outbytes   = B64_maxbytelen(len);
	(*oc) = (oc_ev_membership_t *)g_malloc(outbytes);
	size = base64_to_binary(data_str, len,
			(void *)(*oc), outbytes);

	*quorum = atol(quorum_str);

	LOG_EXIT_INT(size);
	return size;
}


inline void
delete_ece_ev(ece_event_t *ece_ev)
{
	LOG_ENTRY();
	g_free(ece_ev);
	LOG_EXIT_VOID();
}


ece_event_t *
create_ece_ev(gboolean quorum,
		const oc_ev_membership_t *oc,
		ece_event_type_t type,
		int *retsize)
{
	ece_event_t *ece_ev = NULL;
	int i, nodes, start;

	LOG_ENTRY();
	switch(type) {
		case DELTA_JOIN:
			nodes = oc->m_n_in;
			start = oc->m_in_idx;
			break;

		case DELTA_LEAVE:
			nodes = oc->m_n_out;
			start = oc->m_out_idx;
			break;

		case MEMBERSHIP:
			nodes  = oc->m_n_member;
			start = oc->m_memb_idx;
			ECE_ASSERT(nodes);
			break;
		default:
			goto end;
	}

	*retsize = sizeof(ece_event_t)+
			nodes*sizeof(ece_nodeid_t);
	ece_ev = (ece_event_t *)g_malloc0(*retsize);

	ece_ev->type = type;
	ece_ev->transid = oc->m_instance;
	ece_ev->quorum_flag = quorum;
	ece_ev->num_entries = nodes;
	for (i = 0 ; i < nodes; i++) {
		const char *nodeid;
		nodeid = llm_getnodeid(oc->m_array[start+i].node_id);
		ECE_ASSERT(nodeid);
		strcpy((char *)(ece_ev->node+i), nodeid);
	}
end:
	LOG_EXIT_PTR(ece_ev);
	return ece_ev;
}





#define MAGICNO -123456 /* used for debugging purpose */
typedef struct ece_cond_s {
	pthread_cond_t cond;
	char	*node;
	int	ackval;
	int	retval;
} ece_cond_t;

static gboolean cond_in_use;
static GHashTable  *cond_hash = NULL;
static pthread_mutex_t 	cond_mutex;
static pthread_cond_t cond_cond;

void
condition_init(void)
{
	LOG_ENTRY();
	if(cond_hash) {
		g_hash_table_destroy(cond_hash);
	}
	cond_hash =  g_hash_table_new(g_direct_hash,
                                g_direct_equal);
	cond_in_use=FALSE;
	pthread_mutex_init(&cond_mutex, NULL);
	pthread_cond_init(&cond_cond, NULL);
	LOG_EXIT_VOID();
}

static void
condition_lock()
{
	LOG_ENTRY();
   	pthread_mutex_lock(&cond_mutex);
	while(cond_in_use){
		pthread_cond_wait(&cond_cond, &cond_mutex);
	}
	cond_in_use=TRUE;
	pthread_mutex_unlock(&cond_mutex);
	LOG_EXIT_VOID();
}

static void
condition_unlock()
{
	LOG_ENTRY();
   	pthread_mutex_lock(&cond_mutex);
	cond_in_use=FALSE;
	pthread_cond_signal(&cond_cond);
	pthread_mutex_unlock(&cond_mutex);
	LOG_EXIT_VOID();
}


static void
condition_unlock_sleep(pthread_cond_t *cond)
{
	LOG_ENTRY();
   	pthread_mutex_lock(&cond_mutex);
	cond_in_use=FALSE;
	pthread_cond_signal(&cond_cond);
	pthread_cond_wait(cond, &cond_mutex);
	pthread_mutex_unlock(&cond_mutex);
	LOG_EXIT_VOID();
}


/* wait till the given condition is satisfied */
int
condition_wait(char *node, u_int32_t ackval)
{
	int ret;
	ece_cond_t *element;

	LOG_ENTRY();
	
	condition_lock();

	element = g_hash_table_lookup(cond_hash, (void *)ackval);

	if(element) {
		ECE_ASSERT(strcmp(node, element->node) == 0);
		ret = element->retval;
		g_hash_table_remove(cond_hash, (void *) ackval);
		condition_unlock();
	} else {
		element = g_malloc(sizeof(ece_cond_t));
		element->node = (char *)g_strdup(node);
		element->ackval = ackval;
		element->retval = MAGICNO;
		pthread_cond_init(&(element->cond),NULL);
		g_hash_table_insert(cond_hash, (void *)ackval, element);
		condition_unlock_sleep(&(element->cond));
		ret = element->retval;
		ECE_ASSERT(ret != MAGICNO);
	}

	g_free(element->node);
	g_free(element);
	LOG_EXIT_INT(ret);
	return ret;
}

static gboolean
condition_check_1(gpointer key, gpointer value,
			gpointer user_data)
{
	ece_cond_t *element = (ece_cond_t *)value;
	ece_cond_t *data = (ece_cond_t *)user_data;

	if(strcmp(data->node, element->node) == 0) {
		element->retval = data->retval;
		pthread_cond_signal(&(element->cond));
		return TRUE;
	}
	return FALSE;
}


/* wake anybody waiting on the condition to be satisfied */
void
condition_check(const char *node, const u_int32_t ackval, const int retval)
{
	ece_cond_t *element;

	LOG_ENTRY();

	condition_lock();

	if(ackval == 0) {
		ece_cond_t data;
		/* wake up all threads waiting on a response
		 * from 'node'
		 */
		data.node = (char *)g_strdup(node);
		data.retval = retval;
		g_hash_table_foreach_remove(cond_hash,
		         condition_check_1, &data);
		g_free(data.node);
	} else {
		element = (ece_cond_t *)g_hash_table_lookup(
				cond_hash, (void *)ackval);
		if(element) {
			ECE_ASSERT(strcmp(node, element->node) == 0);
			element->retval = retval;
			pthread_cond_signal(&(element->cond));
			g_hash_table_remove(cond_hash, (void *)ackval);
		} else {
			element = g_malloc(sizeof(ece_cond_t));
			element->node = (char *)g_strdup(node);
			element->ackval = ackval;
			element->retval = retval;
			g_hash_table_insert(cond_hash,
				(void *)ackval, element);
		}
	}
	condition_unlock();
	LOG_EXIT_VOID();
}



#ifndef MSG_TRACK_BUFSIZE
#define MSG_TRACK_BUFSIZE 100
#endif

typedef struct msg_track_s {
	u_int32_t ackval;
	gboolean ackflag;
} msg_track_t;

static msg_track_t slavetolocal[MSG_TRACK_BUFSIZE];
static char stl=0;
static msg_track_t slavefromlocal[MSG_TRACK_BUFSIZE];
static char sfl=0;
static msg_track_t daemonfromslave[MSG_TRACK_BUFSIZE];
static char dfs=0;
static msg_track_t d_slavemsgtoremote[MSG_TRACK_BUFSIZE];
static char dstr=0;
static msg_track_t d_remotemsgtoslave[MSG_TRACK_BUFSIZE];
static char drts=0;
static msg_track_t d_localmsgtoremote[MSG_TRACK_BUFSIZE];
static char dltr=0;
static msg_track_t d_localmsgtoslave[MSG_TRACK_BUFSIZE];
static char dlts=0;
static msg_track_t daemonfromremote[MSG_TRACK_BUFSIZE];
static char dfr=0;

void
msg_track(const gboolean strflag,
	const struct ha_msg *h_msg,
	const char *ack_str,
	const gboolean ackflag,
	const msg_enum_t type)
{
	const char * cntl_str;
	int ackval;
	int tmp;
	int indx;
	msg_track_t *arry;

	LOG_ENTRY();

	if(!strflag) {
		assert(cntl_str = ha_msg_value(h_msg,
				ECE_CNTL));

		sscanf(cntl_str,"%d,%d,%d,%d,%d,%d",
			&ackval,
			&tmp,
			&tmp,
			&tmp,
			&tmp,
			&tmp);
	} else {
		ackval = atoi(ack_str);
	}

	switch(type) {
	case STL:
		indx = stl = (1+stl)%MSG_TRACK_BUFSIZE;
		arry = slavetolocal;
		break;
	case SFL:
		indx = sfl = (1+sfl)%MSG_TRACK_BUFSIZE;
		arry = slavefromlocal;
		break;
	case DFS:
		indx = dfs = (1+dfs)%MSG_TRACK_BUFSIZE;
		arry = daemonfromslave;
		break;
	case DSTR:
		indx = dstr = (1+dstr)%MSG_TRACK_BUFSIZE;
		arry = d_slavemsgtoremote;
		break;
	case DRTS:
		indx = drts = (1+drts)%MSG_TRACK_BUFSIZE;
		arry = d_remotemsgtoslave;
		break;
	case DLTR:
		indx = dltr = (1+dltr)%MSG_TRACK_BUFSIZE;
		arry = d_localmsgtoremote;
		break;
	case DLTS:
		indx = dlts = (1+dlts)%MSG_TRACK_BUFSIZE;
		arry = d_localmsgtoslave;
		break;
	case DFR:
		indx = dfr = (1+dfr)%MSG_TRACK_BUFSIZE;
		arry = daemonfromremote;
		break;
	default:
		arry = NULL;
		indx = 0;
	}

	if(arry) {
		arry[indx].ackval = ackval;
		arry[indx].ackflag = ackflag;
		indx = (1+indx)%MSG_TRACK_BUFSIZE;
		arry[indx].ackval = 0xfffffff;
		arry[indx].ackflag = 0;
	}
	LOG_EXIT_VOID();
}
