/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: fsimjfs.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <plugin.h>
#include "fsimjfs.h"

#if __BYTE_ORDER == __BIG_ENDIAN
	#include "jfs_byteorder.h"
void fsim_swap_jfs_superblock( struct superblock * );
void fsim_swap_log_superblock( logsuper_t * );
#else
	#define fsim_swap_jfs_superblock(sb)		do {} while (0)
	#define fsim_swap_log_superblock(logsuper)  do {} while (0)
#endif

int fsim_rw_diskblocks( logical_volume_t *, int, int64_t, int32_t, void *, int );
void set_mkfs_options( option_array_t *, char **, logical_volume_t *, char * );
void set_fsck_options( option_array_t *, char **, logical_volume_t * );

extern char jfsutils_version[10];
extern u_int32_t jfsutils_support;

/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                                   Common Routines                                    +
+                                                                                      +
+--------------------------------------------------------------------------------------*/


/*
 * Get the size limits for this volume.
 */
int fsim_get_volume_limits( struct superblock * sb_ptr,
			    sector_count_t    * min_size,
			    sector_count_t    * max_volume_size,
			    sector_count_t    * max_object_size)
{
	int rc = 0;

	/*
	 * Since JFS does not yet support shrink, min size
	 * is actual file system size.
	 */
	/*
	 * s_size == (number of physical blocks in fs) - (log space) - (fsck workspace)
	 * s_logpxd.len  == number of file system blocks for log
	 * s_fsckpxd.len == number of file system blocks for fsck workspace
	 * So, convert log space and fsck space to physical blocks and add to s_size
	 */
	*max_volume_size = ((u_int64_t) 1 << 40) * (PSIZE/PBSIZE);
	*max_object_size = ((u_int64_t) 1 << 40) * (PSIZE/PBSIZE);
	*min_size        = sb_ptr->s_size + (((sb_ptr->s_logpxd.len + sb_ptr->s_fsckpxd.len)*PSIZE)/PBSIZE);

	return rc;
}


/*
 * Un-Format the volume.
 */
int fsim_unmkfs_jfs( logical_volume_t * volume )
{
	int   fd, rc = 0;

	fd = EngFncs->open_volume(volume, O_RDWR|O_EXCL, 0);
	if (fd < 0) return -fd;

	if ( volume->private_data ) {
		/* clear private data */
		memset( (void *) volume->private_data, 0, SIZE_OF_SUPER );
		/* zero primary superblock */
		rc =  fsim_rw_diskblocks( volume, fd, SUPER1_OFF, SIZE_OF_SUPER, volume->private_data, PUT );
		/* zero secondary superblock */
		rc |= fsim_rw_diskblocks( volume, fd, SUPER2_OFF, SIZE_OF_SUPER, volume->private_data, PUT );
		/* free buffer */
		EngFncs->engine_free(volume->private_data);
		volume->private_data = NULL;
	} else {
		rc = ERROR;
	}

	EngFncs->close_volume(volume,fd);

	return rc;
}


/*
 * Un-Format the external log volume.
 */
int fsim_unmkfs_ext_log( logical_volume_t  * ExtLog_volume )
{
	int  fd, rc = 0;
	char  *buffer = NULL;

	fd = EngFncs->open_volume( ExtLog_volume, O_RDWR|O_EXCL, 0);
	if (fd < 0) return -fd;

	if ( (buffer = EngFncs->engine_alloc(sizeof(logsuper_t))) ) {
		/* zero primary superblock */
		rc =  fsim_rw_diskblocks( ExtLog_volume, fd, LOGPSIZE, sizeof(logsuper_t), buffer, PUT );
		/* free buffer */
		EngFncs->engine_free(buffer);
	} else {
		rc = ENOMEM;
	}

	fd = EngFncs->close_volume(ExtLog_volume, fd);

	return rc;
}


/*
 * Format the volume.
 */
int fsim_mkfs(logical_volume_t * volume, option_array_t * options )
{
	int     rc;
	char   *argv[MKFS_JFS_OPTIONS_COUNT + 7];
	char    logsize[25];
	pid_t   pidm;
	int     status;
	int     fds2[2];
	char    *buffer = NULL;
	int bytes_read = 0;

	if (!(buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN))) {
		return ENOMEM;
	}

	/* open pipe, alloc buffer for collecting fsck.xfs output */
	if ( (rc = pipe(fds2)) ) {
		return rc;
	}
	set_mkfs_options( options, argv, volume, logsize );

	pidm = EngFncs->fork_and_execvp(volume, argv, NULL, fds2, fds2);
	if (pidm != -1) {
		fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);
		while (!(waitpid( pidm, &status, WNOHANG ))) {
			bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
			if (bytes_read > 0) {
				LOG_DETAILS("mkfs.jfs output: \n%s",buffer);
				memset(buffer,0,bytes_read); //clear out message
			}
			usleep(10000); // don't hog all the cpu
		}

		if (WIFEXITED(status)) {
			do {
				bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
				if (bytes_read > 0) {
					LOG_DETAILS("mkfs.jfs output: \n%s",buffer);
				}
			} while (bytes_read > 0);
			/* get mkfs.jfs exit code */
			rc = WEXITSTATUS(status);
			if (rc == 0) {
				LOG_DETAILS("mkfs.jfs completed with exit code %d \n", rc);
			} else {
				LOG_ERROR("mkfs.jfs completed with exit code %d \n", rc);
			}
		} else {
			rc = EINTR;
		}
	} else {
		rc = errno;
	}

	EngFncs->engine_free(buffer);
	close(fds2[0]);
	close(fds2[1]);

	return rc;
}


/*
 * NAME: set_mkfs_options
 *
 * FUNCTION: Build options array (argv) for mkfs.jfs
 *
 * PARAMETERS:
 *      options   - options array passed from EVMS engine
 *      argv      - mkfs options array
 *      vol_name  - volume name on which program will be executed
 *
 */
void set_mkfs_options( option_array_t * options,
		       char ** argv,
		       logical_volume_t * volume,
		       char * logsize )
{
	int i, opt_count = 2;

	argv[0] = "mkfs.jfs";

	/* 'quiet' option */
	argv[1] = "-q";

	for ( i=0; i<options->count; i++ ) {

		if ( options->option[i].is_number_based ) {

			switch (options->option[i].number) {
			
			case MKFS_CHECKBB_INDEX:
				/* 'check for bad blocks' option */
				if ( options->option[i].value.b == TRUE ) {
					argv[opt_count++] = "-c";
				}
				break;

			case MKFS_CASEINSENS_INDEX:
				/* 'case insensitive' option */
				if ( options->option[i].value.b == TRUE ) {
					argv[opt_count++] = "-O";
				}
				break;

			case MKFS_SETVOL_INDEX:
				/* 'set volume name' option */
				if ( options->option[i].value.s ) {
					argv[opt_count++] = "-L";
					argv[opt_count++] = options->option[i].value.s;
				}
				break;

			case MKFS_JOURNAL_VOL_INDEX:
				/* 'set journal volume' option */
				if ( options->option[i].value.s && strcmp(options->option[i].value.s, NO_SELECTION) ) {
					argv[opt_count++] = "-j";
					argv[opt_count++] = options->option[i].value.s;
				}
				break;

			case MKFS_SETLOGSIZE_INDEX:
				/* 'set log size' option */
				if ( options->option[i].value.ui ) {
					sprintf( logsize, "%u", options->option[i].value.ui );
					argv[opt_count++] = "-s";
					argv[opt_count++] = logsize;
				}
				break;

			default:
				break;
			}

		} else {

			if ( !strcmp(options->option[i].name, "badblocks") ) {
				/* 'check for bad blocks' option */
				if ( options->option[i].value.b == TRUE ) {
					argv[opt_count++] = "-c";
				}
			}

			if ( !strcmp(options->option[i].name, "caseinsensitive") ) {
				/* 'case insensitive' option */
				if ( options->option[i].value.b == TRUE ) {
					argv[opt_count++] = "-O";
				}
			}

			if ( !strcmp(options->option[i].name, "vollabel") ) {
				/* 'set volume label' option */
				if ( options->option[i].value.s ) {
					argv[opt_count++] = "-L";
					argv[opt_count++] = options->option[i].value.s;
				}
			}

			if ( !strcmp(options->option[i].name, "journalvol") ) {
				/* 'set journal vol' option */
				if ( options->option[i].value.s && strcmp(options->option[i].value.s, NO_SELECTION) ) {
					argv[opt_count++] = "-j";
					argv[opt_count++] = options->option[i].value.s;
				}
			}

			if ( !strcmp(options->option[i].name, "logsize") ) {
				/* 'set log size' option */
				if ( options->option[i].value.ui ) {
					sprintf( logsize, "%u", options->option[i].value.ui );
					argv[opt_count++] = "-s";
					argv[opt_count++] = logsize;
				}
			}
		}
	}

	argv[opt_count++] = volume->dev_node;
	argv[opt_count] = NULL;

	return;
}


/*
 * Run fsck on the volume.
 */
int fsim_fsck(logical_volume_t * volume, option_array_t * options )
{
	int     rc;
	char   *argv[FSCK_JFS_OPTIONS_COUNT + 3];
	pid_t   pidf;
	int     status, bytes_read;
	int     fds2[2];
	char    *buffer = NULL;

	set_fsck_options( options, argv, volume );

	/* open pipe, alloc buffer for collecting fsck.jfs output */
	if ( (rc = pipe(fds2)) ) {
		return rc;
	}
	if (!(buffer = EngFncs->engine_alloc(MAX_USER_MESSAGE_LEN))) {
		return ENOMEM;
	}

	fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);

	pidf = EngFncs->fork_and_execvp(volume, argv, NULL, fds2, fds2);
	if (pidf != -1) {
		/* wait for child to complete */
		while (!(waitpid( pidf, &status, WNOHANG ))) {
			/* read fsck.jfs output */
			bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
			if (bytes_read > 0) {
				/* display fsck.jfs output */
				MESSAGE(_("fsck.jfs output: \n\n%s"),buffer);
				memset(buffer,0,bytes_read); //clear out message
			}
			usleep(10000); // don't hog all the cpu
		}
		if (WIFEXITED(status)) {
			/* get fsck.jfs exit code */
			bytes_read = read(fds2[0],buffer,MAX_USER_MESSAGE_LEN);
			if (bytes_read > 0) {
				MESSAGE(_("fsck output: \n\n%s"),buffer);
			}
			rc = WEXITSTATUS(status);
			if (rc == 0) {
				LOG_DETAILS("fsck.jfs completed with exit code %d \n", rc);
			} else {
				LOG_ERROR("fsck.jfs completed with exit code %d \n", rc);
			}
		} else {
			rc = EINTR;
		}
	} else {
		rc = errno;
	}

        EngFncs->engine_free(buffer);
	close(fds2[0]);
	close(fds2[1]);

	return rc;
}


/*
 * NAME: set_fsck_options
 *
 * FUNCTION: Build options array (argv) for fsck.jfs
 *
 * PARAMETERS:
 *      options   - options array passed from EVMS engine
 *      argv      - fsck options array
 *      volume    - volume on which program will be executed
 *
 */
void set_fsck_options( option_array_t * options, char ** argv, logical_volume_t * volume )
{
	int i, opt_count = 1;

	argv[0] = "fsck.jfs";

	for ( i=0; i<options->count; i++) {

		if ( options->option[i].is_number_based ) {

			/* 'force check' option and NOT mounted */
			if ( (options->option[i].number == FSCK_FORCE_INDEX) &&
			     (options->option[i].value.b == TRUE)         &&
			     !EngFncs->is_mounted(volume->dev_node, NULL) ) {
				/* set fsck.jfs -f option */
				argv[opt_count++] = "-f";
			}

			/* 'check read only' option or mounted */
			if ( ( (options->option[i].number == FSCK_READONLY_INDEX) &&
			       (options->option[i].value.b == TRUE) ) ||
			     EngFncs->is_mounted(volume->dev_node, NULL) ) {
				/* set fsck.jfs -n option */
				argv[opt_count++] = "-n";
			}

			/* 'omit log replay' option and NOT mounted */
			if ( (options->option[i].number == FSCK_OMITLOG_INDEX) &&
			     (options->option[i].value.b == TRUE)           &&
			     !EngFncs->is_mounted(volume->dev_node, NULL) ) {
				/* set fsck.jfs -o option */
				argv[opt_count++] = "-o";
			}

			/* 'verbose output' option */
			if ( (options->option[i].number == FSCK_VERBOSE_INDEX) &&
			     (options->option[i].value.b == TRUE) ) {
				/* set fsck.jfs -v option */
				argv[opt_count++] = "-v";
			}

			/* 'version' option */
			if ( (options->option[i].number == FSCK_VERSION_INDEX) &&
			     (options->option[i].value.b == TRUE) ) {
				/* set fsck.jfs -V option */
				argv[opt_count++] = "-V";
			}

		} else {

			/* 'force check' option selected and NOT mounted */
			if ( !strcmp(options->option[i].name, "force") &&
			     (options->option[i].value.b == TRUE) &&
			     !EngFncs->is_mounted(volume->dev_node, NULL) ) {
				/* set fsck.jfs -f option */
				argv[opt_count++] = "-f";
			}

			/* 'check read only' option selected or mounted */
			if ( ( (!strcmp(options->option[i].name, "readonly")) &&
			       (options->option[i].value.b == TRUE) ) ||
			     EngFncs->is_mounted(volume->dev_node, NULL) ) {
				/* set fsck.jfs -n option */
				argv[opt_count++] = "-n";
			}

			/* 'omit log replay' option selected and NOT mounted */
			if ( !strcmp(options->option[i].name, "omitlog") &&
			     (options->option[i].value.b == TRUE) &&
			     !EngFncs->is_mounted(volume->dev_node, NULL) ) {
				/* set fsck.jfs -o option */
				argv[opt_count++] = "-o";
			}

			/* 'verbose output' option selected */
			if ( !strcmp(options->option[i].name, "verbose") &&
			     (options->option[i].value.b == TRUE) ) {
				/* set fsck.jfs -v option */
				argv[opt_count++] = "-v";
			}

			/* 'version' option selected */
			if ( !strcmp(options->option[i].name, "version") &&
			     (options->option[i].value.b == TRUE) ) {
				/* set fsck.jfs -V option */
				argv[opt_count++] = "-V";
			}
		}
	}

	argv[opt_count++] = volume->dev_node;
	argv[opt_count]   = NULL;

	return;
}


#if __BYTE_ORDER == __BIG_ENDIAN
/*
 * NAME: fsim_swap_jfs_superblock
 *
 * FUNCTION: Endian swap a JFS superblock
 *
 * PARAMETERS:
 *      sblk   - pointer to superblock
 */
void fsim_swap_jfs_superblock( struct superblock *sblk )
{
	sblk->s_version   = __le32_to_cpu(sblk->s_version);
	sblk->s_size      = __le64_to_cpu(sblk->s_size);
	sblk->s_bsize     = __le32_to_cpu(sblk->s_bsize);
	sblk->s_l2bsize   = __le16_to_cpu(sblk->s_l2bsize);
	sblk->s_l2bfactor = __le16_to_cpu(sblk->s_l2bfactor);
	sblk->s_pbsize    = __le32_to_cpu(sblk->s_pbsize);
	sblk->s_l2pbsize  = __le16_to_cpu(sblk->s_l2pbsize);
	sblk->pad         = __le16_to_cpu(sblk->pad);
	sblk->s_agsize    = __le32_to_cpu(sblk->s_agsize);
	sblk->s_flag      = __le32_to_cpu(sblk->s_flag);
	sblk->s_state     = __le32_to_cpu(sblk->s_state);
	sblk->s_compress  = __le32_to_cpu(sblk->s_compress);

	sblk->s_logdev    = __le32_to_cpu(sblk->s_logdev);
	sblk->s_logserial = __le32_to_cpu(sblk->s_logserial);

	sblk->s_logpxd.len  = __le24_to_cpu(sblk->s_logpxd.len);
	sblk->s_fsckpxd.len = __le24_to_cpu(sblk->s_fsckpxd.len);

	/* struct timestruc_t s_time; */
	sblk->s_time.tv_sec  = __le32_to_cpu(sblk->s_time.tv_sec);
	sblk->s_time.tv_nsec = __le32_to_cpu(sblk->s_time.tv_nsec);

	sblk->s_fsckloglen = __le32_to_cpu(sblk->s_fsckloglen);
	sblk->s_xsize      = __le64_to_cpu(sblk->s_xsize);

	return;
}
#endif	/* BIG ENDIAN */


/*
 * NAME: fsim_get_jfs_superblock
 *
 * FUNCTION: Get and validate a JFS superblock
 *
 * PARAMETERS:
 *      volume   - pointer to volume from which to get the superblock
 *      sb_ptr   - pointer to superblock
 *
 * RETURNS:
 *      (0) for success
 *      != 0 otherwise
 *
 */
int fsim_get_jfs_superblock( logical_volume_t *volume, struct superblock *sb_ptr )
{
	int  fd, rc = 0;

	fd = EngFncs->open_volume(volume, O_RDONLY, 0);
	if (fd < 0) return EIO;

	/* get primary superblock */
	rc = fsim_rw_diskblocks( volume, fd, SUPER1_OFF, SIZE_OF_SUPER, sb_ptr, GET );
	fsim_swap_jfs_superblock( sb_ptr );

	if ( rc == FSIM_SUCCESS ) {
		/* see if primary superblock is JFS */
		if ( ( sb_ptr->s_version > JFS_VERSION ) ||
		     ( memcmp(sb_ptr->s_magic, JFS_MAGIC, sizeof(sb_ptr->s_magic)) != 0 ) ) {
			rc = ENOENT;
		}
	}

	if ( rc ) {  /* can't use the primary superblock */
		/* get secondary superblock */
		rc = fsim_rw_diskblocks( volume, fd, SUPER2_OFF, SIZE_OF_SUPER, sb_ptr, GET );
		fsim_swap_jfs_superblock( sb_ptr );

		if ( rc == FSIM_SUCCESS ) {
			/* see if secondary superblock is JFS */
			if ( ( sb_ptr->s_version > JFS_VERSION ) ||
			     ( memcmp(sb_ptr->s_magic, JFS_MAGIC, sizeof(sb_ptr->s_magic)) != 0 ) ) {
				rc = ENOENT;
			}
		}
	}

	EngFncs->close_volume(volume, fd);

	return rc;
}


#if __BYTE_ORDER == __BIG_ENDIAN
/*
 * NAME: fsim_swap_log_superblock
 *
 * FUNCTION: Endian swaps a JFS external log superblock
 *
 * PARAMETERS:
 *      lsup_t   - pointer to superblock
 */
void fsim_swap_log_superblock( logsuper_t *lsup_t )
{
	lsup_t->magic   = __le32_to_cpu(lsup_t->magic);
	lsup_t->version = __le32_to_cpu(lsup_t->version);
	lsup_t->serial  = __le32_to_cpu(lsup_t->serial);
	lsup_t->size    = __le32_to_cpu(lsup_t->size);
	lsup_t->bsize   = __le32_to_cpu(lsup_t->bsize);
	lsup_t->l2bsize = __le32_to_cpu(lsup_t->l2bsize);
	lsup_t->flag    = __le32_to_cpu(lsup_t->flag);
	lsup_t->state   = __le32_to_cpu(lsup_t->state);
	lsup_t->end     = __le32_to_cpu(lsup_t->end);

	return;
}
#endif	/* BIG ENDIAN */


/*
 * NAME: fsim_get_log_superblock
 *
 * FUNCTION: Get and validate a JFS log superblock
 *
 * PARAMETERS:
 *      dev_node     - pointer to volume from which to get the superblock
 *      log_sb_ptr   - pointer to superblock
 *
 * RETURNS:
 *      (0) for success
 *      != 0 otherwise
 *
 */
int fsim_get_log_superblock( logical_volume_t * volume, logsuper_t *log_sb_ptr )
{
	int  fd, rc = 0;

	fd = EngFncs->open_volume(volume, O_RDONLY, 0);
	if (fd < 0) return EIO;

	/* get log superblock */
	rc = fsim_rw_diskblocks( volume, fd, LOGPSIZE, sizeof(logsuper_t), log_sb_ptr, GET );
	fsim_swap_log_superblock( log_sb_ptr );

	if ( rc == FSIM_SUCCESS ) {
		/* see if superblock is JFS log superblock */
		if ( ( log_sb_ptr->version != LOGVERSION ) ||
		     ( log_sb_ptr->magic   != LOGMAGIC   ) ) {
			rc = ENOENT;
		}
	}

	EngFncs->close_volume(volume, fd);

	return rc;
}


/*
 * NAME: fsim_rw_diskblocks
 *
 * FUNCTION: Read or write specific number of bytes for an opened device.
 *
 * PARAMETERS:
 *      dev_ptr         - file handle of an opened device to read/write
 *      disk_offset     - byte offset from beginning of device for start of disk
 *                        block read/write
 *      disk_count      - number of bytes to read/write
 *      data_buffer     - On read this will be filled in with data read from
 *                        disk; on write this contains data to be written
 *      mode            - GET (read) or PUT (write)
 *
 * RETURNS:
 *      FSIM_SUCCESS (0) for success
 *      EINVAL
 *      EIO
 *
 */
int fsim_rw_diskblocks( logical_volume_t * vol,
			int      dev_ptr,
			int64_t  disk_offset,
			int32_t  disk_count,
			void     *data_buffer,
			int      mode )
{
	int32_t  Bytes_Transferred;

	switch ( mode ) {
	case GET:
		Bytes_Transferred = EngFncs->read_volume(vol,dev_ptr,data_buffer,disk_count,disk_offset);
		break;
	case PUT:
		Bytes_Transferred = EngFncs->write_volume(vol,dev_ptr,data_buffer,disk_count,disk_offset);
		break;
	default:
		return EINVAL;
		break;
	}

	if (Bytes_Transferred < 0) {
		LOG_EXIT_INT(-Bytes_Transferred);
		return -Bytes_Transferred;

	} else if (Bytes_Transferred != disk_count) {
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	return FSIM_SUCCESS;
}


/*
 * Test mkfs.jfs version.
 *    jfsutils must be at version 1.0.9 or greater to function properly
 *    with this FSIM.  That also happens to be the first version in which
 *    mkfs.jfs supported the -V (version) option.  So, run mkfs.jfs with
 *    the -V option,and if it succeeds, the installed version is OK.  If
 *    it fails, pass back the exit code (should be EINVAL) of the failure.
 */
int fsim_test_version( )
{
	int     rc;
	int     dev_rc = 0;
	char   *argv[3];
	pid_t   pidm;
	int     status, bytes_read;
	int     fds2[2];
	char    *buffer = NULL;
	char    *buf_ptr = NULL;
	int     ver_string_size;

	/* open pipe, alloc buffer for collecting fsck.jfs output */
	rc = pipe(fds2);
	if (rc) {
		return(rc);
	}
	if (!(buffer = EngFncs->engine_alloc(PBSIZE))) {
		return(ENOMEM);
	}

	argv[0] = "mkfs.jfs";
	argv[1] = "-V";
	argv[2] = NULL;

	pidm = EngFncs->fork_and_execvp(NULL, argv, NULL, fds2, fds2);
	if (pidm != -1) {
		fcntl(fds2[0], F_SETFL, fcntl(fds2[0], F_GETFL,0) | O_NONBLOCK);
		waitpid( pidm, &status, 0 );
		if ( WIFEXITED(status) ) {
			bytes_read = read(fds2[0],buffer,PBSIZE);
			if (bytes_read > 0) {
				/* check for pre 1.0.9 versions */
				buf_ptr = strstr(buffer, "development");
				if (buf_ptr) {
					dev_rc = ENOENT;
					memset(jfsutils_version, 0, 9);
				} else {
					/* find start and end of version number, store in jfsutils_version */
					buf_ptr = strstr(buffer, "version");
					if (buf_ptr) {
						jfsutils_support |= JFS_UTILS_EXIST;
						buf_ptr = strstr(buf_ptr, " ") + 1;
						ver_string_size = strstr(buf_ptr, ",") - buf_ptr;
						strncpy(jfsutils_version, buf_ptr, ver_string_size);
						/*
						* Even though external log was initially supported
						* in version 1.0.18, there was a mkfs.jfs bug (fixed
						* in 1.0.20) that kept the FSIM from being able to
						* determine if a JFS volume was using an external log.
						* Online expand was added in 1.0.21.
						*/
						if (strcmp(jfsutils_version, "1.0.21") >= 0) {
							jfsutils_support |= JFS_EXTERNAL_LOG;
							jfsutils_support |= JFS_EXPAND;
						} else if (strcmp(jfsutils_version, "1.0.20") >= 0) {
							jfsutils_support |= JFS_EXTERNAL_LOG;
						}
					} else {
						memset(jfsutils_version, 0, 9);
					}
				}
			}
			/* get mkfs.jfs exit code */

			LOG_DEFAULT("mkfs.jfs test version completed with exit code %d \n",WEXITSTATUS(status));
			LOG_DEFAULT("JFS utilities version: %s\n", (jfsutils_version[0] != '\0') ? jfsutils_version : "(none)");
		} else {
			LOG_ERROR("JFS test version can't find utils %d \n", rc);
			rc = 0;//EINTR;
		}
	} else {
		rc = errno;
	}

	EngFncs->engine_free(buffer);

	close(fds2[0]);
	close(fds2[1]);
	
	return(rc|dev_rc);
}


/*
 * NAME: Is_JFS_Log_Vol
 *
 * FUNCTION: Determine if volume is JFS external log
 *
 * PARAMETERS:
 *      log_sb_ptr      - pointer to suspected log superblock
 *
 * RETURNS:
 *      -1 if volume is a JFS external log
 *       0 if volume is not a JFS external log
 */
int Is_JFS_Log_Vol( logsuper_t *log_sb_ptr)
{
	if ( ( jfsutils_support & JFS_EXTERNAL_LOG ) && log_sb_ptr ) {
		if ( (log_sb_ptr->version == LOGVERSION) &&
		     (log_sb_ptr->magic   == LOGMAGIC  )   ) {
			return -1;
		}
	}

	return 0;
}

