/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libs390.so
 *
 *   File: helpers.h
 */
#ifndef S390_HELPERS_HEADER
#define S390_HELPERS_HEADER

extern list_anchor_t  Disk_PrivateData_List;

#include "390segmgr.h"


DISKSEG *          allocate_s390_disk_segment( LOGICALDISK *ld );
void               free_s390_disk_segment( DISKSEG *seg);
int                create_s390_disk_private_data( LOGICALDISK  *ld );
int                delete_s390_disk_private_data( LOGICALDISK *ld );
DISK_PRIVATE_DATA *get_s390_disk_private_data( LOGICALDISK *ld );

DISKSEG *          get_freespace_following_s390_segment( DISKSEG *seg );
DISKSEG *          find_freespace_in_seglist( list_anchor_t  seglist );
int                merge_adjacent_freedisksegs_in_list( list_anchor_t seglist );
int                find_freespace_on_s390_disk( LOGICALDISK *ld );

DISKSEG *          create_s390_metadata_segment( LOGICALDISK  *ld,  lba_t start, sector_count_t size );

int                remove_s390_segment_from_list( list_anchor_t seglist, storage_object_t *seg );
void *             insert_s390_segment_into_list( list_anchor_t seglist, storage_object_t *seg );
void *             insert_s390_segment_into_ordered_list( list_anchor_t seglist, storage_object_t *seg );

DISKSEG *          get_s390_segment_from_minor( LOGICALDISK *ld, int minor );
int                get_next_s390_minor( LOGICALDISK *ld );

int                fixup_390_partition_names( LOGICALDISK *ld );

void               atoe( char *buffer, int length );
void               etoa( char *buffer, int length );
int                init_format1_dscb( LOGICALDISK *ld, DISKSEG *seg );
int                init_format5_dscb( LOGICALDISK *ld );
int                get_vtoc_size( LOGICALDISK *ld );
int                save_390_disk_geometry( LOGICALDISK *ld, DISK_PRIVATE_DATA *disk_pdata );

boolean            i_can_format_disk( DISKSEG *object );

void               prune_s390_segments_from_list( list_anchor_t list );


/*
 *  Called to return the Logical Disk that a segment belongs to.
 *
 *  Can be called with any object.
 *
 *  Returns NULL on failure.
 */
static inline LOGICALDISK * get_logical_disk( storage_object_t *obj )
{
        LOGICALDISK  *ld=NULL;

        if (obj) {

                if (obj->object_type == DISK) {
                        ld = obj;
                }
                else if (obj->private_data) {

                        if ( ((SEG_PRIVATE_DATA *)obj->private_data)->signature == S390_SEG_MGR_PDATA_SIGNATURE ) {

                                ld = ((SEG_PRIVATE_DATA *)obj->private_data)->logical_disk;

                        }

                }

        }

        return ld;
}


/*
 *     LBA addresses are converted into CHS addressees by this formula:
 *
 *        Sector    = ( LBA MOD Sectors Per Track ) + 1
 *        Head      = ( LBA DIV Sectors Per Track ) MOD Heads Per Cylinder
 *        Cylinder  = ( LBA DIV Sectors Per Track ) DIV Heads Per Cylinder
 */
static inline int LBAtoCHS( LOGICALDISK *ld, lba_t  lba, chs_t *chs )
{
        u_int32_t          sectors_per_track;
        u_int32_t          drive_heads;
        u_int32_t          sectors_per_cylinder;
        int                rc;
        DISK_PRIVATE_DATA *disk_pdata = get_s390_disk_private_data( ld );


        if ( (chs != NULL) &&
             (disk_pdata != NULL) &&
             (ld->geometry.sectors_per_track > 0)  ) {

                memset(chs, 0, sizeof(chs_t));

                sectors_per_track    = ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block;
                drive_heads          = ld->geometry.heads;
                sectors_per_cylinder = sectors_per_track * drive_heads;

                chs->sector   = ( lba % sectors_per_track ) + 1;
                chs->head     = ( lba / sectors_per_track ) % drive_heads;
                chs->cylinder =   lba / sectors_per_cylinder;

                rc = 0;
        }
        else {
                rc = EINVAL;
        }

        return rc;
}



/*
 *     CHS addresses are converted into LBA addresses by the following formula:
 *
 *        LBA = (Sector - 1) + (Head * Sectors Per Track) + (Cylinder * Heads Per Cylinder * Sectors Per Track)
 *
 */
static inline int CHStoLBA( LOGICALDISK *ld, chs_t *chs, lba_t *callers_lba )
{
        u_int32_t          sectors_per_track;
        u_int32_t          drive_heads;
        u_int32_t          sectors_per_cylinder;
        lba_t              lba = 0;
        int                rc;
        DISK_PRIVATE_DATA *disk_pdata = get_s390_disk_private_data( ld );


        if (  ( chs != NULL ) &&
              ( callers_lba != NULL ) &&
              ( disk_pdata != NULL ) &&
              ( ld->geometry.sectors_per_track > 0 ) ) {

                sectors_per_track    = ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block;
                drive_heads          = ld->geometry.heads;
                sectors_per_cylinder = sectors_per_track * drive_heads;

                lba = (chs->sector - 1 ) + (chs->head * sectors_per_track) + (chs->cylinder * sectors_per_cylinder );

                rc = 0;
        }
        else {
                rc = EINVAL;
        }

        *callers_lba = lba;

        return rc;
}




/*
 *  Returns TRUE if the LBA starts on a cylinder boundary
 */
static inline boolean  starts_on_cylinder_boundary( LOGICALDISK *ld, lba_t lba )
{
        chs_t  chs;

        if ( LBAtoCHS( ld, lba, &chs ) ) {
                return TRUE;
        }
        else {

                if ( ( chs.sector == 1 ) && ( chs.head == 0 ) ) {
                        return TRUE;
                }
                else {
                        return FALSE;
                }
        }

}



/*
 *  Returns TRUE if the LBA ends on a cylinder boundary
 */
static inline boolean  ends_on_cylinder_boundary( LOGICALDISK *ld, lba_t lba )
{
        chs_t  chs;
        DISK_PRIVATE_DATA *disk_pdata = get_s390_disk_private_data( ld );


        if ( LBAtoCHS( ld, lba, &chs ) ) {
                return TRUE;
        }
        else {

                if ( ( chs.sector == (ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block) ) &&
                     ( chs.head   == ld->geometry.heads-1 ) ) {
                        return TRUE;
                }
                else {
                        return FALSE;
                }

        }

}


/*
 *  Called to calculate the cylinder size for the specified
 *  storage object. Storage object might be either DISK or
 *  SEGMENT type of object.
 */
static inline sector_count_t get_cylinder_size( storage_object_t *obj )
{
        u_int32_t          sectors_per_track=0;
        u_int32_t          drive_heads=0;
        sector_count_t     sectors_per_cylinder=0;
        LOGICALDISK       *ld=NULL;
        DISK_PRIVATE_DATA *disk_pdata=NULL;


        if ( obj ) {

                if (obj->object_type == DISK) {
                        ld = obj;
                }
                else {
                        ld = get_logical_disk(obj);
                }

                if (ld) {

                        disk_pdata           = get_s390_disk_private_data( ld );

                        sectors_per_track    = ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block;
                        drive_heads          = ld->geometry.heads;
                        sectors_per_cylinder = (sector_count_t) sectors_per_track * drive_heads;

                }

        }


        return sectors_per_cylinder;
}


/*
 *  Returns the specified LBA rounded up to a track boundary.
 */
static inline lba_t  roundup_to_track_boundary( LOGICALDISK *ld, lba_t lba )
{
        lba_t              new_lba = lba;
        sector_count_t     extra_sectors=0;
        sector_count_t     sectors_per_track=0;
        DISK_PRIVATE_DATA *disk_pdata;

        disk_pdata = get_s390_disk_private_data( ld );

        if ( disk_pdata ) {

                sectors_per_track = ld->geometry.sectors_per_track * disk_pdata->vsectors_per_block;

                if (sectors_per_track) {

                        extra_sectors = lba % sectors_per_track;

                        if ( extra_sectors != 0) {
                                new_lba = lba + ( sectors_per_track - extra_sectors ) - 1;
                        }

                }

        }

        return new_lba;
}



/*
 *  Returns the specified LBA rounded up to a cylinder boundary.
 */
static inline lba_t  roundup_to_cylinder_boundary( LOGICALDISK *ld, lba_t lba )
{
        lba_t              new_lba = lba;
        sector_count_t     extra_sectors=0;
        sector_count_t     sectors_per_cylinder;


        sectors_per_cylinder = get_cylinder_size(ld);

        if ( sectors_per_cylinder ) {

                extra_sectors = lba % sectors_per_cylinder;

                if ( extra_sectors != 0) {
                        new_lba = lba + ( sectors_per_cylinder - extra_sectors) - 1;
                }

        }

        return new_lba;
}


/*
 *  Returns the specified LBA rounded down to a cylinder boundary.
 */
static inline lba_t  rounddown_to_cylinder_boundary( LOGICALDISK *ld, lba_t lba )
{
        lba_t            new_lba=lba;
        sector_count_t   extra_sectors=0;
        sector_count_t   sectors_per_cylinder;



        sectors_per_cylinder = get_cylinder_size(ld);

        if (sectors_per_cylinder) {

                extra_sectors = lba % sectors_per_cylinder;

                if ( extra_sectors != 0) {
                        new_lba = lba - extra_sectors;
                }

        }

        return new_lba;
}


/*
 *  Returns the s390 metadata segment from the specified logical disk.
 */
static inline storage_object_t *  get_metadata_segment_from_disk( LOGICALDISK *ld )
{
        DISKSEG *md=NULL;
        DISK_PRIVATE_DATA *disk_pdata;


        if (ld) {

                disk_pdata = get_s390_disk_private_data(ld);
                if (disk_pdata) {

                        md = disk_pdata->md;

                }

        }

        return md;
}


/*
 * compute the cyl-cyl-head-head structure
 * from a 390 logical block address
 */
static inline int  blk2cchh( lba_t lba, cchh_t *ptr, struct hd_geometry *geo)
{
        int    rc = EINVAL;

        if (ptr && geo ) {

                if (geo->heads && geo->sectors) {

                        ptr->cc = lba / (geo->heads * geo->sectors);
                        ptr->hh = (lba % (geo->heads * geo->sectors)) / geo->sectors;

                        rc = 0;
                }

        }

        return rc;
}


/*
 * compute the cyl-cyl-head-head structure
 * from a 390 logical block address
 */
static inline int  blk2cchhb( lba_t lba, cchhb_t *ptr, struct hd_geometry *geo)
{
        int    rc = EINVAL;

        if (ptr && geo ) {

                if (geo->heads && geo->sectors) {

                        ptr->cc = lba / (geo->heads * geo->sectors);
                        ptr->hh = (lba % (geo->heads * geo->sectors)) / geo->sectors;
                        ptr->b  = (lba % (geo->heads * geo->sectors)) % geo->sectors;

                        rc = 0;
                }

        }

        return rc;
}




/*
 * compute the block number from a
 * cyl-cyl-head-head-block structure
 *
 *  390 cylinders are numbered 0-n
 *   "  heads      "     "     0-n
 *   "  blocks     "     "     1-n
 *
 */
static inline lba_t  cchhb2blk ( cchhb_t *ptr, struct hd_geometry *geo)
{
        lba_t  lba=0;

        if (ptr && geo) {

                lba = ptr->cc * geo->heads * geo->sectors + ptr->hh * geo->sectors;
                if (ptr->b) {
                        lba += ptr->b - 1;
                }

        }

        return lba;
}

/*
 * compute the block number from a
 * cyl-cyl-head-head structure
 */
static inline lba_t  cchh2blk ( cchh_t *ptr, struct hd_geometry *geo)
{
        lba_t  lba=0;

        if (ptr && geo) {

                lba = ptr->cc * geo->heads * geo->sectors + ptr->hh * geo->sectors;

        }

        return lba;
}


// list of known volume labels
static char disk_label_names[7][8] = {
        {0xd3,0xd5,0xe7,0xf1,0x00},//LNX1
        {0xe5,0xd6,0xd3,0xf1,0x00},//VOL1
        {0xc3,0xd4,0xe2,0xf1,0x00},//CMS1
        {0xc5,0xe5,0xd4,0xe2,0x00} //EVMS - means unformatted
};


static inline ibm_label_type_t  get_partition_type ( char * type )
{
        int i;
        for ( i = 0; i < 4; i ++) {
                if ( ! strncmp(type, disk_label_names[i], 4) )
                        break;
        }
        return i;
}




/*
 *  Called to test if we have private data for a disk.
 */
static inline boolean i_can_modify( storage_object_t *object )
{
        SEG_PRIVATE_DATA    *pdata;

        if (object) {

                pdata = (SEG_PRIVATE_DATA *) object->private_data;

                if (pdata) {

                        if (pdata->signature == S390_SEG_MGR_PDATA_SIGNATURE ) {
                                return TRUE;
                        }

                }
                else {
                        LOG_DEBUG("pdata is null\n");
                }
        }
        else {
                LOG_DEBUG("null object ptr\n");
        }
        return FALSE;
}


/*
 *  Called to see if the VTOC has room left in it for another partition
 */
static inline boolean i_can_add_partition_to_vtoc( LOGICALDISK *ld )
{
        boolean rc=FALSE;
        DISK_PRIVATE_DATA *disk_pdata=NULL;

        if (ld) {

                disk_pdata = get_s390_disk_private_data( ld );

                if (disk_pdata) {

                        if ( disk_pdata->partition_count < disk_pdata->max_partitions_allowed ) {
                                rc = TRUE;
                        }
                }

        }

        return rc;
}

/*
 *  Called to determine if the specified disk is a CDL formatted disk.
 */
static inline boolean isa_compatibility_disk(LOGICALDISK *ld)
{
        boolean rc=FALSE;
        DISK_PRIVATE_DATA *disk_pdata=NULL;

        if (ld) {

                disk_pdata = get_s390_disk_private_data( ld );

                if (disk_pdata) {

                        if ( disk_pdata->disk_layout == ibm_cdl ) {
                                rc = TRUE;
                        }
                }

        }

        return rc;
}


static inline boolean disk_move_pending( storage_object_t *object )
{               
        LOGICALDISK        *ld=get_logical_disk(object);        
        DISK_PRIVATE_DATA  *disk_pdata=NULL;

        if ( ld ) {

                disk_pdata = get_s390_disk_private_data(ld);

                if (disk_pdata) {

                        if (disk_pdata->flags & DISK_HAS_MOVE_PENDING) {
                                return TRUE;
                        }

                }

        }

        return FALSE;
}

#endif
