/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <time.h>
#include <glib.h>
#include <ncurses.h>
#include <panel.h>
#include <frontend.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "dialog.h"
#include "clist.h"
#include "selwin.h"
#include "views.h"
#include "callbacks.h"
#include "enum.h"
#include "readable.h"
#include "logging.h"
#include "backup.h"

/**
 *	append_changed_item - append a changed item to the given clist
 *	@clist: the column list
 *	@item: the pointer to a changed record structure
 *
 *	This routine appends a row pertaining to the name and reason for a
 *	storage object that has changed.
 */
static inline void append_changed_item(struct clist *clist, change_record_t *item)
{
	GPtrArray *text;
	
	text = g_ptr_array_new();
	g_ptr_array_add(text, g_strdup(item->name));
	g_ptr_array_add(text, changes_flag_to_string(item->changes));
	append_item(clist, text, NULL, NULL);
	g_ptr_array_free(text, TRUE);
}

/**
 *	populate_clist_with_changed_objects - populate a clist with item that have changes pending
 *	@clist: the column list
 *
 *	This routine gets the list of things that the engine indicates have changed.
 *	For things that have metadata changes, i.e., not just activation changes,
 *	we place the name and reason for the change as a row in a column list.
 */
static void populate_clist_with_changed_objects(struct clist *clist)
{
	int rc;
	boolean changes_pending;
	change_record_array_t *changes = NULL;
	
	rc = evms_changes_pending(&changes_pending, &changes);
	if (rc == 0) {
		if (changes_pending && changes != NULL) {
			int i;
		
			for (i = 0; i < changes->count; i++) {
				if (changes->changes_pending[i].changes & ~(CHANGE_ACTIVATE |
									    CHANGE_DEACTIVATE |
									    CHANGE_REACTIVATE)) {
					append_changed_item(clist, &(changes->changes_pending[i]));
				}
			}
			evms_free(changes);
		}
	} else {
		log_error("%s: evms_changes_pending() returned error code %d.\n", __FUNCTION__, rc);
	}
}

/**
 *	details_button_activated - display a dialog window with a list of things with pending changes
 *	@item: the menu item that was activated
 *
 *	This routine displays a dialog with the list of objects that indicate they have
 *	some sort of change pending. It is presented when the user activates the "Details"
 *	button from the cannot backup dialog.
 */
static int details_button_activated(struct menu_item *item)
{
	struct selwin *list_dialog;
	struct dialog_window *dialog;

	list_dialog = create_list_dialog(_("Objects with Pending Changes"),
				NULL,
				_("This list shows objects with changes pending"),
				2, NULL);
	dialog = (struct dialog_window *)list_dialog;

	set_clist_column_count(list_dialog->clist, 2);
	set_clist_column_info(list_dialog->clist, 0, calc_clist_column_width(list_dialog->clist, 0.60),
				0,
				CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(list_dialog->clist, 1, calc_clist_column_width(list_dialog->clist, 0.40),
				get_clist_column_end(list_dialog->clist, 0),
				CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(list_dialog->clist, 0, _("Name"));
	print_clist_column_title(list_dialog->clist, 1, _("Reason"));

	populate_clist_with_changed_objects(list_dialog->clist);
	process_modal_dialog(dialog);
	
	return 0;
}

/**
 *	backup_button_activated - invoked to backup metadata
 *	@item: the backup button that invoked this function
 *
 *	This routine is invoked when the user activates the Backup button.
 */
static int backup_button_activated(struct menu_item *item)
{
	int rc;
	struct dialog_window *dialog = item->user_data;

	rc = evms_metadata_backup(NULL);

	update_status(_("Backup"), rc);
	if (rc == 0) {
		dialog->status = DLG_STATUS_CLOSING;
		refresh_views();
	}

	return 0;
}

/**
 *	show_backup_confirmation_dialog - invoked to confirm a backup operation
 *	@title: the window title
 *	@cancel_label: the label text for the cancel button
 *	@prompt: the prompt to display in the window
 *
 *	This routine is invoked to present the user a confirmation dialog for a
 *	backup operation.
 */
static void show_backup_confirmation_dialog(char *title, char *prompt)
{
	struct dialog_window *dialog;
	dialog = create_dialog_window(
				title,
				NULL,
				(dialog_event_handler)NULL,
				(dialog_show_func)NULL,
				(dialog_delete_func)NULL,
				(dialog_delete_cb)NULL,
				_("_Backup"),
				(menuitem_activate_cb)backup_button_activated,
				NULL,
				NULL,
				NULL);

	print_centered(dialog->win, getmaxy(dialog->win) / 2, prompt);

	set_menu_item_sensitivity(dialog->next_button, TRUE);
	set_menu_item_visibility(dialog->prev_button, FALSE);
	set_menu_item_visibility(dialog->cancel_button, TRUE);
	set_horizontal_menu_focus(dialog->menu, dialog->next_button);
	process_modal_dialog(dialog);
}

/**
 *	show_cannot_backup_dialog - invoked to confirm a backup operation
 *	@title: the window title
 *	@cancel_label: the label text for the cancel button
 *	@prompt: the prompt to display in the window
 *
 *	This routine is invoked to present the user a confirmation dialog for a
 *	backup operation.
 */
static void show_cannot_backup_dialog(char *title, char *prompt)
{
	struct dialog_window *dialog;
	dialog = create_dialog_window(
				title,
				NULL,
				(dialog_event_handler)NULL,
				(dialog_show_func)NULL,
				(dialog_delete_func)NULL,
				(dialog_delete_cb)NULL,
				_("_OK"),
				(menuitem_activate_cb)close_window_button_activated,
				_("_Details"),
				(menuitem_activate_cb)details_button_activated,
				NULL);

	print_centered(dialog->win, getmaxy(dialog->win) / 2, prompt);

	set_menu_item_sensitivity(dialog->next_button, TRUE);
	set_menu_item_sensitivity(dialog->prev_button, TRUE);
	set_menu_item_visibility(dialog->cancel_button, FALSE);
	set_horizontal_menu_focus(dialog->menu, dialog->next_button);
	process_modal_dialog(dialog);
}

/**
 *	on_backup_menuitem_activated - callback innvoked when user has selected to backup metadata
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the Actions->Backup menu item to attempt to backup
 *	metadata.
 */
int on_backup_menuitem_activated(struct menu_item *item)
{
	int rc;
	int i;
	boolean changes_pending;
	change_record_array_t * changes;

	rc = evms_changes_pending(&changes_pending, &changes);
	if (rc == 0) {
		if (changes != NULL) {
			/* The Engine cannot backup metadata if metadata changes
			 * are scheduled.  Activation changes are OK. */
			changes_pending = FALSE;
			for (i = 0; !changes_pending && (i < changes->count); i++) {
				if (changes->changes_pending[i].changes & ~(CHANGE_ACTIVATE |
									    CHANGE_DEACTIVATE |
									    CHANGE_REACTIVATE)) {
					changes_pending = TRUE;
				}
			}
		}
		if (changes_pending)
			show_cannot_backup_dialog(_("Cannot Backup Metadata"),
						  _("Metadata cannot be backed up while metadata changes are pending."));
		else
			show_backup_confirmation_dialog(_("Backup Metadata Confirmation"),
							_("Do you wish to backup metadata now or cancel?"));
	} else {
		show_message_dialog(_("Error"), _("Unable to determine if changes are pending!"));
		log_error("%s: evms_changes_pending() returned error code %d.\n", __FUNCTION__, rc);
	}

	return 0;
}
