/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <string.h>
#include <time.h>
#include <glib.h>
#include <frontend.h>
#include <ncurses.h>
#include <panel.h>

#include "common.h"
#include "dlist.h"
#include "menu.h"
#include "window.h"
#include "dialog.h"
#include "clist.h"
#include "selwin.h"
#include "callbacks.h"

static int message_number = 0;
static int new_messages = FALSE;
static int queue_message = FALSE;
static GList *message_queue;

/*
 * Track the number of progress callbacks we have that
 * are still with us.
 */
static guint progress_callbacks_started = 0;

/**
 *	queue_informational_message - places an informational message in queue
 *	@message: the information message to duplicate and queue
 *
 *	This routine allocates a queue element, duplicates the message and adds
 *	the alert message queue element to the queue.
 */
void queue_informational_message(char *message)
{
	if (message != NULL) {
		struct queued_message *queued_message;
		
		queued_message = g_new0(struct queued_message, 1);
		queued_message->timestamp = time(NULL);
		queued_message->number = message_number++;
		queued_message->message = g_strdup(message);
		
		message_queue = g_list_append(message_queue, queued_message);
		new_messages = TRUE;
	}
}

/**
 *	delete_queued_message - deletes a queued informational message
 *	@queued_message: the queued message
 *	@not_used: second parm required for a GFunc function but not used
 *
 *	This routine removes a message in the alert message queue and
 *	frees the memory for the message and the message queue element.
 */
void delete_queued_message(struct queued_message *queued_message, gpointer not_used)
{
	if (queued_message != NULL) {
		g_free(queued_message->message);
		g_free(queued_message);
	}
}

/**
 *	enable_message_queuing - allows queuing of information alerts
 *	@void
 *
 *	This routine simply sets a flag that indicates to the show_user_message()
 *	routine to queue information only alert messages rather than immediately
 *	display them.
 **/
void enable_message_queuing(void)
{
	new_messages = FALSE;
	queue_message = TRUE;
}

/**
 *	disable_message_queuing - stop queuing information only alert messages
 *	@void
 *
 *	This routine simply turns the flags off that indicates that alert messages
 *	should be queued rather than displayed immediately.
 **/
void disable_message_queuing(void)
{
	queue_message = FALSE;
}

/**
 *	report_progress - callback to update statusbar with progress by engine
 *	@progress: the address of the structure containing progress info
 *
 *	This routine is called by the engine to provide operation progress info
 *	typically originated by a plugin.
 **/
int report_progress(progress_t *progress)
{
	if (progress->title) {
		char *text;

		if (progress->id == 0) {
			progress_callbacks_started++;
			progress->id = progress_callbacks_started;
		} else if (progress->count >= progress->total_count) {
			progress_callbacks_started--;
		}

		if (progress->remaining_seconds != 0) {
			char *title;
			unsigned int seconds, minutes, hours;

			seconds = progress->remaining_seconds;
			hours = seconds / 3600;
			seconds %= 3600;
			minutes = seconds / 60;
			seconds %= 60;

			title = g_strdup(progress->title);

			if (hours != 0) {
				text = g_strdup_printf(_("%s  (Time remaining:  %02u:%02u:%02u)"),
							g_strstrip(title),
							hours, minutes, seconds);
			} else {
				text = g_strdup_printf(_("%s  (Time remaining:  %02u:%02u)"),
							g_strstrip(title),
							minutes, seconds);
			}
			g_free(title);
		} else {
			text = g_strdup(progress->title);
		}
		
		print_statusbar_text(text);
		g_free(text);

		if (progress->type == DISPLAY_PERCENT) {
			int startx;
			char percent_complete[6];

			if (progress->total_count == 0) {
				strcpy(percent_complete, "0%");
			} else {
				float percentage;

				percentage = ((float)progress->count / (float)progress->total_count) * 100;
				g_snprintf(percent_complete, sizeof(percent_complete), " %.0f%%",  percentage);
			}
			startx = getmaxx(stdscr) - strlen(percent_complete) - 2;
			mvwaddstr(get_statusbar_win(), 0, startx, percent_complete);
			wrefresh(get_statusbar_win());
		}
	}
 	return 0;
}

/**
 *	on_alert_choice_deleted - callback for when alert choice deleted
 *	@item: the menu item being deleted
 *
 *	This routine simply frees the alert_choice_info struct
 *	associated with a alert choice menu item.
 */
void on_alert_choice_deleted(struct menu_item *item)
{
	g_free(item->user_data);
}

/**
 *	alert_choice_activated - callback for when alert choice is activated
 *	@item: the menu item that was activated
 *
 *	This routine simply updates the alert's return selection
 *	to the index of the activated choice.
 */
int alert_choice_activated(struct menu_item *item)
{
	struct alert_choice_info *info = item->user_data;

	*(info->return_selection) = info->our_index;

	return 0;
}

/**
 *	append_alert_choice - append a choice to the menu of choices for an alert window
 *	@menu: the horizontal menu
 *	@choice: the string containing the choice
 *	@return_selection: where to place index of choice selection. initialized with default choice.
 *	@our_index: the index value for our choice
 *
 *	This routine appends one alert message choice to the horizontal menu of choices.
 *	It sets up the callbacks for the alert choice and also sets focus on this choice
 *	if it is the default choice.
 **/
void append_alert_choice(struct horizontal_menu *menu, char *choice, int *return_selection, int our_index)
{
	char *choice_button;
	struct menu_item *item;
	struct alert_choice_info *info;
	
	info = g_new(struct alert_choice_info, 1);
	info->our_index = our_index;
	info->return_selection = return_selection;

	choice_button = g_strdup_printf("[%s]", choice);

	item = pack_menu_item_at_end(menu, choice_button, 0,
				(menuitem_activate_cb)alert_choice_activated,
				info);

	set_menu_item_delete_cb(item, (menuitem_delete_cb)on_alert_choice_deleted);

	/*
	 * If the default return selection matches this one then make
	 * it the focus item.
	 */
	if (*return_selection == our_index) {
		set_horizontal_menu_focus(menu, item);		
	}

	g_free(choice_button);
}

/**
 *	show_alert_prompt - prompt a user with a set of choices a plug-in has supplied
 *	@message: the alert message to display
 *	@choice_selection: the index of the choice selection
 *	@choices: array of choices
 *
 *	This routine creates a popup window with an alert message a set of choices
 *	for the user. The popup window is removed when the user selects a choice.
 */
void show_alert_prompt(char *message, int *choice_selection, char **choices)
{
	int key = 0, i;
	PANEL *panel;
	WINDOW *win;
	struct horizontal_menu *menu;

	panel = create_centered_popup_window(getmaxy(stdscr) - 2,
					getmaxx(stdscr) - 6, WHITE_BKGD);
	win = panel_window(panel);

	mvwprintw(win, 0, 0, message);

	menu = create_horizontal_menu(win, getmaxx(win) - 4, getmaxy(win) - 2, 1, 1);
	
	for (i = 0; choices[i] != NULL; i++) {
		append_alert_choice(menu, choices[i], choice_selection, i);
	}

	draw_horizontal_menu(menu);
	show_popup_window(panel);

	while (key != KEY_ENTER) {
		key = panel_getch(panel);
		process_horizontal_menu_events(menu, &key);
	}

	delete_horizontal_menu(menu);	
	delete_popup_window(panel);
}

/**
 *	show_informational_message - display a dialog for an informational message received
 *	@number: the message number
 *	@timestamp: the time the message was received
 *	@message: the message string
 *
 *	This routine displays a dialog box containing an informational
 *	message we've received from the engine or a plug-in.
 */
void show_informational_message(int number, time_t timestamp, char *message)
{
	gchar *title;
	
	title = g_strdup_printf(_("Message %d received on %s"), number,
				g_strchomp(ctime(&timestamp)));
	show_message_dialog(title, "%s", message);
	g_free(title);
}

/**
 *	show_user_message - callback to allow engine or engine plugin to alert user
 *	@message: the alert message to display
 *	@choice_selection: the index of the choice selection if choices given
 *	@choices: array of choices for an interactive alert
 *
 *	This routine is called by the engine when it or a plugin needs to display an
 *	alert to a user or when the user needs to be asked for a response from a set
 *	of choices. During engine open, we are asked to queue messages. This allows
 *	the user to receive one dialog to allow them to see the informational messages
 *	that were queued once the main window is up.
 */
int show_user_message(char *message, int *choice_selection, char **choices)
{
	if (choices == NULL) {
		if (queue_message == TRUE) {
			queue_informational_message(message);
		} else {
			show_informational_message(message_number++, time(NULL), message);
		}
	} else {
		show_alert_prompt(message, choice_selection, choices);
	}
	return 0;
}

/**
 *	get_focus_item_data - return the data held by the current focus item
 *	@selwin: the selection window containing a clist
 *
 *	This routine extracts and returns the user data pointer associated
 *	with the current focus item.
 */
gpointer get_focus_item_data(struct selwin *selwin)
{
	GList *element;
	struct clist_item *focus_item;
	
	element = get_clist_focus_item(selwin->clist);
	focus_item = element->data;
	return focus_item->user_data;
}

/**
 *	disable_message_dialog_buttons - disable view and delete buttons from view messages dialog
 *	@dialog: the view messages dialog
 *
 *	This routine makes the "View", "Delete" and "Delete All" buttons on the view
 *	messages dialog insensitive. This is typically called when the message clist
 *	is empty.
 */
void disable_message_dialog_buttons(struct dialog_window *dialog)
{
	struct menu_item *view_button = dialog->user_data;
	
	set_menu_item_sensitivity(view_button, FALSE);
	set_menu_item_sensitivity(dialog->prev_button, FALSE);
	set_menu_item_sensitivity(dialog->next_button, FALSE);	
}

/**
 *	view_message_button_activated - view a single message
 *	@item: the menu item that was activated
 *
 *	This routine is invoked when the View button is activated. It displays
 *	a queued message in popup dialog window.
 */
int view_message_button_activated(struct menu_item *item)
{
	struct queued_message *queued_message;
	struct selwin *selwin = item->user_data;
	
	queued_message = get_focus_item_data(selwin);
	show_informational_message(queued_message->number,
				queued_message->timestamp,
				queued_message->message);
	
	return 0;
}

/**
 *	delete_message_button_activated - delete a single message
 *	@item: the menu item that was activated
 *
 *	This routine is invoked when the Delete button is activated. It
 *	causes the currently focused item/message to be deleted.
 */
int delete_message_button_activated(struct menu_item *item)
{
	struct queued_message *queued_message;
	struct selwin *selwin = item->user_data;
	
	queued_message = get_focus_item_data(selwin);
	delete_queued_message(queued_message, NULL);
	message_queue = g_list_remove(message_queue, queued_message);
	delete_clist_item(selwin->clist, (struct clist_item *)selwin->clist->focus_item->data);
	
	if (message_queue == NULL)
		disable_message_dialog_buttons((struct dialog_window *)selwin);
	return 0;
}

/**
 *	delete_all_messages_button_activated - delete all queued messages
 *	@item: the menu item that was activated
 *
 *	This routine is invoked when the "Delete All" button is activated. It
 *	causes all items/messages in the list to be deleted.
 */
int delete_all_messages_button_activated(struct menu_item *item)
{
	struct selwin *selwin = item->user_data;
	
	delete_all_elements(message_queue, (GFunc)delete_queued_message, NULL);
	message_queue = NULL;
	clear_clist(selwin->clist);
	disable_message_dialog_buttons((struct dialog_window *)selwin);
	
	return 0;
}

/**
 *	format_message_item - return column strings for a row in the messages window clist
 *	@queued_message: the address of the queued message structure
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row/column text for an message placed
 *	in the queued messages window clist.
 */
void format_message_item(struct queued_message *queued_message, GPtrArray *text)
{
	g_ptr_array_add(text, g_strdup_printf("%d", queued_message->number));
	g_ptr_array_add(text, g_strdup_printf("%s", g_strchomp(ctime(&queued_message->timestamp))));
	g_ptr_array_add(text, g_strdup(queued_message->message));
}

/**
 *	populate_message_clist - populate the list of messages
 *	@clist: the messages clist
 *
 *	This routine populates the message clist with informational messages
 *	in the message queue.
 */
void populate_message_clist(struct clist *clist)
{
	GList *element = message_queue;
		
	while (element != NULL) {
		GPtrArray *text;
		struct queued_message *queued_message;

		queued_message = element->data;
		text = g_ptr_array_new();
		format_message_item(queued_message, text);
		append_item(clist, text, element->data, NULL);
		g_ptr_array_free(text, TRUE);

		element = g_list_next(element);
	}
}

/**
 *	show_queued_messages_dialog - creates and display dialog to view and delete queued messages
 *	@void
 *
 *	This routine allows a user to view queued informational messages. Individual or
 *	all messages can also be deleted from this dialog.
 */
void show_queued_messages_dialog(void)
{
	struct selwin *selwin;
	struct menu_item *view_item;
	struct dialog_window *dialog;
	
	selwin = create_selection_window(
				_("Alert/Informational Messages from Plug-ins"),
				NULL,
				_(" "),
				_("_Delete"),
				(menuitem_activate_cb)delete_message_button_activated,
				_("Delete _All"),
				(menuitem_activate_cb)delete_all_messages_button_activated,
				NULL);

	dialog = (struct dialog_window *)selwin;

	view_item = pack_menu_item_at_end(dialog->menu, _("[View]"),
				get_accelerator_char(_("[_View]")),
				(menuitem_activate_cb)view_message_button_activated,
				dialog);

	set_menu_item_sensitivity(dialog->next_button, TRUE);
	set_horizontal_menu_focus(dialog->menu, view_item);
	set_menu_item_activate_cb(dialog->cancel_button,
				(menuitem_activate_cb)close_window_button_activated);
	dialog->user_data = view_item;

	set_clist_select_item_cb(selwin->clist, (clist_select_item_cb)disallow_select_events);
	set_clist_unselect_item_cb(selwin->clist, (clist_unselect_item_cb)disallow_select_events);
	set_clist_column_count(selwin->clist, 3);
	set_clist_column_info(selwin->clist, 0, calc_clist_column_width(selwin->clist, 0.10),
				0,
				CLIST_TEXT_JUSTIFY_LEFT);
	set_clist_column_info(selwin->clist, 1, calc_clist_column_width(selwin->clist, 0.30),
				get_clist_column_end(selwin->clist, 0),
				CLIST_TEXT_JUSTIFY_RIGHT);
	set_clist_column_info(selwin->clist, 2, calc_clist_column_width(selwin->clist, 0.60),
				get_clist_column_end(selwin->clist, 1),
				CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(selwin->clist, 0, _("Number"));
	print_clist_column_title(selwin->clist, 1, _("Timestamp"));
	print_clist_column_title(selwin->clist, 2, _("Message"));

	if (message_queue != NULL)
		populate_message_clist(selwin->clist);
	else
		disable_message_dialog_buttons(dialog);
	process_modal_dialog(dialog);		
}

/**
 *	display_queued_messages - allows the user to view and delete messages we previously queued
 *	@void
 *
 *	This routine allows the user to view information alert messages we received
 *	and queued during some engine operation. It also allows the user to delete
 *	messages.
 **/
void display_queued_messages(void)
{
	if (new_messages == TRUE)
		show_queued_messages_dialog();
}

/**
 *	on_view_messages_menuitem_activated - view queued messages
 *	@item: the menu item that was activated
 *
 *	This routine is invoked when the "Actions->View->Messages" menu item
 *	is invoked.
 */
int on_view_messages_menuitem_activated(struct menu_item *item)
{
	if (message_queue != NULL)
		show_queued_messages_dialog();
	else
		show_message_dialog(_("View Messages"),
				_("There are no informational messages available."));
	return 0;
}
