//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: SolverExpression.hpp,v 1.2 2003/05/04 18:09:01 delpinux Exp $

#ifndef _SOLVER_EXPRESSION_HPP_
#define _SOLVER_EXPRESSION_HPP_

#include <Expression.hpp>

#include <UnknownListExpression.hpp>

#include <DomainExpression.hpp>
#include <MeshExpression.hpp>

#include <SolverOptionsExpression.hpp>
#include <ProblemExpression.hpp>

#include <Variable.hpp>

/*!
  \class SolverExpression

  This class defines the base class of Solver expressions.

  \author Stephane Del Pino
 */
class Solver;
class SolverExpression
  : public Expression
{
protected:
  ReferenceCounting<Solver> __solver;

public:
  enum SolverType {
    solve,
    varsolve,
    variable,
    undefined
  };

private:
  SolverExpression::SolverType __solverType;

public:
  ReferenceCounting<Solver> solver();

  const SolverExpression::SolverType& solverType() const
  {
    return __solverType;
  }

  SolverExpression(const SolverExpression& e);

  SolverExpression(const SolverExpression::SolverType& t);

  virtual ~SolverExpression();
};

class SolverExpressionSolveFEM
  : public SolverExpression
{
private:
  ReferenceCounting<UnknownListExpression> __unknownList;

  ReferenceCounting<MeshExpression> __mesh;

  ReferenceCounting<SolverOptionsExpression> __solverOptions;
  ReferenceCounting<ProblemExpression> __problemExpression;

  std::ostream& put(std::ostream& os) const;

public:
  void execute();

  SolverExpressionSolveFEM(ReferenceCounting<UnknownListExpression> unknownList,
			   ReferenceCounting<MeshExpression> mesh,
			   ReferenceCounting<SolverOptionsExpression> solverOptions,
			   ReferenceCounting<ProblemExpression> problemExpression);

  SolverExpressionSolveFEM(const SolverExpressionSolveFEM& m);

  ~SolverExpressionSolveFEM();
};



class SolverExpressionSolveFDM
  : public SolverExpression
{
private:
  void __setScene(ReferenceCounting<DomainExpression> d) const;

  ReferenceCounting<UnknownListExpression> __unknownList;

  ReferenceCounting<DomainExpression> __domain;
  ReferenceCounting<MeshExpression> __mesh;

  ReferenceCounting<SolverOptionsExpression> __solverOptions;
  ReferenceCounting<ProblemExpression> __problemExpression;

  std::ostream& put(std::ostream& os) const;

public:
  void execute();

  SolverExpressionSolveFDM(ReferenceCounting<UnknownListExpression> unknownList,
			   ReferenceCounting<DomainExpression> domain,
			   ReferenceCounting<MeshExpression> mesh,
			   ReferenceCounting<SolverOptionsExpression> solverOptions,
			   ReferenceCounting<ProblemExpression> problemExpression);

  SolverExpressionSolveFDM(const SolverExpressionSolveFDM& m);

  ~SolverExpressionSolveFDM();
};

class SolverExpressionVariable
  : public SolverExpression
{
private:
  ReferenceCounting<SolverVariable> __solverVariable;

  std::ostream& put(std::ostream& os) const
  {
    os << (*__solverVariable).name() << ": " << (*(*__solverVariable).expression());
    return os;
  }

public:
  void execute()
  {
    ;
  }

  SolverExpressionVariable(ReferenceCounting<SolverVariable> r);

  SolverExpressionVariable(const SolverExpressionVariable& e);

  ~SolverExpressionVariable();
};

class SolverExpressionUndefined
  : public SolverExpression
{
private:
  std::ostream& put(std::ostream& os) const
  {
    os << "undefined solver";
    return os;
  }

public:
  void execute()
  {
    ;
  }

  SolverExpressionUndefined()
    : SolverExpression(SolverExpression::undefined)
  {
    ;
  }

  SolverExpressionUndefined(const SolverExpressionUndefined& m)
    : SolverExpression(m)
  {
    ;
  }

  ~SolverExpressionUndefined()
  {
    ;
  }
};

#endif // _SOLVER_EXPRESSION_HPP_

