//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: BoundaryConditionDiscretizationElimination.hpp,v 1.7 2004/03/27 16:16:07 delpinux Exp $

#ifndef BOUNDARY_CONDITION_DISCRETIZATION_ELIMINATION_HPP
#define BOUNDARY_CONDITION_DISCRETIZATION_ELIMINATION_HPP

#include <BoundaryConditionFDMDiscretization.hpp>
#include <UnAssembledMatrix.hpp>

template<typename MeshType>
class BoundaryConditionDiscretizationElimination
  : public BoundaryConditionFDMDiscretization<MeshType>
{
private:
  class __SetSecondMemberDirichlet
  {
  private:
    const Dirichlet& __dirichlet;

    const size_t __equationNumber;

    const BoundaryConditionDiscretizationElimination<MeshType>& __bc;

  public:

    template <typename SurfaceMeshType>
    void eval(const SurfaceMeshType& surfMesh) const
    {
      typedef FiniteElementTraits<typename SurfaceMeshType::ElementGeometry>
	FEType;

      typedef typename FEType::Transformation ConformTransformation;

      for (typename SurfaceMeshType::const_iterator icell(surfMesh);
	   not(icell.end()); ++icell) {
	const typename SurfaceMeshType::ElementGeometry& cell = *icell;

	const ConformTransformation CT(cell);

	for (size_t k=0; k<CT.numberOfQuadratureVertices(); k++) {
	  TinyVector<3> q = CT.quadratureVertex(k);

	  // Index of the vertex of the mesh the closer to q.
#warning Use of Index is not necessary
	  const Index& iv = __bc.mesh().vertexIndex(q);
	  const Vertex& V = __bc.mesh().vertex(iv);

	  const real_t GValue = __dirichlet.g(V);

	  const size_t I = __bc.__degreeOfFreedomSet(__equationNumber,
						     __bc.mesh().vertexNumber(V));
	  __bc.__dirichletValues[I] = GValue;
	  __bc.__dirichletList[I] = true;
	}
      }
    }

    __SetSecondMemberDirichlet(const Dirichlet &D ,
			       const size_t equationNumber,
			       const BoundaryConditionDiscretizationElimination<MeshType>& bc)
      : __dirichlet(D),
	__equationNumber(equationNumber),
	__bc(bc)
    {
      ;
    }

    __SetSecondMemberDirichlet(const __SetSecondMemberDirichlet& s)
      : __dirichlet(s.__dirichlet),
	__equationNumber(s.__equationNumber),
	__bc(s.__bc)
    {
      ;
    }

    ~__SetSecondMemberDirichlet()
    {
      ;
    }
  };


  template <typename MatrixType,
	    typename VectorType>
  static void
  __DirichletBorderLinearOperator(const BoundaryConditionDiscretizationElimination<MeshType>& bc,
				  MatrixType& A,
				  VectorType& b)
  {
    bc.__dirichletValues.resize(bc.__degreeOfFreedomSet.size());
    bc.__dirichletValues = 0;
    const BoundaryConditionSurfaceMeshAssociation& bcMeshAssociation = *bc.__bcMeshAssociation;

    // Dirichlet on the mesh border ...
    for (size_t i=0; i<bc.problem().numberOfUnknown(); ++i) {
      for (BoundaryConditionSurfaceMeshAssociation
	     ::DirichletMeshAssociation
	     ::const_iterator ibcMesh = bcMeshAssociation.bc(i).begin();
	   ibcMesh != bcMeshAssociation.bc(i).end();
	   ++ibcMesh) {
	const Dirichlet& D = *(*ibcMesh).first;
	const SurfaceMesh& surfMesh = *(*ibcMesh).second;

	BoundaryConditionDiscretizationElimination<MeshType>
	  ::__meshWrapper(surfMesh,
			  typename BoundaryConditionDiscretizationElimination<MeshType>
			  ::__SetSecondMemberDirichlet(D, i, bc));
      }
    }
    if (A.type() != BaseMatrix::unAssembled) {
      //! proceed to clean elimination
      Vector<real_t> y (bc.__dirichletValues.size());
      A.timesX(bc.__dirichletValues,y);
      b -= y;
    }

    for (size_t i=0; i<b.size(); ++i) {
      if (bc.__dirichletList[i]) {
	b[i] = bc.__dirichletValues[i];
      }
    }
  }

public:
  void getDiagonal (BaseVector& Z) const
  {
    Vector<real_t>& z = dynamic_cast<Vector<real_t>&>(Z);

    // Natural Boundary conditions
    BoundaryConditionFDMDiscretization<MeshType>
      ::__StandardGetDiagonalVariationalBorderBilinearOperator(*this, z);
    // Dirichlet on the mesh border ...
    BoundaryConditionFDMDiscretization<MeshType>
      ::__StandardGetDiagonalDirichletBorderBilinearOperator(*this, z);
  }

  void setMatrix (ReferenceCounting<BaseMatrix> givenA,
		  ReferenceCounting<BaseVector> b) const
  {
    switch((*givenA).type()) {
    case BaseMatrix::doubleHashedMatrix: {

      DoubleHashedMatrix& A = dynamic_cast<DoubleHashedMatrix&>(*givenA);

      // Variational Problem's Natural BC
      BoundaryConditionFDMDiscretization<MeshType>
	::__StandardVariationalBorderBilinearOperator(*this, A);

      // Dirichlet on the mesh border ...
      BoundaryConditionFDMDiscretization<MeshType>
	::__StandardDirichletBorderBilinearOperator(*this, A);
      break;
    }

    case BaseMatrix::unAssembled: {
      UnAssembledMatrix& A = dynamic_cast<UnAssembledMatrix&>(*givenA);
      A.setBoundaryConditions(this);
      break;
    }

    default: {
      fferr(0) << __FILE__ << ':' << __LINE__ << ":Not Implemented\n";
      std::exit(1);
    }
    }
  }

  void setSecondMember (ReferenceCounting<BaseMatrix> givenA,
			ReferenceCounting<BaseVector> givenB) const
  {
    Vector<real_t>& b = dynamic_cast<Vector<real_t>&>(*givenB);
    BaseMatrix& A = *givenA;

    //! Natural boundary conditions
    BoundaryConditionFDMDiscretization<MeshType>::
      __StandardVariationalBorderLinearOperator(*this, b);

    //! Dirichlet Boundary conditions
    BoundaryConditionDiscretizationElimination<MeshType>::
      __DirichletBorderLinearOperator(*this, A, b);
  }

  void timesX(const BaseVector& X, BaseVector& Z) const
  {
    const Vector<real_t>& x = dynamic_cast<const Vector<real_t>&>(X);
    Vector<real_t>& z = dynamic_cast<Vector<real_t>&>(Z);

    // Natural Boundary Conditions.
    BoundaryConditionFDMDiscretization<MeshType>::
      __StandardVariationalBorderBilinearOperatorTimesX(*this,
							x, z);

    // Dirichlet on the mesh border ...
    BoundaryConditionFDMDiscretization<MeshType>::
      __StandardDirichletBorderBilinearOperatorTimesX(*this,
						      x, z);
  }

  BoundaryConditionDiscretizationElimination(const Problem& problem,
					     MeshType& mesh,
					     const DegreeOfFreedomSet& dof)
    : BoundaryConditionFDMDiscretization<MeshType>(problem, mesh, dof)
  {
    ;
  }

  ~BoundaryConditionDiscretizationElimination()
  {
    ;
  }
};

#endif // BOUNDARY_CONDITION_DISCRETIZATION_ELIMINATION_HPP

