//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: ConformTransformation.cpp,v 1.8 2004/03/15 21:01:20 delpinux Exp $

#include <Domain.hpp>

#include <ConformTransformation.hpp>
#include <UserFunction.hpp>
#include <QuadratureFormula.hpp>

real_t
ConformTransformationQ1Hexahedron::integrate(const UserFunction& f) const
{
  fferr(2) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
  std::exit(1);
  return 0;
}

//! Computes Xhat, the point which transformed is (x,y,z)
bool
ConformTransformationQ1Hexahedron::invertT(const real_t& x,
					   const real_t& y,
					   const real_t& z,
					   TinyVector<3,real_t>& Xhat) const
{
  // initialization
  for(size_t i=0; i<3; ++i)
    Xhat[i] = 0.5;

  TinyVector<3,real_t> X;
  TinyVector<3,real_t> f;
  TinyVector<3> delta;
  const size_t maxiter = 100;
  size_t niter = 0;

  do {
    niter++;

    //! Computing F(Xhat) - (x,y,z).
    value(Xhat,f);
    fferr(0) << "\n\nIteration N:" << niter << '\n';
    fferr(0) << "Seeking (" << x << ',' << y << ',' << z << ")\n";
    fferr(0) << "F(Xhat) = "  << f << '\n';
    f[0] -= x;
    f[1] -= y;
    f[2] -= z;

    // Evaluation of the Jacobian 
    TinyMatrix<3,3,real_t> J;
    dx(Xhat,X);
    for (size_t i=0; i<3; ++i)
      J(0,i) = X[i];

    dy(Xhat,X);
    for (size_t i=0; i<3; ++i)
      J(1,i) = X[i];

    dz(Xhat,X);
    for (size_t i=0; i<3; ++i)
      J(2,i) = X[i];

    fferr(0) << "f = " << f << '\n';
    fferr(0) << "Xhat = " << Xhat << '\n';
    fferr(0) << "Jacobian:\n";
    fferr(0) << J << '\n';

    delta = f/J;
    fferr(0) << "delta= " << delta << '\n';

    Xhat -= delta;
    fferr(0) << "\t=> new Xhat = " << Xhat << '\n';
    
    if ((niter>maxiter)or(Norm(f)>10)) {
      value(Xhat,X);
      fferr(0) << "false: " <<  Xhat << " -> " << X << '\n';
      return false;
    }
  } while(Norm(f)>1E-2);
  value(Xhat,X);
  fferr(0) << "#### FOUND " << Xhat << " -> " << X << '\n';
  return true;
}

size_t
ConformTransformationQ1Hexahedron::numberOfQuadratureVertices() const
{
  return QuadratureFormulaQ1Hexahedron::instance().numberOfVertices();
}

real_t
ConformTransformationQ1Hexahedron::quadratureWeight(const size_t& i) const
{
  return __H.volume()*QuadratureFormulaQ1Hexahedron::instance().weight(i);
}

TinyVector<3>
ConformTransformationQ1Hexahedron::quadratureVertex(const size_t& i) const
{
  TinyVector<3> X;
  
  const TinyVector<8, TinyVector<3> >& integrationVertices
    = QuadratureFormulaQ1Hexahedron::instance().vertices();

  const TinyVector<3>& qVertex = integrationVertices[i];

  ConformTransformationQ1Hexahedron::value(qVertex,X);

  return X;
}


real_t
ConformTransformationP1Tetrahedron::integrate(const UserFunction& f) const
{
  fferr(2) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
  std::exit(1);
  return 0;
}

size_t
ConformTransformationP1Tetrahedron::numberOfQuadratureVertices() const
{
  return QuadratureFormulaP1Tetrahedron::instance().numberOfVertices();
}

real_t
ConformTransformationP1Tetrahedron::quadratureWeight(const size_t& i) const
{
  return __T.volume()*QuadratureFormulaP1Tetrahedron::instance().weight(i);
}

TinyVector<3>
ConformTransformationP1Tetrahedron::quadratureVertex(const size_t& i) const
{
  TinyVector<3> X;
  
  const TinyVector<4, TinyVector<3> >& integrationVertices
    = QuadratureFormulaP1Tetrahedron::instance().vertices();

  const TinyVector<3>& qVertex = integrationVertices[i];

  ConformTransformationP1Tetrahedron::value(qVertex,X);

  return X;
}

size_t
ConformTransformationQ1CartesianHexahedron::numberOfQuadratureVertices() const
{
  return QuadratureFormulaQ1Hexahedron::instance().numberOfVertices();
}


real_t
ConformTransformationQ1CartesianHexahedron::
quadratureWeight(const size_t& i) const
{
  return __CH.volume()*QuadratureFormulaQ1Hexahedron::instance().weight(i);
}

TinyVector<3>
ConformTransformationQ1CartesianHexahedron::
quadratureVertex(const size_t& i) const
{
  TinyVector<3> X;
  
  const TinyVector<8, TinyVector<3> >& IntegrationVertices
    = QuadratureFormulaQ1Hexahedron::instance().vertices();

  const TinyVector<3>& qVertex = IntegrationVertices[i];

  ConformTransformationQ1CartesianHexahedron::value(qVertex,X);
  return X;
}

real_t
ConformTransformationQ1CartesianHexahedron::
integrateCharacteristic(const Domain& d) const
{
  // Here we use order 4 Lobatto quadrature

  TinyVector<4, real_t>  x;
  x[0] = 0.;
  x[1] = .27639320225002103036; //(1-sqrt(5)/5)/2.;
  x[2] = .72360679774997896964; //(1+sqrt(5)/5)/2.;
  x[3] = 1.;

  TinyVector<4, real_t>  w;
  w[0] = 1./12.;
  w[1] = 5./12.;
  w[2] = 5./12.;
  w[3] = 1./12.;

  real_t sum = 0;
  TinyVector<3, real_t> X_hat;
  TinyVector<3, real_t> X;
  for (unsigned i=0; i<4; ++i) {
    X_hat[0] = x[i];
    for (unsigned j=0; j<4; ++j) {
      X_hat[1] = x[j];
      for (unsigned k=0; k<4; ++k) {
	X_hat[2] = x[k];
	this->value(X_hat, X);
	sum += w[i]*w[j]*w[k] * (d.inside(X) ? 1 : 0);
      }
    }
  }
  return sum;
}


real_t
ConformTransformationP1Triangle::integrate(const UserFunction& f) const
{
  fferr(2) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
  std::exit(1);
  return 0.;
}

size_t
ConformTransformationP1Triangle::numberOfQuadratureVertices() const
{
  return QuadratureFormulaP1Triangle3D::instance().numberOfVertices();
}

real_t
ConformTransformationP1Triangle::quadratureWeight(const size_t& i) const
{
  return __T.volume()*QuadratureFormulaP1Triangle3D::instance().weight(i);
}

TinyVector<3>
ConformTransformationP1Triangle::quadratureVertex(const size_t& i) const
{
  TinyVector<3> X;
  
  const TinyVector<3, TinyVector<3> >& IntegrationVertices
    = QuadratureFormulaP1Triangle3D::instance().vertices();

  const TinyVector<3>& qVertex = IntegrationVertices[i];

  ConformTransformationP1Triangle::value(qVertex,X);
  return X;
}



real_t
ConformTransformationQ1Quadrangle::integrate(const UserFunction& f) const
{
  fferr(2) << __FILE__ << ':' << __LINE__ << ": Not implemented\n";
  std::exit(1);
  return 0.;
}

size_t
ConformTransformationQ1Quadrangle::numberOfQuadratureVertices() const
{
  return QuadratureFormulaQ1Quadrangle3D::instance().numberOfVertices();
}

real_t
ConformTransformationQ1Quadrangle::quadratureWeight(const size_t& i) const
{
  return __Q.volume()*QuadratureFormulaQ1Quadrangle3D::instance().weight(i);
}

TinyVector<3>
ConformTransformationQ1Quadrangle::quadratureVertex(const size_t& i) const
{
  TinyVector<3> X;
  
  const TinyVector<4, TinyVector<3> >& IntegrationVertices
    = QuadratureFormulaQ1Quadrangle3D::instance().vertices();

  const TinyVector<3>& qVertex = IntegrationVertices[i];

  ConformTransformationQ1Quadrangle::value(qVertex,X);
  return X;
}

