/* 

                          Firewall Builder

                 Copyright (C) 2002 NetCitadel, LLC

  Author:  Vadim Kurland     vadim@vk.crocodile.org

  $Id: ipt.cpp,v 1.31 2004/09/28 05:21:01 vkurland Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "config.h"

#include <qsettings.h>

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#include <fstream>
#include <iostream>
#include <algorithm>
#include <functional>

#ifndef _WIN32
#  include <unistd.h>
#else
#  include <direct.h>
#  include <stdlib.h>
#  include <io.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>
#include <assert.h>

#include "PolicyCompiler_ipt.h"
#include "NATCompiler_ipt.h"
#include "OSConfigurator_linux24.h"

#include "fwbuilder/Resources.h"
#include "fwbuilder/FWObjectDatabase.h"
#include "fwbuilder/XMLTools.h"
#include "fwbuilder/FWException.h"
#include "fwbuilder/Firewall.h"
#include "fwbuilder/Interface.h"

#ifdef HAVE_GETOPT_H
  #include <getopt.h>
#else
  #ifdef _WIN32
    #include <getopt.h>
  #else
    #include <stdlib.h>
  #endif
#endif

#include "../common/init.cpp"

using namespace std;
using namespace libfwbuilder;
using namespace fwcompiler;

static const char      *filename       = NULL;
static const char      *wdir           = NULL;
static const char      *fwobjectname   = NULL;
static string           fw_file_name   = "";
static int              dl             = 0;
static int              drp            = -1;
static int              drn            = -1;
static int              verbose        = 0;

class UpgradePredicate: public XMLTools::UpgradePredicate
{
    public:
    virtual bool operator()(const string &msg) const 
    { 
	cout << _("Data file has been created in the old version of Firewall Builder. Use fwbuilder GUI to convert it.") << endl;
	return false;
    }
};

    
void usage(const char *name)
{
    cout << _("Firewall Builder:  policy compiler for Linux 2.4.xx iptables") << endl;
    cout << _("Version ") << VERSION << "-" << RELEASE_NUM << endl;
    cout << _("Usage: ") << name << _(" [-x level] [-v] [-V] [-f filename.xml] [-d destdir] [-m] firewall_object_name") << endl;
}

int main(int argc, char * const *argv)
{   

#ifdef ENABLE_NLS
    setlocale (LC_ALL, "");

    bindtextdomain (PACKAGE, LOCALEDIR);
    textdomain (PACKAGE);
#else
#  ifdef HAVE_SETLOCALE
    setlocale (LC_ALL, "");
#  endif
#endif
    

    if (argc<=1)
    {
        usage(argv[0]);
        exit(1);
    }

    int   opt;

    while( (opt=getopt(argc,argv,"x:vVf:d:r:o:")) != EOF )
    {
        switch(opt)
        {
        case 'd':
            wdir = strdup(optarg);
            break;
        case 'r':
            respath = string(optarg);
            break;
        case 'f':
            filename = strdup(optarg);
            break;
        case 'o':
            fw_file_name = string(optarg);
            break;
        case 'x':
            if (*optarg=='p') {
                ++optarg;
                drp  = atoi(optarg);
            } else {
                if (*optarg=='n') {
                    ++optarg;
                    drn  = atoi(optarg);
                } else {
                    if (isdigit(*optarg))  dl=atoi(optarg);  // increase debug level
                    else {
                        usage(argv[0]);
                        exit(1);
                    }
                }
            }
            break;
        case 'v':
            verbose++;
            break;
        case 'V':
            usage(argv[0]);
            exit(1);
        }
    }
    
    if((argc-1) != optind)
    {
        usage(argv[0]);
        exit(1);
    }

    fwobjectname = strdup( argv[optind++] );

    if (fw_file_name.empty())
        fw_file_name=string(fwobjectname)+".fw";

    if (wdir==0) 	wdir="./";

    if (
#ifdef _WIN32
    _chdir(wdir)
#else
    chdir(wdir)
#endif
    ) {
	cerr << _("Can't change to: ") << wdir << endl;
	exit(1);
    }

    init(argv);

    try
    {
        new Resources(respath+FS_SEPARATOR+"resources.xml");

	/* create database */
	new FWObjectDatabase();

	/* load the data file */
	UpgradePredicate upgrade_predicate; 

	if (verbose) cout << _(" *** Loading data ...");

        FWObjectDatabase::db->setReadOnly( false );
        FWObjectDatabase::db->load( sysfname, &upgrade_predicate, librespath);
        FWObjectDatabase::db->setFileName("");
        FWObjectDatabase *ndb = new FWObjectDatabase();
        ndb->load(filename, &upgrade_predicate,  librespath);
        FWObjectDatabase::db->merge(ndb, NULL);
        delete ndb;
        FWObjectDatabase::db->setFileName(filename);

//	FWObjectDatabase::db->load(filename,  &upgrade_predicate, librespath);
	if (verbose) cout << _(" done\n");

        FWObject *slib = FWObjectDatabase::db->getById("syslib000");
        if ( slib->isReadOnly()) slib->setReadOnly(false);

	/* Review firewall and OS options and generate commands */
	Firewall*  fw=FWObjectDatabase::db->findFirewallByName(fwobjectname);
	FWOptions* options=fw->getOptionsObject();
	string s;

        /* some initial sanity checks */

        list<FWObject*> l2=fw->getByType(Interface::TYPENAME);
        for (list<FWObject*>::iterator i=l2.begin(); i!=l2.end(); ++i) 
        {
            Interface *iface=dynamic_cast<Interface*>(*i);
            assert(iface);

            string::size_type n;
            if ( (n=iface->getName().find("*"))!=string::npos) 
            {
/* this is a special 'wildcard' interface. Its name must end with '*',
 * it must be dynamic and should not have a child IPv4 or
 * physAddress object
 */
                if (n!=iface->getName().length()-1)
                {
                    char errstr[256];
                    sprintf(errstr,
    _("'*' must be the last character in the wildcard's interface name: '%s'."),
                            iface->getName().c_str() );
                    throw FWException(errstr);
                }
/*
  removed test to implement RFE #837238: "unnummbered wildcard interfaces"

                if (!iface->isDyn())
                {
                    char errstr[256];
                    sprintf(errstr,
                            _("Wildcard interface '%s' must be dynamic."),
                            iface->getName().c_str() );
                    throw FWException(errstr);
                }
*/
                list<FWObject*> l3=iface->getByType(physAddress::TYPENAME);
                if (l3.size()>0)
                {
                    char errstr[256];
                    sprintf(errstr,
_("Wildcard interface '%s' should not have a physcal address object attached to it. The physical address object will be ignored.\n"),
                            iface->getName().c_str() );
                    cerr << errstr;
                    for (list<FWObject*>::iterator j=l3.begin(); j!=l3.end(); ++j) 
                        iface->remove(*j);
                }
            }

            if ( iface->isDyn())  
            {
                iface->setBool("use_var_address",true);

                list<FWObject*> l3=iface->getByType(IPv4::TYPENAME);
                if (l3.size()>0)
                {
                    char errstr[256];
                    for (list<FWObject*>::iterator j=l3.begin(); j!=l3.end(); ++j) 
                        if ( FWObjectDatabase::db->findAllReferences(*j).size()!=0 )
                        {
                            sprintf(errstr,
_("Dynamic interface %s has an IP address that is used in the firewall policy rule.\n"),
                                    iface->getName().c_str() );
                            throw FWException(errstr);
                        }

                    sprintf(errstr,
_("Dynamic interface %s should not have an IP address object attached to it. This IP address object will be ignored.\n"),
                            iface->getName().c_str() );
                    cerr << errstr;
                    for (list<FWObject*>::iterator j=l3.begin(); j!=l3.end(); ++j) 
                        iface->remove(*j);
                }
            } else
            {

                list<FWObject*> la=iface->getByType(IPv4::TYPENAME);
                if ( !iface->isDyn() && !iface->isUnnumbered() &&
                     la.empty() )
                {
                    char errstr[256];
                    sprintf(errstr,_("Missing IP address for interface %s\n"),
                            iface->getName().c_str() );
                    throw FWException(errstr);
                }

                for (list<FWObject*>::iterator j=la.begin(); j!=la.end(); ++j) 
                {
                    IPv4 *ipv4 = IPv4::cast(*j);
                    if ( ipv4->getAddress().toString()=="0.0.0.0")
                    {
                        char errstr[256];
                        sprintf(errstr,
                      _("Interface %s has IP address \"0.0.0.0\".\n"),
                                iface->getName().c_str() );
                        throw FWException(errstr);
                    }
                }
            }
        }

	string firewall_dir=options->getStr("firewall_dir");
	if (firewall_dir=="") firewall_dir="/etc";

	bool debug=options->getBool("debug");
	string shell_dbg=(debug)?"set -x":"" ;
	string pfctl_dbg=(debug)?"-v":"";

	OSConfigurator_linux24 *oscnf=NULL;
	if (fw->getStr("host_OS")=="linux24" || fw->getStr("host_OS")=="linksys")
            oscnf=new OSConfigurator_linux24(FWObjectDatabase::db , fwobjectname);
	if (oscnf==NULL)
	    throw FWException(_("Unrecognized host OS ")+fw->getStr("host_OS"));

        oscnf->prolog();

	PolicyCompiler_ipt c( FWObjectDatabase::db , fwobjectname , oscnf );

	c.setDebugLevel( dl );
	c.setDebugRule(  drp );
	c.setVerbose( (bool)(verbose) );

	if ( c.prolog() > 0 )
        {
	    c.compile();
	    c.epilog();
	}

	NATCompiler_ipt n( FWObjectDatabase::db , fwobjectname , oscnf );

	n.setDebugLevel( dl );
	n.setDebugRule(  drn );
	n.setVerbose( (bool)(verbose) );

	if ( n.prolog() > 0 ) {
            oscnf->generateCodeForProtocolHandlers(true);
	    n.compile();
	    n.epilog();
	} else
            oscnf->generateCodeForProtocolHandlers(false);

/*
 * now write generated scripts to files
 */


        char          *timestr;
        time_t         tm;
        struct tm     *stm;

        tm=time(NULL);
        stm=localtime(&tm);
        timestr=strdup(ctime(&tm));
        timestr[ strlen(timestr)-1 ]='\0';

#ifdef _WIN32
        char* user_name=getenv("USERNAME");
#else
        char* user_name=getenv("USER");
#endif

        if (user_name==NULL) {
            cerr << _("Can't figure out your user name, aborting") << endl;
            exit(1);
        }

/*
 * assemble the script and then perhaps post-process it if it should
 * run on Linksys device with sveasoft firmware
 */

        ostringstream script;

	script << "#!/bin/sh "  << endl;

        script << _("#\n\
#  This is automatically generated file. DO NOT MODIFY !\n\
#\n\
#  Firewall Builder  fwb_ipt v") << VERSION << "-" << RELEASE_NUM << _(" \n\
#\n\
#  Generated ") << timestr << " " << tzname[stm->tm_isdst] << _(" by ") 
               << user_name << "\n#\n";

        script << MANIFEST_MARKER << "* " << fw_file_name << endl;
        script << "#" << endl;
        script << "#" << endl;

/* do not put comment in the script if it is intended for linksys */
        if (fw->getStr("host_OS")!="linksys")
        {
            string fwcomment=fw->getComment();
            string::size_type n1,n2;
            n1=n2=0;
            while ( (n2=fwcomment.find("\n",n1))!=string::npos )
            {
                script << "#  " << fwcomment.substr(n1,n2-n1) << endl;
                n1=n2+1;
            }
            script << "#  " << fwcomment.substr(n1) << endl;
            script << "#\n#\n#\n";
        }

        script << shell_dbg << endl;
        script << endl;

        script << "PATH=\"/usr/sbin:/sbin:${PATH}\"" << endl;
        script << "export PATH" << endl;
        script << endl;

	script << oscnf->getCompiledScript();

        script << endl;

//        script << "cd " << firewall_dir << " || exit 1" << endl << endl;
        script << "log \"";
        script << _("Activating firewall script generated ")
               << timestr << " " << tzname[stm->tm_isdst] << _(" by ")
               << user_name;
        script << "\"" << endl;

	script << endl;

        if ( options->getBool("accept_established") ) 
        {
            script << "$IPTABLES -A INPUT   -m state --state ESTABLISHED,RELATED -j ACCEPT"
                   << endl;
            script << "$IPTABLES -A OUTPUT  -m state --state ESTABLISHED,RELATED -j ACCEPT"
                   << endl;
            script << "$IPTABLES -A FORWARD -m state --state ESTABLISHED,RELATED -j ACCEPT"
                   << endl;
            script << endl;
        }


	script << n.getCompiledScript();
	script << c.getCompiledScript();

        oscnf->epilog();
	script << oscnf->getCompiledScript();

	script << endl;

        string sbuf = script.str();

/* starting with 2.0.3 we copy script to linksys using scp and do not
 * need to escape double quotes and '$' anymore
 */

#if 0
        if ( Resources::getTargetOptionBool(fw->getStr("host_OS"),
                                            "escape_everything") )
        {
/* need to escape single and double quotes, as well as '$' in the script */

            string::size_type i;

            i = 0;
            while ( (i=sbuf.find('\"',i))!=string::npos )
            {
                sbuf.replace(i,1,"\\\"");
                i+=2;
            }

            i = 0;
            while ( (i=sbuf.find('\'',i))!=string::npos )
            {
                sbuf.replace(i,1,"\\\'");
                i+=2;
            }

            i = 0;
            while ( (i=sbuf.find('`',i))!=string::npos )
            {
                sbuf.replace(i,1,"\\`");
                i+=2;
            }

            i = 0;
            while ( (i=sbuf.find('$',i))!=string::npos )
            {
                sbuf.replace(i,1,"\\$");
                i+=2;
            }
        }
#endif

#ifdef _WIN32
	ofstream fw_file(fw_file_name.c_str(), ios::out|ios::binary);
#else
	ofstream fw_file(fw_file_name.c_str());
#endif

        fw_file << sbuf << endl;
	fw_file.close();

#ifdef _WIN32
        _chmod(fw_file_name.c_str(),_S_IREAD|_S_IWRITE);
#else
        chmod(fw_file_name.c_str(),S_IXUSR|S_IRUSR|S_IWUSR|S_IRGRP);
#endif
        
        return 0;

    } catch(FWException &ex)  {
	cerr << ex.toString() << endl;
        return 1;
    } catch (std::string s) {
	cerr << s;
        return 1;
    } catch (std::exception ex) {
	cerr << ex.what();
        return 1;
    } catch (...) {
	cerr << _("Unsupported exception");
        return 1;
    }

}









