/**
 * File:          $RCSfile: misc_error.c,v $
 * Module:        Miscellaneous error declarations
 * Part of:       Gandalf Library
 *
 * Revision:      $Revision: 1.17 $
 * Last edited:   $Date: 2002/04/22 13:42:15 $
 * Author:        $Author: pm $
 * Copyright:     (c) 2000 Imagineer Software Limited
 */

/* This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <string.h>
#include <gandalf/common/misc_error.h>
#include <gandalf/common/allocate.h>

/**
 * \addtogroup Common
 * \{
 */

/**
 * \addtogroup CommonError
 * \{
 */

#if 0

typedef struct Gan_ErrorStruct
{
   char *funcname;
   int   code;
   char *message;
   char *filename;
   int   line;
   struct Gan_ErrorStruct *next;
} Gan_ErrorStruct;

static Gan_ErrorStruct *
 gan_error_stack_add ( Gan_ErrorStruct *error_stack,
                       const char *funcname, int code, char *message,
                       const char *filename, int line )
{
   Gan_ErrorStruct *new = gan_malloc_object(Gan_ErrorStruct);

   gan_assert ( new != NULL, "malloc() failed in gan_error_stack_add()" );
   new->funcname = gan_malloc_array ( char, strlen(funcname)+1 );
   strcpy ( new->funcname, funcname );
   new->code = code;
   new->message = gan_malloc_array ( char, strlen(message)+1 );
   strcpy ( new->message, message );
   new->filename = gan_malloc_array ( char, strlen(filename)+1 );
   strcpy ( new->filename, filename );
   new->line = line;
 
   new->next = error_stack;
   return new;
}

static void
 gan_error_stack_free ( Gan_ErrorStruct *error_stack )
{
   Gan_ErrorStruct *next;

   if ( error_stack == NULL ) return;
   next = error_stack->next;
   free ( error_stack );
   gan_error_stack_free ( next );
}

static Gan_ErrorStruct *error_stack = NULL;

void
 gan_err_flush_trace(void)
{
   gan_error_stack_free ( error_stack );
   error_stack = NULL;
}

/* not a user function */
void gan_err_register_expanded ( const char *funcname, int code, char *message,
                                 const char *filename, int line )
{
#if 1
   print_error ( funcname, code, message, filename, line );
   exit(-1);
#endif
   error_stack = gan_error_stack_add ( error_stack, funcname, code, message,
                                       filename, line );
}
#endif

static void print_error ( const char *funcname, int code, const char *message,
                          const char *filename, int line )
{
   fprintf ( stderr, "Gandalf error %d in function %s(): ",
             code, funcname );
   switch ( code )
   {
      case GAN_ERROR_FAILURE:
        fprintf ( stderr, "failed:" );
        break;

      case GAN_ERROR_NOT_IMPLEMENTED:
        fprintf ( stderr, "not implemented" );
        break;

      case GAN_ERROR_REF_OUTSIDE_MATRIX:
        fprintf ( stderr, "reference outside matrix" );
        break;

      case GAN_ERROR_CANT_REALLOC:
        fprintf ( stderr, "can't reallocate" );
        break;

      case GAN_ERROR_MALLOC_FAILED:
        fprintf ( stderr, "memory allocation failed" );
        break;

      case GAN_ERROR_DIFFERENT_DIMS:
        fprintf ( stderr, "dimensions don't match" );
        break;

      case GAN_ERROR_INCOMPATIBLE:
        fprintf ( stderr, "incompatible" );
        break;

      case GAN_ERROR_MATRIX_NOT_SQUARE:
        fprintf ( stderr, "matrix not square" );
        break;

     case GAN_ERROR_INPLACE_TRANSPOSE:
        fprintf ( stderr, "illegal in-place matrix transpose" );
        break;

     case GAN_ERROR_INPLACE_MULTIPLY:
        fprintf ( stderr, "illegal in-place matrix multiply" );
        break;

      case GAN_ERROR_ARRAY_TOO_SMALL:
        fprintf ( stderr, "array too small" );
        break;

      case GAN_ERROR_ILLEGAL_TYPE:
        fprintf ( stderr, "illegal type" );
        break;

      case GAN_ERROR_NO_IMPLICIT_INV:
        fprintf ( stderr, "can't do implicit inverse" );
        break;

      case GAN_ERROR_CLAPACK_ILLEGAL_ARG:
        fprintf ( stderr, "LAPACK function illegal argument" );
        break;

      case GAN_ERROR_CBLAS_ILLEGAL_ARG:
        fprintf ( stderr, "BLAS function illegal argument" );
        break;

      case GAN_ERROR_CBLAS_FAILED:
        fprintf ( stderr, "BLAS function failed" );
        break;

      case GAN_ERROR_NOT_POSITIVE_DEFINITE:
        fprintf ( stderr, "matrix not positive definite" );
        break;

      case GAN_ERROR_DIVISION_BY_ZERO:
        fprintf ( stderr, "attempted division by zero" );
        break;

      case GAN_ERROR_SQRT_OF_NEG_NUMBER:
        fprintf ( stderr, "attempted square-root of negative number" );
        break;

      case GAN_ERROR_NO_CONVERGENCE:
        fprintf ( stderr, "no convergence" );
        break;

      case GAN_ERROR_SINGULAR_MATRIX:
        fprintf ( stderr, "singular matrix" );
        break;

      case GAN_ERROR_IMAGE_TOO_SMALL:
        fprintf ( stderr, "image too small" );
        break;

      case GAN_ERROR_NO_DATA:
        fprintf ( stderr, "no data" );
        break;

      case GAN_ERROR_NOT_ENOUGH_DATA:
        fprintf ( stderr, "not enough data" );
        break;

      case GAN_ERROR_OPENING_FILE:
        fprintf ( stderr, "couldn't open file" );
        break;

      case GAN_ERROR_CORRUPTED_FILE:
        fprintf ( stderr, "corrupted file" );
        break;

      case GAN_ERROR_OUTSIDE_RANGE:
        fprintf ( stderr, "outside range" );
        break;

      case GAN_ERROR_TOO_LARGE:
        fprintf ( stderr, "too large" );
        break;

      case GAN_ERROR_TOO_SMALL:
        fprintf ( stderr, "too small" );
        break;

      case GAN_ERROR_TRUNCATED_FILE:
        fprintf ( stderr, "truncated file" );
        break;

      case GAN_ERROR_NOT_INITIALISED:
        fprintf ( stderr, "not initialised" );
        break;

      case GAN_ERROR_ILLEGAL_ARGUMENT:
        fprintf ( stderr, "illegal argument" );
        break;

      case GAN_ERROR_CCMATH_FAILED:
        fprintf ( stderr, "CCM library function failed" );
        break;

      case GAN_ERROR_NO_SOLUTION:
        fprintf ( stderr, "no solution found" );
        break;

      case GAN_ERROR_READING_FROM_FILE:
        fprintf ( stderr, "reading from file" );
        break;

      case GAN_ERROR_WRITING_TO_FILE:
        fprintf ( stderr, "writing to file" );
        break;

      default:
        fprintf ( stderr, "unknown error" );
        break;
   }

   fprintf ( stderr, " %s\n", message );
   fprintf ( stderr, "Line %d of file %s\n", line, filename );
}

/**
 * \brief Default Gandalf error handling function.
 * \return No value.
 *
 * Call gan_err_set_reporter(gan_err_default_reporter); in your Gandalf
 * program before calling any other Gandalf functions, if you want to use this
 * error handling routine, which just prints an error message and exits.
 */
void
 gan_err_default_reporter(void)
{
   int n, i;

   n = gan_err_get_error_count();
   for (i = 1; i<=n; i++)
   {
      const char *func_name, *message, *file_name;
      int code, line;

      if ((gan_err_get_error(i, &func_name, &code, &file_name, 
                             &line, &message ) != GAN_EC_OK))
         fprintf(stderr, "**** gan_err_get_error unsuccessful\n");
        
      fprintf(stderr, "\nError number %d\n", i );
      print_error ( func_name, code, message, file_name, line );
   }

   abort();
}

/**
 * \}
 */

/**
 * \}
 */
