###############################################################################
#1
#W    stcs.gi  The Matrix Schreier Sims package 
#W             Schreier-Todd-Coxeter-Sims implementation
##
#H    File      : $RCSfile: stcs.gi,v $
#H    Author    : Henrik Brnhielm
#H    Dev start : 2004-07-05 
##
#H    Version   : $Revision: 1.12 $
#H    Date      : $Date: 2004/09/13 06:14:44 $
#H    Last edit : $Author: redstar_ $
##
#H    @(#)$Id: stcs.gi,v 1.12 2004/09/13 06:14:44 redstar_ Exp $
##
## These are the Schreier-Todd-Coxeter-Sims routines, ie Schreier-Sims 
## algorithm with additional calls to Todd-Coxeter coset enumeration to
## possibly speed up the process. It is known to be fast when the input is
## already a base and SGS, and therefore it is good for verifying a proposed
## base and SGS, for example the output of a probabilistic algorithm.
##
###############################################################################

Revision.("matrixss/lib/stcs_gi") := 
  "@(#)$Id: stcs.gi,v 1.12 2004/09/13 06:14:44 redstar_ Exp $";

###############################################################################
##
#F MATRIXSS_SchreierToddCoxeterSims(ssInfo, partialSGS, level, identity, cosetFactor)
##
## The main function for the Schreier-Todd-Coxeter-Sims algorithm. It is very
## similar to ordinary Schreier-Sims algorithm and has a similar interface.
## \beginitems    
## `ssInfo' & main information structure for the current Schreier-Sims run
##    
## `partialSGS' & given partial strong generating set
##    
## `level' & the level of the call to Schreier-Sims
##
## `identity' & the group identity
##
## `cosetFactor' & the quotient of the maximum number of cosets generated 
##                  during coset enumeration and the corresponding orbit size
## \enditems
##
###############################################################################
MATRIXSS_SchreierToddCoxeterSims := 
  function(ssInfo, partialSGS, level, identity, cosetFactor)
    local generator, point, orbit, strip, schreierGenerator, element, 
          action, recursiveLevel, schreierTree, SGS, oldSGS, points, 
          newPoint, oldSchreierTree, newBasePoint, oldOrbit,
          newInverseGenerator, newSGS, freeGroup, cosetTable, word,
          subgroupGens, relation, gens1, gens2, gens3, relations, 
          levelGroup, freeGroupHomo, gen, relHomo, ret, dropoutLevel, residue;
    
    MATRIXSS_DebugPrint(2, ["Schreier-Sims at level ", level]);
    
    action := ssInfo[level].action;
    
    # S^(i + 1) = ssInfo[i].partialSGS
    
    # Find S^(i)        
    # Find the generators from the partial SGS that fixes all points at
    # lower levels.
    if level > 1 then
        SGS := ShallowCopy(ssInfo[level - 1].partialSGS);
    else
        SGS := ShallowCopy(partialSGS);
    fi;
    MakeImmutable(SGS);
    
    # Compute free group for current level
    ssInfo[level].freeGroup := FreeGroup(Length(SGS));
    MATRIXSS_DebugPrint(4, ["Generators : ", List(SGS, i -> i[1])]);
    levelGroup := GroupWithGenerators(List(SGS, i -> i[1]), identity);
    
    # Save mapping between generators of current level group and free group
    gens1 := ShallowCopy(GeneratorsOfGroup(levelGroup));
    gens2 := ShallowCopy(GeneratorsOfGroup(ssInfo[level].freeGroup));
    SortParallel(gens1, gens2);
    ssInfo[level].genMap := Immutable(rec(Generators := gens1, 
                                    FreeGenerators := gens2));
    
    # Compute homomorphism for use in coset enumeration
    freeGroupHomo := GroupHomomorphismByImagesNC(ssInfo[level].freeGroup, 
                             levelGroup, gens2, gens1);
    
    MATRIXSS_DebugPrint(3, ["Saved SGS that fixes first ", level - 1, 
            " points ", Length(SGS)]);
    
    MATRIXSS_DebugPrint(4, ["Base point : ", ssInfo[level].partialBase]);
    MATRIXSS_DebugPrint(9, ["Hash func : ", ssInfo[level].hash]);
    
    # Compute schreier tree for current level
    # Compute \Delta_i = \beta_i^(H^i) = \beta_i^(<S_i>)
    oldSchreierTree := ssInfo[level].schreierTree;
    
    ssInfo[level].schreierTree := 
      MATRIXSS_GetSchreierTree(ssInfo[level].schreierTree,
              ssInfo[level].partialBase, SGS, ssInfo[level].oldSGS,
              action, ssInfo[level].hash, identity);
        
    orbit := Immutable(MATRIXSS_GetOrbit(ssInfo[level].schreierTree.Tree));
    
    MATRIXSS_DebugPrint(4, ["Orbit size for level ", level, " is ", 
            Length(orbit)]); 
    
    # We now want to make sure that SGS also fixes the current level
    
    for point in orbit do
        for generator in SGS do
            
            # Avoid rechecking Schreier generators
            if not MATRIXSS_IsPointInOrbit(oldSchreierTree.Tree, point) or 
               not generator in ssInfo[level].oldSGS then
                
                # Map relations to current free group
                relations := [];
                #gens2 := GeneratorsOfGroup(ssInfo[level].freeGroup);
                gens2 := ssInfo[level].genMap.FreeGenerators;
                
                for element in ssInfo[level].relations do
                    gens1 := GeneratorsOfGroup(element[2]);
                    
                    MATRIXSS_DebugPrint(4, ["Mapping ", element[1],
                            " from ", element[2], " to ",
                            ssInfo[level].freeGroup]);
                    if Length(gens1) <= Length(gens2) then
                        AddSet(relations, 
                               MappedWord(element[1], gens1, 
                                       gens2{[1 .. Length(gens1)]}));
                    fi;
                od;
                
                # Express subgroup generators as words in generators
                # of free group
                subgroupGens := [];
                for element in List(ssInfo[level].partialSGS, i -> i[1]) do
                    element := 
                      PreImagesRepresentative(freeGroupHomo, element);
                    MATRIXSS_DebugPrint(6, ["Adding ", element,
                            " to subgroup gens"]);
                    AddSet(subgroupGens, element);
                    MATRIXSS_DebugPrint(6, ["SubGroup gens : ", 
                            subgroupGens]);
                od;
                
                MATRIXSS_DebugPrint(2, ["Running coset enum with gens : ", 
                        GeneratorsOfGroup(ssInfo[level].freeGroup), 
                        " relations ", relations, " subgroup gens ", 
                        subgroupGens]);
                
                # Perform (interruptible) Todd-Coxeter coset enumeration
                cosetTable := 
                  CosetTableFromGensAndRels(gens2, AsSet(relations), 
                          AsSet(subgroupGens) :
                          max := 1 + Int(cosetFactor * 
                                  MATRIXSS_GetOrbitSize(
                                          ssInfo[level].schreierTree.Tree)),
                          silent);
                
                # If coset enumeration was successful and index of the
                # subgroup was equal to our orbit size, then we know
                # that the subgroup is stabiliser and we can exit
                if cosetTable <> fail then
                    MATRIXSS_DebugPrint(2, ["Nof cosets: ", 
                            Length(cosetTable)]);
                    MATRIXSS_DebugPrint(2, ["Orbit size: ", 
                            MATRIXSS_GetOrbitSize(
                                    ssInfo[level].schreierTree.Tree)]);
                    if Length(cosetTable) = MATRIXSS_GetOrbitSize(
                               ssInfo[level].schreierTree.Tree) then
                        ssInfo[level].oldSGS := SGS;
                        return;
                    fi;
                fi;
                
                # Compute Schreier generator g for current level
                schreierGenerator := 
                  MATRIXSS_GetSchreierGenerator_ToddCoxeter(
                          ssInfo[level].schreierTree.Tree,
                          generator, point, action, identity,
                          ssInfo[level].freeGroup, ssInfo[level].genMap);
                
                MATRIXSS_DebugPrint(6, ["Schreier Generator : ", 
                        schreierGenerator]);
                
                if schreierGenerator[1][1] = identity then
                    continue;
                fi;
                
                # Check if Schreier generator is in stabiliser at 
                # the current level
                # Check if g \in H^(i + 1) = <S^(i + 1)>
                points := [level + 1 .. Length(ssInfo)];
                MATRIXSS_DebugPrint(4, ["Sifting element ", 
                        schreierGenerator[1], " on levels ", points]);
                strip := MATRIXSS_Membership_ToddCoxeter(ssInfo{points},
                                 schreierGenerator[1], 
                                 identity, ssInfo[level].freeGroup);
                MATRIXSS_DebugPrint(6, ["Got sift: ", strip]);
                word := ShallowCopy(strip[1][2]);
                residue := strip[1][1];
                
                # The drop-out level is in range
                # [1 .. Length(ssInfo) + 1 - level]
                # but we want the range given by points
                dropoutLevel := strip[2] + level;
                
                MATRIXSS_DebugPrint(6, ["Word : ", word]);
                
                MakeImmutable(residue);
                MakeImmutable(word);
                
                MATRIXSS_DebugPrint(4, ["Dropout level : ", dropoutLevel]);
                
                if residue[1] <> identity then
                    MATRIXSS_DebugPrint(4, ["Residue found"]);
                    
                    # We have found a Schreier generator which is not in
                    # the stabiliser of the current level, and so we must
                    # add the residue of this generator to our partial SGS
                    # in order to make it into a real SGS
                    
                    newInverseGenerator := Immutable(Reversed(residue));
                    
                    # Add residue to partial SGS
                    # This makes some levels incomplete and so we must
                    # recompute them recursively
                    AddSet(partialSGS, residue);
                    AddSet(partialSGS, newInverseGenerator);
                    
                    # Possibly extend the base if the Schreier generator
                    # fixes all points in our base
                    if dropoutLevel > Length(ssInfo) then
                        MATRIXSS_ExtendBase(ssInfo, residue, identity);
                    fi;
                    
                    # Update partial SGS at each level
                    for recursiveLevel in [level .. dropoutLevel - 1] do
                        if ssInfo[recursiveLevel].action(
                                   ssInfo[recursiveLevel].partialBase,
                                   residue[1]) = 
                           ssInfo[recursiveLevel].partialBase then
                            MATRIXSS_DebugPrint(8, ["Adding ",
                                    residue[1], " and ", residue[2],
                                    " to generators"]);
                            AddSet(ssInfo[recursiveLevel].partialSGS, 
                                   residue);
                            AddSet(ssInfo[recursiveLevel].partialSGS, 
                                   newInverseGenerator);
                        else
                            break;
                        fi;
                    od;
                fi;
                
                for recursiveLevel in [level .. dropoutLevel - 1] do
                    freeGroup := 
                      FreeGroup(Length(ssInfo[recursiveLevel].
                              partialSGS));
                    
                    # Important to use GroupWithGenerators, since we want
                    # all our matrices as generators to create the 
                    # generator mapping, even if some are redundant as
                    # generators (ie some are inverses of each other)
                    levelGroup := 
                      GroupWithGenerators(List(ssInfo[recursiveLevel].
                              partialSGS, i -> i[1]), identity);
                    MATRIXSS_DebugPrint(6, ["Generators : ",
                            GeneratorsOfGroup(levelGroup)]);
                    
                    # Recompute generator mapping, since we have added a
                    # generator
                    gens1 := ShallowCopy(GeneratorsOfGroup(levelGroup));
                    gens2 := ShallowCopy(GeneratorsOfGroup(freeGroup));
                    SortParallel(gens1, gens2);
                    ssInfo[recursiveLevel + 1].genMap := 
                      Immutable(rec(Generators := gens1, 
                              FreeGenerators := gens2));
                    ssInfo[recursiveLevel + 1].freeGroup := freeGroup;
                    
                    # Retrieve generators so that we can map all words to
                    # the same free group
                    gens1 := GeneratorsOfGroup(word[3]);
                    
                    if Length(gens2) >= Length(gens1) then
                        MATRIXSS_DebugPrint(6, ["Looking up ", residue[2],
                                " in ", ssInfo[recursiveLevel + 1].
                                genMap.FreeGenerators]);
                        
                        # The identity is not among our generators
                        if residue[1] <> identity then
                            element := 
                              ssInfo[recursiveLevel + 1].genMap.FreeGenerators[
                                      Position(ssInfo[recursiveLevel + 1].
                                              genMap.Generators, 
                                              residue[2])];
                            
                            # Map the words to the generators in the same
                            # free group                            
                            relation := 
                              schreierGenerator[2][1] * word[2] *
                              MappedWord(element, gens2, 
                                      gens1{[1 .. Length(gens2)]});
                        else
                            relation := schreierGenerator[2][1] * word[2];
                        fi;
                        
                        MATRIXSS_DebugPrint(3, ["Adding relation : ",
                                Immutable([relation, word[3]])]);
                        MATRIXSS_DebugPrint(3, ["Relations: ",
                                ssInfo[recursiveLevel + 1].relations]);
                        
                        # Save relation along with the its free group
                        # so that we can map its generators later
                        AddSet(ssInfo[recursiveLevel + 1].relations,
                            Immutable([relation, word[3]]));
                    fi;
                od;
                
                if residue[1] <> identity then
                    # We must now recompute all levels downward from the
                    # dropout level
                    for recursiveLevel in 
                      Reversed([level + 1 .. dropoutLevel]) do
                        oldSGS := ssInfo[level].oldSGS;
                        ssInfo[level].oldSGS := SGS;
                        MATRIXSS_SchreierToddCoxeterSims(ssInfo, 
                                partialSGS,
                                recursiveLevel, identity, cosetFactor);
                        ssInfo[level].oldSGS := oldSGS;
                    od;
                fi;
            fi;
        od;
    od;
    
    ssInfo[level].oldSGS := SGS;
end;

InstallGlobalFunction(SchreierToddCoxeterSims, function(G)
    local ssInfo, list, generators, level, points, element, ret;
    
    # Get initial set of generators, to be extended to a partial SGS
    generators := GeneratorsOfGroup(G);
        
    # The vector space on which the group acts
    points := FullRowSpace(FieldOfMatrixGroup(G), DimensionOfMatrixGroup(G));
    
    MATRIXSS_DebugPrint(3, ["Group generators : ", generators]);
    
    # Compute initial partial SGS and base and fill ssInfo
    ret := MATRIXSS_GetPartialBaseSGS(generators, Identity(G), points);
    generators := ret[1];
    ssInfo     := ret[2];
    
    MATRIXSS_DebugPrint(3, ["Partial sgs : ", generators]);
    MATRIXSS_DebugPrint(3, ["Initial base length : ", Length(ssInfo)]);
    MATRIXSS_DebugPrint(3, ["Calling recursive Schreier-Sims"]);
    
    # Call Schreier-Sims algorithm for each level (starting from top)
    for level in Reversed([1 .. Length(ssInfo)]) do
        MATRIXSS_SchreierToddCoxeterSims(ssInfo, generators, level, 
                Identity(G), 6/5);
    od;
    
    MATRIXSS_DebugPrint(2, ["Matrix Schreier-Sims done"]);
    MATRIXSS_DebugPrint(2, ["Order is : ", MatrixGroupOrderStabChain(ssInfo)]);
    
    return Immutable(rec(SchreierStructure := ssInfo, SGS := generators));
end);

###############################################################################
#E
