import gtk
import sys
import traceback

from cStringIO import StringIO
from HIGDialog import HIGDialog

# Functions for creating every kind of dialog. They are for internal use.

# define some Dialog icons
_ERROR = gtk.STOCK_DIALOG_ERROR
_INFO = gtk.STOCK_DIALOG_INFO
_QUESTION = gtk.STOCK_DIALOG_QUESTION
_WARNING = gtk.STOCK_DIALOG_WARNING

# we only want to display one dialog at a time, so let's queue them
_dialog_queue = []

# remember the previous message to avoid displaying the same message twice
# in a sequence
_last_message = None


#
# Adds a details button to the given dialog.
#
def _set_details(dialog, details):

    def expand(src, body, icon):
        if (src.get_active()):
            icon.set_from_stock(gtk.STOCK_ZOOM_OUT, gtk.ICON_SIZE_BUTTON)
            body.show()
        else:
            icon.set_from_stock(gtk.STOCK_ZOOM_IN, gtk.ICON_SIZE_BUTTON)
            body.hide()
        dialog.resize(10, 10)
        dialog.set_size_request(-1, -1)

    #vbox1 = dialog.get_children[0]
    vbox1 = dialog.get_vbox()

    vbox2 = gtk.VBox()
    vbox2.show()
    vbox1.pack_start(vbox2)

    align = gtk.Alignment(0.0, 0.0, 0.0, 0.0)
    align.set_property("border-width", 6)
    align.show()

    expander = gtk.ToggleButton()
    hbox = gtk.HBox()
    hbox.show()
    icon = gtk.Image()
    icon.set_from_stock(gtk.STOCK_ZOOM_IN, gtk.ICON_SIZE_BUTTON)
    icon.show()
    hbox.add(icon)
    lbl = gtk.Label(_("Details (%d lines)") % len(details.splitlines()))
    lbl.show()
    hbox.add(lbl)
    expander.add(hbox)

    expander.show()
    align.add(expander)
    vbox2.pack_start(align)

    align = gtk.Alignment(0.0, 0.0, 0.0, 0.0)
    align.set_property("border-width", 6)
    align.show()

    viewport = gtk.ScrolledWindow()
    viewport.set_policy(gtk.POLICY_NEVER, gtk.POLICY_NEVER)
    lbl = gtk.Label(details)
    lbl.set_property("selectable", True)
    lbl.set_property("wrap", True)
    lbl.set_selectable(True)
    lbl.show()

    nil, height = lbl.get_size_request()
    width, nil = vbox1.get_size_request()
    viewport.set_size_request(width, min(height, 320))
    viewport.add_with_viewport(lbl)

    align.add(viewport)
    vbox2.pack_start(align)

    expander.connect("toggled", expand, viewport, icon)


#
# Queues the given dialog for displaying.
#
def _queue_dialog(dialog):

    def proceed(*args):
        _dialog_queue.pop(0)
        if (_dialog_queue): _dialog_queue[0].show()

    dialog.connect("destroy", proceed)

    # display the dialog immediately if there are no others in the queue
    if (not _dialog_queue): dialog.show()
    _dialog_queue.append(dialog)


#
# Displays an error dialog. Errors are critical and the program terminates
# afterwards.
#
def error(primary, secondary):

    dialog = HIGDialog(_ERROR, primary, secondary,
                       (gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE))
    gtk.threads_enter()
    dialog.run()
    gtk.threads_leave()
    sys.exit(1337)


#
# Displays an information dialog.
#
def info(primary, secondary):

    dialog = HIGDialog(_INFO, primary, secondary,
                       (gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE))
    _queue_dialog(dialog)


#
# Displays a question dialog.
#
def question(icon, primary, secondary, *buttons):

    def responder(src, response):
        callback = buttons[response][1]
        if (callback): callback()

    response = 0
    btns = []
    for label, callback in buttons:
        btns.append(label)
        btns.append(response)
        response += 1

    dialog = HIGDialog(_QUESTION, primary, secondary, tuple(btns))
    dialog.connect("response", responder)
    dialog.show()


#
# Displays a warning dialog.
#
def warning(primary, secondary, show_traceback = True, force = False):

    global _last_message

    # don't show the same dialog twice in a sequence
    if (force): _last_message = ""
    if (_last_message == (primary, secondary)): return
    else: _last_message = (primary, secondary)

    dialog = HIGDialog(_WARNING, primary, secondary,
                       (gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE))

    details = ""

    if (show_traceback):
        fd = StringIO()
        traceback.print_exc(file = fd)
        details = fd.getvalue()
        fd.close()

    if (details and not details.startswith("None")):
        _set_details(dialog, details)

    _queue_dialog(dialog)


#
# Use the new filechoose if possible, or fallback to the old one
#
def fileselector(title, callback_ok, callback_cancel, *args):

    def handler(src, response):
        if (response == gtk.RESPONSE_OK):
            if (callback_ok): callback_ok(src, *args)
        else:
            if (callback_cancel): callback_cancel(src, *args)
            else: src.destroy()


    # do we have FileChooserDialog available?
    try:
        fsel = gtk.FileChooserDialog(title, None, gtk.FILE_CHOOSER_ACTION_OPEN,
                                     (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL,
                                      gtk.STOCK_OPEN, gtk.RESPONSE_OK))
        fsel.connect("response", handler)

    # no, then use the old FileSelection
    except:
        def f(btn, fsel, response): handler(fsel, response)

        fsel = gtk.FileSelection()

        if (title): fsel.set_title(title)
        fsel.ok_button.connect("clicked", f, fsel, gtk.RESPONSE_OK)
        fsel.cancel_button.connect("clicked", f, fsel, gtk.RESPONSE_CANCEL)

    fsel.show()

