/* gmoo - a gtk+ based graphical MOO/MUD/MUSH/... client
 * Copyright (C) 1999-2000 Gert Scholten
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <stdlib.h>
#include <sys/time.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <dirent.h>
#include <stdio.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdktypes.h>

#include "config.h"
#ifdef ZLIB
#   include <zlib.h>
#endif
#ifdef ZVT
#   include <zvt/zvtterm.h>
#endif

#include "gtkgmo.h"

#include "world.h"
#include "configdb.h"
#include "settings.h"
#include "misc.h"
#include "rcfile.h"
#include "toolbar.h"
#include "menu.h"
#include "notebook.h"
#include "dproperties.h"
#include "dlogin.h"
#include "dialog.h"
#include "userlist.h"
#include "history.h"
#include "window.h"
#include "editor.h"
#include "net.h"
#include "script.h"
#include "dsend.h"
#include "statusbar.h"
#include "packages.h"

void add_output_and_userlist  (world *w);
void create_and_add_hcontainer(world *w);
void scroll_down(GtkAdjustment *adj);

int max_world_id = 0;

void input_size_allocated(GtkWidget *input, GtkAllocation *alloc, world *w) {
	w->p->input_height = alloc->height;
}

void set_input_height(world *w) {
#define XTRA 12
	int size = gdk_char_height(gtk_widget_get_style(w->input)->font, 'J') + 
		XTRA;
	if(settings->save_input_height && w->p->input_height > 0 &&
	   w->p->input_height > size) {
		size = w->p->input_height;
	}
	gtk_widget_set_usize(w->input, -1, size); 
	gtk_paned_set_position(GTK_PANED(w->vpaned), 
			       w->vpaned->allocation.height - size);
}

char *cat_2_chars(char a, char b) {
	static char s[3];
	s[0] = a; s[1] = b; s[2] = 0;
	return s;
}

void print_own_input(world *w, char *text) {
	char *s = NULL;
	
	if(settings->show_own_input) {
		s = g_malloc(strlen(text) + 15);
		s[0] = '\0';
		strcat(s, "\x1b[0");
		if(settings->show_own_input_fgcolor >= 0 &&
		   settings->show_own_input_fgcolor <= 7) {
			strcat(s, ";");
			strcat(s, cat_2_chars('3', settings->show_own_input_fgcolor +'0'));
		}
		if(settings->show_own_input_bgcolor >= 0 &&
		   settings->show_own_input_bgcolor <= 7) {
			strcat(s, ";");
			strcat(s, cat_2_chars('4', settings->show_own_input_bgcolor +'0'));
		}
		if(settings->show_own_input_bold) {
			strcat(s, ";1");
		}
		if(settings->show_own_input_underline) {
			strcat(s, ";4");
		}
		strcat(s, "m");
		strcat(s, text);
		strcat(s, "\x1b[0m");
	}
	if(w->buffered_data_line && w->buffered_data_line_len) {
		text = g_strconcat(w->buffered_data_line,
				   s ? s : text, NULL);
		g_free(w->buffered_data_line);
		w->buffered_data_line = NULL;
		w->buffered_data_line_len = 0;
		g_free(s);
		s = text;
	}
	if(s) {
		gm_world_println(w, s);
		g_free(s);
	}
}


void enter_callback(GtkWidget *input, world *w) {
	char *text;
	text = gtk_editable_get_chars(GTK_EDITABLE(input), 0,
				      gtk_text_get_length(GTK_TEXT(input)));
	gtk_editable_delete_text(GTK_EDITABLE(input), 0,
				 gtk_text_get_length(GTK_TEXT(input)));
	gm_history_add(w, text);

	print_own_input(w, text);
#ifdef SCRIPT
	if(!gm_script_handle_macro(w, text))
#endif
		gm_world_writeln(w, text);

	if(w->p->log_input) {
		gm_world_println_log(w, text);
	}
	g_free(text);

	if(settings->scroll_on_enter) {
		scroll_down(GTK_GMO(w->output)->adj);
	}
}

#define UP   1
#define DOWN 2
void do_page(int direction, const world *w) {
	GtkAdjustment *adj = GTK_GMO(w->output)->adj;
	float desired;
	float real;

	if(direction == UP) {
		desired = adj->value - adj->page_increment;
		real = MAX(desired, adj->lower);
	} else { /* direction == DOWN */
		desired = adj->value + adj->page_increment;
		real = MIN(desired, adj->upper - adj->page_size);
	}
	gtk_adjustment_set_value(adj, real);
}

void gm_world_clear_screen(world *w) {
	gtk_gmo_clear(GTK_GMO(w->output));
}

int key_callback(GtkWidget *input, GdkEventKey *event, world *w) {
	int handled = TRUE;

	if((event->keyval == GDK_Return || 
	    (event->keyval == GDK_KP_Enter && event->state & GDK_MOD2_MASK)) && 
	   !(event->state & GDK_CONTROL_MASK)) {
		enter_callback(input, w);
	} else if(event->keyval == GDK_Return ||
		  (event->keyval == GDK_KP_Enter &&
		   event->state & GDK_MOD2_MASK)) {
		gtk_text_insert(GTK_TEXT(w->input), NULL, NULL, NULL, "\n", 1);

		/* Tab switching ... */
	} else if((event->keyval == GDK_Tab ||
		   event->keyval == GDK_KP_Tab ||
		   event->keyval == GDK_ISO_Left_Tab) &&
		  event->state & settings->hotkey_mask) {
		if(event->state & GDK_SHIFT_MASK)
			gm_notebook_prev();
		else
			gm_notebook_next();
	} else if(event->keyval >= GDK_0 && event->keyval <= GDK_9 && 
		  event->state & settings->hotkey_mask) {
		gm_notebook_set(event->keyval - GDK_0 > 0 ? event->keyval-GDK_0-1 : 9);
	} else if(settings->hotkey_arrow && event->state & settings->hotkey_mask &&
		  event->keyval == GDK_Left) {
		gm_notebook_prev();
	} else if(settings->hotkey_arrow && event->state & settings->hotkey_mask &&
		  event->keyval == GDK_Right) {
		gm_notebook_next();
		/* END switching ... */

	} else if((event->keyval == GDK_Up && !(event->state & GDK_CONTROL_MASK))||
		  (event->keyval == GDK_p && event->state & GDK_CONTROL_MASK)) {
		gm_history_up(w);
	} else if((event->keyval == GDK_Down && !(event->state&GDK_CONTROL_MASK))||
		  (event->keyval == GDK_n && event->state & GDK_CONTROL_MASK)) {
		gm_history_down(w);
	} else if(event->keyval==GDK_Page_Up && !(event->state&GDK_CONTROL_MASK)) {
		do_page(UP, w);
	} else if(event->keyval==GDK_Page_Down&&!(event->state&GDK_CONTROL_MASK)) {
		do_page(DOWN, w);
	} else if(event->state == GDK_CONTROL_MASK && event->keyval == GDK_l) {
		gm_world_clear_screen(w);

	} else if(event->state == GDK_CONTROL_MASK && event->keyval == GDK_F12) {
		debug = !debug;
		printf("\n\n*** Debug info here has been turned %s ***\n\n", debug?"on":"off");

	} else {
		/* Unhandled key combo */
		handled = FALSE;
	}

	if(handled) {
		gtk_signal_emit_stop_by_name(GTK_OBJECT(input), "key_press_event");
	}
	return handled;
}

void scroll_down(GtkAdjustment *adj) {
	if(adj->value != adj->upper - adj->page_size) {
		gtk_adjustment_set_value(adj, adj->upper - adj->page_size);
	}
}

#ifdef ZVT
void set_term_colors(ZvtTerm *t) {
	zvt_term_set_color_scheme(t, settings->reds, settings->greens,
				  settings->blues);
}

void zvt_term_init(GtkWidget *termw, properties_t *p) {
	ZvtTerm *term = ZVT_TERM(termw);
	zvt_term_set_scrollback(term, settings->buffer_size);
	zvt_term_set_blink(term, FALSE);
	zvt_term_set_bell(term, settings->beep);
	zvt_term_set_scroll_on_output(term, settings->scroll_on_text);

	gtk_signal_connect_after(GTK_OBJECT(term), "realize",
				 GTK_SIGNAL_FUNC(set_term_colors), NULL);
	gtk_signal_connect_object_while_alive(GTK_OBJECT(gm_window_get_window()),
					      "configure_event",
					      GTK_SIGNAL_FUNC(gtk_widget_queue_draw),
					      GTK_OBJECT(term));
}
#endif /* ZVT */

void world_output_resized(GtkGmo *output, int width, int height, world *w) {
	if(debug) printf("World %s resised to %dx%d\n", w->p->name, width, height);
	gm_mcp_update_screensize(w);
}


void create_widgets(world *w) {
	GtkWidget *tmp;
	
	w->vpaned = gtk_vpaned_new ();
	gtk_widget_show (w->vpaned);
	gtk_paned_set_gutter_size(GTK_PANED(w->vpaned),
				  GTK_PANED(w->vpaned)->handle_size);
	gtk_paned_set_position (GTK_PANED (w->vpaned), 201);
	/* ref() this widget ... we'll destroy it when it suits _us_ :) */
	gtk_widget_ref(w->vpaned);

	w->output_scrollw = gtk_scrolled_window_new(NULL, NULL);
	gtk_widget_show(w->output_scrollw);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(w->output_scrollw),
				       GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);

	w->output = gtk_gmo_new();
	gtk_widget_show(w->output);
	gtk_gmo_set_palette(GTK_GMO(w->output), colors);
	gtk_gmo_set_background(GTK_GMO(w->output),
			       (settings->bg_file &&
				settings->bg_filename &&
				settings->bg_filename[0]) ? settings->bg_filename : NULL,
			       settings->bg_tile,
			       settings->bg_trans,
			       settings->bg_tinted,
			       settings->bg_tint_r,
			       settings->bg_tint_g,
			       settings->bg_tint_b);
	gtk_gmo_set_font(GTK_GMO(w->output), NULL, settings->fontname);
	gtk_gmo_set_beep(GTK_GMO(w->output), settings->beep);
	gtk_gmo_set_scroll_on_text(GTK_GMO(w->output), settings->scroll_on_text);
	gtk_gmo_set_max_lines(GTK_GMO(w->output), settings->buffer_size);
	gtk_gmo_set_timestamp(GTK_GMO(w->output), settings->timestamp);
	gtk_signal_connect_object_while_alive(GTK_OBJECT(gm_window_get_window()),
					      "configure_event",
					      GTK_SIGNAL_FUNC(gtk_gmo_refresh_background),
					      GTK_OBJECT(w->output));
	gtk_container_add(GTK_CONTAINER(w->output_scrollw), w->output);

	gtk_signal_connect(GTK_OBJECT(w->output), "resized_in_chars",
			   GTK_SIGNAL_FUNC(world_output_resized), w);


	gm_userlist_new_widgets(w);

	/* Add the widgets */
	create_and_add_hcontainer(w);
	add_output_and_userlist(w);
	/***/

	tmp = gtk_hbox_new(FALSE, 2);
	gtk_paned_pack2(GTK_PANED(w->vpaned), tmp, FALSE, TRUE);
	gtk_widget_show(tmp);

#ifdef __PROMPT__
	w->prompt = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(tmp), w->prompt, FALSE, FALSE, 0);
	gtk_widget_show(w->prompt);
#endif

	w->input = gtk_text_new (NULL, NULL);
	gtk_box_pack_end(GTK_BOX(tmp), w->input, TRUE, TRUE, 0);
	gtk_text_set_editable (GTK_TEXT (w->input), TRUE);
	gtk_widget_show (w->input);
	set_input_height(w);
	gtk_signal_connect_after(GTK_OBJECT(w->input), "size_allocate",
				 GTK_SIGNAL_FUNC(input_size_allocated), w);
	gtk_signal_connect(GTK_OBJECT(w->input), "key_press_event",
			   GTK_SIGNAL_FUNC(key_callback), (void *) w);

	gtk_widget_set_style(w->input, input_style);


	/* The arn't ref()ed, the notebook will destropy them for us */
	w->tab_label = gtk_label_new (w->p->name);
	gtk_widget_show (w->tab_label);
}


/*************** World file access *****************/
#define world_dir(w) g_strconcat(gm_settings_get_worlds_dir(), "/", w, NULL)

int world_exists(const char *name) {
	char *dir = world_dir(name);
	int ret   = exists_dir(dir);
	g_free(dir);
	return ret;
}

extern int checkout_dir(const char *dirname);

GList *get_names_from_order_file() {
	GList *list = NULL;
	char *filename = g_strconcat(gm_settings_get_worlds_dir(),
				     "/" WORLD_ORDER_FILE,
				     NULL);
	FILE *file = fopen(filename, "r");
#define SIZE 256
	char worldname[SIZE];
	char *s;
	int len;

	if(file) {
		while(fgets(worldname, SIZE, file)) {
			len = strlen(worldname);
			if(len > 1 && worldname[len - 1] == '\n')
				worldname[len - 1] = '\0';
			s = g_strconcat(gm_settings_get_worlds_dir(), "/",
					worldname, NULL);
			if(is_dir(s)) {
				list = g_list_append(list, g_strdup(worldname));
			}
			g_free(s);
		}
		fclose(file);
	}
	g_free(filename);

	return list;
}

int str_in_list(GList *list, const char *s) {
	GList *l;
	for(l = list; l; l = g_list_next(l)) {
		if(strcmp(l->data, s) == 0) {
			return TRUE;
		}
	}
	return FALSE;
}

GList *gm_world_get_names() {
	GList *list = NULL;
	char  *dirname = gm_settings_get_worlds_dir();
	DIR   *dir;
	struct dirent *ent;    
	char *s;

	list = get_names_from_order_file();

	if((dir = opendir(dirname))) {
		while((ent = readdir(dir))) {
			if(ent->d_name[0] != '.' && strcmp(ent->d_name, TEMPLATE) != 0) {
				s = g_strconcat(dirname, "/", ent->d_name, NULL);
				if(is_dir(s) && !str_in_list(list, ent->d_name)) {
					list = g_list_prepend(list, g_strdup(ent->d_name));
				}
				g_free(s);
			}
		}
	}
	return list;
}

/* world directory exists */
void world_load_settings(world *w, const char *name) {
	char *dirname = world_dir(name);
	char *filename = g_strconcat(dirname, "/", WORLD_PROPERTIES_FILE, NULL);

	w->p = gm_properties_load(filename);
	w->p->name = g_strdup(name);

	g_free(filename);
	g_free(dirname);
}

properties_t *gm_world_get_settings_by_name(const char *name) {
	char *dirname = world_dir(name);
	char *filename = g_strconcat(dirname, "/", WORLD_PROPERTIES_FILE, NULL);

	properties_t *p = gm_properties_load(filename);
	p->name = g_strdup(name);

	g_free(filename);
	g_free(dirname);

	return p;
}

/* world directory will be created */
void world_save_settings_by_prop(properties_t *p) {
	char *dirname, *filename;

	dirname = world_dir(p->name);
	checkout_dir(dirname);

	filename = g_strconcat(dirname, "/", WORLD_PROPERTIES_FILE, NULL);

	gm_properties_store(p, filename);

	g_free(dirname);
	g_free(filename);
}

void gm_world_save_settings(world *w) {
	if(!w->quick) {
		world_save_settings_by_prop(w->p);
	} else if(debug) {
		printf("\t\tNot saving settings, is a quick connect world\n");
	}
}


void gm_world_remove(const char *name) {
	char *dirname = world_dir(name);
	char *cmd = g_strconcat("rm -rf '", dirname, "'", NULL);
	gm_notebook_close_worlds_with_name(name);
	system(cmd);
	g_free(cmd);
	g_free(dirname);
}

void world_apply_properties(world *w, properties_t *prop) {
	gm_properties_apply(w, gm_properties_copy(prop));
}

int handle_world_exists(const char *worldname) {
	world *w;
	if(strcmp(worldname, TEMPLATE) == 0) {
		gm_dialog_popup(_("Warning"),
				_("Warning !\n"
				  "\n"
				  "\""TEMPLATE"\" is a reserved name,\n"
				  "Please select another name for this world"),
				B_OK,
				TRUE);
		return FALSE;
	}
	if(gm_dialog_popup(_("World exists !"),
			   _("Warning !\n"
			     "\n"
			     "This world already exists, if you enter\n"
			     "`Yes' here, all data that was related\n"
			     "to this world will be deleted.\n"
			     "\n"
			     "Do you wish to replace this world ?"),
			   B_YES | B_NO, TRUE) != B_YES) {
		return FALSE;
	}
	if((w = gm_notebook_find_world(worldname)))
		gm_notebook_close_world(w);
	gm_world_remove(worldname);
	return TRUE;
}

void rename_world(const char *old, const char *new) {
	char *od, *nd, *cmd;
	if(!world_exists(old)) {
		return;
	}

	od = world_dir(old);
	nd = world_dir(new);
	cmd = g_strconcat("mv -f '", od, "' '", nd, "'", NULL);
	if(debug) printf("cmd == >>%s<<\n", cmd);

	system(cmd);

	g_free(cmd);
	g_free(nd);
	g_free(od);
}

int gm_world_apply_properties(world *w, properties_t *p) {
	if(strcmp(p->name, w->p->name) != 0) { /* see if the worldname changed */
		if(world_exists(p->name) &&
		   ! handle_world_exists(p->name)) {
			return FALSE;
		} else {
			rename_world(w->p->name, p->name);
		}
	}
	w->quick = FALSE;
	world_apply_properties(w, p);
	gm_world_save_settings(w);
	return TRUE;
}

int gm_world_apply_properties_by_name(const char *name, properties_t *p) {
	if(strcmp(p->name, name) != 0) { /* see if the worldname changed */
		if(world_exists(p->name) &&
		   !handle_world_exists(p->name)) {
			return FALSE;
		} else {
			rename_world(name, p->name);
		}
	}
	world_save_settings_by_prop(p);
	return TRUE;
}

int gm_world_create_config(properties_t *p) {
	if(world_exists(p->name) &&
	   !handle_world_exists(p->name)) {
		return FALSE;
	}
	world_save_settings_by_prop(p);
	return TRUE;
}

void world_copy(const char *name, const char *newname) {
	char *dirname = world_dir(name);
	char *newdirname = world_dir(newname);
	char *cmd = g_strconcat("cp -af '", dirname, "' '", newdirname, "'", NULL);

	system(cmd);

	g_free(cmd);
	g_free(dirname);
	g_free(newdirname);
}

int gm_world_copy_by_name(const char *name, const char *newname) {
	if(world_exists(newname) &&
	   !handle_world_exists(newname)) {
		return FALSE;
	}
	gm_world_remove(newname);
	world_copy(name, newname);
	return TRUE;
}

void set_world_defaults(world *w) {
	w->server = NULL;

	w->id = max_world_id++;

	w->position = -1;

	w->new_lines = 0;
	w->quick = FALSE;
	w->editors = NULL;
	w->current_editor = NULL;

	w->status_msg = NULL;

	w->log = NULL;

	w->buffered_print_line = NULL;
	w->buffered_write_line = NULL;

	w->buffered_data_line     = NULL;
	w->buffered_data_line_len = 0;

	gm_history_new(w);

	w->mcp = NULL;
	w->userlist = NULL;

	w->dialog_properties = NULL;
	w->dialog_login   = NULL;
	w->dialog_send   = NULL;
}

world *gm_world_create(const char *name) {
	world *w;

	if(!world_exists(name)) return NULL;

	w = g_malloc(sizeof(world));

	world_load_settings(w, name);

	set_world_defaults(w);
	create_widgets(w);

#ifdef SCRIPT
	if(debug) printf("\tNotifying scripts...\n");
	gm_script_world_opened(w);
#endif
	return w;
}

void gm_world_create_quick(const char *hostname, int port) {
	world *w = g_malloc(sizeof(world));
	w->p     = gm_world_get_settings_by_name(TEMPLATE);
	g_free(w->p->name);
	g_free(w->p->hostname);
	w->p->name      = g_strdup_printf("%s:%d", hostname, port);
	w->p->hostname  = g_strdup(hostname);
	w->p->port      = port;

	set_world_defaults(w);

	w->quick = TRUE;

	create_widgets(w);

#ifdef SCRIPT
	if(debug) printf("\tNotifying scripts...\n");
	gm_script_world_opened(w);
#endif

	gm_notebook_add_world(w);
}

void gm_world_save_order(GList *order) {
	char *filename = g_strconcat(gm_settings_get_worlds_dir(),
				     "/" WORLD_ORDER_FILE,
				     NULL);
	FILE *file = fopen(filename, "w");
	GList *l;

	for(l = order; l; l = g_list_next(l)) {
		fputs(l->data, file);
		fputc('\n', file);
	}

	fclose(file);
	g_free(filename);
}

void gm_world_create_template() {
	properties_t *p;
	char *dir;
	char *filename;
	if(world_exists(TEMPLATE)) return;

	dir = world_dir(TEMPLATE);
	checkout_dir(dir);
	filename = g_strconcat(dir, "/", WORLD_PROPERTIES_FILE, NULL);
	p = gm_properties_load(filename);
	gm_properties_store(p, filename);
	gm_properties_free(p);
	g_free(filename);
	g_free(dir);
}


/************ Closing worlds **************/

int gm_world_ok_to_close(world *w) {
	GList *l;
	int ret = TRUE;
	char *s;

	s = g_strdup_printf("If you close world %s now,\n"
			    "all it's editors will be closed\n"
			    "with it, there may be data loss\n"
			    "\nContinue ?", w->p->name);
	if(debug) printf("Checking if its OK to close world \"%s\"\n", w->p->name);
	for(l = w->editors; l != NULL; l = g_list_next(l)) {
		if(!gm_editor_ok_to_close(EDITOR(l->data))) {
			if(gm_dialog_popup(_("Warning"), s, B_YES|B_NO, TRUE) != B_YES) {
				ret = FALSE;
			}
			break;
		}
	}
	g_free(s);
	return ret;
}

void gm_world_close(world *w) {
	GList *l;
	if(debug) printf("Closing world \"%s\" now\n", w->p->name);
	if(debug) printf("\tClosing logs...\n");
	gm_world_close_log(w);
	if(debug) printf("\tDisconnecting...\n");
	gm_world_disconnect(w);
	if(debug) printf("\tSaving settings...\n");
	gm_world_save_settings(w);

	/* gm_world_disconnect will call this.
	 * gm_mcp_close(w->mcp);
	 */

	if(w->editors) {
		if(debug) printf("\tClosing editor(s)...\n");
		l = w->editors;
		do {
			gm_editor_close(EDITOR(l->data));
			l = w->editors;
		} while(l);
	}

#ifdef SCRIPT
	if(debug) printf("\tNotifying scripts...\n");
	gm_script_world_closed(w);
#endif

	if(w->dialog_properties) {
		if(debug) printf("\tHave to close properties dialog...\n");
		gm_dialog_properties_close(w->dialog_properties);
	}
	if(w->dialog_login) {
		if(debug) printf("\tHave to close login dialog...\n");
		gm_dialog_login_close(w->dialog_login->window,
				      w->dialog_login);
	}
	if(w->dialog_send) {
		if(debug) printf("\tHave to close or destroy send dialog...\n");
		gm_dialog_send_close(w);
	}

	if(debug) printf("\tDestroying input history...\n");
	gm_history_destroy(w);

	if(debug) printf("\tClearing world properties...\n");
	gm_properties_free(w->p);

	g_free(w->status_msg);

	g_free(w->buffered_print_line);
	g_free(w->buffered_write_line);
	g_free(w->buffered_data_line);

	g_free(w->server);

	if(debug) printf("\tDestroying widgets...\n");
	gtk_widget_destroy(w->vpaned);

	if(debug) printf("\tFree()ing the world struct itself...\n");
	g_free(w);
	if(debug) printf("Done.\n");
}

/********************* Printing text **********************/

void gm_world_println(world *w, const char *line) {
	int len = strlen(line) + 1;
	gtk_gmo_append_line(GTK_GMO(w->output), line, len);
}

void gm_world_print_buffered(world *w, const char *line) {
	char *s, *t;
	int length;

	if(w->buffered_print_line) {
		s = g_strconcat(w->buffered_print_line, line, NULL);
		g_free(w->buffered_print_line);
	} else {
		s = g_strdup(line);
	}
	if((length = strlen(s))) {
		if(line[length - 1] != '\n') {
			t = strrchr(s, '\n');
			if(t) {
				t[0] = 0;
				gm_world_println(w, s);
				w->buffered_print_line = g_strdup(t + 1);
			} else {
				w->buffered_print_line = g_strdup(s);
			}
		} else {
			s[length - 1] = '\0';
			gm_world_println(w, s);
			w->buffered_print_line = NULL;
		}
	}
	g_free(s);
}

void gm_world_print_buffered_with_len(world *w, const char *line, int length) {
	gm_world_print_buffered(w, line);
}

void gm_world_write(world *w, const char *line) {
	gm_net_send(w, line, strlen(line));
}

void gm_world_write_with_len(world *w, const char *line, int length) {
	gm_net_send(w, line, length);
}

void gm_world_writeln(world *w, const char *line) {
	char *s = g_strconcat(line, "\n", NULL);
	gm_world_write(w, s);
	g_free(s);
}

void gm_world_write_buffered(world *w, const char *line) {
	char *s, *t;
	int length;
	if(w->buffered_write_line) {
		s = g_strconcat(w->buffered_write_line, line, NULL);
		g_free(w->buffered_write_line);
	} else {
		s = g_strdup(line);
	}
	length = strlen(s);
	if(length && line[length - 1] != '\n') {
		t = strrchr(s, '\n');
		if(t) {
			t[0] = 0;
			gm_world_writeln(w, s);
			w->buffered_write_line = g_strdup(t + 1);
		} else {
			w->buffered_write_line = g_strdup(s);
		}
	} else {
		gm_world_write_with_len(w, s, length);
		w->buffered_write_line = NULL;
	}
	g_free(s);
}

void gm_world_write_buffered_with_len(world *w, const char *line, int length) {
	gm_world_write_buffered(w, line);
}

void handle_mcp(world *w, const char *line, int len) {
	if(w->mcp) {
		gm_mcp_handle(w->mcp, line, len);
	} else if(strncasecmp(line, "mcp ", 4) == 0) {
		gm_mcp_open_for_world(w, line, len);
	} else if(debug) {
		printf("\tsession not initialised, and not a connect command\n");
	}
}

void handle_out_of_band(world *w, const char *line, int len) {
	if(line[0] == ' ') { /* NO MCP */
		gm_editor_new_from_oob(w, line + 1);
	} else {
		handle_mcp(w, line, len);
	}
}

void handle_external(world *w, char *line) {
	if(debug) printf("TODO: Handle external command: %s\n", line);
}

char *skip_1st_garbage(const char *line, int len) {
	int i;
	for(i = 0; i < len; i++) {
		if(line[i] == 27) {
			i++;
			if(i < len && line[i] == '[') {
				i++;
				while(i < len &&
				      ((line[i] >= '0' && line[i] <= '9') || line[i] == ';'))
					i++;
			}
		} else if(line[i] != 7) {
			break;
		}
	}
	if(i < len) return (char *) (line + i);
	return NULL;
}

void handle_output_line(world *w, char *line, int len) {
	char *stripped_line = strip_ansi(line, len);
	char *skipped = skip_1st_garbage(line, len);
	char *mcp_line = skipped ? skipped : line;

	if(w->current_editor) {
		if(strcmp(line, ".") == 0) {
			gm_editor_all_data_received(w->current_editor);
			w->current_editor = NULL;
		} else {
			gm_editor_add_line(w->current_editor, line, len);
		}
	} else if(strncmp(mcp_line, "#$#", 3) == 0) {
		handle_out_of_band(w, mcp_line + 3, len - 3 + line - mcp_line);
	} else if(strncmp(mcp_line, "#@#EXTERNAL#@# ", 15) == 0) {
		handle_external(w, mcp_line + 15);
	} else {
		gm_world_println(w, line);
		gm_world_println_log(w, w->p->log_color ? line : stripped_line);

		if(w != gm_notebook_current_world()) {
			w->new_lines++;
		}
		gm_world_update_tab_label(w);
	}
	g_free(stripped_line);
}

#ifdef __PROMPT__
void update_prompt(world *w) {
	char *skipped;
	char *mcp_line;
	char *stripped;
	
	if(w->buffered_data_line && w->buffered_data_line_len) {
		skipped = skip_1st_garbage(w->buffered_data_line,
					   w->buffered_data_line_len);
		mcp_line = skipped ? skipped : w->buffered_data_line;
		if(strncmp(mcp_line, "#$#", 3) == 0 ||
		   strncmp(mcp_line, "#@#EXTERNAM#@#", 15) == 0)
			gtk_label_set_text(GTK_LABEL(w->prompt), "");
		else {
			stripped = strip_ansi(w->buffered_data_line,
					      w->buffered_data_line_len);
			gtk_label_set_text(GTK_LABEL(w->prompt),
					   stripped);
			g_free(stripped);
		}
	} else 
		gtk_label_set_text(GTK_LABEL(w->prompt), "");
}
#endif

void gm_world_handle_output_data(world *w, const char *text, int len) {
	char *s;
	int i;
	int j = 0;
	int free_text;

	if(w->buffered_data_line) {
		text = g_strconcat(w->buffered_data_line, text, NULL);
		len += w->buffered_data_line_len;
		g_free(w->buffered_data_line);
		w->buffered_data_line = NULL;
		free_text = TRUE;
	} else {
		free_text = FALSE;
	}
	s = g_malloc(len + 1);

	for(i = 0; i < len; i++) {
		if(text[i] != '\r') {
			s[j] = text[i];
			if(s[j] == '\n') {
				s[j] = '\0';
				handle_output_line(w, s, j);
				j = 0;
			} else {
				j++;
			}
		}
	}

	if(j) {
		s[j] = '\0';
		w->buffered_data_line = s;
		w->buffered_data_line_len = j;
	} else {
		g_free(s);
	}
	if(free_text) {
		g_free((char *) text);
	}
#ifdef __PROMPT__
	update_prompt(w);
#endif
}

/**************************************************************************/

void gm_world_log_in(world *w) {
	char *s;
	if(gm_world_is_connected(w)) {
		/* The "connect %s %s" string should be replaced
		 * by a user selectable string in the config window */
		s = g_strdup_printf("connect %s %s", w->p->username, w->p->password);
		gm_world_writeln(w, s);
		g_free(s);
	}
}


void gm_world_connect(world *w) {
	if(!gm_world_is_connected(w)) {
		gm_net_connect(w, w->p->hostname, w->p->port);
	} else if(debug) {
		printf("Already connected\n");
	}
}

void gm_world_disconnect(world *w) {
	gm_net_disconnect(w);
}


void gm_world_try_disconnect(world *w) {
	GList *l;
	char *s;
	int disconnect = TRUE;

	s = g_strdup_printf("If you disconnect world %s now,\n"
			    "none of it's editors will be able\n"
			    "to send it's data to the world\n"
			    "\nContinue ?", w->p->name);
	if(debug) printf("Checking if its OK to disconnect world \"%s\"\n",
			 w->p->name);
	for(l = w->editors; l != NULL; l = g_list_next(l)) {
		if(!gm_editor_ok_to_close(EDITOR(l->data))) {
			if(gm_dialog_popup(_("Warning"), s, B_YES|B_NO, TRUE) != B_YES) {
				disconnect = FALSE; break;
			}
			break;
		}
	}
	g_free(s);
	if(disconnect) {
		gm_world_disconnect(w);
	}
}

void gm_world_connecting(world *w) {
	gm_toolbar_world_connecting(w);
	gm_menu_world_connecting(w);
}

void gm_world_connected(world *w) {
	char *s;

	gm_toolbar_world_connected(w);
	gm_menu_world_connected(w);

#ifdef SCRIPT
	gm_script_world_connected(w);
#endif

	gm_world_open_log(w);

	/* Automagic login: */
	if(w->p->autologin && w->p->username && w->p->password) {
		s = g_strdup_printf("\x1b[32m%% Automatically logging in user "
				    "%s\x1b[0m", w->p->username);
		gm_world_println(w, s);
		g_free(s);
		gm_world_log_in(w);
	} else {
		gm_dialog_login_open(w);
	}
}

void gm_world_disconnected(world *w) { 
	gm_mcp_close(w->mcp);
	gm_toolbar_world_disconnected(w);
	gm_menu_world_disconnected(w);
	gm_world_close_log(w);
#ifdef SCRIPT
	gm_script_world_disconnected(w);
#endif
	g_free(w->status_msg);
	w->status_msg = g_strconcat(_("World "), w->p->name, NULL);
	if(w == gm_notebook_current_world())
		gm_statusbar_set(w->status_msg);
}

/*** misc ***/
void gm_world_got_focus(world *w) {
	char *text;
	gtk_widget_grab_focus(w->input);
	w->new_lines = 0;
	gm_world_update_tab_label(w);
	if(w->status_msg) {
		gm_statusbar_set(w->status_msg);
	} else {
		text = g_strconcat(_("World "), w->p->name, NULL);
		gm_statusbar_set(text);
		g_free(text);
	}
}

void gm_world_update_tab_label(world *w) {
	char *s;
	char *name = settings->tab_limit_length ?
		limit_string_to_length(w->p->name, settings->tab_max_length) :
		w->p->name;

	/* Normal */
	if(!w->new_lines || w == gm_notebook_current_world()) {
		gtk_widget_set_rc_style(w->tab_label);

		if(settings->tab_pos_on_tab && w->position > -1) {
			s = g_strdup_printf("(%d) %s", w->position, name);
			gtk_label_set_text(GTK_LABEL(w->tab_label), s);
			g_free(s);
		} else {
			gtk_label_set_text(GTK_LABEL(w->tab_label), name);
		}
	} else { /* Highlited */
		gtk_widget_set_style(w->tab_label, redtab_style);
		if (settings->tabs_show_lines_waiting) { 
			if(settings->tab_pos_on_tab && w->position > -1) {
				s = g_strdup_printf("(%d) %s [%d]", w->position,
						    name, w->new_lines);
			} else {
				s = g_strdup_printf("%s [%d]", name, w->new_lines);
			}
			gtk_label_set_text(GTK_LABEL(w->tab_label), s);
			g_free(s);
		} else {
			if(settings->tab_pos_on_tab && w->position > -1) {
				s = g_strdup_printf("(%d) %s", w->position, name);
				gtk_label_set_text(GTK_LABEL(w->tab_label), s);
				g_free(s);
			} else {
				gtk_label_set_text(GTK_LABEL(w->tab_label), name);
			}
		}
	}
	if(settings->tab_limit_length) {
		g_free(name);
	}
}

void gm_world_set_position(world *w, int pos) {
	w->position = pos < 10 ? (pos + 1) % 10 : -1;
	gm_world_update_tab_label(w);
}

void add_output_and_userlist(world *w) {
	gtk_widget_set_usize(GTK_WIDGET(w->userlist_container),
			     w->p->userlist_width, -1);
	if(GTK_IS_BOX(w->hcontainer)) {
		if(settings->userlist_pos == RIGHT) {
			gtk_box_pack_start(GTK_BOX(w->hcontainer), w->output_scrollw,
					   TRUE, TRUE, 0);
			gtk_box_pack_start(GTK_BOX(w->hcontainer), w->userlist_container,
					   FALSE, TRUE, 0);
		} else {
			gtk_box_pack_start(GTK_BOX(w->hcontainer), w->userlist_container,
					   FALSE, TRUE, 0);
			gtk_box_pack_start(GTK_BOX(w->hcontainer), w->output_scrollw,
					   TRUE, TRUE, 0);
		}
	} else {
		if(settings->userlist_pos == RIGHT) {
			gtk_paned_pack1(GTK_PANED(w->hcontainer), w->output_scrollw,
					TRUE, TRUE);
			gtk_paned_pack2(GTK_PANED(w->hcontainer), w->userlist_container,
					FALSE, TRUE);
		} else {
			gtk_paned_pack1(GTK_PANED(w->hcontainer), w->userlist_container,
					FALSE, TRUE);
			gtk_paned_pack2(GTK_PANED(w->hcontainer), w->output_scrollw,
					TRUE, TRUE);
		}
	}
}

void gm_world_update_userlist_pos(world *w) {
	if(debug) printf("Updating userlist_pos for world %s\n", w->p->name);
	gtk_widget_ref(w->output_scrollw);
	gtk_widget_ref(w->userlist_container);
	gtk_container_remove(GTK_CONTAINER(w->hcontainer), w->output_scrollw);
	gtk_container_remove(GTK_CONTAINER(w->hcontainer), w->userlist_container);

	add_output_and_userlist(w);

	gtk_widget_unref(w->output_scrollw);
	gtk_widget_unref(w->userlist_container);
}

void create_and_add_hcontainer(world *w) {
	if(settings->userlist_resizeable &&
	   GTK_WIDGET_VISIBLE(w->userlist_container)) {
		w->hcontainer = gtk_hpaned_new();
		gtk_paned_set_gutter_size(GTK_PANED(w->hcontainer),
					  GTK_PANED(w->hcontainer)->handle_size);
	} else {
		w->hcontainer = gtk_hbox_new(FALSE, 2);
	}
	gtk_paned_pack1(GTK_PANED(w->vpaned), w->hcontainer, TRUE, TRUE);
	gtk_widget_show(w->hcontainer);
}

void gm_world_update_userlist_resizeable(world *w) {
	if(debug) printf("Updating userlist resizeable for world %s\n", w->p->name);
	gtk_widget_ref(w->output_scrollw);
	gtk_widget_ref(w->userlist_container);
	gtk_container_remove(GTK_CONTAINER(w->hcontainer), w->output_scrollw);
	gtk_container_remove(GTK_CONTAINER(w->hcontainer), w->userlist_container);
	gtk_widget_destroy(w->hcontainer);

	create_and_add_hcontainer(w);
	add_output_and_userlist(w);

	gtk_widget_unref(w->output_scrollw);
	gtk_widget_unref(w->userlist_container);
}


	void gm_world_update_userlist_totals(world *w) {
		if(settings->userlist_totals)
			gtk_widget_show(w->userlist_totals_frame);
		else
			gtk_widget_hide(w->userlist_totals_frame);
	}

	void gm_world_update_userlist_headers(world *w) {
		if(settings->userlist_headers)
			gtk_clist_column_titles_show(GTK_CLIST(w->userlist_clist));
		else
			gtk_clist_column_titles_hide(GTK_CLIST(w->userlist_clist));
	}

void gm_world_update_userlist_objects(world *w) {
	if(debug)printf("Setting column visibility of column 2 in world %s to: %d\n",
			w->p->name, settings->userlist_objects);

	gtk_clist_set_column_visibility(GTK_CLIST(w->userlist_clist),
					2, settings->userlist_objects);
}

void gm_world_update_userlist_friends_color(world *w) {
	if(w->userlist) gm_userlist_update_friends_color(w->userlist);
}


/* logging */
char *create_log_filename(world *w) {
	char *s = NULL;
	time_t t = time(NULL);
	struct tm *time = localtime(&t);
#ifdef ZLIB
	char *extension = g_strdup(w->p->compressed_logs ? ".gz" : "");
#else
	char *extension = g_strdup("");
#endif

	switch(w->p->log_type) {
	case LOG_APPEND:
		s = g_strdup_printf("%s/%s.log%s",
				    gm_settings_get_log_dir(),
				    w->p->name,
				    extension);
		break;
	case LOG_DAY:
		s = g_strdup_printf("%s/%s - %d-%.2d-%.2d.log%s",
				    gm_settings_get_log_dir(),
				    w->p->name,
				    time->tm_year + 1900,
				    time->tm_mon + 1,
				    time->tm_mday,
				    extension);
		break;
	case LOG_CONNECTION:
		s = g_strdup_printf("%s/%s - %d-%.2d-%.2d - %.2d:%.2d.log%s",
				    gm_settings_get_log_dir(),
				    w->p->name,
				    time->tm_year + 1900,
				    time->tm_mon + 1,
				    time->tm_mday + 1,
				    time->tm_hour,
				    time->tm_min,
				    extension);
		break;
	default:
		if(debug) printf("ERROR: Don't know what log type it is ....\n");
	}
	g_free(extension);
	return s;
}

void gm_world_open_log(world *w) {
	char *filename;
	if(w->p->log_use) {
		filename = create_log_filename(w);
#ifdef ZLIB
		if(w->p->compressed_logs)
			w->log = gzopen(filename, "a");
		else
#endif
			w->log = fopen(filename, "a");
		if(w->log == NULL)
			printf("gmoo: Error opening file %s: %s\n", filename,
			       strerror(errno));
		g_free(filename);
	}
}

void gm_world_close_log(world *w) {
	int retval;

	if(w->log) {
#ifdef ZLIB
		if(w->p->compressed_logs) {
			retval = gzclose(w->log);
			if(retval)
				/* gzerror consults errno if it isn't a zlib error */
				printf("gmoo: Error closing file: %s\n",
				       gzerror(w->log, &retval));
		} else {
#endif
			retval = fclose(w->log);
			if(retval)
				printf("gmoo: Error closing file: %s\n", strerror(errno));
#ifdef ZLIB
		}
#endif

		w->log = NULL;
	}
}

void gm_world_println_log(world *w, const char *line) {
	if(w->log) {
		if(strncmp(line, "co ", 3) == 0 ||
		   strncmp(line, "connect ", 8) == 0) {
			gm_world_println(w, "\x1b[32m% Connect string detected, "
					 "not logged\x1b[0m");
		} else {
#ifdef ZLIB
			if(w->p->compressed_logs) {
				gzputs(w->log, line);
				gzputc(w->log, '\n');
			} else {
#endif
				fputs(line, w->log);
				fputc('\n', w->log);
#ifdef ZLIB
			}
#endif
		}
	}
}

int gm_world_is_connected(world *w) {
	return w->server && w->server->connected;
}

int  gm_world_is_connecting(world *w) {
	return w->server && w->server->connecting;
}
