/******************************************************************************\
 gnofin/imp-qif.c   $Revision: 1.11.2.2 $
 Copyright (C) 1999-2000 Marty Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

//#define ENABLE_DEBUG_TRACE

#include "common.h"
#include <libgnomeui/gnome-dialog.h>
#include <libgnomeui/gnome-stock.h>
#include <gtk/gtkframe.h>
#include <gtk/gtkcheckbutton.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include "dialogs.h"
#include "data-if.h"
#include "qif-import.h"

const char ERROR_NO_MEMORY[]   = N_("No memory available for processing QIF file.");
const char ERROR_INVALID_QIF[] = N_("The specified file is not a valid QIF file.");

enum {
  INVALID_TOK,
  TYPEBANK_TOK,
  TYPECASH_TOK,
  TYPECCARD_TOK,
  TYPEINVST_TOK,
  TYPEOTHA_TOK,
  TYPEOTHL_TOK,
  ACCOUNT_TOK,
  TYPECAT_TOK,
  TYPECLASS_TOK,
  TYPEMEMORIZED_TOK,
  OPTIONALLXFR_TOK,
  NONINVEST_DATE_TOK,
  NONINVEST_AMOUNT_TOK,
  NONINVEST_CLEARED_TOK,
  NONINVEST_NUM_TOK,
  NONINVEST_PAYEE_TOK,
  NONINVEST_MEMO_TOK,
  NONINVEST_ADDRESS_TOK,
  NONINVEST_CATEGORY_TOK,
  NONINVEST_CATEGORYINSPLIT_TOK,
  NONINVEST_MEMOINSPLIT_TOK,
  NONINVEST_DOLLARAMOUNT_TOK,
  ENDOFENTRY_TOK,
  ACCOUNTMARKER_TOK,
  TYPEMARKER_TOK,
};

typedef struct {
  char*	name;
  int	value;
} Keyword;

static Keyword keywords[] =
{
  {"!Type:Bank",	TYPEBANK_TOK},
  {"!Type:Cash",	TYPECASH_TOK},
  {"!Type:CCard",	TYPECCARD_TOK},
  {"!Type:Invst",	TYPEINVST_TOK},
  {"!Type:Oth A",	TYPEOTHA_TOK},
  {"!Type:Oth L",	TYPEOTHL_TOK},
  {"!Account",		ACCOUNT_TOK},
  {"!Type:Cat",		TYPECAT_TOK},
  {"!Type:Class",	TYPECLASS_TOK},
  {"!Type:Memorized",	TYPEMEMORIZED_TOK},
  {"!Option:AllXfr",	OPTIONALLXFR_TOK},
  {"D",			NONINVEST_DATE_TOK},
  {"T",			NONINVEST_AMOUNT_TOK},
  {"C",			NONINVEST_CLEARED_TOK},
  {"N",			NONINVEST_NUM_TOK},
  {"P",			NONINVEST_PAYEE_TOK},
  {"M",			NONINVEST_MEMO_TOK},
  {"A",			NONINVEST_ADDRESS_TOK},
  {"L",			NONINVEST_CATEGORY_TOK},
  {"S",			NONINVEST_CATEGORYINSPLIT_TOK},
  {"E",			NONINVEST_MEMOINSPLIT_TOK},
  {"$",			NONINVEST_DOLLARAMOUNT_TOK},
  {"^",			ENDOFENTRY_TOK},
  {"!Type",		TYPEMARKER_TOK},
  {"!Acco",		ACCOUNTMARKER_TOK}
};

#define QIF_NUMOFKEYWORDS sizeof_array(keywords)

#define MAXBUF 512

static gboolean qif_have_day_before_month = FALSE;

static int   qif_find_keyword    (const char *token);
static char *qif_read_file       (FILE *stream, size_t size);
static char *qif_read_line       (char *filebuf, int *eof, char *buf);
static void  qif_trim_left       (char *buf);
static int   qif_find_one_of     (char ch, char *buf);
static char *qif_read_type_bank  (GtkWindow *win, char *fileptr, Bankbook *book, Account *account);
static void  qif_req_n_string    (char *token, char *buf, int count);
static int   qif_prompt_for_type (GtkWindow *win, const char *given_name);

static inline size_t
get_file_size (FILE *file)
{
  struct stat st;
  fstat (fileno (file), &st);
  return st.st_size;
}

//-------------------------------------------------------------------------

gboolean
qif_import (GtkWindow *win, const gchar *filename, Bankbook *book)
{
  AccountInfo acc = {0};
  Account *account;

  char   linebuf[MAXBUF];
  char  *filebuf;
  char  *fileptr;
  FILE  *stream;
  size_t filesize;
  int    token;
  //int    pos;
  int    eof;

  trace ("");

  qif_have_day_before_month = FALSE;

  stream = fopen (filename, "rt");
  if (stream == NULL) {
    dialog_error (win, _("Error importing file: %s\n[%s]"), filename, strerror(errno));
    return FALSE;
  }

  // get length of file
  filesize = get_file_size (stream);
  trace ("file size: %ld", filesize);

  // discard blank lines
  do {
    if (fread (linebuf, sizeof (char), 1, stream) != 1)
    {
      fclose (stream);
      dialog_error (win, _(ERROR_INVALID_QIF));
      return FALSE;
    }
  } while (linebuf[0] != '!');

  // Ensure that we have a valid QIF file
  if (fread (linebuf+1, sizeof (char), 4, stream) != 4) {
    fclose (stream);
    dialog_error (win, _(ERROR_INVALID_QIF));
    return FALSE;
  }

  *(linebuf+5) = 0;  // NULL terminate the 5 character header
  token = qif_find_keyword (linebuf);
  trace ("token: \"%s\" (%d)", linebuf, token);

  switch (token) {
    case TYPEMARKER_TOK:
    case ACCOUNTMARKER_TOK:
      break;
    default:
      dialog_error (win, _(ERROR_INVALID_QIF));
      return FALSE;
  }

  rewind (stream);

  // read file into memory for higher performance in parsing
  if (!(filebuf = qif_read_file (stream, filesize))) {
    dialog_error (win, _(ERROR_NO_MEMORY));
    return FALSE;
  }
  fileptr = filebuf;
  
  acc.name = g_basename (filename);
  acc.notes = _("Imported QIF file");
  account = if_bankbook_insert_account (book, &acc);

  // parse file and build new accounting set
  while (1) {
    // read in next line
    fileptr = qif_read_line (fileptr, &eof, linebuf);

    if (eof) {
      g_free (filebuf);
      return TRUE;
    }

    trace ("line: %s", linebuf);

    qif_trim_left (linebuf);

    // check for a zero length line
    if (strlen (linebuf) == 0)
      continue;

    // get next token
    /*
    pos = qif_find_one_of (' ', linebuf);
    trace ("pos: %d", pos);
    if (pos > 0)
      linebuf[pos] = '\0';
    */

    token = qif_find_keyword (linebuf);
    trace ("token: \"%s\" (%d)", linebuf, token);

top_:
    switch (token) {
      case TYPEBANK_TOK:
      case TYPECASH_TOK:
      case TYPECCARD_TOK:
      case TYPEOTHA_TOK:
      case TYPEOTHL_TOK:
	if (!(fileptr = qif_read_type_bank (win, fileptr, book, account))) {
          g_free (filebuf);
          return FALSE;
	}
	break;
      case TYPEINVST_TOK:
	dialog_error (win, _("Investment account transaction register is not supported."));
	break;
      case ACCOUNT_TOK:
	dialog_error (win, _("Account list is not supported."));
	break;
      case TYPECAT_TOK:
	dialog_error (win, _("Category list is not supported."));
	break;
      case TYPECLASS_TOK:
	dialog_error (win, _("Class list is not supported."));
	break;
      case TYPEMEMORIZED_TOK:
	dialog_error (win, _("Memorized transaction list is not supported."));
	break;
      default:
	trace ("Unknown token");
	token = qif_prompt_for_type (win, linebuf);
	if (token != INVALID_TOK)
	  goto top_;
	break;
    }		
    break;
  }
  g_free (filebuf);
  trace ("exit");
  return TRUE;
}

//-------------------------------------------------------------------------

static void
qif_define_record_types (Bankbook *book)
{
  RecordTypeInfo typ = {0};

  trace ("");

  typ.name = _("ATM");
  typ.description = _("Automated Teller Machine");
  typ.linked = 0;
  typ.numbered = 0;
  typ.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &typ);

  typ.name = _("CC");
  typ.description = _("Check Card");
  typ.linked = 0;
  typ.numbered = 0;
  typ.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &typ);

  typ.name = _("CHK");
  typ.description = _("Check");
  typ.linked = 0;
  typ.numbered = 1;
  typ.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &typ);

  typ.name = _("DEP");
  typ.description = ("Deposit");
  typ.linked = 0;
  typ.numbered = 0;
  typ.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &typ);

  typ.name = _("UNK");
  typ.description = ("Unknown");
  typ.linked = 0;
  typ.numbered = 0;
  typ.sign = RECORD_TYPE_SIGN_ANY;
  if_bankbook_insert_record_type (book, &typ);
}

//-------------------------------------------------------------------------

static gboolean
qif_day_before_month (GtkWindow *win)
{
  static gboolean result = FALSE;

  if (!qif_have_day_before_month)
  {
    GnomeDialog *dialog;
    GtkWidget *frame;
    GtkWidget *button;

    trace ("");

    dialog = GNOME_DIALOG (gnome_dialog_new (_("QIF Import Parameters"),
					     GNOME_STOCK_BUTTON_OK, NULL));
    
    frame = gtk_frame_new (_("QIF Import Parameters"));
    gtk_box_pack_start (GTK_BOX (dialog->vbox), frame, TRUE, TRUE, 0);

    button = gtk_check_button_new_with_label (_("QIF file encodes dates with day before month"));
    gtk_container_set_border_width (GTK_CONTAINER (button), 5);
    gtk_container_add (GTK_CONTAINER (frame), button);

    gtk_widget_show_all (frame);
    
    gnome_dialog_set_parent (dialog, win);
    gnome_dialog_run (dialog);

    result = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (button));

    gnome_dialog_close (dialog);

    qif_have_day_before_month = TRUE;
  }

  return result;
}

//-------------------------------------------------------------------------

static char *
qif_read_type_bank (GtkWindow *win, char *fileptr, Bankbook *book, Account *account)
{
  RecordInfo rec = {0};
  Record *record = NULL;
  int day, month, year;

  char  linebuf[MAXBUF];
  char  tokenbuf[MAXBUF];
  char  typebuf[MAXBUF];
  char *ptr;

  const gchar *type_name = _("UNK");

  int eof;
  int token;
  int pos;

  trace ("");

  qif_define_record_types (book);

  while (1) {
    // read in next line
    ptr = linebuf;
    fileptr = qif_read_line (fileptr, &eof, linebuf);

    if (eof)
      break;

    trace ("text line: %s", linebuf);

    qif_trim_left (linebuf);

    // check for a zero length line
    if (strlen (linebuf) == 0)
      continue;

    // get bank account type
    qif_req_n_string (tokenbuf, linebuf, 1);

    token = qif_find_keyword (tokenbuf);
    trace ("token: \"%s\" (%d)", tokenbuf, token);

    switch (token) {
      case NONINVEST_DATE_TOK:
        pos = qif_find_one_of ('/', linebuf);
	if (pos == 0)
	  pos = qif_find_one_of ('-', linebuf);
        trace ("pos: %d  linebuf: %s", pos, linebuf);

        strncpy (tokenbuf, linebuf, pos);
	tokenbuf[pos] = '\0';  // strncpy does not NULL terminate

        sscanf (tokenbuf, "%d", &month);
        ptr = ptr + (pos+1);   // step past delimator

	/* The QIF format at first glance appears to have a serious
	 * Y2K problem.  Dates are stored as MM/DD/YY.  It seems that
	 * "they" decided to get around this problem by making the
	 * format MM/DD'YY in the case that YY is part of the 21st
	 * century.
	 *
	 * CORRECTION: seems the date format is a user option!!!!
	 *
	 * FIXME: do something about this
	 */

        pos = qif_find_one_of ('/', ptr);
	if (pos == 0)
	  pos = qif_find_one_of ('-', ptr);
	if (pos == 0)
	  pos = qif_find_one_of ('\'', ptr);

        strncpy (tokenbuf, ptr, pos);
	tokenbuf[pos] = '\0';  // strncpy does not NULL terminate

        sscanf (tokenbuf, "%d", &day);
        ptr = ptr + (pos+1);   // step past deliminator

        sscanf (ptr, "%d", &year);

        // Y2K compliance fix
        if ((year < 100) && (year > 80)) {
	  year += 1900;
        }
        else if (year < 81)
	  year += 2000;
	// else
	// assume year is in YYYY format

	/* prompt user to determine if date encodes day before month
	 */
	if (qif_day_before_month (win))
	{
	  int temp = month;
	  month = day;
	  day = temp;
	}

        trace ("month: %d  day: %d  year: %d", month, day, year);
	g_date_clear (&rec.date, 1);
	g_date_set_dmy (&rec.date, day, month, year);
	break;
      case NONINVEST_NUM_TOK:
        // this token encodes the type

        if (*linebuf >= '1' && *linebuf <= '9') {  // check/ref number
          sscanf (linebuf, "%d", &rec.number); 
          trace ("check number: %d", rec.number);
	  type_name = _("CHK");
        }
        else
	{
	  rec.number = 0;
	  // scan for record type
	  if (strlen (linebuf) == 0)
	    type_name = "---"; // hopefully this will be unique enough
	  else
	  if (g_strcasecmp (linebuf, "DEP") == 0)
	    type_name = _("DEP");
	  else
	  if (g_strcasecmp (linebuf, "Visa") == 0)
	    type_name = _("CC");
	  else
	  if (g_strcasecmp (linebuf, "ATM") == 0)
	    type_name = _("ATM");
	  else
	  {
	    strncpy (typebuf, linebuf, sizeof typebuf);
	    type_name = typebuf;
	  }
	}
        break;
      case NONINVEST_AMOUNT_TOK:
        money_parse_f (linebuf, &rec.amount, '.', ',');
        trace ("amount: %d", rec.amount);
        break;
      case NONINVEST_CLEARED_TOK:
        rec.cleared = 1;
        break;
      case NONINVEST_PAYEE_TOK:
        rec.payee = g_strdup (linebuf);
        break;
      case NONINVEST_MEMO_TOK:
	rec.memo = g_strdup (linebuf);
        break;
      case NONINVEST_CATEGORY_TOK:
	rec.category = g_strdup (linebuf);
      	break;
      case ENDOFENTRY_TOK:
        rec.type = if_bankbook_get_record_type_by_name (book, type_name);
	if (rec.type == NULL)
	{
	  // the type_name wasnt recognized... so create it.
	  RecordTypeInfo typ = {0};
	  typ.name = (gchar *) type_name;
	  typ.description = (gchar *) type_name;
	  typ.linked = 0;
	  typ.numbered = 0;
	  typ.sign = RECORD_TYPE_SIGN_ANY;
	  rec.type = if_bankbook_insert_record_type (book, &typ);
	}
	record = if_account_insert_record (account, &rec);
        trace ("record stored");
	
	// clear rec structure
	g_free (rec.payee);
        g_free (rec.memo);
	g_free (rec.category);
	memset (&rec, 0, sizeof rec);

        type_name = _("UNK");
        break;
    }
  }
  return fileptr;
}

//-------------------------------------------------------------------------

static int
qif_find_keyword (const char *tok)
{
  int i;

  trace("");

  for (i=0; i<QIF_NUMOFKEYWORDS; i++) {
    if (g_strcasecmp (tok, keywords[i].name) == 0)
      return keywords[i].value;
  }
  return 0;
}

//-------------------------------------------------------------------------

static char *
qif_read_file (FILE *stream, size_t filesize)
{
  char *filebuf;

  trace ("");

  filebuf = g_new0 (char, filesize);
  if (filebuf == NULL) {
    trace ("Failed to allocate memory for QIF file.");
    return NULL;
  }

  if (fread (filebuf, sizeof (char), filesize, stream) != filesize) {
    trace ("Error reading QIF file.");
    g_free (filebuf);
    return NULL;
  }

  return filebuf;
}

//-------------------------------------------------------------------------

static char *
qif_read_line (char *filebuf, int *eof, char *linebuf)
{
  // FIXME: this function could very easily overrun the linebuf buffer

  char ch = 0;

  trace ("");

  *eof = 0;

  do {
    ch = *filebuf++;
    if ((ch != '\n') && (ch != '\r') && (ch != '\0'))
      *linebuf++ = ch;
    if (ch == '\0') {
      trace ("EOF Reached");
      *eof = 1;
      break;
    }
  } while ((ch != '\n') && (ch != '\r'));

  *linebuf = '\0';
  return filebuf;
}

//-------------------------------------------------------------------------

static void
qif_trim_left (char *buf)
{
  char tempbuf[MAXBUF];
  char *ptr = buf;

  trace ("");

  // FIXME: better to implement this with memmove

  if (*buf == ' ') {
    while (*ptr++ == ' ');
    strcpy (tempbuf, ptr);
    strcpy (buf, tempbuf);
  }
}

//-------------------------------------------------------------------------

static int
qif_find_one_of (char ch, char *buf)
{
  char *ptr;

  trace ("");

  ptr = strchr (buf, ch);
  trace ("ptr: %s", ptr);

  if (ptr == NULL) // didnt find ch terminator
    return 0;

  return (int) (ptr - buf);
}

//-------------------------------------------------------------------------

static void
qif_req_n_string (char *tokenbuf, char *textbuf, int count)
{
  char tempbuf[MAXBUF];

  trace ("");

  strncpy (tokenbuf, textbuf, count);
  *(tokenbuf+count) = 0;

  strcpy (tempbuf, textbuf+count);
  strcpy (textbuf, tempbuf);
}

//-------------------------------------------------------------------------

static int
qif_prompt_for_type (GtkWindow *win, const char *given_name)
{
  trace ("");
  g_return_val_if_fail (given_name, INVALID_TOK);

  if (dialog_question_yes_no (win, _("The QIF file you have specified is of an unknown type.\n"
				     "Would you like to try again, treating it as an ordinary\n"
				     "bank-type QIF file.")) == DIALOG_YES)
    return TYPEBANK_TOK;
  else
    return INVALID_TOK;
}
