/******************************************************************************\
 gnofin/account.c   $Revision: 1.12 $
 Copyright (C) 1999-2000 Darin Fisher

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
\******************************************************************************/

#include "common.h"
#include "account.h"
#include "record.h"
#include "record-type.h"
#include "record-select.h"
#include "data-if.h"
#include "date.h"


/******************************************************************************
 * Lookups
 */

Account *
get_account_by_name (GList *accounts, const gchar *name)
{
  trace ("%s", name);
  g_return_val_if_fail (name, NULL);

  for (; accounts; accounts=accounts->next)
  {
    Account *acc = LIST_DEREF (Account, accounts);

    if (strcmp (acc->name, name) == 0)
      return acc;
  }
  return NULL;
}


/******************************************************************************
 * Account info
 */

guint
account_info_copy (AccountInfo *dest, const AccountInfo *src, guint mask)
{
  trace ("");
  g_return_val_if_fail (src, 0);
  g_return_val_if_fail (dest, 0);

  /* This function copies the account info fields specified by the mask
   * parameter.  If mask == 0, then all writable fields are copied and
   * the true (or actual) event mask is returned. */

  if (mask == 0)
    mask = ACCOUNT_ALL_WRITABLE_FIELDS;
  
  if (mask & ACCOUNT_FIELD_NAME)
    dest->name = g_strdup (src->name);

  if (mask & ACCOUNT_FIELD_NOTES)
    dest->notes = g_strdup (src->notes);

  if (mask & ACCOUNT_FIELD_FOREIGN)
    dest->foreign = src->foreign;

  return mask;
}

void
account_info_clear (AccountInfo *info, guint mask)
{
  trace ("");
  g_return_if_fail (info);

  /* The purpose of this function is to free any dynamic resources
   * associated with the account info structure. */

  if (mask == 0)
    mask = ACCOUNT_ALL_WRITABLE_FIELDS;
  
  if (mask & ACCOUNT_FIELD_NAME)
    g_free (info->name);
  
  if (mask & ACCOUNT_FIELD_NOTES)
    g_free (info->notes);
}

gboolean
account_info_diff (const AccountInfo *a, const AccountInfo *b, guint mask)
{
  trace ("");
  g_return_val_if_fail (a, TRUE);
  g_return_val_if_fail (b, TRUE);
  g_return_val_if_fail (mask, TRUE);

  if (mask & ACCOUNT_FIELD_NAME)
    if (strcmp (a->name, b->name)) return TRUE;

  if (mask & ACCOUNT_FIELD_NOTES)
    if (strcmp (a->notes, b->notes)) return TRUE;
  
  if (mask & ACCOUNT_FIELD_CLEARED_BAL)
    if (a->cleared_bal != b->cleared_bal) return TRUE;
  
  if (mask & ACCOUNT_FIELD_OVERALL_BAL)
    if (a->overall_bal != b->overall_bal) return TRUE;
  
  if (mask & ACCOUNT_FIELD_SORT_FIELD)
    if (a->sort_field != b->sort_field) return TRUE;
  
  if (mask & ACCOUNT_FIELD_SORT_REV)
    if (a->sort_rev != b->sort_rev) return TRUE;

  if (mask & ACCOUNT_FIELD_FOREIGN)
    if (a->foreign != b->foreign) return TRUE;
  
  return FALSE;  /* no difference */
}


/******************************************************************************
 * Sorting
 */

static inline gint
uint_compare (guint a, guint b)
{
  trace ("");

  if (a == b)
    return 0;
  else
    return (a < b) ? -1 : 1;
}

static gint
compare_accounts_by_name (const Account *a, const Account *b)
{
  trace ("");
  g_assert (a->name);
  g_assert (b->name);

  return strcmp (a->name, b->name);
}

static gint
compare_records_by_memo (const Record *a, const Record *b)
{
  trace ("");
  g_assert (a->memo);
  g_assert (b->memo);
  g_assert (a->memo->string);
  g_assert (b->memo->string);

  return strcmp (a->memo->string, b->memo->string);
}

static gint
compare_records_by_category (const Record *a, const Record *b)
{
  trace ("");
  g_assert (a->category);
  g_assert (b->category);
  g_assert (a->category->string);
  g_assert (b->category->string);

  return strcmp (a->category->string, b->category->string);
}

static gint
compare_records_by_payee (const Record *a, const Record *b)
{
  trace ("");
  g_assert (a->payee);
  g_assert (b->payee);
  g_assert (a->payee->string);
  g_assert (b->payee->string);

  return strcmp (a->payee->string, b->payee->string);
}

static gint
compare_records_by_linked_acct (const Record *a, const Record *b)
{
  trace ("");

  if (a->link.acc_name)
  {
    if (b->link.acc_name)
      return strcmp (a->link.acc_name->string, b->link.acc_name->string);
    else
      return 1;
  }
  else if (b->link.acc_name)
    return -1;
  else
    return 0;
}

static gint
compare_records_by_number (const Record *a, const Record *b)
{
  trace ("");

  return uint_compare (record_number (a), record_number (b));
}

static gint
compare_records_by_type (const Record *a, const Record *b)
{
  trace ("");
  g_assert (a->type);
  g_assert (b->type);
  g_assert (a->type->name);
  g_assert (b->type->name);

  return strcmp (a->type->name, b->type->name);
}

static gint
compare_records_by_amount (const Record *a, const Record *b)
{
  trace ("");

  return money_compare (record_amount (a), record_amount (b));
}

static gint
compare_records_by_date (const Record *a, const Record *b)
{
  trace ("");

  return g_date_compare ((GDate *) &a->date, (GDate *) &b->date);
}

static gint
compare_records_by_cleared (const Record *a, const Record *b)
{
  trace ("");

  return uint_compare (a->cleared, b->cleared);
}

typedef gint (* RecordCompareFunc) (const Record *, const Record *);

/* The compare functions listed in preferred order
 */
static RecordCompareFunc record_compare_funcs [] =
{
  compare_records_by_date,
  compare_records_by_cleared,
  compare_records_by_amount,
  compare_records_by_type,
  compare_records_by_number,
  compare_records_by_linked_acct,
  compare_records_by_category,
  compare_records_by_payee,
  compare_records_by_memo,
  NULL
};

static const struct
{
  guint field;
  RecordCompareFunc func;
}
record_compare_table [] =
{
  { RECORD_FIELD_TYPE,            compare_records_by_type        },
  { RECORD_FIELD_DATE,            compare_records_by_date        },
  { RECORD_FIELD_NUMBER,          compare_records_by_number      },
  { RECORD_FIELD_LINKED_ACC_NAME, compare_records_by_linked_acct },
  { RECORD_FIELD_CATEGORY,        compare_records_by_category    },
  { RECORD_FIELD_PAYEE,           compare_records_by_payee       },
  { RECORD_FIELD_MEMO,            compare_records_by_memo        },
  { RECORD_FIELD_CLEARED,         compare_records_by_cleared     },
  { RECORD_FIELD_AMOUNT,          compare_records_by_amount      },
};

static gint
compare_records (const Record *a, const Record *b)
{
  gint i, flip;
  RecordCompareFunc default_func = NULL;
  RecordCompareFunc *f;

  trace ("");
  g_assert (a->parent);
  
  for (i = 0; i < sizeof_array (record_compare_table); ++i)
  {
    if (record_compare_table[i].field == a->parent->sort_field)
    {
      default_func = record_compare_table[i].func;
      break;
    }
  }

  flip = a->parent->sort_rev;

  /* Try default compare func
   */
  if (default_func)
    i = (* default_func) (a, b);
  else
    i = 0;

  /* If default compare func returns 0, try
   * other compare funcs, in preferred order,
   * skipping default_func */
  for (f = record_compare_funcs; (*f) && (i == 0); f++)
  {
    if ((*f) != default_func)
      i = (* (*f)) (a, b);
  }

  return flip ? -i : i;
}


/******************************************************************************
 * Account
 */

Account *
account_new ()
{ 
  Account *account;

  trace ("");

  account = g_new0 (Account, 1);
  if (account)
  {
    account->sort_field = RECORD_FIELD_DATE;
    account->records_select.method = ALL;
    date_now (&account->records_select.date1);
    date_now (&account->records_select.date2);
    account_ref (account);
  }
  return account;
}

void
account_ref (Account *account)
{
  trace ("%p", account);
  g_return_if_fail (account);

  account->ref_count++;
}

void
account_unref (Account *account)
{
  trace ("%p", account);
  g_return_if_fail (account);

  if (account && (--account->ref_count == 0))
    account_destroy (account);
}

void
account_destroy (Account *account)
{
  trace ("%p", account);
  g_return_if_fail (account);

  account->ref_count = 0;

  if (account->parent)
    account_detach (account, FALSE);
  
  while (account->records)
    record_unref (LIST_DEREF (Record, account->records));

  g_free (account->name);
  g_free (account->notes);
  g_free (account);
}

gboolean
account_attach (Account *account, Bankbook *parent, gboolean full)
{
  trace ("%p, %p %s", account, parent, full ? "full" : "");
  g_return_val_if_fail (account, FALSE);
  g_return_val_if_fail (parent, FALSE);
  g_return_val_if_fail (account->parent == NULL, FALSE);  /* Its must not be parented */
  g_return_val_if_fail (account->name, FALSE);            /* Account must be named */

  /* Verify that the account name is unique */
  g_assert (g_list_find_custom (parent->accounts, account,
			       (GCompareFunc) compare_accounts_by_name) == NULL);

  /* Insert account into parent->accounts list, sorted by account name */
#if 0
  parent->accounts =
    g_list_insert_sorted (parent->accounts, account, (GCompareFunc) compare_accounts_by_name);
#else
  parent->accounts = g_list_append (parent->accounts, account);
#endif
  account->parent = parent;

  /* If the account has records, we need to store the objects referenced
   * by the records in the data object.  This is handled by record_attach_children */
  if (full && account->records)
  {
    GList * it;
    for (it=account->records; it; it=it->next)
      record_attach_children (LIST_DEREF (Record, it));
  }

  /* The parent now references this account */
  account_ref (account);
  return TRUE;
}

void
account_detach (Account * account, gboolean full)
{
  trace ("%p %s", account, full ? "full" : "");
  g_return_if_fail (account);
  g_return_if_fail (account->parent);

  if (full)
  {
    GList * it;

    /* Loop through the records, detaching children */
    for (it=account->records; it; it=it->next)
    {
      Record * record = LIST_DEREF (Record, it);

      /* The detach_children method is defined assuming the record has no parent */
      record->parent = NULL;
      record_detach_children (record);
      record->parent = account;
    }
  }

  account->parent->accounts = g_list_remove (account->parent->accounts, account);
  account->parent = NULL;

  account_unref (account);
}

gboolean
account_set_info (Account *account, guint mask, const AccountInfo *info)
{
  trace ("%p", account);
  g_return_val_if_fail (account, FALSE);
  g_return_val_if_fail (account->parent == NULL, FALSE);
  g_return_val_if_fail (!(mask & ACCOUNT_FIELD_OVERALL_BAL), FALSE);
  g_return_val_if_fail (!(mask & ACCOUNT_FIELD_CLEARED_BAL), FALSE);
  g_return_val_if_fail (!(mask & ACCOUNT_FIELD_SORT_FIELD), FALSE);
  g_return_val_if_fail (!(mask & ACCOUNT_FIELD_SORT_REV), FALSE);

  if (mask == 0)
    mask = ACCOUNT_ALL_WRITABLE_FIELDS;

  if (mask & ACCOUNT_FIELD_NAME)
  {
    g_return_val_if_fail (info->name, FALSE);

    /* Free old name and store copy of new one */
    g_free (account->name);
    account->name = g_strdup (info->name);
  }

  if (mask & ACCOUNT_FIELD_NOTES)
  {
    /* Free old notes and store copy of new one, notes may be NULL */
    g_free (account->notes);
    account->notes = g_strdup (info->notes ? info->notes : "");
  }

  if (mask & ACCOUNT_FIELD_FOREIGN)
    account->foreign = info->foreign;

  return TRUE;
}

void
account_get_info (const Account *account, guint mask, AccountInfo *info)
{
  trace ("%p", account);
  g_return_if_fail (account);

  if (mask == 0)
    mask = (guint) -1;

  if (mask & ACCOUNT_FIELD_NAME)
    info->name = account->name;

  if (mask & ACCOUNT_FIELD_NOTES)
    info->notes = account->notes;
  
  if (mask & ACCOUNT_FIELD_OVERALL_BAL)
    info->overall_bal = account->overall_bal;
  
  if (mask & ACCOUNT_FIELD_CLEARED_BAL)
    info->cleared_bal = account->cleared_bal;

  if (mask & ACCOUNT_FIELD_SORT_FIELD)
    info->sort_field = account->sort_field;

  if (mask & ACCOUNT_FIELD_SORT_REV)
    info->sort_rev = account->sort_rev;

  if (mask & ACCOUNT_FIELD_FOREIGN)
    info->foreign = account->foreign;
}

void
account_sort_records (Account *account, guint sort_field, gboolean sort_rev)
{
  GCompareFunc func;

  trace ("%p", account);
  g_return_if_fail (account);
  g_return_if_fail (sort_field != RECORD_FIELD_OVERALL_BAL);
  g_return_if_fail (sort_field != RECORD_FIELD_CLEARED_BAL);
  g_return_if_fail (sort_field != RECORD_FIELD_LINKED_REC);
  g_return_if_fail (sort_field != RECORD_FIELD_LINKED_ACC);

  if ((account->sort_field == sort_field) && 
      (account->sort_rev == sort_rev))
    return;  /* Records appear to be already sorted */

  account->sort_field = sort_field;
  account->sort_rev = sort_rev;

  func = account_get_record_sort_fcn (account);
  account->records = g_list_sort (account->records, func);

  account_recompute_bal (account, 0);
}

GCompareFunc
account_get_record_sort_fcn (const Account *account)
{
  trace ("%p", account);
  return (GCompareFunc) compare_records;
}

void
account_recompute_bal (Account *account, guint start_from)
{
  Record *record;
  money_t amount=0, overall_bal=0, cleared_bal=0;
  GList *it;

  trace ("%p from=%u", account, start_from);
  g_return_if_fail (account);

  if (account->records)
  {
    it = g_list_nth (account->records, start_from);
    if (it == NULL)
      it = g_list_last (account->records);
    if (it->prev)
    {
      record = LIST_DEREF (Record, it->prev);

      overall_bal = record->overall_bal;
      cleared_bal = record->cleared_bal;
    }
    while (it)
    {
      record = LIST_DEREF (Record, it);

      amount = record_amount (record);

      overall_bal += amount;
      if (record->cleared)
	cleared_bal += amount;
    
      record->overall_bal = overall_bal;
      record->cleared_bal = cleared_bal;
    
      it = it->next;
    }
  }
  account->overall_bal = overall_bal;
  account->cleared_bal = cleared_bal;
}

gint
account_index (const Account *account)
{
  trace ("%p", account);
  g_return_val_if_fail (account, 0);
  g_return_val_if_fail (account->parent, 0);

  return g_list_index (account->parent->accounts, (gpointer) account);
}

void
account_dump (const Account *account)
{
  GList *it;

#ifdef MONEY_IS_64BIT
  g_print ("  A - %p[rc=%u] {n=\"%s\",ob=%qd,cb=%qd,sf=%u,sr=%u}\n",
           account,
	   account->ref_count,
	   account->name,
	   account->overall_bal,
	   account->cleared_bal,
	   account->sort_field,
	   account->sort_rev);
#else
  g_print ("  A - %p[rc=%u] {n=\"%s\",ob=%ld,cb=%ld,sf=%u,sr=%u}\n",
           account,
	   account->ref_count,
	   account->name,
	   account->overall_bal,
	   account->cleared_bal,
	   account->sort_field,
	   account->sort_rev);
#endif
  
  for (it=account->records; it; it=it->next)
    record_dump (LIST_DEREF (Record, it));
}

// vim: ts=8 sw=2
