/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2003 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include "gnome-cmd-includes.h"
#include "widget-factory.h"
#include "gnome-cmd-options-dialog.h"
#include "gnome-cmd-data.h"
#include "utils.h"
#include "support.h"
#include "gnome-cmd-main-win-funcs.h"
#include "gnome-cmd-style.h"
#include "gnome-cmd-connection-funcs.h"


static GnomeDialogClass *parent_class = NULL;



struct _GnomeCmdOptionsDialogPrivate
{
};



/***********************************************************************
 *
 *  The General tab
 *
 **********************************************************************/

static GtkWidget*
create_general_tab (GtkWidget *parent)
{
	GtkWidget *frame, *hbox, *vbox, *cat, *cat_box, *table;	
	GtkWidget *radio, *check, *spin, *label;
	
	frame = create_tabframe (parent);
	hbox = create_tabhbox (parent);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	vbox = create_tabvbox (parent);
	gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);	


	/* Right mouse button settings
	 */
	cat_box = create_vbox (parent, FALSE, 0);
	cat = create_category (parent, cat_box, _("Use right mouse button to"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, FALSE, TRUE, 0);

	radio = create_radio (parent, NULL, _("Show popup menu"), "rmb_popup_radio");
	gtk_box_pack_start (GTK_BOX (cat_box), radio, FALSE, TRUE, 0);
	if (gnome_cmd_data_get_right_mouse_button_mode() == RIGHT_BUTTON_POPUPS_MENU)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio), TRUE);
	radio = create_radio (parent, get_radio_group(radio), _("Select files"), "rmb_sel_radio");
	gtk_container_add (GTK_CONTAINER (cat_box), radio);
	if (gnome_cmd_data_get_right_mouse_button_mode() == RIGHT_BUTTON_SELECTS)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio), TRUE);


	/* FilterType settings
	 */
	cat_box = create_vbox (parent, FALSE, 0);
	cat = create_category (parent, cat_box, _("Match filenames using"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, FALSE, TRUE, 0);

	radio = create_radio (parent, NULL, _("Shell syntax"), "ft_shell_radio");
	gtk_container_add (GTK_CONTAINER (cat_box), radio);
	if (gnome_cmd_data_get_filter_type () == FILTER_TYPE_FNMATCH)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio), TRUE);
	radio = create_radio (parent, get_radio_group(radio), _("Regex syntax"), "ft_regex_radio");
	gtk_container_add (GTK_CONTAINER (cat_box), radio);
	if (gnome_cmd_data_get_filter_type () == FILTER_TYPE_REGEX)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio), TRUE);


	
	/* Sort options
	 */
	cat_box = create_vbox (parent, FALSE, 0);
	cat = create_category (parent, cat_box, _("Sorting options"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, FALSE, TRUE, 0);

	check = create_check (parent, _("Case sensitive"), "case_sens_check");
	gtk_box_pack_start (GTK_BOX (cat_box), check, FALSE, TRUE, 0);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (check),
		gnome_cmd_data_get_case_sens_sort ());

	
	/* Directory options
	 */
	table = create_table (parent, 2, 2);
	cat = create_category (parent, table, _("Directory options"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, FALSE, TRUE, 0);

#ifdef HAVE_LIBFAM
	check = create_check (parent, _("Use FAM to detect file changes"), "use_fam");
	gtk_table_attach (GTK_TABLE(table), check, 0, 2, 0, 1, GTK_FILL, 0, 0, 0);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (check),
		gnome_cmd_data_get_use_fam ());
#endif //HAVE_LIBFAM
	label = create_label (parent, _("Directory cache size"));
	table_add (table, label, 0, 1, 0);
	spin = create_spin (parent, "dir_cache_size", 5, 50, gnome_cmd_data_get_dir_cache_size());
	table_add (table, spin, 1, 1, GTK_FILL);
	
	
	return frame;
}

static void
store_general_options (GnomeCmdOptionsDialog *dialog)
{
	GtkWidget *rmb_popup_radio =
		lookup_widget (GTK_WIDGET (dialog), "rmb_popup_radio");
	GtkWidget *ft_regex_radio =
		lookup_widget (GTK_WIDGET (dialog), "ft_regex_radio");
	GtkWidget *case_sens_check =
		lookup_widget (GTK_WIDGET (dialog), "case_sens_check");
#ifdef HAVE_LIBFAM
	GtkWidget *use_fam =
		lookup_widget (GTK_WIDGET (dialog), "use_fam");
#endif //HAVE_LIBFAM
	GtkWidget *dir_cache_size =
		lookup_widget (GTK_WIDGET (dialog), "dir_cache_size");

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (rmb_popup_radio)))
		gnome_cmd_data_set_right_mouse_button_mode (RIGHT_BUTTON_POPUPS_MENU);
	else
		gnome_cmd_data_set_right_mouse_button_mode (RIGHT_BUTTON_SELECTS);

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ft_regex_radio)))
		gnome_cmd_data_set_filter_type (FILTER_TYPE_REGEX);
	else
		gnome_cmd_data_set_filter_type (FILTER_TYPE_FNMATCH);
	
	gnome_cmd_data_set_case_sens_sort (
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (case_sens_check)));

#ifdef HAVE_LIBFAM
	gnome_cmd_data_set_use_fam (
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (use_fam)));
#endif //HAVE_LIBFAM
	gnome_cmd_data_set_dir_cache_size (
		gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (dir_cache_size)));
}



/***********************************************************************
 *
 *  The Data Representation tab
 *
 **********************************************************************/

static void
on_date_format_update                    (GtkButton *button,
										  GtkWidget *options_dialog)
{
    char *format;
	char s[256];
    time_t t;

    GtkWidget *format_entry, *test_label;

    format_entry = lookup_widget (options_dialog, "date_format_entry");
    test_label = lookup_widget (options_dialog, "date_format_test_label");
	
    format = gtk_entry_get_text (GTK_ENTRY (format_entry));

    t = time (NULL);
    strftime (s, sizeof(s), format, localtime (&t));

    gtk_label_set_text (GTK_LABEL (test_label), s);	
}

static GtkWidget*
create_representation_tab (GtkWidget *parent)
{
	GtkWidget *frame, *hbox, *vbox, *cat, *cat_box, *table;	
	GtkWidget *radio, *label, *entry, *button;

	frame = create_tabframe (parent);
	hbox = create_tabhbox (parent);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	vbox = create_tabvbox (parent);
	gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);	


	/* Size display mode
	 *
	 */
	cat_box = create_vbox (parent, FALSE, 0);
	cat = create_category (parent, cat_box, _("Size display mode"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, FALSE, TRUE, 0);

	radio = create_radio (parent, NULL, _("Powered"), "size_powered_radio");
	gtk_container_add (GTK_CONTAINER (cat_box), radio);
	if (gnome_cmd_data_get_size_disp_mode () == GNOME_CMD_SIZE_DISP_MODE_POWERED)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio), TRUE);

	radio = create_radio (parent, get_radio_group (radio), _("Grouped"), "size_grouped_radio");
	gtk_container_add (GTK_CONTAINER (cat_box), radio);
	if (gnome_cmd_data_get_size_disp_mode () == GNOME_CMD_SIZE_DISP_MODE_GROUPED)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio), TRUE);

	radio = create_radio (parent, get_radio_group (radio), _("Plain"), "size_plain_radio");
	gtk_container_add (GTK_CONTAINER (cat_box), radio);
	if (gnome_cmd_data_get_size_disp_mode () == GNOME_CMD_SIZE_DISP_MODE_PLAIN)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio), TRUE);


	/* Permission display mode
	 *
	 */
	cat_box = create_vbox (parent, FALSE, 0);
	cat = create_category (parent, cat_box, _("Permission display mode"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, FALSE, TRUE, 0);

	radio = create_radio (parent, NULL, _("Text (rw-r--r--)"), "perm_text_radio");
	gtk_container_add (GTK_CONTAINER (cat_box), radio);
	if (gnome_cmd_data_get_perm_disp_mode () == GNOME_CMD_PERM_DISP_MODE_TEXT)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio), TRUE);

	radio = create_radio (parent, get_radio_group (radio), _("Number (311)"), "perm_num_radio");
	gtk_container_add (GTK_CONTAINER (cat_box), radio);
	if (gnome_cmd_data_get_perm_disp_mode () == GNOME_CMD_PERM_DISP_MODE_NUMBER)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio), TRUE);

	
	/* Date options
	 * 
	 */
	table = create_table (parent, 2, 3);
	cat = create_category (parent, table, _("Date format"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, FALSE, FALSE, 0);
	
	label = create_label (parent, _("Format:"));
	table_add (table, label, 0, 0, GTK_FILL);

	entry = create_entry (parent, "date_format_entry", gnome_cmd_data_get_date_format());
	gtk_entry_set_text (GTK_ENTRY (entry), gnome_cmd_data_get_date_format ());
	gtk_widget_grab_focus (entry);
	table_add (table, entry, 1, 0, GTK_FILL);	
	
	button = create_button (parent, _("_Test"), GTK_SIGNAL_FUNC (on_date_format_update));
	table_add (table, button, 2, 0, GTK_FILL);

	label = create_label (parent, _("Test result:"));
	table_add (table, label, 0, 1, GTK_FILL);

	label = create_label (parent, "");
	gtk_object_set_data_full (GTK_OBJECT (parent), "date_format_test_label",
							  label, (GtkDestroyNotify)gtk_widget_unref);
	gtk_signal_connect (GTK_OBJECT (label), "realize",
						GTK_SIGNAL_FUNC (on_date_format_update), parent);					
	table_add (table, label, 1, 1, GTK_EXPAND|GTK_FILL);

	label = create_label (parent, _("See the manual page for \"strftime\" for help on how to set the format string."));
	gtk_label_set_line_wrap (GTK_LABEL (label), TRUE);
	table_add (table, label, 1, 2, GTK_FILL);

	return frame;
}

static void
store_representation_options (GnomeCmdOptionsDialog *dialog)
{
	GtkWidget *size_powered_radio = lookup_widget (
		GTK_WIDGET (dialog), "size_powered_radio");
	GtkWidget *size_grouped_radio = lookup_widget (
		GTK_WIDGET (dialog), "size_grouped_radio");
	GtkWidget *perm_text_radio = lookup_widget (
		GTK_WIDGET (dialog), "perm_text_radio");
	GtkWidget *entry = lookup_widget (
		GTK_WIDGET (dialog), "date_format_entry");	
	gchar *format = gtk_entry_get_text (GTK_ENTRY (entry));
	
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (size_powered_radio)))
		gnome_cmd_data_set_size_disp_mode (GNOME_CMD_SIZE_DISP_MODE_POWERED);
	else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (size_grouped_radio)))
		gnome_cmd_data_set_size_disp_mode (GNOME_CMD_SIZE_DISP_MODE_GROUPED);
	else
		gnome_cmd_data_set_size_disp_mode (GNOME_CMD_SIZE_DISP_MODE_PLAIN);

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (perm_text_radio)))
		gnome_cmd_data_set_perm_disp_mode (GNOME_CMD_PERM_DISP_MODE_TEXT);
	else
		gnome_cmd_data_set_perm_disp_mode (GNOME_CMD_PERM_DISP_MODE_NUMBER);

	gnome_cmd_data_set_date_format (format);
	g_free (format);
}



/***********************************************************************
 *
 *  The Layout tab
 *
 **********************************************************************/

static void
on_gfx_changed (GtkButton *btn, GtkWidget *dialog)
{
	GtkWidget *gfx_mime_radio, *icon_frame;
		
	gfx_mime_radio = lookup_widget (GTK_WIDGET (dialog), "gfx_mime_radio");
	icon_frame = lookup_widget (GTK_WIDGET (dialog), "mime_icon_settings_frame");

	if (icon_frame)
		gtk_widget_set_sensitive (
			icon_frame,
			gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfx_mime_radio)));
}

static GtkWidget*
create_layout_tab (GtkWidget *parent)
{
	GtkWidget *frame, *hbox, *vbox, *cat, *cat_box;
	GtkWidget *entry, *radio, *spin, *scale, *table, *label, *fpicker;

	frame = create_tabframe (parent);
	hbox = create_tabhbox (parent);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	vbox = create_tabvbox (parent);
	gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);	
	
	hbox = create_hbox (parent, TRUE, 6);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, TRUE, 0);

	
	/* File listings
	 *
	 */
	table = create_table (parent, 2, 2);
	gtk_table_set_homogeneous (GTK_TABLE (table), TRUE);
	cat = create_category (parent, table, _("File listings"));
	gtk_box_pack_start (GTK_BOX (hbox), cat, TRUE, TRUE, 0);
	
	fpicker = create_font_picker (parent, "list_font_picker");
	table_add (table, fpicker, 1, 0, (GtkAttachOptions)GTK_EXPAND|GTK_FILL);
	gnome_font_picker_set_font_name (
		GNOME_FONT_PICKER (fpicker),
		gnome_cmd_data_get_list_font ());
	gnome_font_picker_set_preview_text (
		GNOME_FONT_PICKER (fpicker),
		"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890");
	spin = create_spin (parent, "row_height_spin", 8, 64,
						gnome_cmd_data_get_list_row_height());
	table_add (table, spin, 1, 1, (GtkAttachOptions)GTK_EXPAND|GTK_FILL);

	label = create_label (parent, _("Font:"));
	table_add (table, label, 0, 0, (GtkAttachOptions)GTK_FILL);
	label = create_label (parent, _("Row height:"));
	table_add (table, label, 0, 1, (GtkAttachOptions)GTK_FILL);
	
	
	/* Main window
	 *
	 */
	table = create_table (parent, 1, 2);
	gtk_table_set_homogeneous (GTK_TABLE (table), TRUE);
	cat = create_category (parent, table, _("Main window"));
	gtk_box_pack_start (GTK_BOX (hbox), cat, TRUE, TRUE, 0);
	
	fpicker = create_font_picker (parent, "misc_font_picker");
	table_add (table, fpicker, 1, 0, (GtkAttachOptions)GTK_EXPAND|GTK_FILL);
	gnome_font_picker_set_font_name (
		GNOME_FONT_PICKER (fpicker),
		gnome_cmd_data_get_misc_font ());
	gnome_font_picker_set_preview_text (
		GNOME_FONT_PICKER (fpicker),
		"abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890");
	
	label = create_label (parent, _("Font:"));
	table_add (table, label, 0, 0, (GtkAttachOptions)GTK_FILL);

	
	/* Graphical mode
	 *
	 */
	cat_box = create_vbox (parent, FALSE, 0);
	cat = create_category (parent, cat_box, _("Graphical mode"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, FALSE, FALSE, 0);

	/*
	 * MIME icon settings
	 */
	table = create_table (parent, 4, 2);
	cat = create_category (parent, table, _("MIME icon settings"));
	gtk_object_set_data (GTK_OBJECT (parent), "mime_icon_settings_frame", cat);
	gtk_box_pack_start (GTK_BOX (vbox), cat, FALSE, FALSE, 0);

	
	/* Graphical mode
	 *
	 */
	radio = create_radio (
		parent, NULL, _("MIME icons"), "gfx_mime_radio");
	gtk_container_add (GTK_CONTAINER (cat_box), radio);
	gtk_signal_connect (GTK_OBJECT (radio), "clicked",
						GTK_SIGNAL_FUNC (on_gfx_changed), parent);
	if (gnome_cmd_data_get_layout () == GNOME_CMD_LAYOUT_MIME_ICONS)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio), TRUE);

	radio = create_radio (
		parent, get_radio_group (radio), _("File type icons"), "gfx_type_radio");
	gtk_container_add (GTK_CONTAINER (cat_box), radio);
	gtk_signal_connect (GTK_OBJECT (radio), "clicked",
						GTK_SIGNAL_FUNC (on_gfx_changed), parent);
	if (gnome_cmd_data_get_layout () == GNOME_CMD_LAYOUT_TYPE_ICONS)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio), TRUE);

	radio = create_radio (
		parent, get_radio_group (radio), _("Text (fastest)"), "gfx_text_radio");
	gtk_container_add (GTK_CONTAINER (cat_box), radio);
	gtk_signal_connect (GTK_OBJECT (radio), "clicked",
						GTK_SIGNAL_FUNC (on_gfx_changed), parent);
	if (gnome_cmd_data_get_layout () == GNOME_CMD_LAYOUT_TEXT)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (radio), TRUE);

	
	/*
	 * MIME icon settings
	 */
	spin = create_spin (
		parent, "iconsize_spin", 8, 64, gnome_cmd_data_get_icon_size());
	table_add (table, spin, 1, 0, (GtkAttachOptions)GTK_FILL);
	scale = create_scale (
		parent, "iconquality_scale", gnome_cmd_data_get_icon_scale_quality (), 0, 3);
	table_add (table, scale, 1, 1, (GtkAttachOptions)GTK_FILL);
	entry = create_file_entry (
		parent, "theme_icondir_entry", gnome_cmd_data_get_theme_icon_dir());
	table_add (table, entry, 1, 2, (GtkAttachOptions)0);
	entry = create_file_entry (
		parent, "doc_icondir_entry", gnome_cmd_data_get_document_icon_dir());
	table_add (table, entry, 1, 3, (GtkAttachOptions)0);
	
	label = create_label (parent, _("Icon size:"));
	table_add (table, label, 0, 0, (GtkAttachOptions)GTK_FILL);
	label = create_label (parent, _("Scaling quality:"));
	table_add (table, label, 0, 1, (GtkAttachOptions)GTK_FILL);
	label = create_label (parent, _("Theme icon directory:"));
	table_add (table, label, 0, 2, (GtkAttachOptions)GTK_FILL);	
	label = create_label (parent, _("Document icon directory:"));
	table_add (table, label, 0, 3, (GtkAttachOptions)GTK_FILL);	

	
	return frame;
}

static void
store_layout_options (GnomeCmdOptionsDialog *dialog)
{
	GtkWidget *gfx_text_radio, *gfx_type_radio;
	GtkWidget *list_font_picker, *misc_font_picker;
	gchar *list_font, *misc_font;
	GtkWidget *iconsize_spin = lookup_widget (
		GTK_WIDGET (dialog), "iconsize_spin");
	GtkWidget *iconquality_scale = lookup_widget (
		GTK_WIDGET (dialog), "iconquality_scale");
	GtkWidget *theme_icondir_entry = lookup_widget (
		GTK_WIDGET (dialog), "theme_icondir_entry");
	GtkWidget *doc_icondir_entry = lookup_widget (
		GTK_WIDGET (dialog), "doc_icondir_entry");
	GtkWidget *row_height_spin = lookup_widget (
		GTK_WIDGET (dialog), "row_height_spin");

	gfx_text_radio = lookup_widget (GTK_WIDGET (dialog),
									"gfx_text_radio");
	gfx_type_radio = lookup_widget (GTK_WIDGET (dialog),
									"gfx_type_radio");
	
	list_font_picker = lookup_widget (GTK_WIDGET (dialog), "list_font_picker");
	misc_font_picker = lookup_widget (GTK_WIDGET (dialog), "misc_font_picker");

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfx_text_radio)))
		gnome_cmd_data_set_layout (GNOME_CMD_LAYOUT_TEXT);
	else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (gfx_type_radio)))
		gnome_cmd_data_set_layout (GNOME_CMD_LAYOUT_TYPE_ICONS);
	else
		gnome_cmd_data_set_layout (GNOME_CMD_LAYOUT_MIME_ICONS);

	list_font = gnome_font_picker_get_font_name (
		GNOME_FONT_PICKER (list_font_picker));
	if (strcmp (list_font, "default") != 0)
		gnome_cmd_data_set_list_font (list_font);
	misc_font = gnome_font_picker_get_font_name (
		GNOME_FONT_PICKER (misc_font_picker));
	if (strcmp (misc_font, "default") != 0)
		gnome_cmd_data_set_misc_font (misc_font);

	gnome_cmd_data_set_theme_icon_dir (
		gtk_entry_get_text (GTK_ENTRY (theme_icondir_entry)));
	gnome_cmd_data_set_document_icon_dir (
		gtk_entry_get_text (GTK_ENTRY (doc_icondir_entry)));
	gnome_cmd_data_set_icon_size (
		gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (iconsize_spin)));

	{
		GtkAdjustment *adj;
		adj = gtk_range_get_adjustment (GTK_RANGE (iconquality_scale));
		gnome_cmd_data_set_icon_scale_quality (adj->value);
	}

	gnome_cmd_data_set_list_row_height (
		gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON (row_height_spin)));
}



/***********************************************************************
 *
 *  The Hide tab
 *
 **********************************************************************/

static GtkWidget*
create_colors_tab (GtkWidget *parent)
{
	GtkWidget *frame, *hbox, *vbox, *cat, *cat_box;
	GtkWidget *table, *label, *cpicker, *check;
	GnomeCmdLayoutColors *colors = gnome_cmd_data_get_layout_colors();

	frame = create_tabframe (parent);
	hbox = create_tabhbox (parent);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	vbox = create_tabvbox (parent);
	gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);	
	

	/* The color pickers
	 *
	 */
	cat_box = create_vbox (parent, FALSE, 12);
	cat = create_category (parent, cat_box, _("Colors"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, FALSE, FALSE, 0);
	
	table = create_table (parent, 4, 3);
	gtk_container_add (GTK_CONTAINER (cat_box), table);
	
	cpicker = create_color_picker (parent, "default_fg");
	table_add (table, cpicker, 1, 1, (GtkAttachOptions)0);
	gnome_color_picker_set_i16 (
		GNOME_COLOR_PICKER (cpicker),
		colors->norm_fg->red, colors->norm_fg->green, colors->norm_fg->blue, 0);
	cpicker = create_color_picker (parent, "default_bg");
	table_add (table, cpicker, 2, 1, (GtkAttachOptions)0);
	gnome_color_picker_set_i16 (
		GNOME_COLOR_PICKER (cpicker),
		colors->norm_bg->red, colors->norm_bg->green, colors->norm_bg->blue, 0);
	cpicker = create_color_picker (parent, "selected_fg");
	table_add (table, cpicker, 1, 2, (GtkAttachOptions)0);
	gnome_color_picker_set_i16 (
		GNOME_COLOR_PICKER (cpicker),
		colors->sel_fg->red, colors->sel_fg->green, colors->sel_fg->blue, 0);
	cpicker = create_color_picker (parent, "selected_bg");
	table_add (table, cpicker, 2, 2, (GtkAttachOptions)0);
	gnome_color_picker_set_i16 (
		GNOME_COLOR_PICKER (cpicker),
		colors->sel_bg->red, colors->sel_bg->green, colors->sel_bg->blue, 0);
	cpicker = create_color_picker (parent, "cursor_fg");
	table_add (table, cpicker, 1, 3, (GtkAttachOptions)0);
	gnome_color_picker_set_i16 (
		GNOME_COLOR_PICKER (cpicker),
		colors->curs_fg->red, colors->curs_fg->green, colors->curs_fg->blue, 0);
	cpicker = create_color_picker (parent, "cursor_bg");
	table_add (table, cpicker, 2, 3, (GtkAttachOptions)0);
	gnome_color_picker_set_i16 (
		GNOME_COLOR_PICKER (cpicker),
		colors->curs_bg->red, colors->curs_bg->green, colors->curs_bg->blue, 0);	

	label = create_label (parent, _("Foreground"));
	table_add (table, label, 1, 0, (GtkAttachOptions)GTK_FILL);
	label = create_label (parent, _("Background"));
	table_add (table, label, 2, 0, (GtkAttachOptions)GTK_FILL);
	label = create_label (parent, _("Default:"));
	table_add (table, label, 0, 1, (GtkAttachOptions)GTK_FILL);
	label = create_label (parent, _("Selected file:"));
	table_add (table, label, 0, 2, (GtkAttachOptions)GTK_FILL);
	label = create_label (parent, _("Cursor:"));
	table_add (table, label, 0, 3, (GtkAttachOptions)GTK_FILL);

	check = create_check (parent,
						  _("Use colors from the LS_COLORS enviroment variable"),
						  "use_ls_colors");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check),
								  gnome_cmd_data_get_use_ls_colors());
	gtk_container_add (GTK_CONTAINER (cat_box), check);

	return frame;
}

static void
store_colors_options (GnomeCmdOptionsDialog *dialog)
{
	GtkWidget *norm_fg = lookup_widget (GTK_WIDGET (dialog), "default_fg");
	GtkWidget *norm_bg = lookup_widget (GTK_WIDGET (dialog), "default_bg");
	GtkWidget *sel_fg = lookup_widget (GTK_WIDGET (dialog), "selected_fg");
	GtkWidget *sel_bg = lookup_widget (GTK_WIDGET (dialog), "selected_bg");
	GtkWidget *curs_fg = lookup_widget (GTK_WIDGET (dialog), "cursor_fg");
	GtkWidget *curs_bg = lookup_widget (GTK_WIDGET (dialog), "cursor_bg");
	GtkWidget *use_ls = lookup_widget (GTK_WIDGET (dialog), "use_ls_colors");
	GnomeCmdLayoutColors *colors = gnome_cmd_data_get_layout_colors();
	gushort a;
	
	gnome_color_picker_get_i16 (
		GNOME_COLOR_PICKER (norm_fg),
		&colors->norm_fg->red, &colors->norm_fg->green, &colors->norm_fg->blue, &a);
	gnome_color_picker_get_i16 (
		GNOME_COLOR_PICKER (norm_bg),
		&colors->norm_bg->red, &colors->norm_bg->green, &colors->norm_bg->blue, &a);
	gnome_color_picker_get_i16 (
		GNOME_COLOR_PICKER (sel_fg),
		&colors->sel_fg->red, &colors->sel_fg->green, &colors->sel_fg->blue, &a);
	gnome_color_picker_get_i16 (
		GNOME_COLOR_PICKER (sel_bg),
		&colors->sel_bg->red, &colors->sel_bg->green, &colors->sel_bg->blue, &a);
	gnome_color_picker_get_i16 (
		GNOME_COLOR_PICKER (curs_fg),
		&colors->curs_fg->red, &colors->curs_fg->green, &colors->curs_fg->blue, &a);
	gnome_color_picker_get_i16 (
		GNOME_COLOR_PICKER (curs_bg),
		&colors->curs_bg->red, &colors->curs_bg->green, &colors->curs_bg->blue, &a);	

	gnome_cmd_data_set_use_ls_colors (
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (use_ls)));
}


/***********************************************************************
 *
 *  The Filter tab
 *
 **********************************************************************/

static GtkWidget*
create_filter_tab (GtkWidget *parent)
{
	GtkWidget *frame;
	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkWidget *cat;
	GtkWidget *cat_box;
	GtkWidget *check;

	frame = create_tabframe (parent);
	hbox = create_tabhbox (parent);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	vbox = create_tabvbox (parent);
	gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);



	
	
	cat_box = create_vbox (parent, FALSE, 0);
	cat = create_category (parent, cat_box, _("Filetypes to hide"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, FALSE, FALSE, 0);
	
	check = create_check (parent, _("Unknown"), "hide_unknown_check");
	gtk_container_add (GTK_CONTAINER (cat_box), check);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (check),
		gnome_cmd_data_get_type_filter (GNOME_VFS_FILE_TYPE_UNKNOWN));
	check = create_check (parent, _("Regular files"), "hide_regular_check");
	gtk_container_add (GTK_CONTAINER (cat_box), check);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (check),
		gnome_cmd_data_get_type_filter (GNOME_VFS_FILE_TYPE_REGULAR));
	check = create_check (parent, _("Directories"), "hide_directory_check");
	gtk_container_add (GTK_CONTAINER (cat_box), check);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (check),
		gnome_cmd_data_get_type_filter (GNOME_VFS_FILE_TYPE_DIRECTORY));
	check = create_check (parent, _("Fifo files"), "hide_fifo_check");
	gtk_container_add (GTK_CONTAINER (cat_box), check);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (check),
		gnome_cmd_data_get_type_filter (GNOME_VFS_FILE_TYPE_FIFO));
	check = create_check (parent, _("Socket files"), "hide_socket_check");
	gtk_container_add (GTK_CONTAINER (cat_box), check);	
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (check),
		gnome_cmd_data_get_type_filter (GNOME_VFS_FILE_TYPE_SOCKET));
	check = create_check (parent, _("Character devices"), "hide_char_check");
	gtk_container_add (GTK_CONTAINER (cat_box), check);	
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (check),
		gnome_cmd_data_get_type_filter (GNOME_VFS_FILE_TYPE_CHARACTER_DEVICE));
	check = create_check (parent, _("Block devices"), "hide_block_check");
	gtk_container_add (GTK_CONTAINER (cat_box), check);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (check),
		gnome_cmd_data_get_type_filter (GNOME_VFS_FILE_TYPE_BLOCK_DEVICE));

	
	cat_box = create_vbox (parent, FALSE, 0);
	cat = create_category (parent, cat_box, _("Also hide"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, FALSE, FALSE, 0);

	check = create_check (parent, _("Hidden files"), "hide_hidden_check");
	gtk_container_add (GTK_CONTAINER (cat_box), check);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (check),
		gnome_cmd_data_get_hidden_filter ());
	check = create_check (parent, _("Symlinks"), "hide_symlink_check");
	gtk_container_add (GTK_CONTAINER (cat_box), check);
	gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (check),
		gnome_cmd_data_get_type_filter (GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK));
	
	return frame;
}

static void
store_filter_options (GnomeCmdOptionsDialog *dialog)
{
	GtkWidget *hide_unknown_check;
	GtkWidget *hide_directory_check;
	GtkWidget *hide_regular_check;
	GtkWidget *hide_socket_check;
	GtkWidget *hide_fifo_check;
	GtkWidget *hide_block_check;
	GtkWidget *hide_char_check;
	GtkWidget *hide_hidden_check;
	GtkWidget *hide_symlink_check;

	hide_unknown_check = lookup_widget (GTK_WIDGET (dialog), "hide_unknown_check");
	hide_regular_check = lookup_widget (GTK_WIDGET (dialog), "hide_regular_check");
	hide_directory_check = lookup_widget (GTK_WIDGET (dialog), "hide_directory_check");
	hide_fifo_check = lookup_widget (GTK_WIDGET (dialog), "hide_fifo_check");
	hide_socket_check = lookup_widget (GTK_WIDGET (dialog), "hide_socket_check");
	hide_char_check = lookup_widget (GTK_WIDGET (dialog), "hide_char_check");
	hide_block_check = lookup_widget (GTK_WIDGET (dialog), "hide_block_check");
	hide_symlink_check = lookup_widget (GTK_WIDGET (dialog), "hide_symlink_check");
	hide_hidden_check = lookup_widget (GTK_WIDGET (dialog), "hide_hidden_check");
	
	gnome_cmd_data_set_type_filter (
		GNOME_VFS_FILE_TYPE_UNKNOWN,
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (hide_unknown_check)));

	gnome_cmd_data_set_type_filter (
		GNOME_VFS_FILE_TYPE_REGULAR,
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (hide_regular_check)));
	
	gnome_cmd_data_set_type_filter (
		GNOME_VFS_FILE_TYPE_DIRECTORY,
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (hide_directory_check)));
	
	gnome_cmd_data_set_type_filter (
		GNOME_VFS_FILE_TYPE_FIFO,
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (hide_fifo_check)));
	
	gnome_cmd_data_set_type_filter (
		GNOME_VFS_FILE_TYPE_SOCKET,
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (hide_socket_check)));
	
	gnome_cmd_data_set_type_filter (
		GNOME_VFS_FILE_TYPE_CHARACTER_DEVICE,
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (hide_char_check)));
	
	gnome_cmd_data_set_type_filter (
		GNOME_VFS_FILE_TYPE_BLOCK_DEVICE,
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (hide_block_check)));
	
	gnome_cmd_data_set_type_filter (
		GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK,
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (hide_symlink_check)));
	
	gnome_cmd_data_set_hidden_filter (
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (hide_hidden_check)));
}



/***********************************************************************
 *
 *  The Programs tab
 *
 **********************************************************************/

static void
add_app_to_list (GtkCList *clist, GnomeCmdApp *app)
{
	gchar *text[2];
	gint row;
	GnomeCmdPixmap *pm;

	pm = gnome_cmd_app_get_pixmap (app);

	text[0] = NULL;
	text[1] = (gchar*)gnome_cmd_app_get_name (app);
	
	row = gtk_clist_append (GTK_CLIST (clist), text);
	if (pm)
		gtk_clist_set_pixmap (GTK_CLIST (clist), row, 0, pm->pixmap, pm->mask);
	gtk_clist_set_row_data (GTK_CLIST (clist), row, app);
}

static void
update_app_in_list (GtkCList *clist, GnomeCmdApp *app)
{
	gint row;
	GnomeCmdPixmap *pm;

	pm = gnome_cmd_app_get_pixmap (app);
	row = gtk_clist_find_row_from_data (clist, app);
	if (pm)
		gtk_clist_set_pixmap (GTK_CLIST (clist), row, 0, pm->pixmap, pm->mask);

	gtk_clist_set_text (clist, row, 1, gnome_cmd_app_get_name (app));
}

static void
on_app_dialog_cancel (GtkButton *button, GtkWidget *dialog)
{
	gtk_widget_destroy (dialog);
}

static void
on_some_files_toggled (GtkToggleButton *btn, GtkWidget *dialog)
{
	GtkWidget *pattern_entry = lookup_widget (dialog, "pattern_entry");

	if (gtk_toggle_button_get_active (btn)) {
		gtk_widget_set_sensitive (pattern_entry, TRUE);
		gtk_widget_grab_focus (pattern_entry);
	}
	else
		gtk_widget_set_sensitive (pattern_entry, FALSE);	
}

static void
get_app_dialog_values (GtkWidget *dialog, gchar **name, gchar **cmd, gchar **icon_path,
					   gint *target, gchar **pattern_string,
					   gboolean *handles_uris, gboolean *handles_multiple,
					   gboolean *requires_terminal)
{
	GtkWidget *name_entry = lookup_widget (dialog, "name_entry");
	GtkWidget *cmd_entry = lookup_widget (dialog, "cmd_entry");
	GtkWidget *icon_entry = lookup_widget (dialog, "icon_entry");	
	GtkWidget *pattern_entry = lookup_widget (dialog, "pattern_entry");	
	GtkWidget *target_files = lookup_widget (dialog, "show_for_all_files");
	GtkWidget *target_dirs = lookup_widget (dialog, "show_for_all_dirs");
	GtkWidget *target_dirs_and_files = lookup_widget (dialog, "show_for_all_dirs_and_files");
	GtkWidget *uris_check = lookup_widget (dialog, "handle_uris");
	GtkWidget *multiple_check = lookup_widget (dialog, "handle_multiple");
	GtkWidget *terminal_check = lookup_widget (dialog, "requires_terminal");

	*name = gtk_entry_get_text (GTK_ENTRY (name_entry));
	*cmd = gtk_entry_get_text (GTK_ENTRY (cmd_entry));
	*icon_path = gnome_icon_entry_get_filename (GNOME_ICON_ENTRY (icon_entry));
	*pattern_string = NULL;
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (target_files)))
		*target = APP_TARGET_ALL_FILES;
	else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (target_dirs)))
		*target = APP_TARGET_ALL_DIRS;
	else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (target_dirs_and_files)))
		*target = APP_TARGET_ALL_DIRS_AND_FILES;
	else {
		*target = APP_TARGET_SOME_FILES;
		*pattern_string = gtk_entry_get_text (GTK_ENTRY (pattern_entry));
	}

	*handles_uris = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (uris_check));
	*handles_multiple = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (multiple_check));
	*requires_terminal = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (terminal_check));
}

static void
on_add_app_dialog_ok (GtkButton *button, GtkWidget *dialog)
{
	GnomeCmdApp *app;
	gint target;
	gboolean handles_uris, handles_multiple, requires_terminal;
	gchar *name, *cmd, *icon_path, *pattern_string;

	GtkWidget *options_dialog = lookup_widget (dialog, "options_dialog");
	GtkWidget *clist = lookup_widget (options_dialog, "app_clist");
	
	get_app_dialog_values (dialog, &name, &cmd, &icon_path,
						   &target, &pattern_string,
						   &handles_uris, &handles_multiple, &requires_terminal);
	if (!name || strlen (name) < 1) return;
	
	app = gnome_cmd_app_new_with_values (name, cmd, icon_path,
										 target, pattern_string,
										 handles_uris, handles_multiple, requires_terminal);
	gnome_cmd_data_add_fav_app (app);
	add_app_to_list (GTK_CLIST (clist), app);
	gtk_widget_destroy (dialog);

	g_free (name);
	g_free (cmd);
	g_free (icon_path);
	g_free (pattern_string);
}

static void
on_edit_app_dialog_ok (GtkButton *button, GtkWidget *dialog)
{
	GnomeCmdApp *app;
	gint target;
	gboolean handles_uris, handles_multiple, requires_terminal;
	gchar *name, *cmd, *icon_path, *pattern_string;

	GtkWidget *options_dialog = lookup_widget (dialog, "options_dialog");
	GtkWidget *clist = lookup_widget (options_dialog, "app_clist");
	
	get_app_dialog_values (dialog, &name, &cmd, &icon_path,
						   &target, &pattern_string,
						   &handles_uris, &handles_multiple, &requires_terminal);
	if (!name || strlen (name) < 1) return;
	
	app = (GnomeCmdApp*)gtk_object_get_data (GTK_OBJECT (options_dialog), "selected_app");
	if (!app) return;

	gnome_cmd_app_set_name (app, name);
	gnome_cmd_app_set_command (app, cmd);
	gnome_cmd_app_set_icon_path (app, icon_path);
	gnome_cmd_app_set_target (app, target);
	if (pattern_string)
		gnome_cmd_app_set_pattern_string (app, pattern_string);
	gnome_cmd_app_set_handles_uris (app, handles_uris);
	gnome_cmd_app_set_handles_multiple (app, handles_multiple);
	gnome_cmd_app_set_requires_terminal (app, requires_terminal);

	update_app_in_list (GTK_CLIST (clist), app);
	gtk_widget_destroy (dialog);

	g_free (name);
	g_free (cmd);
	g_free (icon_path);
	g_free (pattern_string);
}

static GtkWidget *
create_app_dialog (GnomeCmdApp *app, GtkSignalFunc on_ok, GtkSignalFunc on_cancel,
				   GtkWidget *options_dialog)
{
	GtkWidget *frame, *vbox, *hbox, *table, *entry, *label, *cat, *radio, *check;
	GtkWidget *dialog = create_ok_cancel_dialog ("", on_ok, on_cancel, NULL);
	const gchar *s = NULL;

	gtk_object_set_data (GTK_OBJECT (dialog), "options_dialog", options_dialog);	

	frame = create_space_frame (dialog, 6);
	gtk_container_add (GTK_CONTAINER (GNOME_DIALOG (dialog)->vbox), frame);
	hbox = create_hbox (dialog, FALSE, 6);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	vbox = create_vbox (dialog, FALSE, 6);
	gtk_container_add (GTK_CONTAINER (hbox), vbox);
	table = create_table (dialog, 3, 2);

	
	gtk_container_add (GTK_CONTAINER (vbox), table);

	label = create_label (dialog, _("Label:"));
	table_add (table, label, 0, 0, GTK_FILL);
	label = create_label (dialog, _("Command:"));
	table_add (table, label, 0, 1, GTK_FILL);
	label = create_label (dialog, _("Icon:"));
	table_add (table, label, 0, 2, GTK_FILL);

	if (app) s = gnome_cmd_app_get_name (app);
	entry = create_entry (dialog, "name_entry", s);
	table_add (table, entry, 1, 0, GTK_EXPAND|GTK_FILL);
	gtk_widget_grab_focus (entry);
	
	if (app) s = gnome_cmd_app_get_command (app);
	entry = create_entry (dialog, "cmd_entry", s);
	table_add (table, entry, 1, 1, GTK_EXPAND|GTK_FILL);
	
	if (app) s = gnome_cmd_app_get_icon_path (app);
	entry = create_icon_entry (dialog, "icon_entry", s);
	table_add (table, entry, 1, 2, GTK_FILL);

	
	table = create_table (dialog, 3, 1);
	cat = create_category (dialog, table, _("Options"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, FALSE, TRUE, 0);

	check = create_check (dialog, _("Can handle multiple files"), "handle_multiple");
	if (app) gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (check), gnome_cmd_app_get_handles_multiple (app));
	table_add (table, check, 0, 0, GTK_FILL);
	check = create_check (dialog, _("Can handle URIs"), "handle_uris");
	if (app) gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (check), gnome_cmd_app_get_handles_uris (app));
	table_add (table, check, 0, 1, GTK_FILL);
	check = create_check (dialog, _("Requires terminal"), "requires_terminal");
	if (app) gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (check), gnome_cmd_app_get_requires_terminal (app));
	table_add (table, check, 0, 2, GTK_FILL);

	
	table = create_table (dialog, 4, 2);
	cat = create_category (dialog, table, _("Show for"));
	gtk_box_pack_start (GTK_BOX (hbox), cat, FALSE, TRUE, 0);

	radio = create_radio (dialog, NULL, _("All files"),
						  "show_for_all_files");
	if (app) gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (radio),
		gnome_cmd_app_get_target (app) == APP_TARGET_ALL_FILES);
	gtk_table_attach (GTK_TABLE (table), radio, 0, 2, 0, 1, GTK_FILL, 0, 0, 0);
	radio = create_radio (dialog, get_radio_group(radio), _("All directories"),
						  "show_for_all_dirs");
	if (app) gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (radio),
		gnome_cmd_app_get_target (app) == APP_TARGET_ALL_DIRS);
	gtk_table_attach (GTK_TABLE (table), radio, 0, 2, 1, 2, GTK_FILL, 0, 0, 0);
	radio = create_radio (dialog, get_radio_group(radio), _("All directories and files"),
						  "show_for_all_dirs_and_files");
	if (app) gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (radio),
		gnome_cmd_app_get_target (app) == APP_TARGET_ALL_DIRS_AND_FILES);
	gtk_table_attach (GTK_TABLE (table), radio, 0, 2, 2, 3, GTK_FILL, 0, 0, 0);
	radio = create_radio (dialog, get_radio_group(radio), _("Some files"),
						  "show_for_some_files");
	if (app) gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (radio),
		gnome_cmd_app_get_target (app) == APP_TARGET_SOME_FILES);
	gtk_table_attach (GTK_TABLE (table), radio, 0, 2, 3, 4, GTK_FILL, 0, 0, 0);
	if (!app) gtk_toggle_button_set_active (
		GTK_TOGGLE_BUTTON (radio), TRUE);
	gtk_signal_connect (GTK_OBJECT (radio), "toggled",
						GTK_SIGNAL_FUNC (on_some_files_toggled), dialog);
	
	label = create_label (dialog, _("File patterns"));
	table_add (table, label, 0, 4, 0);
	entry = create_entry (dialog, "pattern_entry",
						  app?gnome_cmd_app_get_pattern_string (app):"*.ext1;*.ext2");
	table_add (table, entry, 1, 4, GTK_FILL);
	if (app && gnome_cmd_app_get_target (app) != APP_TARGET_SOME_FILES)
		gtk_widget_set_sensitive (entry, FALSE);

	return dialog;
}

static void
on_app_add (GtkWidget *button, GtkWidget *parent)
{
	GtkWidget *dialog = create_app_dialog (
		NULL, on_add_app_dialog_ok, on_app_dialog_cancel, parent);
	gtk_window_set_title (GTK_WINDOW (dialog), _("New Application"));
}

static void
on_app_edit (GtkWidget *button, GtkWidget *parent)
{
	GnomeCmdApp *app = (GnomeCmdApp*)gtk_object_get_data (
		GTK_OBJECT (parent), "selected_app");
	if (app) {
		GtkWidget *dialog = create_app_dialog (
			app, on_edit_app_dialog_ok, on_app_dialog_cancel, parent);
		gtk_window_set_title (GTK_WINDOW (dialog), _("Edit Application"));
	}
}

static void
on_app_selected (GtkCList *clist, gint row, gint column,
				 GdkEventButton *event, GtkWidget *parent)
{
	GnomeCmdApp *app;
	GtkWidget *remove_button = lookup_widget (parent, "remove_app_button");
	GtkWidget *edit_button = lookup_widget (parent, "edit_app_button");

	app = (GnomeCmdApp*)gtk_clist_get_row_data (clist, row);
	gtk_object_set_data (GTK_OBJECT (parent), "selected_app", app);
	
	gtk_widget_set_sensitive (remove_button, TRUE);
	gtk_widget_set_sensitive (edit_button, TRUE);
}

static void
on_app_moved (GtkCList *clist, gint arg1, gint arg2, GtkWidget *frame)
{
	GList *apps = gnome_cmd_data_get_fav_apps ();
	gpointer data;

	if (!apps
		|| MAX (arg1, arg2) >= g_list_length (apps)
		|| MIN (arg1, arg2) < 0
		|| arg1 == arg2)
		return;

	data = g_list_nth_data (apps, arg1);
	apps = g_list_remove (apps, data);
	
	apps = g_list_insert (apps, data, arg2);

	gnome_cmd_data_set_fav_apps (apps);
}

static void
on_app_remove (GtkWidget *button, GtkWidget *frame)
{
	GnomeCmdApp *app;
	GtkCList *clist = GTK_CLIST (lookup_widget (frame, "app_clist"));

	if (clist->focus_row >= 0) {
		app = (GnomeCmdApp*)gtk_clist_get_row_data (clist, clist->focus_row);
		gnome_cmd_data_remove_fav_app (app);
		gtk_clist_remove (clist, clist->focus_row);
	}
}

static void
on_app_move_up (GtkWidget *button, GtkWidget *frame)
{
	GtkCList *clist = GTK_CLIST (lookup_widget (frame, "app_clist"));

	if (clist->focus_row >= 1) {
		gtk_clist_row_move (clist, clist->focus_row, clist->focus_row-1);
	}
}

static void
on_app_move_down (GtkWidget *button, GtkWidget *frame)
{
	GtkCList *clist = GTK_CLIST (lookup_widget (frame, "app_clist"));

	if (clist->focus_row >= 0) {
		gtk_clist_row_move (clist, clist->focus_row, clist->focus_row+1);
	}
}

static GtkWidget*
create_programs_tab (GtkWidget *parent)
{
	GtkWidget *frame, *hbox, *vbox, *cat, *table;	
	GtkWidget *entry, *button, *label, *clist, *bbox, *check;
	GList *apps;
	
	frame = create_tabframe (parent);
	hbox = create_tabhbox (parent);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	vbox = create_tabvbox (parent);
	gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);	


	check = create_check (parent, _("Always download remote files before opening in external programs"),
						  "honor_expect_uris");
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check),
								  !gnome_cmd_data_get_honor_expect_uris ());
	cat = create_category (parent, check, _("MIME applications"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, FALSE, FALSE, 0);
	
	table = create_table (parent, 4, 2);
	cat = create_category (parent, table, _("Standard programs"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, FALSE, FALSE, 0);

	label = create_label (parent, _("Viewer:"));
	table_add (table, label, 0, 0, GTK_FILL);
	label = create_label (parent, _("Editor:"));
	table_add (table, label, 0, 1, GTK_FILL);
	label = create_label (parent, _("Differ:"));
	table_add (table, label, 0, 2, GTK_FILL);
	label = create_label (parent, _("Terminal:"));
	table_add (table, label, 0, 3, GTK_FILL);
	
	entry = create_entry (parent, "viewer", gnome_cmd_data_get_viewer());	
	table_add (table, entry, 1, 0, GTK_EXPAND|GTK_FILL);
	entry = create_entry (parent, "editor", gnome_cmd_data_get_editor());
	table_add (table, entry, 1, 1, GTK_EXPAND|GTK_FILL);
	entry = create_entry (parent, "differ", gnome_cmd_data_get_differ());
	table_add (table, entry, 1, 2, GTK_EXPAND|GTK_FILL);
	entry = create_entry (parent, "term", gnome_cmd_data_get_term());
	table_add (table, entry, 1, 3, GTK_EXPAND|GTK_FILL);

	
	/*
	 * Other favorite apps frame
	 */

	hbox = create_hbox (parent, FALSE, 0);
	cat = create_category (parent, hbox, _("Other favorite apps"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, FALSE, TRUE, 0);


	clist = create_clist (parent, "app_clist", 2, 16, on_app_selected, on_app_moved);
	create_clist_column (clist, 0, 20, "");
	create_clist_column (clist, 1, 150, _("Label"));
	gtk_box_pack_start (GTK_BOX (hbox), clist, TRUE, TRUE, 0);

	bbox = create_vbuttonbox (parent);
	gtk_box_pack_start (GTK_BOX (hbox), bbox, FALSE, TRUE, 0);
	
	button = create_button (parent, _("_Add..."), on_app_add);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_container_add (GTK_CONTAINER (bbox), button);

	button = create_button (parent, _("_Edit..."), on_app_edit);
	gtk_object_set_data (GTK_OBJECT (parent), "edit_app_button", button);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_widget_set_sensitive (button, FALSE);
	gtk_container_add (GTK_CONTAINER (bbox), button);

	button = create_button (parent, _("_Remove"), on_app_remove);
	gtk_object_set_data (GTK_OBJECT (parent), "remove_app_button", button);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_widget_set_sensitive (button, FALSE);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	
	button = create_stock_button (
		parent, GNOME_STOCK_BUTTON_UP, GTK_SIGNAL_FUNC (on_app_move_up));
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_container_add (GTK_CONTAINER (bbox), button);

	button = create_stock_button (
		parent, GNOME_STOCK_BUTTON_DOWN, GTK_SIGNAL_FUNC (on_app_move_down));
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_container_add (GTK_CONTAINER (bbox), button);

	
	clist = gtk_object_get_data (GTK_OBJECT (parent), "app_clist");
	apps = gnome_cmd_data_get_fav_apps ();
	while (apps) {
		GnomeCmdApp *app = (GnomeCmdApp*)apps->data;
		add_app_to_list (GTK_CLIST (clist), app);
		apps = apps->next;
	}
	
	
	return frame; 
}

static void
store_programs_options (GnomeCmdOptionsDialog *dialog)
{
	GtkWidget *entry1 = lookup_widget (GTK_WIDGET (dialog), "viewer");
	GtkWidget *entry2 = lookup_widget (GTK_WIDGET (dialog), "editor");
	GtkWidget *entry3 = lookup_widget (GTK_WIDGET (dialog), "differ");
	GtkWidget *entry5 = lookup_widget (GTK_WIDGET (dialog), "term");
	GtkWidget *check = lookup_widget (GTK_WIDGET (dialog), "honor_expect_uris");

	gnome_cmd_data_set_viewer (gtk_entry_get_text (GTK_ENTRY (entry1)));
	gnome_cmd_data_set_editor (gtk_entry_get_text (GTK_ENTRY (entry2)));
	gnome_cmd_data_set_differ (gtk_entry_get_text (GTK_ENTRY (entry3)));
	gnome_cmd_data_set_term (gtk_entry_get_text (GTK_ENTRY (entry5)));

	gnome_cmd_data_set_honor_expect_uris (
		!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (check)));
}



/***********************************************************************
 *
 *  The Devices tab
 *
 **********************************************************************/

static void
add_device_to_list (GtkCList *clist, GnomeCmdDevice *dev)
{
	gchar *text[2];
	gint row;
	GnomeCmdPixmap *pm;

	pm = gnome_cmd_device_get_pixmap (dev);

	text[0] = NULL;
	text[1] = (gchar*)gnome_cmd_device_get_alias (dev);
	
	row = gtk_clist_append (GTK_CLIST (clist), text);
	if (pm)
		gtk_clist_set_pixmap (GTK_CLIST (clist), row, 0, pm->pixmap, pm->mask);
	gtk_clist_set_row_data (GTK_CLIST (clist), row, dev);
}

static void
update_device_in_list (GtkCList *clist, GnomeCmdDevice *dev,
					   gchar *alias, gchar *device_fn, gchar *mountp, gchar *icon_path)
{
	gint row;
	GnomeCmdPixmap *pm;

	gnome_cmd_device_set_alias (dev, alias);
	gnome_cmd_device_set_device_fn (dev, device_fn);
	gnome_cmd_device_set_mountp (dev, mountp);
	gnome_cmd_device_set_icon_path (dev, icon_path);

	pm = gnome_cmd_device_get_pixmap (dev);
	row = gtk_clist_find_row_from_data (clist, dev);
	gtk_clist_set_text (GTK_CLIST (clist), row, 1, alias);
	gtk_clist_set_text (GTK_CLIST (clist), row, 2, device_fn);
	gtk_clist_set_text (GTK_CLIST (clist), row, 3, mountp);

	if (pm)
		gtk_clist_set_pixmap (GTK_CLIST (clist), row, 0, pm->pixmap, pm->mask);
	else
		gtk_clist_set_pixmap (GTK_CLIST (clist), row, 0, NULL, NULL);
}

static void
on_device_dialog_cancel (GtkButton *button, GtkWidget *dialog)
{
	gtk_widget_destroy (dialog);
}

static void
get_device_dialog_values (GtkWidget *dialog, gchar **alias, gchar **device,
						  gchar **mountp, gchar **icon_path)
{
	GtkWidget *alias_entry = lookup_widget (dialog, "alias_entry");
	GtkWidget *device_entry = lookup_widget (dialog, "device_entry");
	GtkWidget *mountp_entry = lookup_widget (dialog, "mountp_entry");
	GtkWidget *icon_entry = lookup_widget (dialog, "device_iconentry");

	*alias = gtk_entry_get_text (GTK_ENTRY (alias_entry));
	*device = gtk_entry_get_text (GTK_ENTRY (device_entry));
	*mountp = gtk_entry_get_text (GTK_ENTRY (mountp_entry));
	*icon_path = gnome_icon_entry_get_filename (GNOME_ICON_ENTRY (icon_entry));
}

static void
on_add_device_dialog_ok (GtkButton *button, GtkWidget *dialog)
{
	GnomeCmdDevice *dev;
	gchar *alias, *device, *mountp, *icon_path;

	GtkWidget *options_dialog = lookup_widget (dialog, "options_dialog");
	GtkWidget *clist = lookup_widget (options_dialog, "device_clist");
	
	get_device_dialog_values (dialog, &alias, &device, &mountp, &icon_path);
	if (!alias || strlen (alias) < 1) return;
	
	dev = gnome_cmd_device_new_with_values (alias, device, mountp, icon_path);
	gnome_cmd_data_add_device (dev);
	add_device_to_list (GTK_CLIST (clist), dev);
	gtk_widget_destroy (dialog);

	gnome_cmd_connection_device_added (dev);

	g_free (alias);
	g_free (device);
	g_free (mountp);
	g_free (icon_path);
}

static void
on_edit_device_dialog_ok (GtkButton *button, GtkWidget *dialog)
{
	GnomeCmdDevice *dev;
	gchar *alias, *device, *mountp, *icon_path;

	GtkWidget *options_dialog = lookup_widget (dialog, "options_dialog");
	GtkWidget *clist = lookup_widget (options_dialog, "device_clist");
	
	get_device_dialog_values (dialog, &alias, &device, &mountp, &icon_path);
	if (!alias || strlen (alias) < 1) return;
	
	dev = (GnomeCmdDevice*)gtk_object_get_data (GTK_OBJECT (options_dialog), "selected_device");
	if (!dev) return;

	update_device_in_list (GTK_CLIST (clist), dev, alias, device, mountp, icon_path);
	gtk_widget_destroy (dialog);

	g_free (alias);
	g_free (device);
	g_free (mountp);
	g_free (icon_path);
}

static GtkWidget *
create_device_dialog (GnomeCmdDevice *dev, GtkSignalFunc on_ok, GtkSignalFunc on_cancel,
					  GtkWidget *options_dialog)
{
	GtkWidget *frame, *table, *entry, *label;
	GtkWidget *dialog = create_ok_cancel_dialog ("", on_ok, on_cancel, NULL);
	const gchar *s = NULL;
	
	gtk_window_set_title (GTK_WINDOW (dialog), "");
	gtk_object_set_data (GTK_OBJECT (dialog), "options_dialog", options_dialog);	

	frame = create_space_frame (dialog, 6);
	gtk_container_add (GTK_CONTAINER (GNOME_DIALOG (dialog)->vbox), frame);
	table = create_table (dialog, 4, 2);
	gtk_container_add (GTK_CONTAINER (frame), table);

	label = create_label (dialog, _("Alias:"));
	table_add (table, label, 0, 0, GTK_FILL);
	label = create_label (dialog, _("Device:"));
	table_add (table, label, 0, 1, GTK_FILL);
	label = create_label (dialog, _("Mount point:"));
	table_add (table, label, 0, 2, GTK_FILL);
	label = create_label (dialog, _("Icon:"));
	table_add (table, label, 0, 3, GTK_FILL);

	if (dev) s = gnome_cmd_device_get_alias (dev);
	entry = create_entry (dialog, "alias_entry", s);
	table_add (table, entry, 1, 0, GTK_EXPAND|GTK_FILL);
	gtk_widget_grab_focus (entry);

	if (dev) s = gnome_cmd_device_get_device_fn (dev);
	entry = create_file_entry (dialog, "device_entry", s);
	table_add (table, entry, 1, 1, GTK_EXPAND|GTK_FILL);
	
	if (dev) s = gnome_cmd_device_get_mountp (dev);
	entry = create_file_entry (dialog, "mountp_entry", s);
	table_add (table, entry, 1, 2, GTK_EXPAND|GTK_FILL);
	
	if (dev) s = gnome_cmd_device_get_icon_path (dev);
	entry = create_icon_entry (dialog, "device_iconentry", s);
	table_add (table, entry, 1, 3, GTK_FILL);

	return dialog;
}

static void
on_device_add (GtkWidget *button, GtkWidget *parent)
{
	GtkWidget *dialog = create_device_dialog (
		NULL, on_add_device_dialog_ok, on_device_dialog_cancel, parent);
	gtk_window_set_title (GTK_WINDOW (dialog), _("New Device"));
}

static void
on_device_edit (GtkWidget *button, GtkWidget *parent)
{
	GnomeCmdDevice *dev = (GnomeCmdDevice*)gtk_object_get_data (
		GTK_OBJECT (parent), "selected_device");
	if (dev) {
		GtkWidget *dialog = create_device_dialog (
			dev, on_edit_device_dialog_ok, on_device_dialog_cancel, parent);
		gtk_window_set_title (GTK_WINDOW (dialog), _("Edit Device"));
	}
}

static void
on_device_remove (GtkWidget *button, GtkWidget *frame)
{
	GnomeCmdDevice *dev;
	GtkCList *clist = GTK_CLIST (lookup_widget (frame, "device_clist"));

	if (clist->focus_row >= 0) {
		dev = (GnomeCmdDevice*)gtk_clist_get_row_data (clist, clist->focus_row);
		gnome_cmd_data_remove_device (dev);
		gtk_clist_remove (clist, clist->focus_row);
		gnome_cmd_connection_device_removed (dev);
	}
}

static void
on_device_selected (GtkCList *clist, gint row, gint column,
				    GdkEventButton *event, GtkWidget *parent)
{
	GnomeCmdDevice *dev;	
	GtkWidget *remove_button = lookup_widget (parent, "remove_device_button");
	GtkWidget *edit_button = lookup_widget (parent, "edit_device_button");

	dev = (GnomeCmdDevice*)gtk_clist_get_row_data (clist, row);
	gtk_object_set_data (GTK_OBJECT (parent), "selected_device", dev);
	
	gtk_widget_set_sensitive (remove_button, TRUE);
	gtk_widget_set_sensitive (edit_button, TRUE);
}

static void
on_device_moved (GtkCList *clist, gint arg1, gint arg2, GtkWidget *frame)
{
	GList *list = gnome_cmd_data_get_devices ();
	gpointer data;

	if (!list
		|| MAX (arg1, arg2) >= g_list_length (list)
		|| MIN (arg1, arg2) < 0
		|| arg1 == arg2)
		return;

	data = g_list_nth_data (list, arg1);
	list = g_list_remove (list, data);
	
	list = g_list_insert (list, data, arg2);

	gnome_cmd_data_set_devices (list);
}

static void
on_device_move_up (GtkWidget *button, GtkWidget *frame)
{
	GtkCList *clist = GTK_CLIST (lookup_widget (frame, "device_clist"));

	gtk_clist_row_move (clist, clist->focus_row, clist->focus_row-1);
}

static void
on_device_move_down (GtkWidget *button, GtkWidget *frame)
{
	GtkCList *clist = GTK_CLIST (lookup_widget (frame, "device_clist"));

	gtk_clist_row_move (clist, clist->focus_row, clist->focus_row+1);	
}

static GtkWidget *
create_devices_tab (GtkWidget *parent)
{
	GtkWidget *frame, *hbox, *vbox, *cat, *cat_box;	
	GtkWidget *button, *clist, *bbox, *check;
	GList *devices;
	
	frame = create_tabframe (parent);
	hbox = create_tabhbox (parent);
	gtk_container_add (GTK_CONTAINER (frame), hbox);
	vbox = create_tabvbox (parent);
	gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);	

	
	cat_box = create_vbox (parent, FALSE, 0);
	cat = create_category (parent, cat_box, _("Devices"));
	gtk_box_pack_start (GTK_BOX (vbox), cat, FALSE, TRUE, 0);

	hbox = create_hbox (parent, FALSE, 0);
	gtk_container_add (GTK_CONTAINER (cat_box), hbox);

	clist = create_clist (parent, "device_clist", 2, 24, on_device_selected, on_device_moved);
	create_clist_column (clist, 0, 26, "");
	create_clist_column (clist, 1, 40, _("Alias"));
	gtk_box_pack_start (GTK_BOX (hbox), clist, TRUE, TRUE, 0);

	bbox = create_vbuttonbox (parent);
	gtk_box_pack_start (GTK_BOX (hbox), bbox, FALSE, TRUE, 0);
	
	button = create_button (parent, _("_Add..."), on_device_add);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_container_add (GTK_CONTAINER (bbox), button);

	button = create_button (parent, _("_Edit..."), on_device_edit);
	gtk_object_set_data (GTK_OBJECT (parent), "edit_device_button", button);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_widget_set_sensitive (button, FALSE);
	gtk_container_add (GTK_CONTAINER (bbox), button);

	button = create_button (parent, _("_Remove"), on_device_remove);
	gtk_object_set_data (GTK_OBJECT (parent), "remove_device_button", button);
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_widget_set_sensitive (button, FALSE);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	
	button = create_stock_button (
		parent, GNOME_STOCK_BUTTON_UP, GTK_SIGNAL_FUNC (on_device_move_up));
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_container_add (GTK_CONTAINER (bbox), button);

	button = create_stock_button (
		parent, GNOME_STOCK_BUTTON_DOWN, GTK_SIGNAL_FUNC (on_device_move_down));
	GTK_WIDGET_SET_FLAGS (button, GTK_CAN_DEFAULT);
	gtk_container_add (GTK_CONTAINER (bbox), button);

	check = create_check (parent, _("Show only the icons"), "device_only_icon");
	gtk_container_add (GTK_CONTAINER (cat_box), check);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check),
								  gnome_cmd_data_get_device_only_icon());
  
	check = create_check (parent, _("Skip mounting (useful when using super-mount)"),
						  "skip_mounting");
	gtk_container_add (GTK_CONTAINER (cat_box), check);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (check),
								  gnome_cmd_data_get_skip_mounting());
  

	clist = gtk_object_get_data (GTK_OBJECT (parent), "device_clist");
	devices = gnome_cmd_data_get_devices ();
	while (devices) {
		GnomeCmdDevice *dev = (GnomeCmdDevice*)devices->data;
		add_device_to_list (GTK_CLIST (clist), dev);
		devices = devices->next;
	}
	
	return frame;
}


static void
store_devices_options (GtkWidget *dialog)
{
	GtkWidget *device_only_icon = lookup_widget (dialog, "device_only_icon");
	GtkWidget *skip_mounting = lookup_widget (dialog, "skip_mounting");

	gnome_cmd_data_set_device_only_icon (
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (device_only_icon)));
	gnome_cmd_data_set_skip_mounting (
		gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (skip_mounting)));
}


static void
on_options_dialog_close (GtkButton *button, GtkWidget *dialog)
{
	GnomeCmdOptionsDialog *options_dialog = GNOME_CMD_OPTIONS_DIALOG (dialog);
		
	store_general_options (options_dialog);
	store_representation_options (options_dialog);
	store_layout_options (options_dialog);
	store_colors_options (options_dialog);
	store_filter_options (options_dialog);
	store_programs_options (options_dialog);
	store_devices_options (dialog);

	gtk_widget_destroy (dialog);

	gnome_cmd_style_create ();
	gnome_cmd_main_win_update_style (main_win);

	gnome_cmd_data_save ();
}



/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *object)
{
	GnomeCmdOptionsDialog *dialog = GNOME_CMD_OPTIONS_DIALOG (object);

	g_free (dialog->priv);

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
map (GtkWidget *widget)
{
	if (GTK_WIDGET_CLASS (parent_class)->map != NULL)
		GTK_WIDGET_CLASS (parent_class)->map (widget);
}


static void
class_init (GnomeCmdOptionsDialogClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = GTK_OBJECT_CLASS (class);
	widget_class = GTK_WIDGET_CLASS (class);

	parent_class = gtk_type_class (gnome_dialog_get_type ());

	object_class->destroy = destroy;

	widget_class->map = map;
}

static void
init (GnomeCmdOptionsDialog *dialog)
{
	GtkWidget *options_dialog;
	GtkWidget *close_button;
	
	dialog->priv = g_new (GnomeCmdOptionsDialogPrivate, 1);
	options_dialog = GTK_WIDGET (dialog);

	
	gtk_object_set_data (GTK_OBJECT (options_dialog), "options_dialog",
						 options_dialog);
	gtk_window_set_position (GTK_WINDOW (options_dialog), GTK_WIN_POS_CENTER);
	gtk_window_set_title (GTK_WINDOW (options_dialog), _("Options"));

	
	dialog->notebook = gtk_notebook_new ();
	gtk_widget_ref (dialog->notebook);
	gtk_object_set_data_full (GTK_OBJECT (options_dialog), "notebook",
							  dialog->notebook, (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (dialog->notebook);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox), dialog->notebook,
						TRUE, TRUE, 0);

	gtk_container_add (GTK_CONTAINER (dialog->notebook),
					   create_general_tab (options_dialog));
	gtk_container_add (GTK_CONTAINER (dialog->notebook),
					   create_representation_tab (options_dialog));
	gtk_container_add (GTK_CONTAINER (dialog->notebook),
					   create_layout_tab (options_dialog));
	gtk_container_add (GTK_CONTAINER (dialog->notebook),
					   create_colors_tab (options_dialog));
	gtk_container_add (GTK_CONTAINER (dialog->notebook),
					   create_filter_tab (options_dialog));
	gtk_container_add (GTK_CONTAINER (dialog->notebook),
					   create_programs_tab (options_dialog));
	gtk_container_add (GTK_CONTAINER (dialog->notebook),
					   create_devices_tab (options_dialog));
	

	gtk_notebook_set_tab_label (
		GTK_NOTEBOOK (dialog->notebook),
		gtk_notebook_get_nth_page (GTK_NOTEBOOK (dialog->notebook),
								   GNOME_CMD_OPTIONS_DIALOG_TAB_GENERAL),
		gtk_label_new (_("General")));

	gtk_notebook_set_tab_label (
		GTK_NOTEBOOK (dialog->notebook),
		gtk_notebook_get_nth_page (GTK_NOTEBOOK (dialog->notebook),
								   GNOME_CMD_OPTIONS_DIALOG_TAB_REPRESENTATION),
		gtk_label_new (_("Data Representation")));
	
	gtk_notebook_set_tab_label (
		GTK_NOTEBOOK (dialog->notebook),
		gtk_notebook_get_nth_page (GTK_NOTEBOOK (dialog->notebook),
								   GNOME_CMD_OPTIONS_DIALOG_TAB_LAYOUT),
		gtk_label_new (_("Layout")));

	gtk_notebook_set_tab_label (
		GTK_NOTEBOOK (dialog->notebook),
		gtk_notebook_get_nth_page (GTK_NOTEBOOK (dialog->notebook),
								   GNOME_CMD_OPTIONS_DIALOG_TAB_COLORS),
		gtk_label_new (_("Colors")));

	gtk_notebook_set_tab_label (
		GTK_NOTEBOOK (dialog->notebook),
		gtk_notebook_get_nth_page (GTK_NOTEBOOK (dialog->notebook),
								   GNOME_CMD_OPTIONS_DIALOG_TAB_FILTERS),
		gtk_label_new (_("Filters")));

	gtk_notebook_set_tab_label (
		GTK_NOTEBOOK (dialog->notebook),
		gtk_notebook_get_nth_page (GTK_NOTEBOOK (dialog->notebook),
								   GNOME_CMD_OPTIONS_DIALOG_TAB_PROGRAMS),
		gtk_label_new (_("Programs")));

	gtk_notebook_set_tab_label (
		GTK_NOTEBOOK (dialog->notebook),
		gtk_notebook_get_nth_page (GTK_NOTEBOOK (dialog->notebook),
								   GNOME_CMD_OPTIONS_DIALOG_TAB_DEVICES),
		gtk_label_new (_("Devices")));
	


	
	gnome_dialog_append_button (GNOME_DIALOG (options_dialog),
								GNOME_STOCK_BUTTON_CLOSE);
	close_button = g_list_last (GNOME_DIALOG (options_dialog)->buttons)->data;
	gtk_widget_ref (close_button);
	gtk_object_set_data_full (GTK_OBJECT (options_dialog),
							  "close_button", close_button,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (close_button);
	GTK_WIDGET_SET_FLAGS (close_button, GTK_CAN_DEFAULT);

	
	gtk_signal_connect (GTK_OBJECT (close_button), "clicked",
						GTK_SIGNAL_FUNC (on_options_dialog_close), options_dialog);
}




/***********************************
 * Public functions
 ***********************************/

GtkType
gnome_cmd_options_dialog_get_type         (void)
{
	static GtkType dlg_type = 0;

	if (dlg_type == 0)
	{
		GtkTypeInfo dlg_info =
		{
			"GnomeCmdOptionsDialog",
			sizeof (GnomeCmdOptionsDialog),
			sizeof (GnomeCmdOptionsDialogClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		dlg_type = gtk_type_unique (gnome_dialog_get_type (), &dlg_info);
	}
	return dlg_type;
}


GtkWidget*
gnome_cmd_options_dialog_new (void)
{	
	GnomeCmdOptionsDialog *dialog;

	dialog = gtk_type_new (gnome_cmd_options_dialog_get_type ());

	return GTK_WIDGET (dialog);
}


void
gnome_cmd_options_dialog_set_tab (GnomeCmdOptionsDialog *dialog, GnomeCmdOptionsDialogTab tab)
{
	gtk_notebook_set_page (GTK_NOTEBOOK (dialog->notebook), tab);
}

