# GNU Enterprise Common - GNUe Schema Definition - Schema & Script Generator
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Scripter.py 7007 2005-02-11 16:18:59Z reinhard $

import string

from gnue.common.schema import GSParser
from gnue.common.utils.FileUtils import openResource
from gnue.common.apps.GClientApp import *
from gnue.common.datasources import GDataSource
from gnue.common.apps import errors


# =============================================================================
# Exceptions
# =============================================================================

class Error (errors.SystemError):
  pass

class MissingKeyError (errors.ApplicationError):
  def __init__ (self, tableName):
    msg = u_("Data row of table '%s' has no key fields") % tableName
    errors.ApplicationError.__init__ (self, msg)

class CircularReferenceError (errors.ApplicationError):
  def __init__ (self):
    msg = u_("Tables have circular or unresolveable references")
    errors.ApplicationError.__init__ (self, msg)

class CircularFishHookError (errors.ApplicationError):
  def __init__ (self, table):
    msg = u_("The table '%s' contains circular data-references") % table
    errors.ApplicationError.__init__ (self, msg)

# =============================================================================
# Load GNUe Schema Definition files and create a database schema for it
# =============================================================================

class Scripter (GClientApp):

  from gnue.common import VERSION

  COMMAND = "gnue-schema"
  NAME    = "GNUe Schema Scripter"
  USAGE   = "[options] gsd-file [gsd-file gsd-file ...]"
  SUMMARY = _("GNUe Schema Scripter creates database schemas based on GNUe "
              "Schema Definitions.")


  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, connections = None):
    self.addCommandOption ('connection', 'c', argument='connectionname',
        default = "gnue",
        help = _("Use the connection <connectionname> for creating the schema"))

    self.addCommandOption ('output','o', argument='filename',
        help = _("Also send the code for creating the schema to this file."))

    self.addCommandOption ('file-only', 'f', default = False,
        help = _("If this flag is set, only code is sent to the output file "
                 "and the schema is not created automatically."))

    self.addCommandOption ('mode', 'm', argument='both|schema|data',
        default = 'both',
        help = _("Mode of operation. If mode is 'schema', only schema "
                 "creation is done. If mode is 'data' only data integration "
                 "is done."))

    self.addCommandOption ('username', 'u', argument="user",
        help = _("Set the username for the database. If the database is to be "
                 "created, this username will be it's owner."))

    self.addCommandOption ('password', 'p', argument="password",
        help = _("Set the password for the database."))

    self.addCommandOption ('createdb', 'd', default = False,
        help = _("If this option is set, the database will be created before "
                 "any schema creation is done. There must be a username "
                 "either from the given connection-configuration or from the "
                 "command line. This user becomes the owner of the database "
                 "and will be implicitly created."))

    self.addCommandOption ('yes', 'y', default = False,
        help = _("If this option is set, the program runs in batch-mode, "
                 "which means all questions are answered with 'yes' "
                 "automatically."))

    ConfigOptions = {}
    GClientApp.__init__ (self, connections, 'schema', ConfigOptions)



  # ---------------------------------------------------------------------------
  # Main program
  # ---------------------------------------------------------------------------

  def run (self):
    """
    This is the main function of the whole process. It verifies the given
    options, loads all schema definitions and then logs into the connection to
    perform all actions requested.
    """

    self.__checkOptions ()
    self.__loadInputFiles ()


    # If we have to process data rows let's have a look if there's a key
    # defined for all tables and rows
    if self.__doData and len (self.tabledata.keys ()):
      self.verifyDataKeys ()


    # If the user wants to create the database, first go and ask if she's
    # really sure about
    if self.OPTIONS ['createdb']:
      res = self.__ask (u_("You are about to create the new database '%s'. " \
                           "Continue") % self.connection.name,
                           [u_("y"), u_("n")], "n")
      if res == u_("n"):
        return

      creator = self.connection.getSchemaCreator ()
      if creator is not None:
        try:
          creator.createDatabase ()

        finally:
          creator.close ()

    # Process schema information (if requested)
    asked = False
    if self.__doSchema:
      if not self.OPTIONS ['file-only']:
        asked = True
        res   = self.__ask (u_("You are about to change the database '%s'. " \
            "Continue") % self.connection.name, [u_("y"), u_("n")], u_("n"))
        if res == u_("n"):
          return

      self.executeAndGenerateCode ()


    # Process data information (if requested)
    if self.__doData and len (self.tabledata.keys ()):
      if not asked:
        res = self.__ask (u_("You are about to change the database '%s'. " \
            "Continue") % self.connection.name, [u_("y"), u_("n")], u_("n"))
        if res == u_("n"):
          return

      self.updateData ()




  # ---------------------------------------------------------------------------
  # Walk through all command line options
  # ---------------------------------------------------------------------------

  def __checkOptions (self):
    """
    This function checks wether the given command line arguments and options
    are usefull or not.
    """

    if not len (self.ARGUMENTS):
      raise StartupError, u_("No input file specified.")

    try:
      self._files = []

      for filename in self.ARGUMENTS:
        self._files.append (openResource (filename))

    except IOError:
      raise StartupError, \
          u_("Unable to open input file: %s") % errors.getException () [2]


    self.outfile = self.OPTIONS ['output']

    if self.OPTIONS ['file-only'] and self.outfile is None:
      raise StartupError, \
          u_("Output to file only requested, but no filename specified.")

    self.__doSchema = self.OPTIONS ['mode'].lower () in ['both', 'schema']
    self.__doData   = self.OPTIONS ['mode'].lower () in ['both', 'data'] and \
                          not self.OPTIONS ['file-only']

    if not (self.__doSchema or self.__doData):
      raise StartupError, \
          u_("Mode of operation must be one of 'both', 'schema' or 'data'.")

    cName = self.OPTIONS ['connection']
    self.connection = self.connections.getConnection (cName)

    # if a username is given on the command line we pass both username and
    # password to the connection parameters. If the password is not set, it
    # defaults to an empty string.
    username = self.connection.parameters.get ('username', 'gnue')
    password = self.connection.parameters.get ('password', 'gnue')

    if self.OPTIONS ['username'] is not None:
      username = self.OPTIONS ['username']

    if self.OPTIONS ['password'] is not None:
      password = self.OPTIONS ['password']

    self.connection.parameters ['username'] = username
    self.connection.parameters ['password'] = password


  # ---------------------------------------------------------------------------
  # Load all input files
  # ---------------------------------------------------------------------------

  def __loadInputFiles (self):
    """
    This function loads all files given in the argument list and builds the
    dictionaries with the tabledefinition and -data. The sequence 'tableOrder'
    lists all tablenames in an order to be created without violating
    referential constraints.
    """

    self.tables    = {}
    self.tabledata = {}
    self.unordered = []

    for item in range (len (self._files)):
      print o (u_("Loading gsd file '%s' ...") % self.ARGUMENTS [item])

      try:
        schema = GSParser.loadFile (self._files [item])
        schema.walk (self.__iterateObjects)

      finally:
        self._files [item].close ()

    tables      = {}
    self.fishes = {}

    for table in self.tables.values ():
      # first add the table itself to the dictionary
      tablename = table ['name'].lower ()
      if not tables.has_key (tablename):
        tables [tablename] = []

      # if the table has constraints, add a reference to this table
      if table.has_key ('constraints') and len (table ['constraints']):
        for constraint in table ['constraints']:
          ref = constraint.get ('reftable')
          if ref:
            refname = ref.lower ()
            if refname == tablename:
              if not self.fishes.has_key (tablename):
                self.fishes [tablename] = []

              self.fishes [tablename].append ((constraint ['fields'],
                                               constraint ['reffields']))
            else:
              tables [tablename].append (refname)

    # now get the proper order to create/modify the tables
    self.tableOrder = []
    add = self.__shrinkList (tables, CircularReferenceError)
    while len (add):
      self.tableOrder.extend (add)
      add = self.__shrinkList (tables, CircularReferenceError)

    # Now make sure to append all tables to the tableOrder, which are not yet
    # listed. These could be the case if no table-definition is given for a
    # table.
    for item in self.unordered:
      if not item in self.tableOrder:
        self.tableOrder.append (item)


  # ---------------------------------------------------------------------------
  # Return all items from the dictionary which have no dependencies
  # ---------------------------------------------------------------------------

  def __shrinkList (self, dataDict, circularError, *args):
    """
    This function returns a sequence of all keys without any dependencies.
    If no such items were found but there are still entries in the dictionary
    a CircularReferenceError will be raised.

    @param dataDict: dictionary to extract items from. The values are sequences
        with keys which are referencing to an item
    @param circularError: exception class which will be raised if there are
        circular references
    @return: sequence of all keys which do not have any dependency
    """

    result = []

    for (key, references) in dataDict.items ():
      if not len (references):
        # if an item has no dependencies add it to the result
        result.append (key)

        # remove it from all other entries it is referred to
        for ref in dataDict.values ():
          while key in ref:
            ref.remove (key)

        # and finally remove it from the dictionary
        del dataDict [key]

    # if no entry without a dependency was found, but there are still entries
    # in the dictionary, they *must* have circular references
    if not len (result) and len (dataDict.keys ()):
      raise circularError, args

    return result




  # ---------------------------------------------------------------------------
  # Get a dictionary with all keys listed in tags and values from sObject
  # ---------------------------------------------------------------------------

  def fetchTags (self, sObject, tags):
    """
    This function creates a dictionary with all attributes from sObject listed
    in tags, where the attributenames are the keys.

    @param sObject: Schema object to retriev attributes from
    @param tags: list of all attributes to retrieve
    @return: dictionary with the attribute names as keys and their values
    """
    res = {}
    for item in tags:
      if hasattr (sObject, item):
        res [item] = getattr (sObject, item)
    return res


  # ---------------------------------------------------------------------------
  # iteration over all schema objects in the document tree
  # ---------------------------------------------------------------------------

  def __iterateObjects (self, sObject):
    """
    This is the master iteration function, it runs over all top level objects
    in the GSObject tree and processes GSTable- and GSTableData-subtrees.

    @param sObject: current Schema object to be processed
    """

    if sObject._type == "GSTable":
      tableKey = sObject.name.lower ()
      if not self.tables.has_key (tableKey):
        self.tables [tableKey] = {'name': sObject.name}

      sObject.walk (self.__schemaFields, defs = self.tables [tableKey])

    elif sObject._type == "GSTableData":
      tableKey = sObject.tablename.lower ()
      if not self.tabledata.has_key (tableKey):
        self.tabledata [tableKey] = {'name': sObject.tablename,
                                     'rows': [],
                                     'defi': {'fields': [], 'key': []}}
        if not tableKey in self.unordered:
          self.unordered.append (tableKey)

      sObject.walk (self.__dataRows, defs = self.tabledata [tableKey])

    return


  # ---------------------------------------------------------------------------
  # Process the fields of a GSTable
  # ---------------------------------------------------------------------------

  def __schemaFields (self, sObject, defs):
    """
    This function iterates over all child elements of a GSTable instance and
    converts this subtree into a table definition dictionary given by the
    parameter defs.

    @param sObject: current schema object to be processed
    @param defs: dictionary of the table definition, which has to be extended
        by this function.
    """

    # process a regular field of a table
    if sObject._type == "GSField":
      fDef = self.fetchTags (sObject, ['name', 'type', 'default',
                        'defaultwith', 'length', 'precision', 'nullable'])
      if not defs.has_key ('fields'):
        defs ['fields'] = []

      found = False
      for item in defs ['fields']:
        if item ['name'].lower () == sObject.name.lower ():
          found = True
          break

      if not found:
        defs ['fields'].append (fDef)

    # add a primary key definition to the table definition
    elif sObject._type == "GSPrimaryKey":
      pkDef = {'name': sObject.name, 'fields': []}
      defs ['primarykey'] = pkDef
      sObject.walk (self.__schemaPrimaryKey, defs = pkDef)

    # start an index definition and process it's fields
    elif sObject._type == "GSIndex":
      if not defs.has_key ('indices'):
        defs ['indices'] = []

      found = False
      for item in defs ['indices']:
        if item ['name'].lower () == sObject.name.lower ():
          found = True
          break

      if not found:
        indexDef = {'name'  : sObject.name,
                    'unique': hasattr (sObject, "unique") and sObject.unique,
                    'fields': []}
        defs ['indices'].append (indexDef)

        sObject.walk (self.__schemaIndex, defs = indexDef)

    # create constraints
    elif sObject._type == "GSConstraint":
      # for unique-constraints we use a 'unique index'
      if sObject.type == "unique":
        if not defs.has_key ('indices'):
          defs ['indices'] = []

        found = False
        for item in defs ['indices']:
          if item ['name'].lower () == sObject.name.lower ():
            found = True
            break

        if not found:
          cDef = {'name'  : sObject.name,
                  'unique': True,
                  'fields': []}
          defs ['indices'].append (cDef)

      # for all other types of constraints we use a ConstraintDefinition
      else:
        if not defs.has_key ('constraints'):
          defs ['constraints'] = []

        found = False
        for item in defs ['constraints']:
          if item ['name'].lower () == sObject.name.lower ():
            found = True
            break

        if not found:
          cDef = self.fetchTags (sObject, ['name', 'type'])
          cDef ['fields'] = []
          defs ['constraints'].append (cDef)

      if not found:
        sObject.walk (self.__schemaConstraint, defs = cDef)


  # ---------------------------------------------------------------------------
  # Iterate over all fields of a primary key
  # ---------------------------------------------------------------------------

  def __schemaPrimaryKey (self, sObject, defs):
    """
    This function converts all GSPKField instances into a primary key
    definition (dictionary).

    @param sObject: current schema object to be processed
    @param defs: dictionary describing the primary key
    """
    if sObject._type == "GSPKField":
      defs ['fields'].append (sObject.name)


  # ---------------------------------------------------------------------------
  # Iterate over all fields of an index
  # ---------------------------------------------------------------------------

  def __schemaIndex (self, sObject, defs):
    """
    This function converts all GSIndexField instances into a index definition
    (dictionary).

    @param sObject: current schema object to be processed
    @param defs: dictionary describing the index
    """
    if sObject._type == "GSIndexField":
      defs ['fields'].append (sObject.name)


  # ---------------------------------------------------------------------------
  # Iterate over all children of a constraint definition
  # ---------------------------------------------------------------------------

  def __schemaConstraint (self, sObject, defs):
    """
    This function converts constraint fields and references into a constraint
    definition (dictionary).

    @param sObject: current schema object to be processed
    @param defs: dictionary describing the constraint
    """
    if sObject._type == "GSConstraintField":
      defs ['fields'].append (sObject.name)

    elif sObject._type == "GSConstraintRef":
      defs ['reftable'] = sObject.table
      if not defs.has_key ('reffields'):
        defs ['reffields'] = []
      defs ['reffields'].append (sObject.name)


  # ---------------------------------------------------------------------------
  # Iterate over all rows of a tabledata definition
  # ---------------------------------------------------------------------------

  def __dataRows (self, sObject, defs):
    """
    This function converts a GSRow instance into an element in the given data
    dictionary.

    @param sObject: current schema object to be processed
    @param defs: sequence of row definitions of the corresponding table
    """
    if sObject._type == "GSRow":
      defs ['rows'].append ({'key': [], 'fields': {}})
      sObject.walk (self.__dataValues, defs ['rows'][-1])

    elif sObject._type == "GSColumn":
      defi = defs ['defi']
      defi ['fields'].append (sObject.field)

      if hasattr (sObject, 'key') and sObject.key:
        defi ['key'].append (sObject.field)


  # ---------------------------------------------------------------------------
  # Iterate over all values of a row definition
  # ---------------------------------------------------------------------------
  def __dataValues (self, sObject, defs):
    """
    This function translates a GSValue instance into an element of a row
    definition dictionary.

    @param sObject: current schema object to be processed.
    @param defs: row definition dictionary to be extended.
    """
    if sObject._type == "GSValue":
      defs ['fields'][sObject.field] = sObject.value
      if hasattr (sObject, "key") and sObject.key:
        defs ['key'].append (sObject.field)


  # ---------------------------------------------------------------------------
  # Make sure all rows in the table data sequence have a valid key
  # ---------------------------------------------------------------------------

  def verifyDataKeys (self):
    """
    This function iterates over all data rows and makes sure they all have an
    appropriate key specified. Given all data rows of a table, there must be at
    least on row with the key-attribute set on it's fields. If there is no
    row with key information, but a table definition with a primary key this
    key will be used.

    @raise MissingKeyError: If no key information could be found for all rows
        of a table this exception will be raised.
    """
    for item in self.tabledata.values ():
      tableName = item ['name']
      rows      = item ['rows']
      tableKey  = None
      missing   = False

      # is there a definition with keys ?
      if len (item ['defi']['key']):
        tableKey = item ['defi']['key']
        continue

      # is a key missing or does at least one row specify a key?
      for row in rows:
        if len (row ['key']):
          tableKey = row ['key']
        else:
          missing = True

      if not missing:
        continue

      # is there a table definition with a valid primary key available ?
      if tableKey is None:
        tableDef = self.tables.get (tableName.lower ())
        if tableDef is not None and tableDef.has_key ('primarykey'):
          tableKey = tableDef ['primarykey']['fields']


      # if at least one key was available, fill up all missing keys with this
      # one.
      if tableKey is not None:
        for row in rows:
          if not len (row ['key']):
            row ['key'] = tableKey
        continue

      raise MissingKeyError, (tableName)


  # ---------------------------------------------------------------------------
  # Execute and generate the code
  # ---------------------------------------------------------------------------

  def executeAndGenerateCode (self):
    """
    This function logs into the given connection and calls it for an update of
    it's schema according to the loaded table definitions. Additionally the
    schema creation code is generated by this call, which will be stored in the
    given output file (if requested by options).
    """

    self.connections.loginToConnection (self.connection)

    print o(u_("Updating schema ..."))

    # create a list of table definitions ordered to avoid integrity errors
    tables = []
    for name in self.tableOrder:
      if self.tables.has_key (name):
        tables.append (self.tables [name])

    code = self.connection.updateSchema (tables, self.OPTIONS ['file-only'])

    if self.outfile is not None:
      dest = open (self.outfile, 'w')

      for item in code:
        for line in item:
          dest.write (line + "\n")

      dest.close ()


  # ---------------------------------------------------------------------------
  # Update connection with table data 
  # ---------------------------------------------------------------------------

  def updateData (self):
    """
    This function updates the backend with data. Every row is checked using the
    specified key-fields and updated if it exists otherwise inserted.
    """

    print o(u_("Updating data ..."))

    if not len (self.tableOrder):
      self.tableOrder = self.tabledata.keys ()

    # make sure to use the proper table order for inserting data
    tablelist = []
    for name in self.tableOrder:
      if self.tabledata.has_key (name):
        tablelist.append (self.tabledata [name])

    for table in tablelist:
      tablename = table ['name']
      rows      = table ['rows']

      if not len (rows):
        continue

      (datasource, keyFields) = self.__openSource (table)

      print o (u_("  updating table '%s' ...") % tablename)
      stat = [0, 0, 0]

      if self.fishes.has_key (tablename.lower ()):
        rows = self.__fishSort (table)
      else:
        rows = table ['rows']

      for row in rows:
        condition = {}

        for keyField in keyFields:
          condition [keyField] = row ['fields'][keyField]

        gDebug (8, "Condition: %s" % condition)
        resultSet = datasource.createResultSet (condition)

        if resultSet.firstRecord () is None:
          resultSet.insertRecord ()
          modIx = 0
        else:
          modIx = 1

        doPost = False

        for (fieldName, fieldValue) in row ['fields'].items ():
          if resultSet.current.getField (fieldName) != fieldValue:
            resultSet.current.setField (fieldName, fieldValue)
            doPost = True

        if doPost:
          resultSet.post ()
        else:
          modIx += 1

        stat [modIx] += 1

      if stat [0] + stat [1]:
        self.connections.commitAll ()

      print o (u_("    Rows: %(ins)d inserted, %(upd)d updated, %(kept)d "
                  "unchanged.") \
               % {'ins': stat [0], 'upd': stat [1], 'kept': stat [2]})


  # ---------------------------------------------------------------------------
  # Prepare a datasource
  # ---------------------------------------------------------------------------

  def __openSource (self, table):
    """
    Create a new datasource instance for the given table using all fields
    requested either by a table definition, or a table description.

    @param table: name of the table to create a datasource for
    @return: tuple with datasource instance and a sequence of all key-fields
    """

    # First we use all fields and keys as described by a table definition
    fieldList = table ['defi']['fields']
    keyFields = table ['defi']['key']

    # Now add all fields, which are only given in the rows of tabledata tags
    for row in table ['rows']:
      for field in row ['fields'].keys ():
        if not field in fieldList:
          fieldList.append (field)

      for field in row ['key']:
        if not field in keyFields:
          keyFields.append (field)

    source = GDataSource.DataSourceWrapper (connections = self.connections,
        attributes = {'name'    : "dts_%s" % table ['name'],
                      'database': self.OPTIONS ['connection'],
                      'table'   : table ['name']},
        fields      = fieldList,
        unicodeMode = True)

    return (source, keyFields)


  # ---------------------------------------------------------------------------
  # Sort rows of table with a fish-hook
  # ---------------------------------------------------------------------------

  def __fishSort (self, tabledef):
    """
    If a table has fishhooks, make sure to re-order all data rows so they can
    be inserted without conflicting constraints.

    @param tabledef: tabledata definition to sort the rows of
    @return: sequence of row-dictionaries
    """
    
    tablename = tabledef ['name']
    fishes    = self.fishes [tablename.lower ()]

    # First create an empty map of all fields to keep track of
    fishdict  = {}
    for (fields, reffields) in fishes:
      for field in fields:
        fishdict [field] = {}

    # Now we create a dictionary of all data rows, where we use a tuple of all
    # key-fields as dictionary key. Additionaly update the mapping with the
    # backreferences (fishdict)
    rowByKey = {}

    for row in tabledef ['rows']:
      key = tuple ([row ['fields'][item] for item in row ['key']])
      rowByKey [key] = row ['fields']

      for (fields, reffields) in fishes:
        for ix in range (len (fields)):
          fishdict [fields [ix]][row ['fields'][reffields [ix]]] = key
            
    # In the next step we create a dictionary to track all dependencies between
    # the rows.
    sortdict = {}

    for (key, data) in rowByKey.items ():
      if not sortdict.has_key (key):
        sortdict [key] = []

      for (fields, reffields) in fishes:
        for item in fields:
          # If a reference field has a value, we get the key of that record,
          # and add it as a dependency of the current row, since that record
          # has to be inserted before the current one.
          if data.get (item) is not None:
            ref = fishdict [item].get (data [item])
            if ref is not None:
              sortdict [key].append (ref)

    # Now, since we have a dictionary mapping all dependencies, let's create an
    # ordered list of all rows. All items holding an empty sequence, do *not*
    # depend on another row, so we can just add them. Aside from that we can
    # remove all these rows from the remaining dependency-sequences, since this
    # dependency is fullfilled.
    order = []
    add   = self.__shrinkList (sortdict, CircularFishHookError, tablename)

    while len (add):
      order.extend (add)
      add = self.__shrinkList (sortdict, CircularFishHookError, tablename)

    # The sequence 'order' now contains all rows in the proper order to be
    # inserted. To be done, we have to create a sequence of rows-dicts.
    result = []
    for key in order:
      result.append ({'fields': rowByKey [key]})

    return result



  # ---------------------------------------------------------------------------
  # Ask a question with a set of valid options and a default
  # ---------------------------------------------------------------------------

  def __ask (self, question, options, default):
    """
    This function asks for a question, allowing a set of answers, using a
    default-value if the user just presses <Enter>.

    @param question: string with the question to ask
    @param options: sequence of allowed options, i.e. ['y', 'n']
    @param default: string with the default option

    @return: string with the option selected
    """

    if self.OPTIONS ['yes']:
      return u_("y")

    answer  = None
    default = default.lower ()
    lopts   = map (string.lower, options)

    dopts   = []
    for item in lopts:
      dopts.append (item == default and item.upper () or item)

    while True:
      print o(question), o("[%s]:" % string.join (dopts, ",")),
      answer = raw_input ().lower () or default

      if answer in lopts:
        break

    return answer


# =============================================================================
# If executed directly, start the scripter
# =============================================================================
if __name__ == '__main__':
  Scripter ().run ()
