# GNU Enterprise Forms - GF Class Hierarchy - Block
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: GFBlock.py 7090 2005-03-04 17:47:02Z reinhard $
"""
Classes making up the Block object
"""

from gnue.forms.GFObjects.GFDataSource import GFDataSource

from gnue.common.apps import errors
from gnue.common.datasources import GConditions
from GFContainer import GFContainer
from gnue.common import events

import string

# These should really go somewhere else
TRUE = 1
FALSE = 0


############################################################
# GFBlock
#
#
class GFBlock(GFContainer, events.EventAware):
  def __init__(self, parent=None):
    GFContainer.__init__(self, parent, 'GFBlock')

    self.mode = 'normal'
    self._convertAsterisksToPercent = gConfigForms('AsteriskWildcard')

    self._inits = [self.__initialize]

    #self._datasource = None
    self._resultSet = None
    self._dataSourceLink = None

    self._currentRecord = 0
    self._recordCount = 0
    self._queryDefaults = {}
    self._queryValues = {}
    self._lastQueryValues = {}

    self._gap = 0
    self._rows = 1

    self._scrollbars= []

    # Populated by GFEntry's initialize
    self._entryList = []

    # Populated by GFField's initialize
    self._fieldMap = {}
    self._fieldList = []

    #
    # Trigger exposure
    #
    self._validTriggers = {
##                  'ON-SWITCH':      'On-Switch',    ## Deprecated
                  'ON-NEWRECORD':   'On-NewRecord',
                  'PRE-COMMIT':     'Pre-Commit',
                  'POST-COMMIT':    'Post-Commit',
                  'PRE-QUERY':      'Pre-Query',
                  'POST-QUERY':     'Post-Query',
                  'PRE-MODIFY':     'Pre-Modify',
                  'PRE-INSERT':     'Pre-Insert',
                  'PRE-DELETE':     'Pre-Delete',
                  'PRE-UPDATE':     'Pre-Update',
                  'PRE-FOCUSOUT':   'Pre-FocusOut',
                  'POST-FOCUSOUT':  'Post-FocusOut',
                  'PRE-FOCUSIN':    'Pre-FocusIn',
                  'POST-FOCUSIN':   'Post-FocusIn',
                  'PRE-CHANGE':     'Pre-Change',
                  'POST-CHANGE':    'Post-Change',
                  }


    self._triggerGlobal = 1
    self._triggerFunctions={
        'clear':{'function':self.processClear,
                  'description':''},
        'getResultSet':{'function':self.getResultSet,
                  'description':''},
        'commit':{'function':self.commit,
                  'description':''},
        'newRecord':{'function':self.newRecord,
                      'description':''},
        'deleteRecord':{'function':self.deleteRecord,
                        'description':''},
        'duplicateRecord':{'function':self.duplicateRecord,
                        'description':'Duplicates the current (non-empty) record into a new record.'},
        'gotoRecord':{'function':self.jumpRecord,
                      'description':''},
        'firstRecord':{'function':self.firstRecord,
                        'description':'Navigates the block to the first record it contains.'},
        'isEmpty':{'function':self.isEmpty,
                    'description':'Returns True if block is empty.'},
        'isSaved':{'function':self.isSaved,
                    'description':'Returns True if block contains no modified records.'},
        'lastRecord':{'function':self.lastRecord,
                  'description':'Navigates the block to the last record it contains.'},
        'nextRecord':{'function':self.nextRecord,
                      'description':'Navigates the block to the next record in sequence.'},
        'prevRecord':{'function':self.prevRecord,
                      'description':'Navigates the block to the previous record in sequence.'},
        'jumpRecords':{'function':self.jumpRecords,
                      'description':'Navigates the specified number of records.'},
        'rollback':{'function':self.processRollback,
                    'description':'Clears all records regardless of state from the block'},
        'initQuery':{'function':self.initQuery,
                    'description':'Prepares the block for query input.'},
        'copyQuery':{'function':self.copyQuery,
                    'description':'Prepares the block for query input.'},
        'cancelQuery':{'function':self.cancelQuery,
                    'description':'Cancels query input.'},
        'executeQuery':{'function':self.processQuery,
                        'description':'Executes the current query.'},
        'call': {'function'   : self.callFunction,
                 'description': 'Executes a function of the datasource'},
        'update': {'function': self.updateCurrentRecordSet}
        }

    self._triggerProperties={
          'parent':        {'get':self.getParent},
          'autoCommit':    {'set':self.triggerSetAutoCommit,
                            'get':self.triggerGetAutoCommit },
          'queryable':     {'set':self.triggerSetQueryable,
                            'get':self.triggerGetQueryable },
          'editable':      {'set':self.triggerSetEditable,
                            'get':self.triggerGetEditable },
          'autoCreate':    {'set':self.triggerSetAutoCreate,
                            'get':self.triggerGetAutoCreate },
          'deletable':     {'set':self.triggerSetDeletable,
                            'get':self.triggerGetDeletable },
          'transparent':   {'set':self.triggerSetTransparent,
                            'get':self.triggerGetTransparent },
          'autoNextRecord':{'set':self.triggerSetAutoNextRecord,
                            'get':self.triggerGetAutoNextRecord },
       }


  # Iterator support (Python 2.2+)
  def __iter__(self):
    """
    Iterator support for Python 2.2+

    Allows you to do:
      for foo in myBlock:
        ....
    """

    return _BlockIter(self)


  def _buildObject(self):

    if hasattr(self, 'rows'):
      self._rows = self.rows

    if hasattr(self, 'rowSpacer'):
      self._gap = self.rowSpacer

    if hasattr(self, 'restrictDelete') and self.restrictDelete:
      self.deletable = False
      del self.__dict__['restrictDelete']

    if hasattr(self, 'restrictInsert') and self.restrictInsert:
      self.editable = 'update'
      del self.__dict__['restrictInsert']

    if hasattr(self,'datasource'):
      self.datasource = string.lower(self.datasource)

    return GFContainer._buildObject(self)

  #
  # Primary initialization
  #
  def __initialize(self):
    self._form = form = self.findParentOfType('GFForm')
    self._logic = logic = self.findParentOfType('GFLogic')

    self._lastValues = {}

    logic._blockList.append(self)
    logic._blockMap[self.name] = self

    # Initialize our events system
    events.EventAware.__init__(self, form._instance.eventController)

    # Create a stub/non-bound datasource if we aren't bound to one
    if not hasattr(self,'datasource') or not self.datasource:
      ds = GFDataSource(self._form)
      self.datasource = ds.name = "__dts_%s" % id(self)
      form._datasourceDictionary[ds.name] = ds
      ds._buildObject()
      ds.phaseInit()

    self._dataSourceLink = form._datasourceDictionary.get (self.datasource)
    if self._dataSourceLink is None:
      raise errors.ApplicationError, \
          u_("Datasource '%(datasource)s' in block '%(block)s' not found") \
          % {'datasource': self.datasource,
             'block': self.name}

    # Get min and max child rows, if applicable
    try:
      self._minChildRows = self._dataSourceLink.detailmin
    except AttributeError:
      self._minChildRows = 0
    try:
      self._maxChildRows = self._dataSourceLink.detailmax
    except AttributeError:
      self._maxChildRows = None

    # We will monitor our own resultSet changes
    self._dataSourceLink.registerResultSetListener(self._loadResultSet)

    self.walk(self.__setChildRowSettings)

  def __setChildRowSettings(self, object):
    if hasattr(object,'rows'):
      rows = object._rows = object.rows
    else:
      rows = object._rows = self._rows

    if hasattr(object,'rowSpacer'):
      object._gap = object.rowSpacer
    else:
      object._gap = self._gap

  #
  #
  #
  def getFocusOrder(self):
    list = []
    for field in self._children:
      try:
        list += field._entryList
      except AttributeError:
        pass # Triggers, etc

    return GFContainer.getFocusOrder(self,list)

  #
  # isSaved
  #
  #
  def isSaved(self):
    """
    Returns True if the datasource the block is associated
    with is not pending any changes.
    """
    # TODO: Find all uses of isSaved() and replace with: not isPending()
    return not self.isPending()

  def isPending(self):
    """
    Returns True if the datasource the block is associated
    with is pending any changes.
    """
    return self._resultSet and self._resultSet.isPending()

  #
  # deleteRecord
  #
  def deleteRecord(self):
    """
    Doesn't really delete the record but marks it for
    deletion during next commit
    """
    self._resultSet.current.delete()

  #
  #
  def _loadResultSet(self, resultSet):
    """
    Load and process a resultset
    """
    self._resultSet = resultSet
    self._currentRecord = -1
    self._recordCount = 0

    # This makes the resultSet call our currentRecordMoved method
    resultSet.registerListener (self)

    # FIXME: Is this used anywhere?
    resultSet._block = self

    if self._resultSet.firstRecord():
      # self.switchRecord(0)            # happens via currentRecordMoved
      self._recordCount = self._resultSet.getRecordCount()
    else:
      # This means no results were returned from a query.
# #       if self._minChildRows:
# #         # If datasource.detailmin is set, create some blank
# #         # (but non-empty) records
# #         while self._recordCount < self._minChildRows:
# #           self.newRecord()
# #           # Force the record to no longer be
# #           field = self._fieldList[0]
# #           field.setValue(field.getValue())
# #         self.firstRecord()
# #       else:
        # Otherwise, give us a single empty record to work from
        self.newRecord()



  #
  # isEmpty()
  #
  def isEmpty(self):
    return self._resultSet.current.isEmpty()

  #
  # getResultSet()
  #
  def getResultSet(self):
    return self._resultSet


  # This gets called by the resultSet whenever the current record changes
  def currentRecordMoved (self):
    self.switchRecord (self._resultSet.getRecordNumber() - self._currentRecord)

  #
  #
  #
  def switchRecord(self, adjustment):
    """
    Moves the proper record into editing position
    """
    self._currentRecord = self._resultSet.getRecordNumber()
    for field in self._fieldList:
      field.gotNewCurrentRecord()
      for entry in field._entryList:
        # This loop is probably better somewhere else
        entry.recalculateVisible( adjustment, self._currentRecord, self._recordCount)
      self._form.updateUIEntry(field)
      self._form.refreshUIEvents()

    # Adjusting scrollbars
    for sb in self._scrollbars:
      sb.adjustScrollbar(self._currentRecord, self._recordCount)

  #
  # newRecord
  #
  # Adds a record to the current records in memory
  #
  def newRecord(self):

    # Focus out
    self.processTrigger('PRE-FOCUSOUT')
    if self.autoCommit and self._resultSet.current:
      self._form.commit()
    self.processTrigger('POST-FOCUSOUT')

    if self._resultSet.insertRecord():

      # Set the defaultToLast fields
      for field, value in self._lastValues.items():
        if value is not None:
          self._resultSet.current.setField(field, value, 0)

      self._recordCount = self._resultSet.getRecordCount()
      # self.switchRecord(1)            # happens via currentRecordMoved
      # Focus in
      self.processTrigger('PRE-FOCUSIN')
      self.processTrigger('ON-NEWRECORD')

      self.processTrigger('POST-FOCUSIN')

  def duplicateRecord(self, exclude=(), include=()):
    self._resultSet.duplicateRecord(exclude=exclude, include=include)

  def isLastRecord(self):
    return self._resultSet.isLastRecord()

  def isFirstRecord(self):
    return self._resultSet.isFirstRecord()

  def nextRecord(self):
    if not self._resultSet.isLastRecord():

      # Do FocusOut triggers
      self.processTrigger('PRE-FOCUSOUT')
      if self.autoCommit:
        self._form.commit()
      self.processTrigger('POST-FOCUSOUT')

      self._resultSet.nextRecord()
      self._recordCount = self._resultSet.getRecordCount()
      # self.switchRecord(1)            # happens via currentRecordMoved

      # Focus in
      self.processTrigger('PRE-FOCUSIN')
      self.processTrigger('POST-FOCUSIN')

    elif self.autoCreate and not self.isEmpty() and not self.editable in ('update','N'):
      self.newRecord()
      # Go to first field
      self._form.findAndChangeFocus(self._entryList[0])



  def lastRecord(self):
    if not self._resultSet.isLastRecord():
      # Do FocusOut triggers
      self.processTrigger('PRE-FOCUSOUT')
      self.processTrigger('POST-FOCUSOUT')

      self._resultSet.lastRecord()
      # jump = self._resultSet.getRecordNumber () - self._currentRecord
      # self.switchRecord (jump)        # happens via currentRecordMoved

      # Focus in
      self.processTrigger('PRE-FOCUSIN')
      self.processTrigger('POST-FOCUSIN')

  def firstRecord(self):
    if not self._resultSet.isFirstRecord():
      # Do FocusOut triggers
      self.processTrigger('PRE-FOCUSOUT')
      self.processTrigger('POST-FOCUSOUT')

      self._resultSet.firstRecord()
      # self.switchRecord(0)            # happens via currentRecordMoved

      # Focus in
      self.processTrigger('PRE-FOCUSIN')
      self.processTrigger('POST-FOCUSIN')

  def prevRecord(self):
    if not self._resultSet.isFirstRecord():

      # Do FocusOut triggers
      self.processTrigger('PRE-FOCUSOUT')
      self.processTrigger('POST-FOCUSOUT')

      self._resultSet.prevRecord()
      # self.switchRecord(-1)           # happens via currentRecordMoved

      # Focus in
      self.processTrigger('PRE-FOCUSIN')
      self.processTrigger('POST-FOCUSIN')

  def commit(self):
    print "DEPRECIATION WARNING: the use of block.commit () is depreciated.", \
          "Please use form.commit () instead."
    self._form.commit ()

    #Commented out (dimas)
    #Does not work properly anyway
    #self._form.changeFocus(self)
    #self._form.commit()

    #TODO: Add error handling
    #TODO: Check how triggers performed
    #TODO: original code is in GFForm.commit
    #self._precommitRecord = self._currentRecord
    #self.processCommit()

  def jumpRecord(self, recordNumber):
    # If recordNumber is negative, move relative to last record
    if recordNumber < 0:
      recordNumber += self._resultSet.getRecordCount()
    if recordNumber < 0:
      raise "Invalid record number"

    if recordNumber != self._resultSet.getRecordNumber():
      # Focus out
      self.processTrigger('PRE-FOCUSOUT')
      self.processTrigger('POST-FOCUSOUT')

      if not self._resultSet.setRecord(recordNumber):
        self._resultSet.lastRecord()

      # jump = self._resultSet.getRecordNumber() - self._currentRecord
      # self._currentRecord = self._resultSet.getRecordNumber()
      # self.switchRecord(jump)         # happens via currentRecordMoved

      # Focus in
      self.processTrigger('PRE-FOCUSIN')
      self.processTrigger('POST-FOCUSIN')


  def jumpRecords(self, adjustment):
    targetRecord = self._resultSet.getRecordNumber() + adjustment

    if targetRecord < 0:
      targetRecord = 0
    elif targetRecord > self._resultSet.getRecordCount():
      targetRecord = self._resultSet.getRecordCount()

    self.jumpRecord(targetRecord)
    self._form._instance.updateRecordCounter(self._form)

  #
  # processCommit
  #
  def processCommit(self):
    gDebug (4, "processing commit on block %s"%self.name,1)

    self.mode='commit'

    self._resultSet.setRecord(self._precommitRecord)

    # Step thru all the records looking for records with uncommited 
    # changes that had default values set.  If one is found then those
    # fields are reset to their current values *with* modification tracking
    # on.  The default population on new records is to set the value without
    # marking the record as modified which prevents their value from being saved.
    for record in self._resultSet._cachedRecords:
      if record.isPending():
        for field in self._lastValues.keys():
          record.setField(field, record.getField(field), 1)

    # Backstep thru the record adjusting for any prior records that
    # will be deleted.  Keeps the record postition properly adjusted
    # during deletes.
    for record in self._resultSet._cachedRecords:
      if record.isDeleted():
        self._resultSet.prevRecord()
        self._precommitRecord -= 1
      if record == self._resultSet.current:
        break

    if not self._dataSourceLink.hasMaster():
      self._resultSet.post()
      # The real commit will be done by the form once per connection
      # self._dataSourceLink._dataObject.commit()

  #
  # Called after the commit on the backend is through.
  #
  def finalizeCommit (self):

    # Our recordCount might have changed if records were deleted
    self._recordCount = self._resultSet.getRecordCount()

    # If all records were deleted, create an empty new one, otherwise make sure
    # we move the record pointer to the correct record
    if not self._recordCount:
      self.newRecord()
    else:
      self.jumpRecord(self._precommitRecord)

    self.mode='normal'


  #
  # processClear
  #
  def processClear(self):
    self._dataSourceLink._dataObject.rollback()
    if not self._dataSourceLink.hasMaster():
      self._dataSourceLink.createEmptyResultSet()
    else:
      self._dataSourceLink.createEmptyResultSet(masterRecordSet=self._resultSet._masterRecordSet)
    self._currentRecord = 0
    self._recordCount = 0
    # FIXME: does this happen via currentRecordMoved?
    self.switchRecord(0)


  #
  # processRollback
  #
  # if recovering=False, then the user requested a rollback
  # if recovering=True, then a commit or such failed and we need to clean up
  # (but not lose state information)
  #
  def processRollback (self, recovering = False, backendRollback = True):

    # if called from GFForm the connection's rollback () has been executed
    # already. But if called from a trigger code we do it here
    if backendRollback:
      if self._dataSourceLink._dataObject._connection is not None:
        self._dataSourceLink._dataObject._connection.rollback ()

    if not recovering:
      self._currentRecord = 0
      self._recordCount   = 0

      if not self._dataSourceLink.hasMaster ():
        self._dataSourceLink.createEmptyResultSet ()
      else:
        # if we have a master make sure to keep in touch with it
        self._dataSourceLink.createEmptyResultSet ( \
            masterRecordSet = self._resultSet._masterRecordSet)

    # FIXME: does this happen via currentRecordMoved?
    self.switchRecord (0)


  #
  # initQuery and processQuery
  #
  def initQuery(self):

    # If Enter-Query is hit once, enter query mode
    # If Enter-Query is hit twice, bring back conditions from last query.
    # If Enter-Query is hit thrice, cancel the query and go into normal mode.

    if self.mode != 'query':
        self.mode = 'query'
        self._query2 = int(gConfigForms("RememberLastQuery"))
        self._queryValues = {}
        self._queryValues.update(self._queryDefaults)
        self.switchRecord(0)

  def copyQuery(self):
    self._query2 = 0
    self._queryValues = {}
    self._queryValues.update(self._lastQueryValues)
    self.switchRecord(0)

  def cancelQuery(self):
    self.mode = 'normal'
    self.switchRecord(0)

  def processQuery(self):
    # Set the maxList to a single master block
    # as a placeholder
    maxList = [self._dataSourceLink._dataObject]
    while maxList[0]._masterObject:
      maxList = [maxList[0]._masterObject]

    # Find the longest master/detail chain that
    # contains query values.  This will become
    # the chain that is queried
    for block in self._logic._blockList:
      if block._queryValues.keys():
        templist = [block._dataSourceLink._dataObject]
        while templist[-1]._masterObject:
          templist.append(templist[-1]._masterObject)
        if len(maxList) < len(templist): maxList = templist

    # Store block states
    for block in self._logic._blockList:
      block.mode = 'normal'
      block._lastQueryValues = {}
      block._lastQueryValues.update(block._queryValues)

    # graft in the sloppy query stuff if needed
    for dataobject in maxList:
      for block in self._logic._blockList:
        if dataobject == block._dataSourceLink._dataObject:
          break
      for entry in block._entryList:
        if hasattr(entry._field,'sloppyQuery') and block._queryValues.has_key(entry._field):
          block._queryValues[entry._field] = "%"+ string.join(list(block._queryValues[entry._field]),"%")+"%"

    # Build the sql required by the detail blocks
    SQL = ""
    for dataobject in maxList[:-1]:

      for block in self._logic._blockList:
        if dataobject == block._dataSourceLink._dataObject:
          break

      block.processTrigger('PRE-QUERY')
      for field in block._fieldList:
        field.processTrigger('PRE-QUERY')

      conditions = _generateConditional(block)
      SQL = self._dataSourceLink.getQueryString(conditions,1,SQL)

    for block in self._logic._blockList:
     if maxList[-1] == block._dataSourceLink._dataObject:
      break
    rootBlock = block

    conditions = _generateConditional(rootBlock)
    rootBlock._dataSourceLink.createResultSet(conditions, sql=SQL)

    rootBlock._recordCount = rootBlock._resultSet.getRecordCount()

    for block in self._logic._blockList:
      block.processTrigger('POST-QUERY')
      for field in block._fieldList:
        field.processTrigger('POST-QUERY')

    # Adjusting scrollbars
    for sb in self._scrollbars:
      sb.adjustScrollbar(self._currentRecord, self._recordCount)


  def registerScrollbar(self, sb):
    self._scrollbars.append(sb)

  # ---------------------------------------------------------------------------
  # Call a datasource's function
  # ---------------------------------------------------------------------------

  def callFunction (self, name, parameters):
    try:
      res = self._dataSourceLink.callFuncOfCurrentRecordsetEntry (name,
                        parameters)
    finally:
      self.switchRecord (0)

    return res


  # ---------------------------------------------------------------------------
  # Update the current datasource's record set
  # ---------------------------------------------------------------------------

  def updateCurrentRecordSet (self):
    try:
      self._dataSourceLink.updateCurrentRecordSet ()

    finally:
      self.switchRecord (0)


  ###################################################################
  #
  # Trigger settable stuff
  #
  def triggerSetAutoCommit(self, value):
    self.autoCommit = not not value # Force boolean

  def triggerGetAutoCommit(self):
    return self.autoCommit

  def triggerSetQueryable(self, value):
    self.queryable = not not value # Force boolean

  def triggerGetQueryable(self):
    return self.queryable

  def triggerSetEditable(self, value):
    assert (value in ('Y','new','update','N'))
    self.editable = value

  def triggerGetEditable(self):
    return self.editable

  def triggerSetAutoCreate(self, value):
    self.autoCreate = not not value # Force boolean

  def triggerGetAutoCreate(self):
    return self.autoCreate

  def triggerSetDeletable(self, value):
    self.deletable = not not value # Force boolean

  def triggerGetDeletable(self):
    return self.deletable

  def triggerSetTransparent(self, value):
    self.transparent = not not value # Force boolean

  def triggerGetTransparent(self):
    return self.transparent

  def triggerSetAutoNextRecord(self, value):
    self.autoNextRecord = not not value # Force boolean

  def triggerGetAutoNextRecord(self):
    return self.autoNextRecord



#
# _generateConditional
#
# Creates a GCondtional tree based upon the values currently
# stored in the form.
#
def _generateConditional(block):
    conditionLike = {}
    conditionEq = {}
    # Get all the user-supplied parameters from the entry widgets
    for entry, val in block._queryValues.items():
        if entry._bound and entry.isQueryable() and len(str(val)):
          if entry.typecast == 'text':
            if block._convertAsterisksToPercent:
              try:
                val = str(val).replace('*','%')
              except ValueError:
                pass
            if (val.find('%') >= 0 or val.find('_') >= 0):
              conditionLike[entry.field] = val
            else:
              conditionEq[entry.field] = val
          else:
            conditionEq[entry.field] = val

    if len(conditionLike.keys()) and len(conditionEq.keys()):
##        GDebug.printMesg(5,'Combining like w/and (%s, %s)' % (conditionLike, conditionEq))
        conditions = GConditions.combineConditions( \
            GConditions.buildConditionFromDict(conditionLike, GConditions.GClike), \
            GConditions.buildConditionFromDict(conditionEq, GConditions.GCeq) )

    elif len(conditionLike.keys()):
        conditions = GConditions.buildConditionFromDict(conditionLike, GConditions.GClike)

    elif len(conditionEq.keys()):
        conditions = GConditions.buildConditionFromDict(conditionEq, GConditions.GCeq)

    else:
        conditions = {}

    return conditions





# A simple resultset iterator
# Lets you use ResultSets as:
#
#   for record in myResultSet:
#      blah
#
# NOTE: Python 2.2+  (but it won't get called in
#    Python 2.1 or below, so not a problem)
#
class _BlockIter:
  def __init__(self, block):
    self.block = block
    self.new = True
    self.done = False

  def __iter__(self):
    return self

  def next(self):
    if self.done:
      raise StopIteration
    elif self.new:
      self.block.jumpRecord(0)
      self.new = False
    elif self.block._resultSet.isLastRecord():
      self.done = True
      raise StopIteration
    else:
      self.block.nextRecord()

    if self.block.isEmpty() and self.block._resultSet.isLastRecord():
      self.done = True
      raise StopIteration

    return self.block

